package org.nuiton.topia.persistence;

/*
 * #%L
 * ToPIA :: Persistence
 * $Id: TopiaEntities.java 2982 2014-01-17 22:38:39Z athimel $
 * $HeadURL: https://svn.nuiton.org/topia/tags/topia-3.0-beta-4/topia-persistence/src/main/java/org/nuiton/topia/persistence/TopiaEntities.java $
 * %%
 * Copyright (C) 2004 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Function;
import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.Iterables;

/**
 * Utility class that provides useful methods for {@link org.nuiton.topia.persistence.TopiaEntity} manipulation.
 *
 * @author Brendan Le Ny : leny@codelutin.com
 * @author Arnaud Thimel : thimel@codelutin.com
 * @since 3.0
 */
public class TopiaEntities {

    /**
     * Guava's {@link com.google.common.base.Function} to get a topiaId from any
     * {@link org.nuiton.topia.persistence.TopiaEntity}
     */
    protected static final Function<TopiaEntity, String> GET_TOPIA_ID = new Function<TopiaEntity, String>() {
        @Override
        public String apply(TopiaEntity input) {
            return input == null ? null : input.getTopiaId();
        }
    };

    /**
     * Creates a Guava's {@link com.google.common.base.Predicate} which tests if a
     * {@link org.nuiton.topia.persistence.TopiaEntity} has the given {@code topiaId}
     *
     * @param id the expected topiaId
     * @return the created Predicate
     */
    protected static Predicate<TopiaEntity> entityHasId(String id) {
        Predicate<String> equalsPredicate = Predicates.equalTo(id);
        Predicate<TopiaEntity> result = Predicates.compose(equalsPredicate, GET_TOPIA_ID);
        return result;
    }

    /**
     * Iterate over the given {@link java.lang.Iterable} looking for a {@link org.nuiton.topia.persistence.TopiaEntity}
     * having the given {@code topiaId}.
     *
     * @param entities the {@link java.lang.Iterable} instance to iterate through
     * @param id       the expected {@code topiaId}
     * @param <E>      the type of {@link org.nuiton.topia.persistence.TopiaEntity} of the given Iterable
     * @return the found {@link org.nuiton.topia.persistence.TopiaEntity} or null
     */
    public static <E extends TopiaEntity> E findByTopiaId(Iterable<E> entities, String id) {
        E result;
        if (entities == null) {
            result = null;
        } else {
            Optional<E> eOptional = Iterables.tryFind(entities, entityHasId(id));
            result = eOptional.orNull();
        }
        return result;
    }

    /**
     * Function to obtain {@link org.nuiton.topia.persistence.TopiaEntity#getTopiaId()} from any entity.
     *
     * @since 2.6.12
     */
    public static Function<TopiaEntity, String> getTopiaIdFunction() {
        return GET_TOPIA_ID;
    }

}
