/*
 * *##% 
 * ToPIA :: Persistence
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.topia.generator;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.Template;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.models.object.*;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.persistence.TopiaDAO;
import org.nuiton.topia.persistence.TopiaDAOImpl;
import org.nuiton.util.StringUtil;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;






/**
 * Created: 13 déc. 2009
 *
 * @author Tony Chemit <chemit@codelutin.com> Copyright Code Lutin
 * @version $Revision: 1733 $
 *          <p/>
 *          Mise a jour: $Date: 2009-12-20 19:41:57 +0100 (dim. 20 déc. 2009) $ par :
 *          $Author: tchemit $
 * @since 2.3.0
 * @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.topia.generator.DAOAbstractTransformer"
 */
public class DAOAbstractTransformer extends ObjectModelTransformerToJava {

    /**
     * Logger
     */
    private static final Log log = LogFactory.getLog(DAOAbstractTransformer.class);
    
    @Override
    public void transformFromClass(ObjectModelClass clazz) {
        if (!clazz.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_ENTITY)) {
            return;
        }

        String clazzName = clazz.getName();

        ObjectModelClass result = createAbstractClass(clazzName + "DAOAbstract<E extends " + clazzName + ">", clazz.getPackageName());

        // super class

        String extendClass = "";
        for (ObjectModelClass parent : clazz.getSuperclasses()) {
            extendClass = parent.getQualifiedName();
            if (parent.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_ENTITY)) {
                extendClass += "DAOImpl<E>";
                // in java no multi-inheritance
                break;
            }
        }
        if (extendClass.length() == 0) {
            extendClass = TopiaDAOImpl.class.getName() + "<E>";

        }
        if (log.isDebugEnabled()) {
            log.debug("super class = "+extendClass);
        }
        setSuperClass(result, extendClass);

        addInterface(result, TopiaDAO.class.getName() + "<E>");

        // imports

        Collection<ObjectModelOperation> DAOoperations = getDAOOperations(clazz);
        if (isCollectionNeeded(DAOoperations)) {
            addImport(result, Collection.class);
        }
        if (isSetNeeded(DAOoperations)) {
            addImport(result, Set.class);
        }
        addImport(result, List.class);
        addImport(result, Arrays.class);
        addImport(result, TopiaException.class);
        addImport(result, TopiaContextImplementor.class);

        boolean enableSecurity = (
                clazz.hasTagValue(TopiaGeneratorUtil.TAG_SECURITY_CREATE) ||
                        clazz.hasTagValue(TopiaGeneratorUtil.TAG_SECURITY_LOAD) ||
                        clazz.hasTagValue(TopiaGeneratorUtil.TAG_SECURITY_UPDATE) ||
                        clazz.hasTagValue(TopiaGeneratorUtil.TAG_SECURITY_DELETE)
        );

        if (enableSecurity) {
            addImport(result, ArrayList.class);
            addImport(result, java.security.Permission.class);
            addImport(result, "org.nuiton.topia.taas.entities.TaasAuthorizationImpl");
            addImport(result, "org.nuiton.topia.taas.jaas.TaasPermission");
            addImport(result, "org.nuiton.topia.taas.TaasUtil");
            addImport(result, TopiaDAO.class);

            //FIXME : how to do static imports ?
//import static org.nuiton.topia.taas.TaasUtil.CREATE;
//import static org.nuiton.topia.taas.TaasUtil.DELETE;
//import static org.nuiton.topia.taas.TaasUtil.LOAD;
//import static org.nuiton.topia.taas.TaasUtil.UPDATE;
        }

        ObjectModelOperation op;

        // getEntityClass

        op = addOperation(result,"getEntityClass","Class<E>", ObjectModelModifier.PUBLIC);
        setOperationBody(op,""
+"\n"
+"        return (Class<E>)"+clazzName+".class;\n"
+"    "
        );


        generateDAOOperations(result, DAOoperations);

        // delete
        
        op = addOperation(result, "delete", "void", ObjectModelModifier.PUBLIC);
        addException(op, TopiaException.class);
        addParameter(op, "E", "entity");
        StringBuilder body = new StringBuilder();
        String modelName = StringUtils.capitalize(model.getName());
        String providerFQN = getOutputProperties().getProperty(Template.PROP_DEFAULT_PACKAGE) + "." + modelName + "DAOHelper.getImplementationClass";
        for (ObjectModelAttribute attr : clazz.getAttributes()) {
        	String attrType = attr.getType();
        	String reverseAttrName = attr.getReverseAttributeName();
            ObjectModelAttribute reverse = attr.getReverseAttribute();
            if (!attr.hasAssociationClass() && reverse != null && reverse.isNavigable()
                    && GeneratorUtil.isNMultiplicity(attr) && GeneratorUtil.isNMultiplicity(reverse)) {
                // On doit absolument supprimer pour les relations many-to-many
                // le this de la collection de l'autre cote

            	String attrDBName = TopiaGeneratorUtil.getDBName(attr);
            	String attrClassifierDBName = TopiaGeneratorUtil.getDBName(attr.getClassifier());
            	String attrJoinTableName = TopiaGeneratorUtil.getManyToManyTableName(attr);
            	String attrReverseDBName = TopiaGeneratorUtil.getReverseDBName(attr);
                body.append(""
+"\n"
+"        {\n"
+"            List<"+attrType+"> list = getContext().getHibernate().createSQLQuery(\n"
+"                    \"SELECT main.topiaid \" +\n"
+"                    \"from "+attrClassifierDBName+" main, "+attrJoinTableName+" secondary \" +\n"
+"                    \"where main.topiaid=secondary."+attrDBName+"\" +\n"
+"                    \" and secondary."+attrReverseDBName+"='\" + entity.getTopiaId() + \"'\")\n"
+"                    .addEntity(\"main\", "+providerFQN+"("+attrType+".class)).list();\n"
+"            for ("+attrType+" item : list) {\n"
+"                item.remove"+StringUtils.capitalize(reverseAttrName)+"(entity);\n"
+"            }\n"
+"        }\n"
+""
                );
            } else  if (!attr.hasAssociationClass() && reverse != null
                    && reverse.isNavigable() && !GeneratorUtil.isNMultiplicity(reverse)) {
                // On doit mettre a null les attributs qui ont cet objet sur les
                // autres entites en one-to-*
                // TODO peut-etre qu'hibernate est capable de faire ca tout seul ?
            	// THIMEL: J'ai remplacé reverse.getName() par reverseAttrName sans certitude
                body.append(""
                +"\n"
+"                {\n"
+"                List<"+attrType+"> list = getContext()\n"
+"                            .getDAO("+attrType+".class)\n"
+"                            .findAllByProperties(\""+reverseAttrName+"\", entity);\n"
+"                    for ("+attrType+" item : list) {\n"
+"                        item.set"+StringUtils.capitalize(reverseAttrName)+"(null);\n"
+""
                );
            	if(attr.isAggregate()){
                    body.append(""
+"\n"
+"            			item.delete();\n"
+""
                            );
            	}
                body.append(""
+"\n"
+"                    }\n"
+"                }\n"
+""
                );

            }
        }
        body.append(""
+"\n"
+"        super.delete(entity);\n"
+"    "
        );
        setOperationBody(op,body.toString());
        

        for (ObjectModelAttribute attr : clazz.getAttributes()) {
            if (!attr.isNavigable()) {
                continue;
            }
            String attrName = attr.getName();

            if (!GeneratorUtil.isNMultiplicity(attr)) {
                generateNoNMultiplicity(result, attr, false);
            } else {
                generateNMultiplicity(result, attr);
            }
        }

        if (clazz instanceof ObjectModelAssociationClass) {
            ObjectModelAssociationClass assocClass = (ObjectModelAssociationClass)clazz;
            for (ObjectModelAttribute attr : assocClass.getParticipantsAttributes()) {
                if (attr != null) {
                    if (!GeneratorUtil.isNMultiplicity(attr)) {
                        generateNoNMultiplicity(result, attr, true);
                    } else {
                        generateNMultiplicity(result, attr);
                    }
                }
            }
        }

        if(enableSecurity) {

            // getRequestPermission

            op = addOperation(result,"getRequestPermission","List<Permission>",ObjectModelModifier.PUBLIC);
            setDocumentation(op,"Retourne les permissions a verifier pour l'acces a l'entite pour le service Taas");
            addException(op, TopiaException.class);
            addParameter(op,String.class,"topiaId");
            addParameter(op,int.class,"actions");
            StringBuilder buffer = new StringBuilder();
            buffer.append(""
+"\n"
+"        List<Permission> resultPermissions = new ArrayList<Permission>();\n"
+"        if ((actions & TaasUtil.CREATE) == TaasUtil.CREATE) {\n"
+""
                    );
            buffer.append(generateSecurity(result, clazz, TopiaGeneratorUtil.TAG_SECURITY_CREATE));
            buffer.append(""
+"\n"
+"        }\n"
+"        if ((actions & TaasUtil.LOAD) == TaasUtil.LOAD) {\n"
+""
                    );
            buffer.append(generateSecurity(result, clazz, TopiaGeneratorUtil.TAG_SECURITY_LOAD));
            buffer.append(""
+"\n"
+"        }\n"
+"        if ((actions & TaasUtil.UPDATE) == TaasUtil.UPDATE) {\n"
+""
                    );
            buffer.append(generateSecurity(result, clazz, TopiaGeneratorUtil.TAG_SECURITY_UPDATE));
            buffer.append(""
+"\n"
+"        }\n"
+"        if ((actions & TaasUtil.DELETE) == TaasUtil.DELETE) {\n"
+""
                    );
            buffer.append(generateSecurity(result, clazz, TopiaGeneratorUtil.TAG_SECURITY_DELETE));
            buffer.append(""
+"\n"
+"        }\n"
+"        return resultPermissions;\n"
+"    "
                    );
            
            setOperationBody(op,buffer.toString());
            
            // THIMEL : Le code suivant doit pouvoir être déplacé dans DAODelegator ?

            // getRequestPermission


            op = addOperation(result,"getRequestPermission","List<Permission>",ObjectModelModifier.PROTECTED);
            addParameter(op,String.class,"topiaId");
            addParameter(op,int.class,"actions");
            addParameter(op,String.class,"query");
            addParameter(op,Class.class,"daoClass");
            addException(op,TopiaException.class);
            setDocumentation(op,"Retourne les permissions a verifier pour l'acces a l'entite pour le service Taas");
            setOperationBody(op,""
+"    TopiaContextImplementor context = getContext();\n"
+"    List<String> result = context.find(query, \"id\", topiaId);\n"
+"\n"
+"    List<Permission> resultPermissions = new ArrayList<Permission>();\n"
+"    for (String topiaIdPermission : result) {\n"
+"        TopiaDAO dao = context.getDAO(daoClass);\n"
+"        List<Permission> permissions = dao.getRequestPermission(topiaIdPermission, actions);\n"
+"        if(permissions != null) {\n"
+"            resultPermissions.addAll(permissions);\n"
+"        } else {\n"
+"            TaasPermission permission = new TaasPermission(topiaIdPermission, actions);\n"
+"            resultPermissions.add(permission);\n"
+"        }\n"
+"    }\n"
+"    return resultPermissions;\n"
+"    "
            );
        }
    }

    /**
     * Generation of DAO operations signatures from class.
     * These operations are abstract and identified by <<dao>> stereotype in the model.
     * The developper must defined these methods in the DAOImpl associated to this DAOAbstract.
     * @param result clazz where to add operations
     * @param operations operations to generate
     */
    private void generateDAOOperations(ObjectModelClass result, Collection<ObjectModelOperation> operations) {
        for (ObjectModelOperation op : operations) {

            //TODO: add to transformer cloneOperation
            
            ObjectModelOperation op2;
            op2 = addOperation(result, op.getName(), op.getReturnType(), ObjectModelModifier.ABSTRACT, ObjectModelModifier.toValue(op.getVisibility()));
            setDocumentation(op2, op.getDocumentation());

            // parameters
            
            for (ObjectModelParameter param : op.getParameters()) {
                ObjectModelParameter param2 = addParameter(op2, param.getType(), param.getName());
                setDocumentation(param2, param.getDocumentation());
            }

            // exceptions 
            Set<String> exceptions = op.getExceptions();
            exceptions.add(TopiaException.class.getName());
            for (String exception : exceptions) {
                addException(op2,exception);
            }
        }
    }


    private String generateSecurity(ObjectModelClass result, ObjectModelClass clazz, String securityTagName) {
        StringBuilder buffer = new StringBuilder();
        
    	if (clazz.hasTagValue(securityTagName)) {
    		String security = clazz.getTagValue(securityTagName);
            Pattern propertiesPattern = Pattern
                .compile("((?:[_a-zA-Z0-9]+\\.)+(?:_?[A-Z][_a-zA-Z0-9]*\\.)+)attribute\\.(?:([_a-z0-9][_a-zA-Z0-9]*))#(?:(create|load|update|delete))");
            String[] valuesSecurity = security.split(":");

            for (String valueSecurity : valuesSecurity) {
                Matcher matcher = propertiesPattern.matcher(valueSecurity);
                matcher.find();
                // className is fully qualified name of class
                String className = matcher.group(1);
                className = StringUtil.substring(className, 0, -1); // remove ended
                // .
                // target is class, attribute or operation
                String attributeName = matcher.group(2);
                String actions = matcher.group(3).toUpperCase();

                String query = "";
                String daoClass = "";
                if(className.equals(clazz.getQualifiedName())) {
                    query = "select " + attributeName + ".topiaId from " + clazz.getQualifiedName() + " where topiaId = :id";
                    daoClass = clazz.getAttribute(attributeName).getClassifier().getQualifiedName();
                } else {
                    query = "select at.topiaId from " + className + " at inner join at." + attributeName + " cl where cl.topiaId = :id";
                    daoClass = className;
                }
                buffer.append(""
+"\n"
+"              resultPermissions.addAll(getRequestPermission(topiaId,\n"
+"                                                            "+actions+",\n"
+"                                                            \""+query+"\",\n"
+"                                                            "+daoClass+".class));\n"
+""
                );
            }
        } else {
            buffer.append(""
+"            return null;\n"
+"    "
            );
        }
        return buffer.toString();
    }

    protected void generateNoNMultiplicity(ObjectModelClass result, ObjectModelAttribute attr, boolean isAssoc)  {
    	String attrName = attr.getName();
    	String attrType = attr.getType();
        String propertyName = attrName;
        if (!isAssoc && attr.hasAssociationClass()) {
            propertyName = TopiaGeneratorUtil.toLowerCaseFirstLetter(attr.getAssociationClass().getName()) + "." + propertyName;
        }
        ObjectModelOperation op;
        op = addOperation(result, "findBy" + StringUtils.capitalize(attrName), "E", ObjectModelModifier.PUBLIC);
        addException(op, TopiaException.class);
        addParameter(op, attrType, "v");
        setDocumentation(op, "Retourne le premier élément trouvé ayant comme valeur pour l'attribut "+ attrName + " le paramètre.");
        setOperationBody(op,""
+"\n"
+"        E result = findByProperty(\""+propertyName+"\", v);\n"
+"        return result;\n"
+"    "
        );

        op = addOperation(result, "findAllBy" + StringUtils.capitalize(attrName), "List<E>", ObjectModelModifier.PUBLIC);
        addException(op, TopiaException.class);
        addParameter(op, attrType, "v");
        setDocumentation(op, "Retourne les éléments ayant comme valeur pour l'attribut " + attrName + " le paramètre.");
        setOperationBody(op,""
+"\n"
+"        List<E> result = findAllByProperty(\""+propertyName+"\", v);\n"
+"        return result;\n"
+"    "
        );

        if (attr.hasAssociationClass()) {
            String assocClassName = attr.getAssociationClass().getName();
            String assocClassFQN = attr.getAssociationClass().getQualifiedName();
            op = addOperation(result, "findBy" + StringUtils.capitalize(assocClassName), "E", ObjectModelModifier.PUBLIC);
            addException(op, TopiaException.class);
            addParameter(op, assocClassFQN, "value");
            setDocumentation(op, "Retourne le premier élément trouvé ayant comme valeur pour l'attribut " + TopiaGeneratorUtil.toLowerCaseFirstLetter(assocClassName) + " le paramètre.");
            setOperationBody(op, ""
+"\n"
+"        E result = findByProperty(\""+TopiaGeneratorUtil.toLowerCaseFirstLetter(assocClassName)+"\", value);\n"
+"        return result;\n"
+"    "
            );

            op = addOperation(result, "findAllBy" + StringUtils.capitalize(assocClassName), "List<E>", ObjectModelModifier.PUBLIC);
            addException(op, TopiaException.class);
            addParameter(op, assocClassFQN, "value");
            setDocumentation(op, "Retourne les éléments ayant comme valeur pour l'attribut " + TopiaGeneratorUtil.toLowerCaseFirstLetter(assocClassName) + " le paramètre.");
            setOperationBody(op,""
+"\n"
+"        List<E> result = findAllByProperty(\""+TopiaGeneratorUtil.toLowerCaseFirstLetter(assocClassName)+"\", value);\n"
+"        return result;\n"
+"    "
            );
        }
    }

    protected void generateNMultiplicity(ObjectModelClass result, ObjectModelAttribute attr) {
    	String attrName = attr.getName();
    	String attrType = attr.getType();
        ObjectModelOperation op;

        op = addOperation(result, "findContains" + StringUtils.capitalize(attrName), "E", ObjectModelModifier.PUBLIC);
        addException(op, TopiaException.class);
        addParameter(op, attrType + "...", "v");
        setDocumentation(op, "Retourne le premier élément trouvé dont l'attribut " +attrName + " contient le paramètre." );
        setOperationBody(op,""
+"\n"
+"        E result = findContainsProperties(\""+TopiaGeneratorUtil.toLowerCaseFirstLetter(attrName)+"\", Arrays.asList(v));\n"
+"        return result;\n"
+"    "
        );

        op = addOperation(result, "findAllContains" + StringUtils.capitalize(attrName), "List<E>", ObjectModelModifier.PUBLIC);
        addException(op, TopiaException.class);
        addParameter(op, attrType + "...", "v");
        setDocumentation(op, "Retourne les éléments trouvé dont l'attribut " + attrName + " contient le paramètre.");
        setOperationBody(op, ""
+"\n"
+"        List<E> results = findAllContainsProperties(\""+TopiaGeneratorUtil.toLowerCaseFirstLetter(attrName)+"\", Arrays.asList(v));\n"
+"        return results;\n"
+"    "
        );
    }

    private boolean isCollectionNeeded(Collection<ObjectModelOperation> operations) {
        return isImportNeeded(operations, "Collection");
    }

    private boolean isSetNeeded(Collection<ObjectModelOperation> operations) {
        return isImportNeeded(operations, "Set");
    }

    private boolean isImportNeeded(Collection<ObjectModelOperation> operations, String importName) {
        for (ObjectModelOperation op : operations) {
            if (op.getReturnType().contains(importName)) {
                return true;
            }
            for (ObjectModelParameter param : op.getParameters()) {
                if (param.getType().contains(importName)) {
                    return true;
                }
            }
        }
        return false;
    }

    public static Collection<ObjectModelOperation> getDAOOperations(ObjectModelClass clazz) {
        Collection<ObjectModelOperation> results = new ArrayList<ObjectModelOperation>();
        for (ObjectModelOperation op : clazz.getOperations()) {
            if (op.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_DAO)) {
                results.add(op);
            }
        }
        return results;
    }
}
