/* *##%
 * Copyright (C) 2009 Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package org.nuiton.widget.editor;

import java.awt.BorderLayout;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;

import javax.swing.JPanel;
import javax.swing.event.CaretListener;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.FileUtil;
import org.fife.ui.rsyntaxtextarea.RSyntaxTextArea;
import org.fife.ui.rsyntaxtextarea.SyntaxConstants;
import org.fife.ui.rtextarea.RTextScrollPane;

/**
 * RSyntaxTextArea editor implementation.
 *
 * @author chatellier
 * @version $Revision: 1.0 $
 * 
 * Last update : $Date: 15 mai 2009 $
 * By : $Author: chatellier $
 */
public class RSyntaxEditor extends JPanel implements EditorInterface, DocumentListener {

    /** serialVersionUID. */
    private static final long serialVersionUID = 5880160718377536089L;

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(RSyntaxEditor.class);
    
    protected RSyntaxTextArea editor;
    
    protected boolean isModified = false;
    
    public RSyntaxEditor() {
        editor = new RSyntaxTextArea();

        RTextScrollPane sp = new RTextScrollPane(editor);
        setLayout(new BorderLayout());
        add(sp, BorderLayout.CENTER);
    }

    @Override
    public boolean accept(File file) {
        String ext = FileUtil.extension(file);
        boolean result = "java".equalsIgnoreCase(ext);
        result = result || "xml".equalsIgnoreCase(ext);
        result = result || "sql".equalsIgnoreCase(ext);
        result = result || "r".equalsIgnoreCase(ext);
        return result;
    }
    
    /*
     * @see org.nuiton.widget.editor.DefaultEditor#open(java.io.File)
     */
    @Override
    public boolean open(File file) {
        
        try {
            editor.getDocument().removeDocumentListener(this);
            
            Reader in = new BufferedReader(new FileReader(file));
            // editor.read(in, file);
            // String text = editor.getText();
    
            String text = "";
            char c;
            int last = 0;
    
            while ((last = in.read()) != -1) {
                c = (char) last;
                // on peut avoir \r\n (windows) \r (macos) \n (unix)
                if ('\r' == c) { // pour windows et macos on remplace par \n
                    in.mark(1);
                    last = in.read();
                    if (last != -1) {
                        if ('\n' != (char) last) {
                            // on a seulement \r on remet le dernier caractere
                            // lu
                            in.reset();
                        }
                        // dans tous les cas \r ou \r\n on remplace par \n
                        c = '\n';
                    }
                }
                text += c;
            }
            
            String ext = FileUtil.extension(file);
            if ("java".equalsIgnoreCase(ext)) {
                editor.setSyntaxEditingStyle(SyntaxConstants.SYNTAX_STYLE_JAVA);
            } else if ("xml".equalsIgnoreCase(ext)) {
                editor.setSyntaxEditingStyle(SyntaxConstants.SYNTAX_STYLE_XML);
            } else if ("sql".equalsIgnoreCase(ext)) {
                editor.setSyntaxEditingStyle(SyntaxConstants.SYNTAX_STYLE_SQL);
            } else if ("r".equalsIgnoreCase(ext)) {
                editor.setSyntaxEditingStyle(SyntaxConstants.SYNTAX_STYLE_PERL);
            }

            editor.setText(text);
            editor.setCaretPosition(0);
            editor.getDocument().addDocumentListener(this);
            isModified = false;
            return true;
        } catch (FileNotFoundException eee) {
            if (log.isWarnEnabled()) {
                log.warn("Can't find file: " + file, eee);
            }
        } catch (IOException eee) {
            if (log.isWarnEnabled()) {
                log.warn("Can't open file: " + file, eee);
            }
        }
        return false;
    }

    /*
     * @see org.nuiton.widget.editor.EditorInterface#addDocumentListener(javax.swing.event.DocumentListener)
     */
    @Override
    public void addDocumentListener(DocumentListener listener) {
        editor.getDocument().addDocumentListener(listener);
    }

    /*
     * @see org.nuiton.widget.editor.EditorInterface#removeDocumentListener(javax.swing.event.DocumentListener)
     */
    @Override
    public void removeDocumentListener(DocumentListener listener) {
        editor.getDocument().removeDocumentListener(listener);
    }

    /*
     * @see org.nuiton.widget.editor.EditorInterface#addCaretListener(javax.swing.event.CaretListener)
     */
    @Override
    public void addCaretListener(CaretListener listener) {
        editor.addCaretListener(listener);
    }

    /*
     * @see org.nuiton.widget.editor.EditorInterface#removeCaretListener(javax.swing.event.CaretListener)
     */
    @Override
    public void removeCaretListener(CaretListener listener) {
        editor.removeCaretListener(listener);
    }

    /*
     * @see org.nuiton.widget.editor.EditorInterface#getText()
     */
    @Override
    public String getText() {
        String result = editor.getText();
        return result;
    }

    /*
     * @see org.nuiton.widget.editor.EditorInterface#isModified()
     */
    @Override
    public boolean isModified() {
        return isModified;
    }

    /*
     * @see org.nuiton.widget.editor.EditorInterface#saveAs(java.io.File)
     */
    @Override
    public boolean saveAs(File file) {
        Writer out = null;
        try {
            FileOutputStream outf = new FileOutputStream(file);
            out = new OutputStreamWriter(outf, "utf-8");
            editor.write(out);
            isModified = false;
            return true;
        } catch (IOException eee) {
            if (log.isWarnEnabled()) {
                log.warn("Can't save file", eee);
            }
        }
        finally {
            if (out != null) {
                try {
                    out.close();
                } catch (IOException e) {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't save file", e);
                    }
                }
            }
        }
        return false;
    }

    /*
     * @see org.nuiton.widget.editor.EditorInterface#setText(java.lang.String)
     */
    @Override
    public void setText(String text) {
        editor.setText(text);
    }

    /*
     * @see javax.swing.JComponent#setEnabled(boolean)
     */
    @Override
    public void setEnabled(boolean b) {
        super.setEnabled(b);
        editor.setEnabled(b);
    }

    /*
     * @see javax.swing.event.DocumentListener#changedUpdate(javax.swing.event.DocumentEvent)
     */
    @Override
    public void changedUpdate(DocumentEvent e) {
        isModified = true;
    }

    /*
     * @see javax.swing.event.DocumentListener#insertUpdate(javax.swing.event.DocumentEvent)
     */
    @Override
    public void insertUpdate(DocumentEvent e) {
        isModified = true;
    }

    /*
     * @see javax.swing.event.DocumentListener#removeUpdate(javax.swing.event.DocumentEvent)
     */
    @Override
    public void removeUpdate(DocumentEvent e) {
        isModified = true;
    }

    /*
     * @see org.nuiton.widget.editor.EditorInterface#copy()
     */
    @Override
    public void copy() {
        editor.copy();
    }

    /*
     * @see org.nuiton.widget.editor.EditorInterface#cut()
     */
    @Override
    public void cut() {
        editor.cut();
    }

    /*
     * @see org.nuiton.widget.editor.EditorInterface#paste()
     */
    @Override
    public void paste() {
        editor.paste();
    }
}
