/*
 * *##% 
 * JRedmine client
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.jredmine;

import java.io.InputStream;
import org.nuiton.io.rest.RestClient;
import org.nuiton.io.rest.RestClientConfiguration;
import org.nuiton.jredmine.rest.RedmineRestClient;

/**
 * Technical contrat te be implemented in a {@link RedmineService}.
 *
 * Any concrete implentation of a {@link RedmineService} should also implements
 * this service.
 *
 * @author chemit
 * @since 1.0.0
 */
public interface RedmineServiceImplementor {

    /**
     * Tests if the service is loogued to the redmine service.
     * 
     * @return {@code true} is service is init and loggued to Redmine service,
     * {@code false} otherwise.
     */
    boolean isInit();

    /**
     * Initialize the service given a redmine client already initialized.
     * 
     * @param client the redmine client to be used by the service
     * @return the initialized service
     * @throws RedmineServiceException
     * @see RedmineRestClient
     */
    RedmineServiceImplementor init(RestClient client) throws RedmineServiceException;

    /**
     * Initialize the service given a client configuration.
     * 
     * @param configuration the configuration to be used to init the internal redmine client
     * @return the initialized service
     * @throws RedmineServiceException
     */
    RedmineServiceImplementor init(RestClientConfiguration configuration) throws RedmineServiceException;

    /**
     * Close the service and destroy any connexions to the redmine service.
     *
     * @throws RedmineServiceException
     */
    void destroy() throws RedmineServiceException;

    /**
     * Generic method to obtain a single data from a redmine server.
     * 
     * @param <T> the type of data to obtain
     * @param type the type of data to obtain
     * @param args the parameters to obtain the data
     * @return the obtained data
     * @throws RedmineServiceException
     */
    <T> T getData(Class<T> type, Object... args) throws RedmineServiceException;

    /**
     * Generic method to obtain a array of data from a redmine server.
     *
     * @param <T> the type of data to obtain
     * @param type the type of data to obtain
     * @param args the parameters to obtain the datas
     * @return the obtained datas
     * @throws RedmineServiceException
     */
    <T> T[] getDatas(Class<T> type, Object... args) throws RedmineServiceException;

    /**
     * Generic method to obtain an input stream of a data from a redmine server.
     *
     * @param <T> the type of data to obtain
     * @param type the type of data to obtain
     * @param args the parameters to obtain the data
     * @return the input stream containing the data
     * @throws RedmineServiceException
     */
    <T> InputStream getDataStream(Class<T> type, Object... args) throws RedmineServiceException;

    /**
     * Generic method to obtain an input stream of an array of data from a redmine server.
     *
     * @param <T> the type of data to obtain
     * @param type the type of data to obtain
     * @param args the parameters to obtain the datas
     * @return the input stream containing the datas
     * @throws RedmineServiceException
     */
    <T> InputStream getDatasStream(Class<T> type, Object... args) throws RedmineServiceException;
}
