/*
 * #%L
 * JAXX :: Widgets
 * 
 * $Id: BeanComboBoxHandler.java 2076 2010-09-07 12:26:00Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.2.1/jaxx-widgets/src/main/java/jaxx/runtime/swing/editor/bean/BeanComboBoxHandler.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package jaxx.runtime.swing.editor.bean;

import jaxx.runtime.SwingUtil;
import jaxx.runtime.decorator.DecoratorUtils;
import jaxx.runtime.decorator.JXPathDecorator;
import jaxx.runtime.decorator.MultiJXPathDecorator;
import jaxx.runtime.swing.JAXXButtonGroup;
import jaxx.runtime.swing.renderer.DecoratorListCellRenderer;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.autocomplete.ObjectToStringConverter;

import javax.swing.JComponent;
import javax.swing.JPopupMenu;
import javax.swing.text.Document;
import javax.swing.text.JTextComponent;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.List;

/**
 * Le handler d'un {@link BeanComboBox}.
 * <p/>
 * Note: ce handler n'est pas stateless et n'est donc pas partageable entre plusieurs ui.
 *
 * @author tchemit <chemit@codelutin.com>
 * @param <O> le type des objet contenus dans le modèle du composant.
 * @see BeanComboBox
 */
public class BeanComboBoxHandler<O> implements PropertyChangeListener {

    public static final Log log = LogFactory.getLog(BeanComboBoxHandler.class);

    /** ui if the handler */
    protected BeanComboBox<O> ui;

    /** the mutator method on the property of boxed bean in the ui */
    protected Method mutator;

    /** the original document of the combbo box editor (keep it to make possible undecorate) */
    protected Document originalDocument;

    /** the convertor used to auto-complete */
    protected ObjectToStringConverter convertor;

    /** the decorator of data */
    protected MultiJXPathDecorator<O> decorator;

    protected boolean init;

    public BeanComboBoxHandler(BeanComboBox<O> ui) {
        this.ui = ui;
    }

    protected final FocusListener EDITOR_TEXT_COMP0NENT_FOCUSLISTENER = new FocusListener() {

        @Override
        public void focusGained(FocusEvent e) {
            if (log.isDebugEnabled()) {
                log.debug("close popup from " + e);
            }
            ui.getPopup().setVisible(false);
        }

        @Override
        public void focusLost(FocusEvent e) {
        }
    };

    private final BeanUIUtil.PopupHandler popupHandler = new BeanUIUtil.PopupHandler() {

        @Override
        public JPopupMenu getPopup() {
            return ui.getPopup();
        }

        @Override
        public JComponent getInvoker() {
            return ui.getChangeDecorator();
        }
    };

    /**
     * Initialise le handler de l'ui
     *
     * @param decorator le decorateur a utiliser
     * @param data      la liste des données a gérer
     */
    public void init(JXPathDecorator<O> decorator, List<O> data) {

        if (init) {
            throw new IllegalStateException("can not init the handler twice");
        }
        init = true;

        JAXXButtonGroup indexes = ui.getIndexes();

        this.decorator = BeanUIUtil.createDecorator(decorator);

        // init combobox renderer base on given decorator
        ui.getCombobox().setRenderer(new DecoratorListCellRenderer(this.decorator));

        convertor = BeanUIUtil.newDecoratedObjectToStringConverter(this.decorator);

        // keep a trace of original document (to make possible reverse autom-complete) 
        JTextComponent editorComponent = (JTextComponent) ui.getCombobox().getEditor().getEditorComponent();
        originalDocument = editorComponent.getDocument();

        // build popup
        popupHandler.preparePopup(ui.getSelectedToolTipText(),
                               ui.getNotSelectedToolTipText(),
                               ui.getI18nPrefix(),
                               ui.getPopupTitleText(),
                               indexes,
                               ui.getPopupLabel(),
                               ui.getSortUp(),
                               ui.getSortDown(),
                               this.decorator);

        ui.autoComplete = true;

        ui.addPropertyChangeListener(this);

        // set datas
        ui.setData(data);

        // select sort button
        indexes.setSelectedButton(ui.getIndex());
    }

    /** Toggle the popup visible state. */
    public void togglePopup() {
        popupHandler.togglePopup();
    }

    /**
     * Modifie l'état autoComplete de l'ui.
     *
     * @param oldValue l'ancienne valeur
     * @param newValue la nouvelle valeur
     */
    protected void setAutoComplete(Boolean oldValue, Boolean newValue) {
        oldValue = oldValue != null && oldValue;
        newValue = newValue != null && newValue;
        if (oldValue.equals(newValue)) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("autocomplete state : <" + oldValue + " to " + newValue + ">");
        }
        if (!newValue) {
            JTextComponent editorComponent = (JTextComponent) ui.getCombobox().getEditor().getEditorComponent();
            editorComponent.removeFocusListener(EDITOR_TEXT_COMP0NENT_FOCUSLISTENER);
            BeanUIUtil.undecorate(ui.getCombobox(), originalDocument);
        } else {
            BeanUIUtil.decorate(ui.getCombobox(), convertor);
            JTextComponent editorComponent = (JTextComponent) ui.getCombobox().getEditor().getEditorComponent();
            editorComponent.addFocusListener(EDITOR_TEXT_COMP0NENT_FOCUSLISTENER);
        }
    }

    /**
     * Modifie l'index du décorateur
     *
     * @param oldValue l'ancienne valeur
     * @param newValue la nouvelle valeur
     */
    protected void setIndex(Integer oldValue, Integer newValue) {
        if (newValue == null || newValue.equals(oldValue)) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("check state : <" + oldValue + " to " + newValue + ">");
        }
        updateUI(newValue, ui.isReverseSort());
    }

    /**
     * Modifie l'index du décorateur
     *
     * @param oldValue l'ancienne valeur
     * @param newValue la nouvelle valeur
     */

    protected void setSortOrder(Boolean oldValue, Boolean newValue) {

        if (newValue == null || newValue.equals(oldValue)) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("check state : <" + oldValue + " to " + newValue + ">");
        }

        updateUI(ui.getIndex(), newValue);
    }

    @SuppressWarnings({"unchecked"})
    protected void updateUI(int index, boolean reversesort) {

        // change decorator context
        decorator.setContextIndex(index);

        // keep selected item
        Object previousSelectedItem = ui.getSelectedItem();
        Boolean wasAutoComplete = ui.isAutoComplete();

        if (wasAutoComplete) {
            ui.setAutoComplete(false);
        }

        // remove autocomplete
        if (previousSelectedItem != null) {
            ui.getCombobox().setSelectedItem(null);
            ui.selectedItem = null;
        }

        List<O> data = ui.getData();
        try {
            // Sort data with the decorator jxpath tokens.
            DecoratorUtils.sort(decorator,
                                data,
                                index,
                                reversesort
            );

        } catch (Exception eee) {
            log.warn(eee.getMessage(), eee);
        }

        // reload the model
        SwingUtil.fillComboBox(ui.getCombobox(), data, null);

        if (wasAutoComplete) {
            ui.setAutoComplete(true);
        }

        if (previousSelectedItem != null) {
            ui.setSelectedItem(previousSelectedItem);
        }

        ui.getCombobox().requestFocus();
    }

    /**
     * Modifie la valeur sélectionnée dans la liste déroulante.
     *
     * @param oldValue l'ancienne valeur
     * @param newValue la nouvelle valeur
     */
    protected void setSelectedItem(Object oldValue, Object newValue) {
        if (ui.getBean() == null) {
            return;
        }

        if (newValue == null) {
            if (ui.getCombobox().getSelectedItem() == null) {
                return;
            }
            ui.getCombobox().setSelectedItem(null);

            if (ui.isAutoComplete()) {
                ui.setAutoComplete(false);
                ui.setAutoComplete(true);
            }

            if (oldValue == null) {
                return;
            }
        }
        if (log.isDebugEnabled()) {
            log.debug(ui.getProperty() + " on " + ui.getBean().getClass() + " :: " + oldValue + " to " + newValue);
        }

        try {
            Method mut = getMutator();
            if (mut != null) {
                mut.invoke(ui.getBean(), newValue);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /** @return le document de l'éditeur avant complétion. */
    public Document getOriginalDocument() {
        return originalDocument;
    }

    public MultiJXPathDecorator<O> getDecorator() {
        return decorator;
    }

    public Class<?> getTargetClass() {
        Method m = getMutator();
        return m == null ? null : m.getParameterTypes()[0];
    }

    /** @return le mutateur a utiliser pour modifier le bean associé. */
    protected Method getMutator() {
        if (mutator == null) {
            Object bean = ui.getBean();
            if (bean == null) {
                throw new NullPointerException("could not find bean in " + ui);
            }
            String property = ui.getProperty();
            if (property == null) {
                throw new NullPointerException("could not find property in " + ui);
            }

            try {
                PropertyDescriptor descriptor = PropertyUtils.getPropertyDescriptor(bean, property);
                if (descriptor != null) {
                    mutator = descriptor.getWriteMethod();
                }
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
        return mutator;
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        String propertyName = evt.getPropertyName();

        if (BeanComboBox.PROPERTY_SELECTED_ITEM.equals(propertyName)) {
            setSelectedItem(evt.getOldValue(), evt.getNewValue());
            return;
        }

        if (BeanComboBox.PROPERTY_AUTO_COMPLETE.equals(propertyName)) {

            setAutoComplete((Boolean) evt.getOldValue(), (Boolean) evt.getNewValue());
            return;
        }

        if (BeanListHeader.PROPERTY_INDEX.equals(propertyName)) {

            // decorator index has changed, force reload of data in ui
            setIndex((Integer) evt.getOldValue(),
                     (Integer) evt.getNewValue());
            return;
        }

        if (BeanListHeader.PROPERTY_REVERSE_SORT.equals(propertyName)) {

            // sort order has changed, force reload of data in ui
            setSortOrder((Boolean) evt.getOldValue(),
                         (Boolean) evt.getNewValue());
            return;
        }

        if (BeanListHeader.PROPERTY_DATA.equals(propertyName)) {

            // list has changed, force reload of index
            setIndex(-1, ui.getIndex());
        }
    }
}
