package org.nuiton.jaxx.swing.extra;

/*
 * #%L
 * JAXX :: Runtime
 * %%
 * Copyright (C) 2008 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.jaxx.swing.extra.table.renderer.I18nTableCellRenderer;

import javax.swing.JComponent;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableModel;
import java.awt.Component;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Rectangle;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.util.EventObject;

/**
 * Some usefull methods on {@link JTable}.
 *
 * Created on 12/4/14.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @since 2.18
 */
public class JTables {

    /** Logger. */
    private static final Log log = LogFactory.getLog(JTables.class);

    public static void selectFirstCellOnFirstRowAndStopEditing(JTable table) {

        // select first cell
        doSelectCell(table, 0, 0);

        if (table.isEditing()) {

            // but no edit it
            table.getCellEditor().stopCellEditing();
        }
    }

    public static void selectFirstCellOnLastRow(JTable table) {

        // select first cell
        doSelectCell(table, table.getRowCount() - 1, 0);
    }

    public static void selectFirstCellOnRow(JTable table, int row, boolean stopEdit) {

        // select first cell
        doSelectCell(table, row, 0);

        if (stopEdit && table.isEditing()) {

            table.getCellEditor().stopCellEditing();
        }
    }

    public static void doSelectCell(JTable table,
                                    int rowIndex,
                                    int columnIndex) {

        int rowCount = table.getRowCount();
        if (rowCount == 0) {

            // no row, can not selected any cell
            if (log.isWarnEnabled()) {
                log.warn("No row in table, can not select any cell");
            }
            return;
        }
        int columnCount = table.getColumnCount();
        if (columnCount == 0) {

            // no column, can not selected any cell
            if (log.isWarnEnabled()) {
                log.warn("No column in table, can not select any cell");
            }
            return;
        }
        if (columnIndex > columnCount) {
            if (log.isWarnEnabled()) {
                log.warn(String.format("ColumnIndex: %s is more than columnCount %s", columnIndex, columnCount));
            }
            columnIndex = columnCount - 1;
        }
        if (columnIndex < 0) {
            columnIndex = 0;
        }
        if (rowIndex >= rowCount) {
            if (log.isWarnEnabled()) {
                log.warn(String.format("RowIndex: %s is more than rowCount %s", rowIndex, rowCount));
            }
            rowIndex = rowCount - 1;
        }
        if (rowIndex < 0) {
            rowIndex = 0;
        }

        table.setColumnSelectionInterval(columnIndex, columnIndex);
        table.setRowSelectionInterval(rowIndex, rowIndex);
        table.editCellAt(rowIndex, columnIndex);
    }

    /**
     * Add {@link java.awt.event.KeyListener} to focus next editable cell on TAB key
     *
     * @param table to add TAB {@link java.awt.event.KeyListener}
     */
    public static void makeTableTabFocusable(final JTable table) {
        table.setCellSelectionEnabled(true);
        table.setSurrendersFocusOnKeystroke(true);
        table.addKeyListener(new KeyAdapter() {

            @Override
            public void keyReleased(KeyEvent e) {
                if (e.getKeyCode() == KeyEvent.VK_TAB) {

                    // get table informations
                    int selectedColumn = table.getSelectedColumn();
                    int selectedRow = table.getSelectedRow();

                    if (log.isDebugEnabled()) {
                        log.debug("Selected was row[" + selectedRow + "] column[" + selectedColumn + "]");
                    }

                    int columnCount = table.getColumnCount();
                    int rowCount = table.getRowCount();

                    // search on current line
                    for (int toSelectColumn = selectedColumn; toSelectColumn < columnCount; toSelectColumn++) {

                        if (editCell(table, selectedRow, toSelectColumn)) {
                            return;
                        }
                    }

                    // search on other lines
                    for (int toSelectRow = selectedRow; toSelectRow < rowCount; toSelectRow++) {
                        for (int toSelectColumn = 0; toSelectColumn < columnCount; toSelectColumn++) {

                            if (editCell(table, toSelectRow, toSelectColumn)) {
                                return;
                            }
                        }
                    }
                }
            }
        });
    }

    /**
     * Used to edit a cell of a given table.
     *
     * @param table   the table to edit
     * @param row     row index of cell to editing
     * @param colummn column index of cell to editing
     * @return {@code false} if for any reason the cell cannot be edited,
     * or if the indices are invalid
     */
    public static boolean editCell(JTable table, int row, int colummn) {

        boolean result = false;
        if (table.isCellEditable(row, colummn)) {

            if (table.isEditing()) {

                int editingRow = table.getEditingRow();
                int editingColumn = table.getEditingColumn();

                // stop edition
                TableCellEditor cellEditor = table.getCellEditor(editingRow,
                                                                 editingColumn);
                cellEditor.stopCellEditing();
            }

            // select row
            table.setColumnSelectionInterval(colummn, colummn);
            table.setRowSelectionInterval(row, row);

            // edit cell
            result = table.editCellAt(row, colummn, new EventObject(table));
            Component component = table.getEditorComponent();
            component.requestFocus();

            if (log.isDebugEnabled()) {
                log.debug("Select row[" + row + "] column[" + colummn +
                          "] return : " + result);
            }
        }
        return result;
    }

    public static void ensureRowIndex(TableModel model, int rowIndex)
            throws ArrayIndexOutOfBoundsException {
        if (rowIndex < -1 || rowIndex >= model.getRowCount()) {
            throw new ArrayIndexOutOfBoundsException(
                    "the rowIndex was " + rowIndex + ", but should be int [0,"
                    + (model.getRowCount() - 1) + "]");
        }
    }

    public static void ensureColumnIndex(TableModel model, int index)
            throws ArrayIndexOutOfBoundsException {
        if (index < -1 || index >= model.getColumnCount()) {
            throw new ArrayIndexOutOfBoundsException(
                    "the columnIndex was " + index + ", but should be int [0,"
                    + (model.getColumnCount() - 1) + "]");
        }
    }

    /**
     * Add to a given table a selec tion model listener to always scroll to
     * current cell selection.
     *
     * @param table the table
     * @since 2.5.3
     */
    public static void scrollToTableSelection(final JTable table) {

        table.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
            @Override
            public void valueChanged(ListSelectionEvent e) {
                ListSelectionModel listSelectionModel =
                        (ListSelectionModel) e.getSource();
                int firstIndex = e.getFirstIndex();
                int lastIndex = e.getLastIndex();
                Integer newSelectedRow = null;

                if (listSelectionModel.isSelectionEmpty()) {

                    // no selection
                } else if (listSelectionModel.isSelectedIndex(firstIndex)) {

                    // use first index
                    newSelectedRow = firstIndex;
                } else if (listSelectionModel.isSelectedIndex(lastIndex)) {

                    // use last index
                    newSelectedRow = lastIndex;
                }
                if (newSelectedRow != null) {
                    Rectangle rect = table.getCellRect(newSelectedRow, 0, true);
                    table.scrollRectToVisible(rect);
                }
            }
        });
    }

    public static void setI18nTableHeaderRenderer(JTable table,
                                                  String... libelles) {
        I18nTableCellRenderer defaultRenderer =
                new I18nTableCellRenderer(
                        table.getTableHeader().getDefaultRenderer(), libelles);
        table.getTableHeader().setDefaultRenderer(defaultRenderer);
    }

    /**
     * Return the selected rows of the table in the model coordinate or empty
     * array if selection is empty.
     *
     * @param table the table to seek
     * @return the selected rows of the table in the model coordinate or empty
     * array if selection is empty.
     * @since 2.5.29
     */
    public static int[] getSelectedModelRows(JTable table) {
        int[] selectedRows = table.getSelectedRows();
        int length = selectedRows.length;
        int[] result = new int[length];
        for (int i = 0; i < length; i++) {
            int selectedRow = selectedRows[i];
            result[i] = table.convertRowIndexToModel(selectedRow);
        }
        return result;
    }

    /**
     * Return the selected row of the table in the model coordinate or
     * {@code -1} if selection is empty.
     *
     * @param table the table to seek
     * @return the selected row of the table in the model coordinate or
     * {@code -1} if selection is empty.
     * @since 2.5.29
     */
    public static int getSelectedModelRow(JTable table) {
        int result = table.getSelectedRow();
        if (result != -1) {
            // can convert to model coordinate
            result = table.convertRowIndexToModel(result);
        }
        return result;
    }

    /**
     * Return the selected column of the table in the model coordinate or
     * {@code -1} if selection is empty.
     *
     * @param table the table to seek
     * @return the selected column of the table in the model coordinate or
     * {@code -1} if selection is empty.
     * @since 2.5.29
     */
    public static int getSelectedModelColumn(JTable table) {
        int result = table.getSelectedColumn();
        if (result != -1) {
            // can convert to model coordinate
            result = table.convertColumnIndexToModel(result);
        }
        return result;
    }

    /**
     * Select the given row index {@code rowIndex} (from the model coordinate)
     * in the selection of the given table.
     *
     * @param table    the table where to set the selection
     * @param rowIndex the row index in the model coordinate to set as selection
     * @since 2.5.29
     */
    public static void setSelectionInterval(JTable table, int rowIndex) {

        int rowViewIndex = table.convertRowIndexToView(rowIndex);
        table.getSelectionModel().setSelectionInterval(rowViewIndex, rowViewIndex);
    }

    /**
     * Add the given row index {@code rowIndex} (from the model coordinate)
     * in the selection of the given table.
     *
     * @param table    the table where to set the selection
     * @param rowIndex the row index in the model coordinate to add to selection
     * @since 2.5.29
     */
    public static void addRowSelectionInterval(JTable table, int rowIndex) {

        int rowViewIndex = table.convertRowIndexToView(rowIndex);
        table.getSelectionModel().addSelectionInterval(rowViewIndex, rowViewIndex);
    }

    public static int computeTableColumnWidth(JTable table,
                                              Font font,
                                              int columnIndex, String suffix) {
        int width = 0;
        if (font == null) {
            font = table.getFont();
        }
//        if (font == null) {
//            TableColumn column = table.getColumnModel().getColumn(columnIndex);
//            font = ((JComponent) column.getCellRenderer()).getFont();
//        }
        FontMetrics fontMetrics = table.getFontMetrics(font);
        for (int i = 0, rowCount = table.getRowCount(); i < rowCount; i++) {
            String key = (String) table.getModel().getValueAt(i, 0);
            int w = fontMetrics.stringWidth(key + suffix);
            if (w > width) {
                width = w;
            }
        }
        return width;
    }

    public static void fixTableColumnWidth(JTable table,
                                           int columnIndex,
                                           int width) {
        TableColumn column = table.getColumnModel().getColumn(columnIndex);
        column.setMaxWidth(width);
        column.setMinWidth(width);
        column.setWidth(width);
        column.setPreferredWidth(width);
    }

    public static void setTableColumnEditor(JTable table,
                                            int columnIndex,
                                            TableCellEditor editor) {
        TableColumn column = table.getColumnModel().getColumn(columnIndex);
        column.setCellEditor(editor);
    }

    public static void setTableColumnRenderer(JTable table,
                                              int columnIndex,
                                              TableCellRenderer editor) {
        TableColumn column = table.getColumnModel().getColumn(columnIndex);
        column.setCellRenderer(editor);
    }

    public static TableCellRenderer newStringTableCellRenderer(
            final DefaultTableCellRenderer renderer,
            final int length,
            final boolean tooltip) {

        return new DefaultTableCellRenderer() {

            private static final long serialVersionUID = 1l;

            @Override
            public Component getTableCellRendererComponent(
                    JTable table,
                    Object value,
                    boolean isSelected,
                    boolean hasFocus,
                    int row,
                    int column) {

                renderer.getTableCellRendererComponent(
                        table,
                        value,
                        isSelected,
                        hasFocus,
                        row,
                        column
                );
                String val = renderer.getText();
                String val2 = val;
                if (val.length() > length) {
                    val2 = val.substring(0, length - 3) + "...";
                }

                JComponent comp = (JComponent)
                        super.getTableCellRendererComponent(
                                table,
                                val2,
                                isSelected,
                                hasFocus,
                                row,
                                column
                        );
                if (tooltip) {
                    comp.setToolTipText(val);
                }
                return comp;
            }
        };
    }
}
