package jaxx.runtime;

import jaxx.Base64Coder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.DefaultListCellRenderer;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JList;
import javax.swing.UIManager;
import java.awt.Component;
import java.awt.Dimension;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeListenerProxy;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EventListener;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;

public class Util {

    public static final String DEFAULT_ICON_PATH = "/icons/";
    public static final String DEFAULT_ICON_PATH_PROPERTY = "default.icon.path";

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private final Log log = LogFactory.getLog(Util.class);


    // Maps root objects to lists of event listeners
    private static Map<Object, WeakReference<List<EventListenerDescriptor>>> eventListeners = new WeakHashMap<Object, WeakReference<List<EventListenerDescriptor>>>();
    private static Map<JAXXObject, WeakReference<List<DataBindingUpdateListener>>> dataBindingUpdateListeners = new WeakHashMap<JAXXObject, WeakReference<List<DataBindingUpdateListener>>>();


    private static class EventListenerDescriptor {
        Class listenerClass;
        String listenerMethodName;
        String methodName;
        Object eventListener;
    }


    /**
     * Decodes the serialized representation of a JAXXObjectDescriptor.  The string must be a byte-to-character mapping
     * of the binary serialization data for a JAXXObjectDescriptor.  See the comments in JAXXCompiler.createJAXXObjectDescriptorField
     * for the rationale behind this (admittedly ugly) approach.
     *
     * @param descriptor descriptor to decode
     * @return the dedoced descriptor
     */
    public static JAXXObjectDescriptor decodeJAXXObjectDescriptor(String descriptor) {
        try {
            return (JAXXObjectDescriptor) Base64Coder.deserialize(descriptor, false);
            /*byte[] data = new byte[descriptor.length()];
            // copy low-order bytes into the array.  The high-order bytes should all be zero.
            System.arraycopy(descriptor.getBytes(), 0, data, 0, data.length);
            //descriptor.getBytes(0, descriptor.length(), data, 0);
            ObjectInputStream in = new ObjectInputStream(new ByteArrayInputStream(data));
            return (JAXXObjectDescriptor) in.readObject();*/
        }
        catch (IOException e) {
            throw new RuntimeException("Internal error: can't-happen error", e);
        }
        catch (ClassNotFoundException e) {
            throw new RuntimeException("Internal error: can't-happen error", e);
        }
    }


    public static JAXXObjectDescriptor decodeCompressedJAXXObjectDescriptor(String descriptor) {
        try {
            return (JAXXObjectDescriptor) Base64Coder.deserialize(descriptor, true);

            /*byte[] data = new byte[descriptor.length()];
            // copy low-order bytes into the array.  The high-order bytes should all be zero.
            System.arraycopy(descriptor.getBytes(), 0, data, 0, data.length);
            //descriptor.getBytes(0, descriptor.length(), data, 0); 
            ObjectInputStream in = new ObjectInputStream(new GZIPInputStream(new ByteArrayInputStream(data)));
            return (JAXXObjectDescriptor) in.readObject();*/
        }
        catch (IOException e) {
            throw new RuntimeException("Internal error: can't-happen error", e);
        }
        catch (ClassNotFoundException e) {
            throw new RuntimeException("Internal error: can't-happen error", e);
        }
    }


    public static Object getEventListener(Class<? extends EventListener> listenerClass, final String listenerMethodName, final Object methodContainer, final String methodName) {
        WeakReference<List<EventListenerDescriptor>> ref = eventListeners.get(methodContainer);
        List<EventListenerDescriptor> descriptors = ref != null ? ref.get() : null;
        if (descriptors == null) {
            descriptors = new ArrayList<EventListenerDescriptor>();
            eventListeners.put(methodContainer, new WeakReference<List<EventListenerDescriptor>>(descriptors));
        } else {
            for (EventListenerDescriptor descriptor : descriptors) {
                if (listenerClass == descriptor.listenerClass &&
                        (listenerMethodName == null ? descriptor.listenerMethodName == null : listenerMethodName.equals(descriptor.listenerMethodName)) &&
                        methodName.equals(descriptor.methodName)) {
                    return descriptor.eventListener;
                }
            }
        }

        // else we need to create a new listener
        final EventListenerDescriptor descriptor = new EventListenerDescriptor();
        descriptor.listenerClass = listenerClass;
        descriptor.listenerMethodName = listenerMethodName;
        descriptor.methodName = methodName;
        try {
            final List<Method> listenerMethods = Arrays.asList(listenerClass.getMethods());
            Method listenerMethod = null;
            if (listenerMethodName != null) {
                for (Method listenerMethod1 : listenerMethods) {
                    if ((listenerMethod1).getName().equals(listenerMethodName)) {
                        listenerMethod = listenerMethod1;
                        break;
                    }
                }
            }
            if (listenerMethodName != null && listenerMethod == null) {
                throw new IllegalArgumentException("no method named " + listenerMethodName + " found in class " + listenerClass.getName());
            }
            Class[] parameterTypes = listenerMethods.get(0).getParameterTypes();
            Class<?> methodContainerClass = methodContainer.getClass();
            final Method targetMethod = methodContainerClass.getMethod(methodName, parameterTypes);
            descriptor.eventListener = Proxy.newProxyInstance(listenerClass.getClassLoader(),
                    new Class[]{listenerClass},
                    new InvocationHandler() {
                        public Object invoke(Object proxy, Method method, Object[] args) {
                            String methodName = method.getName();
                            if ((listenerMethodName == null && listenerMethods.contains(method)) || methodName.equals(listenerMethodName)) {
                                try {
                                    return targetMethod.invoke(methodContainer, args);
                                }
                                catch (IllegalAccessException e) {
                                    throw new RuntimeException(e);
                                }
                                catch (InvocationTargetException e) {
                                    throw new RuntimeException(e);
                                }
                            }
                            if (methodName.equals("toString")) {
                                return toString();
                            }
                            if (methodName.equals("equals")) {
                                return descriptor.eventListener == args[0];
                            }
                            if (methodName.equals("hashCode")) {
                                return hashCode();
                            }
                            return null;
                        }
                    });
            descriptors.add(descriptor);
            return descriptor.eventListener;
        }
        catch (NoSuchMethodException e) {
            throw new RuntimeException(e);
        }
    }


    public static Object getEventListener(Class<? extends EventListener> listenerClass, final Object methodContainer, final String methodName) {
        return getEventListener(listenerClass, null, methodContainer, methodName);
    }


    public static DataBindingUpdateListener getDataBindingUpdateListener(JAXXObject object, String bindingName) {
        WeakReference<List<DataBindingUpdateListener>> ref = dataBindingUpdateListeners.get(object);
        List<DataBindingUpdateListener> listeners = ref == null ? null : ref.get();
        if (listeners == null) {
            listeners = new ArrayList<DataBindingUpdateListener>();
            dataBindingUpdateListeners.put(object, new WeakReference<List<DataBindingUpdateListener>>(listeners));
        } else {
            for (DataBindingUpdateListener listener : listeners) {
                if (bindingName.equals(listener.getBindingName())) {
                    return listener;
                }
            }
        }
        DataBindingUpdateListener listener = new DataBindingUpdateListener(object, bindingName);
        listeners.add(listener);
        return listener;
    }


    public static void setComponentWidth(Component component, int width) {
        component.setSize(width, component.getHeight());
        if (component instanceof JComponent) {
            JComponent jcomponent = (JComponent) component;
            jcomponent.setPreferredSize(new Dimension(width, jcomponent.getPreferredSize().height));
            jcomponent.setMinimumSize(new Dimension(width, jcomponent.getPreferredSize().height));
            if (jcomponent.isDisplayable()) {
                jcomponent.revalidate();
            }
        }
    }


    public static void setComponentHeight(Component component, int height) {
        component.setSize(component.getWidth(), height);
        if (component instanceof JComponent) {
            JComponent jcomponent = (JComponent) component;
            jcomponent.setPreferredSize(new Dimension(jcomponent.getPreferredSize().width, height));
            jcomponent.setMinimumSize(new Dimension(jcomponent.getPreferredSize().width, height));
            if (jcomponent.isDisplayable()) {
                jcomponent.revalidate();
            }
        }
    }


    public static boolean assignment(boolean value, String name, JAXXObject src) {
        src.firePropertyChange(name.trim(), null, "dummy value");
        return value;
    }


    public static byte assignment(byte value, String name, JAXXObject src) {
        src.firePropertyChange(name.trim(), null, "dummy value");
        return value;
    }


    public static short assignment(short value, String name, JAXXObject src) {
        src.firePropertyChange(name.trim(), null, "dummy value");
        return value;
    }


    public static int assignment(int value, String name, JAXXObject src) {
        src.firePropertyChange(name.trim(), null, "dummy value");
        return value;
    }


    public static long assignment(long value, String name, JAXXObject src) {
        src.firePropertyChange(name.trim(), null, "dummy value");
        return value;
    }


    public static float assignment(float value, String name, JAXXObject src) {
        src.firePropertyChange(name.trim(), null, "dummy value");
        return value;
    }


    public static double assignment(double value, String name, JAXXObject src) {
        src.firePropertyChange(name.trim(), null, "dummy value");
        return value;
    }


    public static char assignment(char value, String name, JAXXObject src) {
        src.firePropertyChange(name.trim(), null, "dummy value");
        return value;
    }


    public static java.lang.Object assignment(java.lang.Object value, String name, JAXXObject src) {
        src.firePropertyChange(name.trim(), null, "dummy value");
        return value;
    }

    /**
     * Compute the string representation of an object.
     * <p/>
     * Return empty string if given object is null
     *
     * @param value the value to write
     * @return the string representation of the given object or an empty string if object is null.
     */
    public static String getStringValue(Object value) {
        String result;
        result = value == null ? "" : value.toString();
        return result;
    }

    /**
     * Test if a type of entry exists in a given context and throw an IllegalArgumentException if not found.
     * <p/>
     * If entry is found, return his value in context.
     *
     * @param <T> the type of required data
     * @param context the context to test
     * @param def     the definition of the entry to seek in context
     * @return the value from the context
     * @throws IllegalArgumentException if the entry is not found in context.
     */
    public static <T> T checkJAXXContextEntry(JAXXContext context, JAXXContextEntryDef<T> def) throws IllegalArgumentException {

        T value = def.getContextValue(context);

        if (value == null) {
            throw new IllegalArgumentException("the context entry [" + def + "] ] was not found in context " + context);
        }

        return value;
    }

    /**
     * Convinient method to apply more than one binding on a JAXX ui.
     *
     * @param src      the ui to treate
     * @param bindings the list of binding to process.
     */
    public static void applyDataBinding(JAXXObject src, String... bindings) {
        for (String binding : bindings) {
            src.applyDataBinding(binding);
        }
    }

    /**
     * Convinient method to process more than one binding on a JAXX ui.
     *
     * @param src      the ui to treate
     * @param bindings the list of binding to process.
     */
    public static void processDataBinding(JAXXObject src, String... bindings) {
        for (String binding : bindings) {
            src.processDataBinding(binding);
        }
    }

    /**
     * Convinient method to remove more than one binding on a JAXX ui.
     *
     * @param src      the ui to treate
     * @param bindings the list of binding to process.
     */
    public static void removeDataBinding(JAXXObject src, String... bindings) {
        for (String binding : bindings) {
            src.removeDataBinding(binding);
        }
    }

    public static <O> DefaultListCellRenderer newDecoratedListCellRenderer(final Decorator<O> decorator) {
        return new DefaultListCellRenderer() {

            private static final long serialVersionUID = 1L;

            @Override
            public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
                String decorated;
                if (value instanceof String) {
                    decorated = (String) value;
                } else {
                    decorated = decorator.toString(value);
                }
                return super.getListCellRendererComponent(list, decorated, index, isSelected, cellHasFocus);
            }
        };
    }

    public static ImageIcon createIcon(String path) {
        java.net.URL imgURL = Util.class.getResource(path);
        if (imgURL != null) {
            return new ImageIcon(imgURL);
        } else {
            throw new IllegalArgumentException("could not find icon " + path);
        }
    }

    /**
     * @param path the location of icons in root directory icons
     * @return the icon at {@link #getIconPath()}+path
     */
    public static ImageIcon createImageIcon(String path) {
        String iconPath = getIconPath();
        return createIcon(iconPath + path);
    }

    /**
     * @param key the key of the icon to retreave from {@link UIManager}
     * @return the icon, or <code>null if no icon found in {@link UIManager}
     */
    public static Icon getUIManagerIcon(String key) {
        return UIManager.getIcon(key);
    }

    /**
     * retreave for the {@link UIManager} the icon prefixed by <code>action.</code>
     *
     * @param key the key of the action icon to retreave from {@link UIManager}
     * @return the icon, or <code>null if no icon found in {@link UIManager}
     */
    public static Icon getUIManagerActionIcon(String key) {
        return getUIManagerIcon("action." + key);
    }


    public static ImageIcon createActionIcon(String name) {
        String iconPath = getIconPath();
        return createIcon(iconPath + "action-" + name + ".png");
    }

    public static ImageIcon createI18nIcon(String name) {
        String iconPath = getIconPath();
        return createIcon(iconPath + "i18n/" + name + ".png");
    }

    /**
     * detects all PropertychangedListener added by Jaxx uis
     * (should be a {@link DataBindingListener}
     *
     * @param propertyNames the array of property names to find
     * @param listeners     the array of listeners to filter
     * @return the filtered listeners
     */
    public static PropertyChangeListener[] findJaxxPropertyChangeListener(String[] propertyNames, PropertyChangeListener... listeners) {
        if (listeners == null || listeners.length == 0) {
            return new PropertyChangeListener[0];
        }
        List<String> pNames = Arrays.asList(propertyNames);

        List<PropertyChangeListener> toRemove = new ArrayList<PropertyChangeListener>();

        for (PropertyChangeListener listener : listeners) {
            String pName = null;
            PropertyChangeListenerProxy plistener = null;
            if (listener instanceof PropertyChangeListenerProxy) {
                plistener = (PropertyChangeListenerProxy) listener;
                if (!pNames.contains(plistener.getPropertyName())) {
                    // not on the good property
                    continue;
                }
                listener = (PropertyChangeListener) plistener.getListener();
                pName = plistener.getPropertyName();
            }
            if (plistener != null && pName != null && listener instanceof DataBindingListener) {
                if (log.isDebugEnabled()) {
                    log.debug("find config listener to remove  [" + pName + "] : " + listener);
                }
                toRemove.add(plistener);
                //toRemove.add(listener);
            }
        }
        return toRemove.toArray(new PropertyChangeListener[toRemove.size()]);
    }

    private static String getIconPath() {
        String iconPath = UIManager.getString(DEFAULT_ICON_PATH_PROPERTY);
        if (iconPath == null) {
            iconPath = DEFAULT_ICON_PATH;
        } else {
            if (!iconPath.endsWith("/")) {
                iconPath += "/";
            }
        }
        return iconPath;
    }

}
