package jaxx.runtime;

import java.util.List;

/**
 *
 * Some usefull methods on {@link Decorator} to store and obtain decorators.
 *
 * Use the method {@link #registerDecorator(String, Decorator)} to register a new
 * decorator.
 *
 * Use the method {@link #getDecorator(String)} to obtain a registred
 * decorator based on his registred name.
 *
 * Use the method {@link #getDecorator(Class, tring)} to obtain a registred
 * decorator based on the type of decorator and the registred name.
 *
 * @author tony
 * @since 1.3
 */
public class DecoratorUtils {

    protected static List<DecoratorContext<?>> cache;

    public static <T> void registerDecorator(String context, Decorator<T> decorator) {
        DecoratorContext<T> result = getDecoratorContext(decorator.getInternalClass(), context);

        if (result != null) {
            throw new IllegalArgumentException("there is an already register decorator " + result);
        }

        if (cache == null) {
            cache = new java.util.ArrayList<DecoratorContext<?>>();
        }
        cache.add(new DecoratorContext<T>(context, decorator));
    }

    public static <T> Decorator<T> getDecorator(String context) {
        Decorator<T> result = getDecorator(null, context);
        return result;
    }

    public static <T> Decorator<T> getDecorator(Class<T> type, String context) {
        DecoratorContext<T> decoratorContext = getDecoratorContext(type, context);
        Decorator<T> result = decoratorContext == null ? null : decoratorContext.getDecorator();
        return result;
    }

    public static void clear() {
        if (cache != null) {
            cache.clear();
        }
    }
    
    @SuppressWarnings({"unchecked"})
    protected static <T> DecoratorContext<T> getDecoratorContext(Class<T> type, String context) {
        DecoratorContext<T> result = null;
        if (cache != null) {
            for (DecoratorContext<?> d : cache) {
                if (type == null) {
                    if (d.accept(context)) {
                        result = (DecoratorContext<T>) d;
                        break;
                    }
                    continue;
                }
                if (d.accept(type, context)) {
                    result = (DecoratorContext<T>) d;
                    break;
                }
            }
        }
        return result;
    }

    public static class DecoratorContext<T> {

        final String context;
        final Decorator<T> decorator;

        public DecoratorContext(String context, Decorator<T> decorator) {
            this.context = context;
            this.decorator = decorator;
        }

        public String getContext() {
            return context;
        }

        public Decorator<T> getDecorator() {
            return decorator;
        }

        public Class<T> getType() {
            return decorator.getInternalClass();
        }

        public boolean accept(Class<?> type, String context) {
            return getType().isAssignableFrom(type) && accept(context);
        }

        public boolean accept(String context) {
            return ((this.context == null && context == null) || (this.context != null && this.context.equals(context)));
        }

        @Override
        public String toString() {
            return super.toString() + "<type: " + getType().getName() + ", context :" + context + ">";
        }
    }
}
