/*
 * *##% 
 * JAXX Compiler
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.compiler.reflect;

import jaxx.compiler.CompilerException;
import jaxx.compiler.JAXXCompiler;
import jaxx.compiler.JAXXEngine;
import jaxx.compiler.java.parser.JavaParser;
import jaxx.compiler.java.parser.JavaParserTreeConstants;
import jaxx.compiler.java.parser.ParseException;
import jaxx.compiler.java.parser.SimpleNode;
import jaxx.compiler.tags.TagManager;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.Reader;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

// TODO: need to unify this implementation with the parsing in ScriptManager
public class JavaFileParser {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private final Log log = LogFactory.getLog(JavaFileParser.class);
    private JAXXCompiler compiler;
    private String className;
    private String packageName = null;
    private String superclass = "java.lang.Object";
    private List<MethodDescriptor> methods = new ArrayList<MethodDescriptor>();
    private List<FieldDescriptor> fields = new ArrayList<FieldDescriptor>();

    private JavaFileParser(ClassLoader classLoader) {
        compiler = JAXXEngine.createDummyCompiler(classLoader);
    }

    public static ClassDescriptor parseJavaFile(String displayName, Reader src, ClassLoader classLoader) throws ClassNotFoundException {
        //FIXME-TC20091205 : no now JAXX can look to interfaces, must add them
        // has some limitations -- it reports all members as public, leaves getDeclaredMethod and getDeclaredField
        // undefined, and doesn't report interfaces.  It's safe to leave those the way they are for now, because
        // JAXX doesn't look at any of those for non-JAXX classes.
        JavaFileParser parser = new JavaFileParser(classLoader);
        if (log.isDebugEnabled()) {
            log.debug("starting parsing : " + displayName);
        }
        try {
            parser.doParse(displayName, src);
        } catch (Exception e) {
            log.error(e.getMessage());
            throw new RuntimeException(e);
        }

        List<MethodDescriptor> publicMethods = parser.methods;
        List<FieldDescriptor> publicFields = parser.fields;
        //List/*<MethodDescriptor>*/ declaredMethods = new ArrayList/*<MethodDescriptor>*/(publicMethods);
        //List/*<FieldDescriptor>*/ declaredFields = new ArrayList/*<FieldDescriptor>*/(publicFields);
        Iterator<MethodDescriptor> methods = publicMethods.iterator();
        while (methods.hasNext()) {
            MethodDescriptor method = methods.next();
            if (!Modifier.isPublic(method.getModifiers())) {
                methods.remove();
            }
        }
        Iterator<FieldDescriptor> fields = publicFields.iterator();
        while (fields.hasNext()) {
            FieldDescriptor field = fields.next();
            if (!Modifier.isPublic(field.getModifiers())) {
                fields.remove();
            }
        }
        ClassDescriptor superclassDescriptor = ClassDescriptorLoader.getClassDescriptor(parser.superclass, classLoader);
        publicMethods.addAll(Arrays.asList(superclassDescriptor.getMethodDescriptors()));
        publicFields.addAll(Arrays.asList(superclassDescriptor.getFieldDescriptors()));
        //FIXME-TC20091205 : now JAXX can look to interfaces, must add them
        //Set<String> interfaces = new HashSet<String>();
        //ClassDescriptor[] superclassInterfaces = superclassDescriptor.getInterfaces();
        //for (ClassDescriptor superclassInterface : superclassInterfaces) {
        //interfaces.add(superclassInterface.getName());
        //}
        return new ClassDescriptor(parser.className, parser.packageName, parser.superclass, new String[0], false, false, null, null, classLoader,
                publicMethods.toArray(new MethodDescriptor[publicMethods.size()]),
                publicFields.toArray(new FieldDescriptor[publicFields.size()])) {

            @Override
            public FieldDescriptor getDeclaredFieldDescriptor(String name) throws NoSuchFieldException {
                throw new NoSuchFieldException(name);
            }

            @Override
            public MethodDescriptor getDeclaredMethodDescriptor(String name, ClassDescriptor... parameterTypes) throws NoSuchMethodException {
                throw new NoSuchMethodException(name);
            }
        };
    }

    private void doParse(String displayName, Reader src) {
        try {
            JavaParser p = new JavaParser(src);
            p.CompilationUnit();
            SimpleNode node = p.popNode();
            if (node != null) {
                scanCompilationUnit(node);
                return;
            }
            throw new CompilerException("Internal error: null node parsing Java file from " + src);
        } catch (ParseException e) {
            throw new CompilerException("Error parsing Java source code " + displayName + ": " + e.getMessage());
        }
    }

    private void scanCompilationUnit(SimpleNode node) {
        for (int i = 0; i < node.jjtGetNumChildren(); i++) {
            SimpleNode child = node.getChild(i);
            int nodeType = child.getId();
            if (nodeType == JavaParserTreeConstants.JJTPACKAGEDECLARATION) {
                packageName = child.getChild(1).getText().trim();
                compiler.addImport(packageName + ".*");
            } else if (nodeType == JavaParserTreeConstants.JJTIMPORTDECLARATION) {
                String text = child.getText().trim();
                if (text.startsWith("import")) {
                    text = text.substring("import".length()).trim();
                }
                if (text.endsWith(";")) {
                    text = text.substring(0, text.length() - 1);
                }
                compiler.addImport(text);
            } else if (nodeType == JavaParserTreeConstants.JJTTYPEDECLARATION) {
                scanCompilationUnit(child);
                //TC-20091204 : add enum support
//            } else if (nodeType == JavaParserTreeConstants.JJTCLASSORINTERFACEDECLARATION) {
            } else if (nodeType == JavaParserTreeConstants.JJTCLASSORINTERFACEDECLARATION || nodeType== JavaParserTreeConstants.JJTENUMDECLARATION) {
                scanClass(child);
            }
        }
    }

    // scans the main ClassOrInterfaceDeclaration
    private void scanClass(SimpleNode node) {
        boolean isInterface = node.firstToken.image.equals("interface");
        className = node.firstToken.next.image;
        if (packageName != null) {
            className = packageName + "." + className;
        }
        for (int i = 0; i < node.jjtGetNumChildren(); i++) {
            SimpleNode child = node.getChild(i);
            int nodeType = child.getId();
            if (nodeType == JavaParserTreeConstants.JJTEXTENDSLIST) {
                if (!isInterface) {
                    assert child.jjtGetNumChildren() == 1 : "expected ExtendsList to have exactly one child for a non-interface class";
                    String rawName = child.getChild(0).getText().trim();
                    superclass = TagManager.resolveClassName(rawName, compiler);
                    if (superclass == null) {
                        throw new CompilerException("Could not find class: " + rawName);
                    }
                }
            } else if (nodeType == JavaParserTreeConstants.JJTCLASSORINTERFACEBODY) {
                scanClassNode(child);
            }
        }
    }

    // scans class body nodes
    private void scanClassNode(SimpleNode node) {
        int nodeType = node.getId();
        if (nodeType == JavaParserTreeConstants.JJTMETHODDECLARATION) {
            String returnType = null;
            String name = null;
            List<String> parameterTypes = new ArrayList<String>();
            //List<String> parameterNames = new ArrayList<String>();
            for (int i = 0; i < node.jjtGetNumChildren(); i++) {
                SimpleNode child = node.getChild(i);
                int type = child.getId();
                if (type == JavaParserTreeConstants.JJTRESULTTYPE) {
                    returnType = TagManager.resolveClassName(child.getText().trim(), compiler);
                } else if (type == JavaParserTreeConstants.JJTMETHODDECLARATOR) {
                    name = child.firstToken.image.trim();
                    SimpleNode formalParameters = child.getChild(0);
                    assert formalParameters.getId() == JavaParserTreeConstants.JJTFORMALPARAMETERS;
                    for (int j = 0; j < formalParameters.jjtGetNumChildren(); j++) {
                        SimpleNode parameter = formalParameters.getChild(j);
                        String rawParameterType = parameter.getChild(1).getText().trim().replaceAll("\\.\\.\\.", "[]");
                        String parameterType = TagManager.resolveClassName(rawParameterType, compiler);
                        if (parameterType == null && JAXXCompiler.STRICT_CHECKS) {
                            throw new CompilerException("could not find class '" + rawParameterType + "'");
                        }
                        parameterTypes.add(parameterType);
                        //parameterNames.add(parameter.getChild(2).getText().trim());
                    }
                }
            }
            methods.add(new MethodDescriptor(name, Modifier.PUBLIC, returnType, parameterTypes.toArray(new String[parameterTypes.size()]), compiler.getClassLoader())); // TODO: determine the actual modifiers
        } else if (nodeType == JavaParserTreeConstants.JJTCLASSORINTERFACEDECLARATION) {
            // TODO: handle inner classes
        } else if (nodeType == JavaParserTreeConstants.JJTCONSTRUCTORDECLARATION) {
            // TODO: handle constructors
        } else if (nodeType == JavaParserTreeConstants.JJTFIELDDECLARATION) {
            String text = node.getText();
            String declaration = text;
            int equals = text.indexOf("=");
            if (equals != -1) {
                declaration = declaration.substring(0, equals);
            }
            declaration = declaration.trim();
            String[] declarationTokens = declaration.split("\\s");
            //boolean isFinal = Arrays.asList(declarationTokens).contains("final");
            //boolean isStatic = Arrays.asList(declarationTokens).contains("static");
            String name = declarationTokens[declarationTokens.length - 1];
            if (name.endsWith(";")) {
                name = name.substring(0, name.length() - 1).trim();
            }
            String cName = declarationTokens[declarationTokens.length - 2];
            String type = TagManager.resolveClassName(cName, compiler);
            fields.add(new FieldDescriptor(name, Modifier.PUBLIC, type, compiler.getClassLoader())); // TODO: determine the actual modifiers
        } else {
            for (int i = 0; i < node.jjtGetNumChildren(); i++) {
                SimpleNode child = node.getChild(i);
                scanClassNode(child);
            }
        }
    }
}
