/*
 * *##% 
 * JAXX Compiler
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.compiler.tags.swing;

import jaxx.compiler.CompilerException;
import jaxx.compiler.CompiledObject;
import jaxx.compiler.I18nHelper;
import jaxx.compiler.JAXXCompiler;
import jaxx.compiler.reflect.ClassDescriptorLoader;
import jaxx.compiler.tags.TagHandler;
import jaxx.runtime.swing.TabInfo;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

import javax.swing.Icon;
import javax.swing.JTabbedPane;
import java.awt.Color;
import java.io.IOException;

public class TabHandler implements TagHandler {

    @Override
    public void compileFirstPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException {
        compileChildrenFirstPass(tag, compiler);
    }

    @Override
    public void compileSecondPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException {
        if (!ClassDescriptorLoader.getClassDescriptor(JTabbedPane.class).isAssignableFrom(compiler.getOpenComponent().getObjectClass())) {
            compiler.reportError("tab tag may only appear within JTabbedPane tag");
            return;
        }

        JTabbedPaneHandler.CompiledTabbedPane tabs = (JTabbedPaneHandler.CompiledTabbedPane) compiler.getOpenComponent();

        String id = tag.getAttribute("id");
        if (id == null || id.length() == 0) {
            id = compiler.getAutoId(ClassDescriptorLoader.getClassDescriptor(TabInfo.class));
        }
        TabInfo tabInfo = new TabInfo(id);
        CompiledObject compiledTabInfo = new CompiledObject(id, ClassDescriptorLoader.getClassDescriptor(TabInfo.class), compiler);
        compiler.registerCompiledObject(compiledTabInfo);
        //id = tabInfo.getId();
        tabs.tabInfo = tabInfo;
        setAttributes(compiledTabInfo, tabs, tag, compiler);
        compileChildrenSecondPass(tag, compiler);
        tabs.tabInfo = null;
    }

    public static void setAttribute(CompiledObject compiledTabInfo, JTabbedPaneHandler.CompiledTabbedPane tabs, String name, String value, JAXXCompiler compiler) throws CompilerException {
        value = value.trim();
        TabInfo tabInfo = tabs.tabInfo;
        String id = tabInfo.getId();
        String binding = compiler.processDataBindings(value);
        if (binding != null) {
            compiler.registerDataBinding(binding, id + "." + name, id + ".set" + org.apache.commons.lang.StringUtils.capitalize(name) + "(" + binding + ");");
            return;
        }

        String valueCode = compiler.getJavaCode(value);

        // add i18n support
        if (I18nHelper.isI18nableAttribute(name, compiler)) {
            value = valueCode = I18nHelper.addI18nInvocation(id, name, valueCode, compiler);
        }

        if (name.equals("title")) {
            tabInfo.setTitle(value);
            compiledTabInfo.appendInitializationCode(id + ".setTitle(" + valueCode + ");");
            //compiledTabInfo.appendInitializationCode(id + ".setTitle(" + TypeManager.getJavaCode(value) + ");");
        } else if (name.equals("toolTipText")) {
            tabInfo.setToolTipText(value);
            compiledTabInfo.appendInitializationCode(id + ".setToolTipText(" + valueCode + ");");
            //compiledTabInfo.appendInitializationCode(id + ".setToolTipText(" + TypeManager.getJavaCode(value) + ");");
        } else if (name.equals("icon")) {
            Icon icon = (Icon) compiler.convertFromString(value, Icon.class);
            tabInfo.setIcon(icon);
            compiledTabInfo.appendInitializationCode(id + ".setIcon(" + compiler.getJavaCode(icon) + ");");
        } else if (name.equals("enabled")) {
            boolean enabled = (Boolean) compiler.convertFromString(value, Boolean.class);
            tabInfo.setEnabled(enabled);
            compiledTabInfo.appendInitializationCode(id + ".setEnabled(" + enabled + ");");
        } else if (name.equals("disabledIcon")) {
            Icon disabledIcon = (Icon) compiler.convertFromString(value, Icon.class);
            tabInfo.setDisabledIcon(disabledIcon);
            compiledTabInfo.appendInitializationCode(id + ".setDisabledIcon(" + compiler.getJavaCode(disabledIcon) + ");");
        } else if (name.equals("mnemonic")) {
            int mnemonic = (Character) compiler.convertFromString(value, char.class);
            tabInfo.setMnemonic(mnemonic);
            compiledTabInfo.appendInitializationCode(id + ".setMnemonic(" + mnemonic + ");");
        } else if (name.equals("displayedMnemonicIndex")) {
            int displayedMnemonicIndex = (Integer) compiler.convertFromString(value, int.class);
            tabInfo.setDisplayedMnemonicIndex(displayedMnemonicIndex);
            compiledTabInfo.appendInitializationCode(id + ".setDisplayedMnemonicIndex(" + displayedMnemonicIndex + ");");
        } else if (name.equals("foreground")) {
            Color foreground = (Color) compiler.convertFromString(value, Color.class);
            tabInfo.setForeground(foreground);
            compiledTabInfo.appendInitializationCode(id + ".setForeground(" + compiler.getJavaCode(foreground) + ");");
        } else if (name.equals("background")) {
            Color background = (Color) compiler.convertFromString(value, Color.class);
            tabInfo.setBackground(background);
            compiledTabInfo.appendInitializationCode(id + ".setBackground(" + compiler.getJavaCode(background) + ");");
        } else if (name.equals("id")) {
            // ignore, already handled
        } else {
            compiler.reportError("The <tab> tag does not support the attribute '" + name + "'");
        }
    }

    public void setAttributes(CompiledObject compiledTabInfo, JTabbedPaneHandler.CompiledTabbedPane tabs, Element tag, JAXXCompiler compiler) throws CompilerException {
        NamedNodeMap children = tag.getAttributes();
        for (int i = 0; i < children.getLength(); i++) {
            Attr attribute = (Attr) children.item(i);
            String name = attribute.getName();
            String value = attribute.getValue();
            if (!name.startsWith("xmlns") && !JAXXCompiler.JAXX_INTERNAL_NAMESPACE.equals(attribute.getNamespaceURI())) {
                setAttribute(compiledTabInfo, tabs, name, value, compiler);
            }
        }
    }

    protected void compileChildrenFirstPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException {
        NodeList children = tag.getChildNodes();
        for (int i = 0; i < children.getLength(); i++) {
            Node node = children.item(i);
            int nodeType = node.getNodeType();
            if (nodeType == Node.ELEMENT_NODE) {
                Element child = (Element) node;
                compileChildTagFirstPass(child, compiler);
            } else if (nodeType == Node.TEXT_NODE || nodeType == Node.CDATA_SECTION_NODE) {
                String text = ((Text) node).getData().trim();
                if (text.length() > 0) {
                    compiler.reportError("tag '" + tag.getLocalName() + "' may not contain text ('" + ((Text) node).getData().trim() + "')");
                }
            }
        }
    }

    protected void compileChildTagFirstPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException {
        compiler.compileFirstPass(tag);
    }

    protected void compileChildrenSecondPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException {
        NodeList children = tag.getChildNodes();
        for (int i = 0; i < children.getLength(); i++) {
            Node node = children.item(i);
            int nodeType = node.getNodeType();
            if (nodeType == Node.ELEMENT_NODE) {
                Element child = (Element) node;
                compileChildTagSecondPass(child, compiler);
            } else if (nodeType == Node.TEXT_NODE || nodeType == Node.CDATA_SECTION_NODE) {
                String text = ((Text) node).getData().trim();
                if (text.length() > 0) {
                    compiler.reportError("tag '" + tag.getLocalName() + "' may not contain text ('" + ((Text) node).getData().trim() + "')");
                }
            }
        }
    }

    protected void compileChildTagSecondPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException {
        compiler.compileSecondPass(tag);
    }
}
