/*
 * *##% 
 * JAXX Compiler
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.compiler.css;

import jaxx.compiler.binding.DataSource;
import jaxx.compiler.binding.DataBinding;
import jaxx.compiler.*;
import jaxx.runtime.css.Selector;
import jaxx.runtime.css.Rule;
import jaxx.runtime.css.Stylesheet;
import jaxx.compiler.java.parser.JavaParser;
import jaxx.compiler.java.parser.JavaParserTreeConstants;
import jaxx.compiler.java.parser.SimpleNode;
import jaxx.compiler.reflect.ClassDescriptor;
import jaxx.compiler.reflect.ClassDescriptorLoader;
import jaxx.compiler.reflect.MethodDescriptor;
import jaxx.compiler.tags.DefaultObjectHandler;
import jaxx.compiler.tags.TagManager;

import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * A helper class to compute {@link Stylesheet}, {@link Rule} and {@link Selector}
 * and extract all the compiler logic from this class.
 * <p/>
 * In that way we can make the compiler as a single module and a runtime as another module.
 *
 * @author chemit
 */
public class StylesheetHelper {

    public static void applyTo(CompiledObject object, JAXXCompiler compiler, Stylesheet stylesheet, Stylesheet overrides) throws CompilerException {
        Map<String, String> overriddenProperties;
        if (overrides != null) {
            overriddenProperties = getApplicableProperties(overrides, object);
            //overriddenProperties = overrides.getApplicableProperties(s,object);
        } else {
            overriddenProperties = null;
        }

        Map<String, String> properties = getApplicableProperties(stylesheet, object);
        if (properties != null) {
            if (overriddenProperties != null) {
                properties.keySet().removeAll(overriddenProperties.keySet());
            }
            DefaultObjectHandler handler = TagManager.getTagHandler(object.getObjectClass());
            for (Map.Entry<String, String> e : properties.entrySet()) {
                String value = e.getValue();
                if (value.equals(Rule.INLINE_ATTRIBUTE) || value.equals(Rule.DATA_BINDING)) {
                    continue;
                }
                handler.setAttribute(object, e.getKey(), e.getValue(), false, compiler);
            }
        }

        Rule[] pseudoClasses = getApplicablePseudoClasses(stylesheet, object);
        if (pseudoClasses != null) {
            Map<String, Map<String, String>> combinedPseudoClasses = new LinkedHashMap<String, Map<String, String>>();
            for (Rule pseudoClass1 : pseudoClasses) {
                Selector[] selectors = pseudoClass1.getSelectors();
                for (Selector selector : selectors) {
                    if (appliesTo(selector, object) == Selector.PSEUDOCLASS_APPLIES) {
                        properties = pseudoClass1.getProperties();
                        String pseudoClass = selector.getPseudoClass();
                        // TODO: overrides by downstream pseudoclasses are not handled
                        Map<String, String> combinedProperties = combinedPseudoClasses.get(pseudoClass);
                        if (combinedProperties == null) {
                            combinedProperties = new HashMap<String, String>();
                            combinedPseudoClasses.put(pseudoClass, combinedProperties);
                        }
                        combinedProperties.putAll(properties);
                    }
                }
            }

            int count = 0;
            for (Map.Entry<String, Map<String, String>> e : combinedPseudoClasses.entrySet()) {
                applyPseudoClass(e.getKey(), e.getValue(), object, compiler, count++);
            }
        }
    }

    /**
     * Replaces all references to the variable "object" with the actual object ID.
     *
     * @param code ?
     * @param id   ?
     * @return ?
     * @throws CompilerException ?
     */
    public static String replaceObjectReferences(String code, String id) throws CompilerException {
        JavaParser p = new JavaParser(new StringReader(code + ";"));
        p.Expression();
        jaxx.compiler.java.parser.SimpleNode node = p.popNode();
        scanNode(node, id);
        return node.getText();
    }

    public static void scanNode(SimpleNode node, String id) {
        if (node.getId() == JavaParserTreeConstants.JJTNAME) {
            String name = node.getText();
            if (name.equals("object") || (name.indexOf(".") != -1 && name.substring(0, name.indexOf(".")).trim().equals("object"))) {
                node.firstToken.image = id;
            }
        } else {
            int count = node.jjtGetNumChildren();
            for (int i = 0; i < count; i++) {
                scanNode(node.getChild(i), id);
            }
        }
    }

    public static void compilePseudoClassAdd(String pseudoClass, CompiledObject object, String propertyCode, JAXXCompiler compiler) throws CompilerException {

        if (pseudoClass.startsWith("{")) {
            pseudoClass = pseudoClass.substring(1, pseudoClass.length() - 1).trim();
            pseudoClass = replaceObjectReferences(pseudoClass, object.getJavaCode());
            String dest = object.getId() + ".style." + pseudoClass + ".add";
            String destCode = compiler.getJavaCode(dest);
            if (compiler.hasProcessDataBinding()) {
                compiler.appendProcessDataBinding("else ");
            }
            compiler.appendProcessDataBinding("if ($dest.equals(" + destCode + ")) { if (" + pseudoClass + ") { " + propertyCode + "} }");
            new DataSource(dest, pseudoClass, compiler).compile("new DataBindingListener(" + compiler.getRootObject().getJavaCode() + ", " + destCode + ")");
            compiler.appendInitDataBindings("applyDataBinding(" + destCode + ");");
            return;
        }

        MouseEventEnum constant = MouseEventEnum.valueOf(pseudoClass);

        String property = null;
        switch (constant) {
            case mousedown:
                property = "mousePressed";
                break;
            case mouseout:
                property = "mouseExited";
                break;
            case mouseover:
                property = "mouseEntered";
                break;
            case mouseup:
                property = "mouseReleased";
                break;
        }

        ClassDescriptor mouseListenerDescriptor = ClassDescriptorLoader.getClassDescriptor(MouseListener.class);
        ClassDescriptor mouseEventDescriptor = ClassDescriptorLoader.getClassDescriptor(MouseEvent.class);

        try {
            MethodDescriptor addMouseListener = object.getObjectClass().getMethodDescriptor("addMouseListener", mouseListenerDescriptor);
            MethodDescriptor methodDescriptor = mouseListenerDescriptor.getMethodDescriptor(property, mouseEventDescriptor);
            object.addEventHandler("style." + pseudoClass + ".add", addMouseListener, methodDescriptor, propertyCode, compiler);

        } catch (NoSuchMethodException e) {
            compiler.reportError("mouseover pseudoclass cannot be applied to object " + object.getObjectClass().getName() + " (no addMouseListener method)");
        }
    }

    public static void compilePseudoClassRemove(String pseudoClass, CompiledObject object, String propertyCode, JAXXCompiler compiler) throws CompilerException {
        if (pseudoClass.startsWith("{")) {
            pseudoClass = pseudoClass.substring(1, pseudoClass.length() - 1).trim();
            pseudoClass = replaceObjectReferences(pseudoClass, object.getJavaCode());
            String dest = object.getId() + ".style." + pseudoClass + ".remove";
            String destCode = compiler.getJavaCode(dest);
            if (compiler.hasProcessDataBinding()) {
                compiler.appendProcessDataBinding("else ");
            }
            compiler.appendProcessDataBinding("if ($dest.equals(" + destCode + ")) { if (" + invert(pseudoClass) + ") { " + propertyCode + "} }");
            new DataSource(dest, pseudoClass, compiler).compile("new DataBindingListener(" + compiler.getRootObject().getJavaCode() + ", " + destCode + ")");
            compiler.appendInitDataBindings("applyDataBinding(" + destCode + ");");
            return;
        }

        MouseEventEnum constant = MouseEventEnum.valueOf(pseudoClass);

        String property = null;
        switch (constant) {
            case mousedown:
                property = "mousePressed";
                break;
            case mouseout:
                property = "mouseReleased";
                break;
            case mouseover:
                property = "mouseExited";
                break;
            case mouseup:
                property = "mousePressed";
                break;
        }

        ClassDescriptor mouseListenerDescriptor = ClassDescriptorLoader.getClassDescriptor(MouseListener.class);
        ClassDescriptor mouseEventDescriptor = ClassDescriptorLoader.getClassDescriptor(MouseEvent.class);

        try {
            MethodDescriptor addMouseListener = object.getObjectClass().getMethodDescriptor("addMouseListener", mouseListenerDescriptor);
            MethodDescriptor methodDescriptor = mouseListenerDescriptor.getMethodDescriptor(property, mouseEventDescriptor);
            object.addEventHandler("style." + pseudoClass + ".remove", addMouseListener, methodDescriptor, propertyCode, compiler);

        } catch (NoSuchMethodException e) {
            compiler.reportError("mouseover pseudoclass cannot be applied to object " + object.getObjectClass().getName() + " (no addMouseListener method)");
        }
    }

    public static String invert(String javaCode) {
        javaCode = javaCode.trim();
        return javaCode.startsWith("!") ? javaCode.substring(1) : "!(" + javaCode + ")";
    }

    public static String unwrap(ClassDescriptor type, String valueCode) {
        if (type == ClassDescriptorLoader.getClassDescriptor(boolean.class)) {
            return "((java.lang.Boolean) " + valueCode + ").booleanValue()";
        } else if (type == ClassDescriptorLoader.getClassDescriptor(byte.class)) {
            return "((java.lang.Byte) " + valueCode + ").byteValue()";
        } else if (type == ClassDescriptorLoader.getClassDescriptor(short.class)) {
            return "((java.lang.Short) " + valueCode + ").shortValue()";
        } else if (type == ClassDescriptorLoader.getClassDescriptor(int.class)) {
            return "((java.lang.Integer) " + valueCode + ").intValue()";
        } else if (type == ClassDescriptorLoader.getClassDescriptor(long.class)) {
            return "((java.lang.Long) " + valueCode + ").longValue()";
        } else if (type == ClassDescriptorLoader.getClassDescriptor(float.class)) {
            return "((java.lang.Float) " + valueCode + ").floatValue()";
        } else if (type == ClassDescriptorLoader.getClassDescriptor(double.class)) {
            return "((java.lang.Double) " + valueCode + ").doubleValue()";
        } else if (type == ClassDescriptorLoader.getClassDescriptor(char.class)) {
            return "((java.lang.Character) " + valueCode + ").charValue()";
        } else {
            return valueCode;
        }
    }

    public static void applyPseudoClass(String pseudoClass, Map<String, String> properties,
            CompiledObject object, JAXXCompiler compiler, int priority) throws CompilerException {
        if (pseudoClass.indexOf("[") != -1) {
            pseudoClass = pseudoClass.substring(0, pseudoClass.indexOf("["));
        }
        final StringBuffer buffer = new StringBuffer();

        DefaultObjectHandler handler = TagManager.getTagHandler(object.getObjectClass());
        boolean valueDeclared = false;
        String eol = JAXXCompiler.getLineSeparator();
        for (Map.Entry<String, String> e : properties.entrySet()) {
            String property = e.getKey();
            ClassDescriptor type = handler.getPropertyType(object, property, compiler);
            String dataBinding = compiler.processDataBindings(e.getValue());
            String valueCode;
            if (dataBinding != null) {
                valueCode = "new jaxx.runtime.css.DataBinding(" + compiler.getJavaCode(object.getId() + "." + property + "." + priority) + ")";
                DataBinding dataBinding1 = new DataBinding(dataBinding, object.getId() + "." + property + "." + priority, handler.getSetPropertyCode(object.getJavaCode(), property, "(" + JAXXCompiler.getCanonicalName(type) + ") " + dataBinding, compiler));
                dataBinding1.compile(compiler, false);
            } else {
                try {
                    Class<?> typeClass = type != null ? ClassDescriptorLoader.getClass(type.getName(), type.getClassLoader()) : null;
                    valueCode = compiler.getJavaCode(compiler.convertFromString(e.getValue(), typeClass));
                } catch (ClassNotFoundException ex) {
                    compiler.reportError("could not find class " + type.getName());
                    return;
                }
            }
            if (!valueDeclared) {
                buffer.append("java.lang.Object ");
                valueDeclared = true;
            }
            buffer.append("value = jaxx.runtime.css.Pseudoclasses.applyProperty(").append(compiler.getOutputClassName()).append(".this, ").append(object.getJavaCode()).append(", ").append(compiler.getJavaCode(property)).append(", ").append(valueCode).append(", jaxx.runtime.css.Pseudoclasses.wrap(").append(handler.getGetPropertyCode(object.getJavaCode(), property, compiler)).append("), ").append(priority).append(");").append(eol);
            buffer.append("if (!(value instanceof jaxx.runtime.css.DataBinding)) {").append(eol);
            String unwrappedValue = unwrap(type, "value");
            buffer.append("    ").append(handler.getSetPropertyCode(object.getJavaCode(), property, "(" + JAXXCompiler.getCanonicalName(type) +
                    ") " + unwrappedValue, compiler)).append(eol);
            buffer.append("}").append(eol);
        }

        if (pseudoClass.equals("focused")) {
            pseudoClass = "{ object.hasFocus() }";
        } else if (pseudoClass.equals("unfocused")) {
            pseudoClass = "{ !object.hasFocus() }";
        } else if (pseudoClass.equals("enabled")) {
            pseudoClass = "{ object.isEnabled() }";
        } else if (pseudoClass.equals("disabled")) {
            pseudoClass = "{ !object.isEnabled() }";
        } else if (pseudoClass.equals("selected")) {
            pseudoClass = "{ object.isSelected() }";
        } else if (pseudoClass.equals("deselected")) {
            pseudoClass = "{ !object.isSelected() }";
        }

        compilePseudoClassAdd(pseudoClass, object, buffer.toString(), compiler);

        buffer.setLength(0);
        valueDeclared = false;
        for (Map.Entry<String, String> e : properties.entrySet()) {
            String property = e.getKey();
            ClassDescriptor type = handler.getPropertyType(object, property, compiler);
            String dataBinding = compiler.processDataBindings(e.getValue());
            String valueCode;
            if (dataBinding != null) {
                valueCode = "new jaxx.runtime.css.DataBinding(" + compiler.getJavaCode(object.getId() + "." + property + "." + priority) + ")";
                DataBinding dataBinding1 = new DataBinding(dataBinding, object.getId() + "." + property + "." + priority, handler.getSetPropertyCode(object.getJavaCode(), property, "(" + JAXXCompiler.getCanonicalName(type) + ") " + dataBinding, compiler));
                dataBinding1.compile(compiler, false);
            } else {
                try {
                    Class<?> typeClass = type != null ? ClassDescriptorLoader.getClass(type.getName(), type.getClassLoader()) : null;
                    valueCode = compiler.getJavaCode(compiler.convertFromString(e.getValue(), typeClass));
                } catch (ClassNotFoundException ex) {
                    compiler.reportError("could not find class " + type.getName());
                    return;
                }
            }
            if (!valueDeclared) {
                buffer.append("java.lang.Object ");
                valueDeclared = true;
            }
            buffer.append("value = jaxx.runtime.css.Pseudoclasses.removeProperty(").append(compiler.getOutputClassName()).append(".this, ").append(object.getJavaCode()).append(", ").append(compiler.getJavaCode(property)).append(", ").append(valueCode).append(", jaxx.runtime.css.Pseudoclasses.wrap(").append(handler.getGetPropertyCode(object.getJavaCode(), property, compiler)).append("), ").append(priority).append(");").append(eol);
            buffer.append("if (!(value instanceof jaxx.runtime.css.DataBinding)) {").append(eol);
            String unwrappedValue = unwrap(type, "value");
            buffer.append("    ").append(handler.getSetPropertyCode(object.getJavaCode(), property, "(" + JAXXCompiler.getCanonicalName(type) +
                    ") " + unwrappedValue, compiler)).append(eol);
            buffer.append("}").append(eol);
        }
        compilePseudoClassRemove(pseudoClass, object, buffer.toString(), compiler);
    }

    public static Map<String, String> getApplicableProperties(Stylesheet s, CompiledObject object) throws CompilerException {
        DefaultObjectHandler handler = TagManager.getTagHandler(object.getObjectClass());
        Map<String, String> result = null;
        for (Rule rule : s.getRules()) {
            int apply = appliesTo(rule, object);
            if (apply == Selector.ALWAYS_APPLIES || apply == Selector.ALWAYS_APPLIES_INHERIT_ONLY) {
                if (result == null) {
                    result = new HashMap<String, String>();
                }
                for (Map.Entry<String, String> entry : rule.getProperties().entrySet()) {
                    String property = entry.getKey();
                    if (apply == Selector.ALWAYS_APPLIES || handler.isPropertyInherited(property)) {
                        result.put(property, entry.getValue());
                    }
                }
            }
        }
        return result;
    }

    public static Rule[] getApplicablePseudoClasses(Stylesheet s, CompiledObject object) throws CompilerException {
        List<Rule> result = null;
        for (Rule rule : s.getRules()) {
            if (appliesTo(rule, object) == Selector.PSEUDOCLASS_APPLIES) {
                if (result == null) {
                    result = new ArrayList<Rule>();
                }
                result.add(rule);
            }
        }
        return result != null ? result.toArray(new Rule[result.size()]) : null;
    }

    public static Rule inlineAttribute(CompiledObject object, String propertyName, boolean dataBinding) {
        Map<String, String> properties = new HashMap<String, String>();
        properties.put(propertyName, dataBinding ? Rule.DATA_BINDING : Rule.INLINE_ATTRIBUTE);
        return new Rule(new Selector[]{new Selector(null, null, null, object.getId(), true)}, properties);
    }

    public static int appliesTo(Rule rule, CompiledObject object) throws CompilerException {
        int appliesTo = Selector.NEVER_APPLIES;
        for (Selector selector : rule.getSelectors()) {
            appliesTo = Math.max(appliesTo(selector, object), appliesTo);
            if (appliesTo == Selector.ALWAYS_APPLIES || appliesTo == Selector.ALWAYS_APPLIES_INHERIT_ONLY) {
                break;
            }
        }
        return appliesTo;
    }

    public static int appliesTo(Selector selector, CompiledObject object) {
        boolean inheritOnly = false;
        CompiledObject parent = object;
        String javaClassName = selector.getJavaClassName();
        String styleClass = selector.getStyleClass();
        String pseudoClass = selector.getPseudoClass();
        String id = selector.getId();

        while (parent != null) {
            boolean classMatch = (javaClassName == null);
            if (!classMatch) {
                ClassDescriptor javaClass = parent.getObjectClass();
                do {
                    String name = javaClass.getName();
                    if (name.equals(javaClassName) || name.substring(name.lastIndexOf(".") + 1).equals(javaClassName)) {
                        classMatch = true;
                        break;
                    }
                    javaClass = javaClass.getSuperclass();
                } while (javaClass != null);
            }

            boolean styleClassMatch = (styleClass == null || styleClass.equals(parent.getStyleClass()));

            String objectId = parent.getId();
            objectId = objectId.substring(objectId.lastIndexOf(".") + 1);
            boolean idMatch = (id == null || (' ' + objectId + ' ').indexOf(' ' + id + ' ') > -1);

            if (classMatch && styleClassMatch && idMatch) {
                if (pseudoClass != null) {
                    return inheritOnly ? Selector.PSEUDOCLASS_APPLIES_INHERIT_ONLY : Selector.PSEUDOCLASS_APPLIES;
                } else {
                    return inheritOnly ? Selector.ALWAYS_APPLIES_INHERIT_ONLY : Selector.ALWAYS_APPLIES;
                }
            }

            parent = parent.getParent();
            inheritOnly = true;
        }
        return Selector.NEVER_APPLIES;
    }

    public enum MouseEventEnum {

        mouseover,
        mouseout,
        mousedown,
        mouseup
    }
}
