/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.tags.swing;

import jaxx.CompilerException;
import jaxx.compiler.CompiledObject;
import jaxx.compiler.JAXXCompiler;
import jaxx.reflect.ClassDescriptor;
import jaxx.reflect.ClassDescriptorLoader;
import jaxx.tags.DefaultComponentHandler;
import org.w3c.dom.Element;

import javax.swing.JSplitPane;
import java.awt.Component;

public class JSplitPaneHandler extends DefaultComponentHandler {
    public JSplitPaneHandler(ClassDescriptor beanClass) {
        super(beanClass);
        ClassDescriptorLoader.checkSupportClass(getClass(), beanClass, JSplitPane.class);
    }

    protected Component createRawComponent(Element tag) {
        return new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
    }

    /**
     * Add support for <code>orientation="vertical"</code> and <code>orientation="horizontal"</code>.  The
     * values required by the JAXXBeanInfo are the unwieldy <code>vertical_split</code> and <code>horizontal_split</code>
     * (which are also recognized).
     */
    @Override
    protected int constantValue(String key, String value) {
        if (key.equals("orientation")) {
            value = value.trim().toLowerCase();
            if (value.equals("horizontal") || value.equals("horizontal_split")) {
                return JSplitPane.HORIZONTAL_SPLIT;
            }
            if (value.equals("vertical") || value.equals("vertical_split")) {
                return JSplitPane.VERTICAL_SPLIT;
            }
            throw new IllegalArgumentException("orientation must be 'horizontal' or 'vertical', found '" + value + "'");
        }
        return super.constantValue(key, value);
    }

    @Override
    public CompiledObject createCompiledObject(String id, JAXXCompiler compiler) throws CompilerException {
        return new CompiledObject(id, getBeanClass(), compiler) {
            private int count;

            @Override
            public void addChild(CompiledObject child, String constraints, JAXXCompiler compiler) throws CompilerException {
                if (constraints != null) {
                    compiler.reportError("JSplitPane does not accept constraints");
                }
                if (count == 0) {
                    super.addChild(child, "JSplitPane.LEFT", compiler);
                } else if (count == 1) {
                    super.addChild(child, "JSplitPane.RIGHT", compiler);
                } else {
                    compiler.reportError("JSplitPane is limited to two children");
                }
                count++;
            }
        };
    }
}