/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.tags;

import jaxx.CompilerException;
import jaxx.UnsupportedAttributeException;
import jaxx.compiler.JAXXCompiler;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.StringWriter;

/**
 * Handles the <code>&lt;script&gt;</code> tag.
 *
 * @author Ethan Nicholas
 */
public class ScriptHandler implements TagHandler {
    public void compileFirstPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException {
        File scriptFile = null;
        NamedNodeMap attributes = tag.getAttributes();
        for (int i = 0; i < attributes.getLength(); i++) {
            Attr attribute = (Attr) attributes.item(i);
            String name = attribute.getName();
            String attrValue = attribute.getValue();
            if (name.equals("source")) {
                scriptFile = new File(compiler.getBaseDir(), attrValue.replace('/', File.separatorChar));
                StringWriter scriptBuffer = new StringWriter();
                try {
                    FileReader in = new FileReader(scriptFile);
                    char[] readBuffer = new char[2048];
                    int c;
                    while ((c = in.read(readBuffer)) > 0) {
                        scriptBuffer.write(readBuffer, 0, c);
                    }
                }
                catch (FileNotFoundException e) {
                    compiler.reportError("script file not found: " + scriptFile);
                }
                compiler.registerScript(scriptBuffer.toString(), scriptFile);
            } else
            if (!name.startsWith("xmlns") && !JAXXCompiler.JAXX_INTERNAL_NAMESPACE.equals(attribute.getNamespaceURI()))
                throw new UnsupportedAttributeException(name);
        }

        StringBuffer script = new StringBuffer();
        NodeList children = tag.getChildNodes();
        for (int i = 0; i < children.getLength(); i++) {
            Node child = children.item(i);
            switch (child.getNodeType()) {
                case Node.ELEMENT_NODE:
                    compiler.reportError("<script> tag may not contain child elements: " + tag);
                case Node.TEXT_NODE: // fall through
                case Node.CDATA_SECTION_NODE:
                    script.append(((Text) child).getData());
            }
        }

        String scriptString = script.toString().trim();
        if (scriptString.length() > 0) {
            if (scriptFile != null) {
                compiler.reportError("<script> tag has both a source attribute and an inline script");
            }
            compiler.registerScript(script.toString());
        }
    }


    public void compileSecondPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException {
    }
}
