/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.tags;

import jaxx.CompilerException;
import jaxx.UnsupportedAttributeException;
import jaxx.compiler.CompiledObject;
import jaxx.compiler.I18nHelper;
import jaxx.compiler.JAXXCompiler;
import jaxx.reflect.ClassDescriptor;
import jaxx.reflect.ClassDescriptorLoader;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Element;

import java.awt.Component;
import java.awt.Container;
import java.awt.event.ComponentListener;
import java.awt.event.ContainerListener;
import java.awt.event.FocusListener;
import java.beans.IntrospectionException;
import java.io.IOException;
import java.lang.reflect.Field;
import org.w3c.dom.Attr;
import org.w3c.dom.NamedNodeMap;

public class DefaultComponentHandler extends DefaultObjectHandler {

    /** log */
    protected static final Log log = LogFactory.getLog(DefaultComponentHandler.class);
    private String containerDelegate;

    public DefaultComponentHandler(ClassDescriptor beanClass) {
        super(beanClass);
        ClassDescriptorLoader.checkSupportClass(getClass(), beanClass, Component.class);
    }

    @Override
    protected void init() throws IntrospectionException {
        if (jaxxBeanInfo == null) {
            super.init();

            containerDelegate = (String) getJAXXBeanInfo().getJAXXBeanDescriptor().getValue("containerDelegate");
            if (containerDelegate == null && ClassDescriptorLoader.getClassDescriptor(Container.class).isAssignableFrom(getBeanClass().getSuperclass())) {
                containerDelegate = ((DefaultComponentHandler) TagManager.getTagHandler(getBeanClass().getSuperclass())).getContainerDelegate();
            }
        }
    }

    @Override
    protected void configureProxyEventInfo() {
        super.configureProxyEventInfo();
        addProxyEventInfo("hasFocus", FocusListener.class);
        addProxyEventInfo("isVisible", ComponentListener.class);
        addProxyEventInfo("getBounds", ComponentListener.class);
        addProxyEventInfo("getLocation", ComponentListener.class);
        addProxyEventInfo("getLocationOnScreen", ComponentListener.class);
        addProxyEventInfo("getSize", ComponentListener.class);
        addProxyEventInfo("getX", ComponentListener.class);
        addProxyEventInfo("getY", ComponentListener.class);
        addProxyEventInfo("getWidth", ComponentListener.class);
        addProxyEventInfo("getHeight", ComponentListener.class);
        if (ClassDescriptorLoader.getClassDescriptor(Container.class).isAssignableFrom(getBeanClass())) {
            addProxyEventInfo("getComponentCount", ContainerListener.class);
        }
    }

    @Override
    protected void setDefaults(CompiledObject object, Element tag, JAXXCompiler compiler) throws CompilerException {
        super.setDefaults(object, tag, compiler);
        setAttribute(object, "name", object.getId(), false, compiler);
        openComponent(object, tag, compiler);
    }

    @Override
    public void compileFirstPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException {
        super.compileFirstPass(tag, compiler);
    }

    @Override
    public void compileSecondPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException {
        super.compileSecondPass(tag, compiler);
        closeComponent(compiler.getOpenComponent(), tag, compiler);
    }

    protected void openComponent(CompiledObject object, Element tag, JAXXCompiler compiler) throws CompilerException {
        String constraints = tag.getAttribute("constraints");
        if (constraints != null && constraints.length() > 0) {
            compiler.openComponent(object, constraints);
        } else {
            compiler.openComponent(object);
        }
    }

    protected void closeComponent(CompiledObject object, Element tag, JAXXCompiler compiler) throws CompilerException {
        compiler.closeComponent(object);
    }

    @Override
    public boolean isPropertyInherited(String property) throws UnsupportedAttributeException {
        return property.equals("font") || property.startsWith("font-") || property.equals("foreground") || property.equals("enabled");
    }

    @Override
    public ClassDescriptor getPropertyType(CompiledObject object, String propertyName, JAXXCompiler compiler) throws CompilerException {
        if (propertyName.equals("x") || propertyName.equals("y") || propertyName.equals("width") || propertyName.equals("height") ||
                "font-size".equals(propertyName)) {
            return ClassDescriptorLoader.getClassDescriptor(Integer.class);
        }
        if (propertyName.equals("font-face") || propertyName.equals("font-style") || propertyName.equals("font-weight")) {
            return ClassDescriptorLoader.getClassDescriptor(String.class);
        }
        return super.getPropertyType(object, propertyName, compiler);
    }

    @Override
    public String getGetPropertyCode(String id, String name, JAXXCompiler compiler) throws CompilerException {
        if (name.equals("font-face")) {
            return id + ".getFont().getFontName()";
        }
        if (name.equals("font-size")) {
            return id + ".getFont().getSize()";
        }
        if (name.equals("font-weight")) {
            return "(" + id + ".getFont().getStyle() & Font.BOLD) != 0 ? \"bold\" : \"normal\"";
        }
        if (name.equals("font-style")) {
            return "(" + id + ".getFont().getStyle() & Font.ITALIC) != 0 ? \"italic\" : \"normal\"";
        }
        return super.getGetPropertyCode(id, name, compiler);
    }

    @Override
    public String getSetPropertyCode(String id, String name, String valueCode, JAXXCompiler compiler) throws CompilerException {
        if (name.equals("x")) {
            return id + ".setLocation(" + valueCode + ", " + id + ".getY());";
        }
        if (name.equals("y")) {
            return id + ".setLocation(" + id + ".getX(), " + valueCode + ");";
        }
        if (name.equals("width")) { // need to optimize case when both width and height are being assigned
            return "jaxx.runtime.Util.setComponentWidth(" + id + "," + valueCode + ");\n";
        }
        if (name.equals("height")) {
            return "jaxx.runtime.Util.setComponentHeight(" + id + "," + valueCode + ");\n";
        }
        if (name.equals("font-face")) {
            return "if (" + id + ".getFont() != null) " + id + ".setFont(new Font(" + valueCode + ", " + id + ".getFont().getStyle(), " + id + ".getFont().getSize()));";
        }
        if (name.equals("font-size")) {
            return "if (" + id + ".getFont() != null) " + id + ".setFont(" + id + ".getFont().deriveFont((float) " + valueCode + "));";
        }
        if (name.equals("font-weight")) {
            if (valueCode.equals("\"bold\"")) {
                return "if (" + id + ".getFont() != null) " + id + ".setFont(" + id + ".getFont().deriveFont(" + id + ".getFont().getStyle() | Font.BOLD));";
            }
            if (valueCode.equals("\"normal\"")) {
                return "if (" + id + ".getFont() != null) " + id + ".setFont(" + id + ".getFont().deriveFont(" + id + ".getFont().getStyle() & ~Font.BOLD));";
            }
            if (!valueCode.startsWith("\"")) {
                return "if (" + id + ".getFont() != null) { if ((" + valueCode + ").equals(\"bold\")) " +
                        id + ".setFont(" + id + ".getFont().deriveFont(" + id + ".getFont().getStyle() | Font.BOLD)); else " +
                        id + ".setFont(" + id + ".getFont().deriveFont(" + id + ".getFont().getStyle() & ~Font.BOLD)); }";
            }
            compiler.reportError("font-weight must be either \"normal\" or \"bold\", found " + valueCode);
            return "";
        }
        if (name.equals("font-style")) {
            if (valueCode.equals("\"italic\"")) {
                return "if (" + id + ".getFont() != null) " + id + ".setFont(" + id + ".getFont().deriveFont(" + id + ".getFont().getStyle() | Font.ITALIC));";
            }
            if (valueCode.equals("\"normal\"")) {
                return "if (" + id + ".getFont() != null) " + id + ".setFont(" + id + ".getFont().deriveFont(" + id + ".getFont().getStyle() & ~Font.ITALIC));";
            }
            if (!valueCode.startsWith("\"")) {
                return "if (" + id + ".getFont() != null) { if ((" + valueCode + ").equals(\"italic\")) " +
                        id + ".setFont(" + id + ".getFont().deriveFont(" + id + ".getFont().getStyle() | Font.ITALIC)); else " +
                        id + ".setFont(" + id + ".getFont().deriveFont(" + id + ".getFont().getStyle() & ~Font.ITALIC)); }";
            }
            compiler.reportError("font-style must be either \"normal\" or \"italic\", found " + valueCode);
            return "";
        }
        if (ClassDescriptorLoader.getClassDescriptor(Container.class).isAssignableFrom(getBeanClass()) && name.equals("layout")) { // handle containerDelegate (e.g. contentPane on JFrame)
            String containerDelegate = (String) getJAXXBeanInfo().getJAXXBeanDescriptor().getValue("containerDelegate");
            if (containerDelegate != null) {
                return id + '.' + containerDelegate + "().setLayout(" + valueCode + ");";
            }
        }
        // ajout du support i18n
        if (I18nHelper.isI18nableAttribute(name, compiler)) {
            valueCode = I18nHelper.addI18nInvocation(id, name, valueCode, compiler);
        }

        return super.getSetPropertyCode(id, name, valueCode, compiler);
    }

    @Override
    public void setAttribute(CompiledObject object, String propertyName, String stringValue, boolean inline, JAXXCompiler compiler) {

        if (propertyName.startsWith("_")) {
            // client property
            if (stringValue.startsWith("{")) {
                stringValue = stringValue.substring(1, stringValue.length() - 1);
            }
            object.addClientProperty(propertyName.substring(1), stringValue);
            //TC-20090327 rather not generating code here
            //object.appendAdditionCode(object.getJavaCode() + ".putClientProperty(\"" + propertyName.substring(1) + "\", " + stringValue + ");");
            return;
        }
        if ("icon".equals(propertyName)) {
            if (!(stringValue.startsWith("{") || stringValue.endsWith("}"))) {
                // this is a customized icon, add the icon creation code
                if (compiler.getOptions().isUseUIManagerForIcon()) {
                    stringValue = "{" + jaxx.runtime.Util.class.getName() + ".getUIManagerIcon(\"" + stringValue + "\")}";
                } else {
                    stringValue = "{" + jaxx.runtime.Util.class.getName() + ".createImageIcon(\"" + stringValue + "\")}";
                }
            }
        } else if ("actionIcon".equals(propertyName)) {
            // customized actionIcon property
            if (stringValue.startsWith("{") && stringValue.endsWith("}")) {
                // there is a script to define the action icon, this is forbidden
                compiler.reportError("the actionIcon does not support script, remove braces..., fix the file " + compiler.getOutputClassName());
                return;
            }
            propertyName = "icon";
            if (compiler.getOptions().isUseUIManagerForIcon()) {
                stringValue = "{" + jaxx.runtime.Util.class.getName() + ".getUIManagerActionIcon(\"" + stringValue + "\")}";
            } else {
                stringValue = "{" + jaxx.runtime.Util.class.getName() + ".createActionIcon(\"" + stringValue + "\")}";
            }
        }
        super.setAttribute(object, propertyName, stringValue, inline, compiler);
    }

    @Override
    protected void scanAttributesForDependencies(Element tag, JAXXCompiler compiler) {
        super.scanAttributesForDependencies(tag, compiler);
        // check for clientProperty attributes
        //FIXME make this works,... it seems jaxx compiler does not come here ?
        //FIXME see the the firstPassHandler in JAXXCompiler ?
        NamedNodeMap children = tag.getAttributes();
        for (int i = 0, max = children.getLength(); i < max; i++) {
            Attr attr = (Attr) children.item(i);
            String name = attr.getName();
            if (!name.startsWith("_")) {
                continue;
            }
            String value = attr.getValue();
            if (value.startsWith("{")) {
                compiler.reportWarning(tag, "an clientProperty attribute " + name.substring(1) + " does not required curly value but was : " + value, 0);
            }
        }

    }

    /**
     * Maps string values onto integers, so that int-valued enumeration properties can be specified by strings.  For
     * example, when passed a key of 'alignment', this method should normally map the values 'left', 'center', and
     * 'right' onto SwingConstants.LEFT, SwingConstants.CENTER, and SwingConstants.RIGHT respectively.
     * <p/>
     * You do not normally need to call this method yourself; it is invoked by {@link #convertFromString} when an
     * int-valued property has a value which is not a valid number.  By default, this method looks at the
     * <code>enumerationValues</code> value of the <code>JAXXPropertyDescriptor</code>.
     *
     * @param key   the name of the int-typed property
     * @param value the non-numeric value that was specified for the property
     * @throws IllegalArgumentException if the property is an enumeration, but the value is not valid
     * @throws NumberFormatException    if the property is not an enumeration
     */
    @Override
    protected int constantValue(String key, String value) {
        if ((key.equals("mnemonic") || key.equals("displayedMnemonic"))) {
            if (value.length() == 1) {
                return value.charAt(0);
            }
            try {
                Field vk = java.awt.event.KeyEvent.class.getField(value);
                return (Integer) vk.get(null);
            } catch (NoSuchFieldException e) {
                throw new IllegalArgumentException("mnemonics must be either a single character or the name of a field in KeyEvent (found: '" + value + "')");
            } catch (IllegalAccessException e) {
                throw new RuntimeException(e);
            }
        }
        return super.constantValue(key, value);
    }

    /**
     * Returns <code>true</code> if this component can contain other components.  For children to be
     * allowed, the component must be a subclass of <code>Container</code> and its <code>JAXXBeanInfo</code>
     * must not have the value <code>false</code> for its <code>isContainer</code> value.
     *
     * @return <code>true</code> if children are allowed
     */
    public boolean isContainer() {
        boolean container = ClassDescriptorLoader.getClassDescriptor(Container.class).isAssignableFrom(getBeanClass());
        if (container) {
            try {
                init();
                if (Boolean.FALSE.equals(getJAXXBeanInfo().getJAXXBeanDescriptor().getValue("isContainer"))) {
                    container = false;
                }
            } catch (IntrospectionException e) {
                throw new RuntimeException(e);
            }
        }
        return container;
    }

    public String getContainerDelegate() {
        try {
            init();
            return containerDelegate;
        } catch (IntrospectionException e) {
            throw new RuntimeException(e);
        }

    }
}