package jaxx.introspection;

import jaxx.CompilerException;
import jaxx.reflect.ClassDescriptor;
import jaxx.reflect.MethodDescriptor;

/**
 * Mirrors the class <code>java.beans.PropertyDescriptor</code>.  JAXX uses its own introspector rather than the built-in
 * <code>java.beans.Introspector</code> so that it can introspect {@link jaxx.reflect.ClassDescriptor},
 * not just <code>java.lang.Class</code>.
 */
public class JAXXPropertyDescriptor extends JAXXFeatureDescriptor {
    private ClassDescriptor propertyType;
    private MethodDescriptor readMethod;
    private MethodDescriptor writeMethod;
    private boolean bound;

    public JAXXPropertyDescriptor(ClassDescriptor classDescriptor, String propertyName) {
        this(classDescriptor, propertyName, null, null);
    }


    public JAXXPropertyDescriptor(ClassDescriptor classDescriptor, String propertyName,
                                  MethodDescriptor readMethod, MethodDescriptor writeMethod) {
        this(classDescriptor, propertyName, readMethod, writeMethod, false);
    }


    public JAXXPropertyDescriptor(ClassDescriptor classDescriptor, String propertyName,
                                  MethodDescriptor readMethod, MethodDescriptor writeMethod,
                                  boolean bound) {
        super(classDescriptor, propertyName);
        this.readMethod = readMethod;
        this.writeMethod = writeMethod;
        this.bound = bound;
    }


    public MethodDescriptor getReadMethodDescriptor() {
        if (readMethod == null) {
            try {
                readMethod = getClassDescriptor().getMethodDescriptor("get" + capitalize(getName()));
            }
            catch (NoSuchMethodException e) {
                try {
                    readMethod = getClassDescriptor().getMethodDescriptor("is" + capitalize(getName()));
                }
                catch (NoSuchMethodException e2) {
                }
            }
        }
        return readMethod;
    }


    public MethodDescriptor getWriteMethodDescriptor() {
        if (writeMethod == null) {
            try {
                String methodName = "set" + capitalize(getName());
                MethodDescriptor read = getReadMethodDescriptor();
                if (read != null) {
                    writeMethod = getClassDescriptor().getMethodDescriptor(methodName, read.getReturnType());
                } else {
                    throw new CompilerException("Internal error: requesting 'set' method for property of unknown type: '" + getName() + "' (in " + getClassDescriptor() + ")");
                }
            }
            catch (NoSuchMethodException e) {
            }
        }
        return writeMethod;
    }


    public ClassDescriptor getPropertyType() {
        if (propertyType == null) {
            MethodDescriptor read = getReadMethodDescriptor();
            if (read != null)
                propertyType = read.getReturnType();
            else {
                MethodDescriptor write = getWriteMethodDescriptor();
                propertyType = write.getParameterTypes()[0];
            }
        }
        return propertyType;
    }


    public boolean isBound() {
        return bound;
    }


    public void setBound(boolean bound) {
        this.bound = bound;
    }
}