/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.compiler;

import jaxx.compiler.JavaMethod.MethodOrder;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.List;
import java.util.Map.Entry;

/**
 * A Java source file being generated for output.  Once the class is completely initialized, use the
 * {@link #toString} method to generate source code for it.
 */
public class JavaFile {

    protected static final String GETTER_PATTERN = "return %1$s;";

    protected static final String BOOLEAN_GETTER_PATTERN = "return %1$s !=null && %1$s;";

    protected static final String SETTER_PATTERN = "%1$s oldValue = this.%2$s;\nthis.%2$s = newValue;\nfirePropertyChange(\"%2$s\", oldValue, newValue);";

    private int modifiers;
    private String className;
    private List<String> imports = new ArrayList<String>();
    private List<JavaField> fields = new ArrayList<JavaField>();
    private List<JavaMethod> methods = new ArrayList<JavaMethod>();
    private List<JavaFile> innerClasses = new ArrayList<JavaFile>();
    private String superClass;
    private List<String> interfaces;
    private StringBuffer rawBodyCode = new StringBuffer();
    private boolean superclassIsJAXXObject;
    private boolean abstractClass;
    private String genericType;
    private String superGenericType;


    public JavaFile() {
    }


    public JavaFile(int modifiers, String className, String superClass) {
        this(modifiers, className, superClass, null);
    }


    public JavaFile(int modifiers, String className, String superClass, List<String> interfaces) {
        this.modifiers = modifiers;
        this.className = className;
        this.superClass = superClass;
        this.interfaces = interfaces;
    }


    public void addImport(String importString) {
        imports.add(importString);
    }

    public void addImport(Class importString) {
        imports.add(importString.getName());
    }


    public String[] getImports() {
        return imports.toArray(new String[imports.size()]);
    }


    public int getModifiers() {
        return modifiers;
    }


    public void setModifiers(int modifiers) {
        this.modifiers = modifiers;
    }


    public String getClassName() {
        return className;
    }


    public void setClassName(String className) {
        this.className = className;
    }


    public String getSuperClass() {
        return superClass;
    }


    public void setSuperClass(String superClass) {
        this.superClass = superClass;
    }


    public List<String> getInterfaces() {
        if (interfaces == null) {
            interfaces = new ArrayList<String>();
        }
        return interfaces;
    }


    public void setInterfaces(List<String> interfaces) {
        this.interfaces = interfaces;
    }

    public void setGenericType(String genericType) {
        this.genericType = genericType;
    }

    public void addMethod(JavaMethod method) {
        methods.add(method);
    }


    public JavaMethod[] getMethods() {
        return methods.toArray(new JavaMethod[methods.size()]);
    }

    public void addField(JavaField field) {
        addField(field, false);
    }

    public void addField(JavaField field, boolean javaBean) {
        addSimpleField(field);
        String id = field.getName();
        String capitalizedName = org.apache.commons.lang.StringUtils.capitalize(id);
        // add getter file
        String content = String.format(GETTER_PATTERN, id);
        addMethod(new JavaMethod(
                Modifier.isProtected(field.getModifiers()) ? Modifier.PUBLIC : Modifier.PROTECTED,
                field.getType(), "get" + capitalizedName, null, null, content));

        if (javaBean) {
            // add full javabean support
            if (Boolean.class.getName().equals(field.getType())) {
                content = String.format(BOOLEAN_GETTER_PATTERN, id);
                addMethod(new JavaMethod(Modifier.PUBLIC, field.getType(), "is" + capitalizedName, null, null, content));
            }
            content = String.format(SETTER_PATTERN, field.getType(), id);
            JavaArgument arg = new JavaArgument(field.getType(), "newValue");
            addMethod(new JavaMethod(Modifier.PUBLIC, "void", "set" + capitalizedName, new JavaArgument[]{arg}, null, content));
        }
    }

    public void addSimpleField(JavaField field) {
        fields.add(field);
    }


    public JavaField[] getFields() {
        return fields.toArray(new JavaField[fields.size()]);
    }


    public static String addIndentation(String source, int indentation, String lineSeparator) {
        return indent(source, indentation, false, lineSeparator);
    }


    public static String setIndentation(String source, int indentation, String lineSeparator) {
        return indent(source, indentation, true, lineSeparator);
    }


    public static String indent(String source, int indentation, boolean trim, String lineSeparator) {
        if (trim) {
            source = source.trim();
        }
        char[] spaces = new char[indentation];
        Arrays.fill(spaces, ' ');
        StringBuffer result = new StringBuffer();
        String[] lines = source.split(System.getProperty("line.separator") + "|\n");
        for (int i = 0; i < lines.length; i++) {
            if (i > 0) {
                result.append(lineSeparator);
            }
            result.append(spaces);
            result.append(trim ? lines[i].trim() : lines[i]);
        }
        return result.toString();
    }


    public void addBodyCode(String bodyCode) {
        rawBodyCode.append(bodyCode);
    }


    public String getClassBody(String lineSeparator) {
        StringBuffer result = new StringBuffer();
        if (fields.size() > 0) {
            java.util.Collections.sort(fields); // sort fields

            for (JavaField field : fields) {
                result.append(addIndentation(field.toString(lineSeparator), 4, lineSeparator));
                result.append(lineSeparator);
            }

            result.append(lineSeparator);
        }

        if (rawBodyCode.length() > 0) {
            result.append(addIndentation("/* begin raw body code */\n", 4, lineSeparator));
            String s = rawBodyCode.toString();
            if (!s.startsWith(lineSeparator)) {
                result.append(lineSeparator);
            }
            result.append(addIndentation(s, 4, lineSeparator));
            result.append(lineSeparator);
            result.append(addIndentation("/* end raw body code */", 4, lineSeparator));
            result.append(lineSeparator);
        }

        for (JavaFile innerClass : innerClasses) {
            result.append(addIndentation(innerClass.toString(), 4, lineSeparator));
            result.append(lineSeparator).append(lineSeparator);
        }

        EnumMap<MethodOrder, List<JavaMethod>> map = JavaMethod.getSortedMethods(methods);
        for (Entry<MethodOrder, List<JavaMethod>> entry : map.entrySet()) {
            List<JavaMethod> list = entry.getValue();
            entry.getKey().generate(result, list, lineSeparator);
            list.clear();
        }
        map.clear();

        return result.toString();
    }


    public String getClassDefinition(String lineSeparator) {
        StringBuffer result = new StringBuffer();
        result.append(getModifiersText(modifiers));
        if (abstractClass) {
            result.append("abstract ");
        }
        result.append("class ");
        result.append(className.substring(className.lastIndexOf(".") + 1));
        if (genericType != null) {
            result.append('<').append(genericType).append('>');
        }
        result.append(" extends ");
        result.append(superClass);
        if (superGenericType != null) {
            result.append('<').append(superGenericType).append('>');
        }
        if (interfaces != null && !interfaces.isEmpty()) {
            result.append(" implements ").append(interfaces.get(0));
            for (int i = 1; i < interfaces.size(); i++) {
                /*if (i > 0) {
                    result.append(", ");
                }*/
                result.append(", ").append(interfaces.get(i));
            }
        }
        result.append(" {");
        result.append(lineSeparator);
        result.append(getClassBody(lineSeparator));
        result.append("}");
        return result.toString();
    }


    public static String getModifiersText(int modifiers) {
        if (modifiers == 0) {
            return "";
        } else {
            return Modifier.toString(modifiers) + ' ';
        }
    }


    /**
     * Returns the Java source code for this class.
     *
     * @param lineSeparator line separator
     * @return a complete Java file for this class
     */
    public String toString(String lineSeparator) {
        StringBuffer result = new StringBuffer();
        if (className.indexOf(".") != -1) {
            result.append("package ").append(className.substring(0, className.lastIndexOf("."))).append(";");
            result.append(lineSeparator);
            result.append(lineSeparator);
        }

        if (imports.size() > 0) {
            for (String anImport : imports) {
                result.append("import ");
                result.append(anImport);
                result.append(';');
                result.append(lineSeparator);
            }
            result.append(lineSeparator);
        }

        result.append(getClassDefinition(lineSeparator));
        return result.toString();
    }

    public void addInterface(String canonicalName) {
        if (interfaces == null || !interfaces.contains(canonicalName)) {
            getInterfaces().add(canonicalName);
        }
    }

    public void addInterfaces(String[] canonicalNames) {
        if (canonicalNames == null) {
            return;
        }
        for (String canonicalName : canonicalNames) {
            if (interfaces == null || !interfaces.contains(canonicalName)) {
                getInterfaces().add(canonicalName);
            }
        }
    }

    public boolean isSuperclassIsJAXXObject() {
        return superclassIsJAXXObject;
    }

    public void setSuperclassIsJAXXObject(boolean superclassIsJAXXObject) {
        this.superclassIsJAXXObject = superclassIsJAXXObject;
    }

    public void setAbstractClass(boolean abstractClass) {
        this.abstractClass = abstractClass;
    }


    public void setSuperGenericType(String superGenericType) {
        this.superGenericType = superGenericType;
    }
}