package jaxx.compiler;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.SortedMap;
import jaxx.compiler.JAXXCompilerLaunchor.LifeCycle;
import org.nuiton.util.StringUtil;

/**
 * Pour profiler les temps d'execution pendant une compilation.
 *
 * @author tony
 * @since 1.3
 */
public class JAXXProfile {

    void clear() {
        entries.clear();
        compilers.clear();
    }

    protected class CompilerEntry {

        JAXXCompiler compiler;
        SortedMap<String, Long> times;

        public CompilerEntry(JAXXCompiler compiler) {
            this.compiler = compiler;
            times = new java.util.TreeMap<String, Long>();
        }
    }

    public static class ProfileResult {

        long min, max, average, total;
        Map<JAXXCompiler, Long> delta;
        List<Long> times;

        ProfileResult(String label, Map<JAXXCompiler, Long> delta) {
            this.delta = delta;
            times = new java.util.ArrayList<Long>(delta.values());
            java.util.Collections.sort(times);
            min = times.get(0);
            max = times.get(times.size() - 1);
            total = 0;
            average = 0;
            for (Long t : times) {
                total += t;
            }
            average = total / times.size();
        }

        public long getTime(JAXXCompiler compiler) {
            for (Entry<JAXXCompiler, Long> entry : delta.entrySet()) {
                if (entry.getKey().equals(compiler)) {
                    return entry.getValue();
                }
            }
            throw new IllegalArgumentException("could not find time for compiler " + compiler);
        }

        public void clear() {
            times.clear();
            delta.clear();
        }

        public JAXXCompiler getCompiler(Long l) {
            for (Entry<JAXXCompiler, Long> entry : delta.entrySet()) {
                if (entry.getValue().equals(l)) {
                    return entry.getKey();
                }
            }
            throw new IllegalArgumentException("could not find compiler for time " + l);
        }
    }
    SortedMap<Integer, CompilerEntry> entries;
    List<JAXXCompiler> compilers;

    public JAXXProfile() {
        compilers = new java.util.ArrayList<JAXXCompiler>();
        entries = new java.util.TreeMap<Integer, CompilerEntry>();
    }

    public void addTime(JAXXCompiler compiler, String key) {
        CompilerEntry e = getEntry(compiler);
        e.times.put(key, System.nanoTime());
    }

    public Map<JAXXCompiler, Long> getDelta(String label, String keyOne, String keyTwo) {
        Map<JAXXCompiler, Long> result = new java.util.HashMap<JAXXCompiler, Long>();
        for (java.util.Map.Entry<Integer, CompilerEntry> e : entries.entrySet()) {
            JAXXCompiler c = getCompiler(e.getKey());
            CompilerEntry entry = e.getValue();
            Long t0 = entry.times.get(keyOne);
            Long t1 = entry.times.get(keyTwo);
            if (t0 == null) {
                throw new NullPointerException("could not find time for " + keyOne + " on compiler " + c.getOutputClassName());
            }
            if (t1 == null) {
                throw new NullPointerException("could not find time for " + keyTwo + " on compiler " + c.getOutputClassName());
            }
            long delta = t1 - t0;
            result.put(c, delta);
        }
        return result;
    }

    public ProfileResult newProfileResult(LifeCycle step) {
        ProfileResult result;
        String name = step.name();
        Map<JAXXCompiler, Long> delta = getDelta(name, name + "_start", name + "_end");
        result = new ProfileResult(name, delta);
        return result;
    }

    public ProfileResult newProfileResult(ProfileResult... toCumul) {
        ProfileResult result;
        Map<JAXXCompiler, Long> delta = new java.util.HashMap<JAXXCompiler, Long>();
        for (JAXXCompiler c : compilers) {
            long total = 0;
            for (ProfileResult cumul : toCumul) {
                long time = cumul.getTime(c);
                total += time;
            }
            delta.put(c, total);
        }
        result = new ProfileResult("all", delta);
        return result;
    }

    public StringBuilder computeProfileReport() {

        StringBuilder buffer = new StringBuilder();

        if (compilers.isEmpty()) {
            return buffer.append("no jaxx file treated, no profile report");
        }

        // compute max size of the fqn of a compiled file
        int maxLength = 0;
        for (JAXXCompiler compiler : compilers) {
            int l = compiler.getOutputClassName().length();
            if (l > maxLength) {
                maxLength = l;
            }
        }

        ProfileResult cfp = newProfileResult(LifeCycle.compile_first_pass);
        ProfileResult csp = newProfileResult(LifeCycle.compile_second_pass);
        ProfileResult ssp = newProfileResult(LifeCycle.stylesheet_pass);
        ProfileResult gp = newProfileResult(LifeCycle.generate_pass);
        ProfileResult total = newProfileResult(cfp, csp, ssp, gp);

        String reportPattern = "\n|%1$-" + maxLength + "s|%2$15s|%3$15s|%4$15s|%5$15s|%6$15s|";

        char[] tmpC = new char[maxLength];
        Arrays.fill(tmpC, '-');
        String line = String.format(reportPattern,
                new String(tmpC),
                "---------------",
                "---------------",
                "---------------",
                "---------------",
                "---------------");

        buffer.append(line);

        buffer.append(String.format(reportPattern,
                "(files / stats) \\ passes",
                "compile round 1", "compile round 2", "stylesheet", "generation", "all passes"));

        buffer.append(line);

        // affiche les temps de tous les fichiers en temp total croissant
        for (Long l : total.times) {
            JAXXCompiler c = total.getCompiler(l);
            printReportLine(buffer, reportPattern, c.getOutputClassName(), cfp.getTime(c), csp.getTime(c), ssp.getTime(c), gp.getTime(c), total.getTime(c));
        }

        buffer.append(line);

        if (compilers.size() > 1) {
            printReportLine(buffer, reportPattern, "total (" + compilers.size() + " files)", cfp.total, csp.total, ssp.total, gp.total, total.total);

            buffer.append(line);

            printReportLine2(buffer, reportPattern, "min", cfp.min, csp.min, ssp.min, gp.min, total.min);
            printReportLine2(buffer, reportPattern, "max", cfp.max, csp.max, ssp.max, gp.max, total.max);
            printReportLine(buffer, reportPattern, "average", cfp.average, csp.average, ssp.average, gp.average, total.average);
            buffer.append(line);
        }
        cfp.clear();
        csp.clear();
        ssp.clear();
        gp.clear();
        total.clear();

        return buffer;
    }
    public static final String TIME_PATTERN = "%1$9s - %2$2d%%";

    protected void printReportLine(StringBuilder buffer, String reportPattern, String label, long firstPassCounter, long secondPassCounter, long cssCounter, long generatorCounter, long totalCounter) {
        float percentCFP = ((float) firstPassCounter / totalCounter) * 100;
        float percentCSP = ((float) secondPassCounter / totalCounter) * 100;
        float percentCSSP = ((float) cssCounter / totalCounter) * 100;
        float percentGP = ((float) generatorCounter / totalCounter) * 100;
        String strCFP = String.format(TIME_PATTERN, StringUtil.convertTime(firstPassCounter), (int) percentCFP);
        String strCSP = String.format(TIME_PATTERN, StringUtil.convertTime(secondPassCounter), (int) percentCSP);
        String strCSSP = String.format(TIME_PATTERN, StringUtil.convertTime(cssCounter), (int) percentCSSP);
        String strGP = String.format(TIME_PATTERN, StringUtil.convertTime(generatorCounter), (int) percentGP);

        buffer.append(String.format(reportPattern, label, strCFP, strCSP, strCSSP, strGP, StringUtil.convertTime(totalCounter)));
    }

    protected void printReportLine2(StringBuilder buffer, String reportPattern, String label, long firstPassCounter, long secondPassCounter, long cssCounter, long generatorCounter, long totalCounter) {
        buffer.append(String.format(reportPattern, label, StringUtil.convertTime(firstPassCounter), StringUtil.convertTime(secondPassCounter), StringUtil.convertTime(cssCounter), StringUtil.convertTime(generatorCounter), StringUtil.convertTime(totalCounter)));
    }

    protected CompilerEntry getEntry(JAXXCompiler compiler) {
        int key = compiler.getOutputClassName().hashCode();
        CompilerEntry result = entries.get(key);
        if (result == null) {
            result = new CompilerEntry(compiler);
            entries.put(key, result);
            compilers.add(compiler);
        }
        return result;
    }

    protected JAXXCompiler getCompiler(int hasCode) {
        for (JAXXCompiler c : compilers) {
            if (hasCode == c.getOutputClassName().hashCode()) {
                return c;
            }
        }
        return null;
    }
}
