package jaxx.compiler;

import jaxx.Base64Coder;
import jaxx.CompilerException;
import jaxx.reflect.ClassDescriptor;
import jaxx.reflect.ClassDescriptorLoader;
import jaxx.reflect.FieldDescriptor;
import jaxx.reflect.MethodDescriptor;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.JAXXObjectDescriptor;
import jaxx.types.TypeManager;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.IOException;
import java.lang.reflect.Modifier;
import static java.lang.reflect.Modifier.FINAL;
import static java.lang.reflect.Modifier.PROTECTED;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * This class is a refactoring of the {@link jaxx.compiler.JAXXCompiler}.
 * <p/>
 * We delegate now the generation of a {@link jaxx.runtime.JAXXObject} to this class, the
 * {@link jaxx.compiler.JAXXCompiler} now only deals with the compilation of files.
 *
 * @author chemit
 */
public class JAXXObjectGenerator implements Generator {

    /** log */
    protected static final Log log = LogFactory.getLog(JAXXObjectGenerator.class);
    protected static final JavaField ACTIVE_BINDINGS_FIELD = JavaField.newField(PROTECTED,
            "java.util.List<Object>", "$activeBindings", "new ArrayList<Object>()");
    protected static final JavaField BINDING_SOURCES_FIELD = JavaField.newField(PROTECTED,
            "java.util.Map<String,Object>", "$bindingSources", "new HashMap<String,Object>()");
    protected static final JavaField OBJECT_MAP_FIELD = JavaField.newField(PROTECTED,
            "Map<String,Object>", "$objectMap", "new HashMap<String,Object>()");
    protected static final JavaField ALL_COMPONENTS_CREATED_FIELD = JavaField.newField(java.lang.reflect.Modifier.PRIVATE,
            "boolean", "allComponentsCreated");
    protected static final JavaField CONTEXT_INITIALIZED = JavaField.newField(java.lang.reflect.Modifier.PRIVATE,
            "boolean", "contextInitialized", "true");
    protected static final JavaField PREVIOUS_VALUES_FIELD = JavaField.newField(0,
            "java.util.Map", "$previousValues", "new java.util.HashMap()");
    protected static final JavaField DELEGATE_CONTEXT_FIELD = JavaField.newField(PROTECTED,
            "jaxx.runtime.JAXXContext", "delegateContext");
    protected static final JavaField PROPERTY_CHANGE_SUPPORT_FIELD = JavaField.newField(0,
            "java.beans.PropertyChangeSupport", "$propertyChangeSupport");
    protected static final JavaMethod GET_CONTEXT_VALUE_METHOD = JavaMethod.newMethod(java.lang.reflect.Modifier.PUBLIC, "<T> T", "getContextValue",
            "return delegateContext.getContextValue(clazz, null);",
            new JavaArgument("Class<T>", "clazz"));
    protected static final JavaMethod GET_CONTEXT_VALUE_NAMED_METHOD = JavaMethod.newMethod(java.lang.reflect.Modifier.PUBLIC, "<T> T", "getContextValue",
            "return delegateContext.getContextValue(clazz, name);",
            new JavaArgument("Class<T>", "clazz"), new JavaArgument("String", "name"));
    protected static final JavaMethod SET_CONTEXT_VALUE_NAMED_METHOD = JavaMethod.newMethod(java.lang.reflect.Modifier.PUBLIC, "<T> void", "setContextValue",
            "delegateContext.setContextValue(o, name);",
            new JavaArgument("T", "o"), new JavaArgument("String", "name"));
    protected static final JavaMethod SET_CONTEXT_VALUE_METHOD = JavaMethod.newMethod(java.lang.reflect.Modifier.PUBLIC, "<T> void", "setContextValue",
            "delegateContext.setContextValue(o, null);",
            new JavaArgument("T", "o"));
    protected static final JavaMethod REMOVE_CONTEXT_VALUE_NAMED_METHOD = JavaMethod.newMethod(java.lang.reflect.Modifier.PUBLIC, "<T> void", "removeContextValue",
            "delegateContext.removeContextValue(clazz, name);",
            new JavaArgument("Class<T>", "clazz"), new JavaArgument("String", "name"));
    protected static final JavaMethod REMOVE_CONTEXT_VALUE_METHOD = JavaMethod.newMethod(java.lang.reflect.Modifier.PUBLIC, "<T> void", "removeContextValue",
            "delegateContext.removeContextValue(clazz, null);",
            new JavaArgument("Class<T>", "clazz"));
    protected static final JavaMethod GET_PARENT_CONTAINER_MORE_METHOD = JavaMethod.newMethod(java.lang.reflect.Modifier.PUBLIC, "<O extends Container> O", "getParentContainer",
            "return delegateContext.getParentContainer(source, clazz);",
            new JavaArgument("Object", "source"), new JavaArgument("Class<O>", "clazz"));
    protected static final JavaMethod GET_PARENT_CONTAINER_METHOD = JavaMethod.newMethod(java.lang.reflect.Modifier.PUBLIC, "<O extends Container> O", "getParentContainer",
            "return delegateContext.getParentContainer(clazz);",
            new JavaArgument("Class<O>", "clazz"));
    protected static final JavaMethod GET_OBJECT_BY_ID_METHOD = JavaMethod.newMethod(java.lang.reflect.Modifier.PUBLIC, "java.lang.Object", "getObjectById",
            "return $objectMap.get(id);",
            new JavaArgument("String", "id"));
    protected static final JavaMethod GET_JAXX_OBJECT_DESCRIPTOR_METHOD = JavaMethod.newMethod(java.lang.reflect.Modifier.PUBLIC | java.lang.reflect.Modifier.STATIC, "jaxx.runtime.JAXXObjectDescriptor", "$getJAXXObjectDescriptor",
            "return jaxx.runtime.Util.decodeCompressedJAXXObjectDescriptor($jaxxObjectDescriptor);");
    protected static final JavaMethod PROCESS_DATA_BINDING_METHOD = JavaMethod.newMethod(java.lang.reflect.Modifier.PUBLIC, "void", "processDataBinding",
            "processDataBinding(dest, false);",
            new JavaArgument("String", "dest"));
    protected static final JavaMethod FIRE_PROPERTY_CHANGE_METHOD = JavaMethod.newMethod(java.lang.reflect.Modifier.PUBLIC, "void", "firePropertyChange",
            "super.firePropertyChange(propertyName, oldValue, newValue);",
            new JavaArgument("String", "propertyName"), new JavaArgument("Object", "oldValue"), new JavaArgument("Object", "newValue"));
    protected static final JavaMethod FIRE_PROPERTY_CHANGE_NAMED_METHOD = JavaMethod.newMethod(java.lang.reflect.Modifier.PUBLIC, "void", "firePropertyChange",
            "$getPropertyChangeSupport().firePropertyChange(propertyName, oldValue, newValue);",
            new JavaArgument("String", "propertyName"), new JavaArgument("Object", "oldValue"), new JavaArgument("Object", "newValue"));
    protected static final JavaMethod GET_PROPERTY_CHANGE_SUPPORT_METHOD = JavaMethod.newMethod(0, "java.beans.PropertyChangeSupport", "$getPropertyChangeSupport",
            "if ($propertyChangeSupport == null)\n" +
            "    $propertyChangeSupport = new PropertyChangeSupport(this);\n" +
            "return $propertyChangeSupport;");
    protected static final JavaMethod ADD_PROPERTY_CHANGE_SUPPORT_METHOD = JavaMethod.newMethod(java.lang.reflect.Modifier.PUBLIC, "void", "addPropertyChangeListener",
            "$getPropertyChangeSupport().addPropertyChangeListener(listener);",
            new JavaArgument("java.beans.PropertyChangeListener", "listener"));
    protected static final JavaMethod ADD_PROPERTY_CHANGE_SUPPORT_NAMED_METHOD = JavaMethod.newMethod(java.lang.reflect.Modifier.PUBLIC, "void", "addPropertyChangeListener",
            "$getPropertyChangeSupport().addPropertyChangeListener(property, listener);",
            new JavaArgument("String", "property"), new JavaArgument("java.beans.PropertyChangeListener", "listener"));
    protected static final JavaMethod REMOVE_PROPERTY_CHANGE_SUPPORT_METHOD = JavaMethod.newMethod(java.lang.reflect.Modifier.PUBLIC, "void", "removePropertyChangeListener",
            "$getPropertyChangeSupport().removePropertyChangeListener(listener);",
            new JavaArgument("java.beans.PropertyChangeListener", "listener"));
    protected static final JavaMethod REMOVE_PROPERTY_CHANGE_SUPPORT_NAMED_METHOD = JavaMethod.newMethod(java.lang.reflect.Modifier.PUBLIC, "void", "removePropertyChangeListener",
            "$getPropertyChangeSupport().removePropertyChangeListener(property, listener);",
            new JavaArgument("String", "property"), new JavaArgument("java.beans.PropertyChangeListener", "listener"));

    @Override
    public void finalizeCompiler(CompiledObject root, JAXXCompiler compiler, JavaFile javaFile, String packageName, String className) {

        String fullClassName = packageName != null ? packageName + "." + className : className;
        if (root == null) {
            throw new CompilerException("root tag must be a class tag");
        }
        //Map<String, CompiledObject> objects = compiler.getObjects();
        ClassDescriptor superclass = root.getObjectClass();
        boolean superclassIsJAXXObject = ClassDescriptorLoader.getClassDescriptor(JAXXObject.class).isAssignableFrom(superclass);
        javaFile.setModifiers(Modifier.PUBLIC);
        javaFile.setClassName(fullClassName);
        javaFile.setSuperClass(JAXXCompiler.getCanonicalName(superclass));
        javaFile.setSuperclassIsJAXXObject(superclassIsJAXXObject);

        javaFile.addInterfaces(compiler.getExtraInterfaces());
        javaFile.setAbstractClass(compiler.isAbstractClass());
        javaFile.setGenericType(compiler.getGenericType());
        javaFile.setSuperGenericType(compiler.getSuperGenericType());

        for (CompiledObject object : compiler.getObjects().values()) {
            CompiledObjectDecorator decorator = object.getDecorator();
            decorator.finalizeCompiler(compiler, root, object, javaFile, packageName, className, fullClassName);

            /*if (!object.isOverride() && !(object instanceof ScriptInitializer)) {
            String id = object.getId();
            int access = id.startsWith("$") ? Modifier.PRIVATE : Modifier.PROTECTED;
            if (object == root) {
            javaFile.addField(new JavaField(access, fullClassName, id, "this"));
            } else {
            //TC -20081017 can have generic on compiled Object
            javaFile.addField(JavaField.newField(access, JAXXCompiler.getCanonicalName(object), id), object.isJavaBean());
            }
            }

            if (!compiler.inlineCreation(object) && object != root) {
            javaFile.addMethod(JavaMethod.newMethod(Modifier.PROTECTED, "void", object.getCreationMethodName(), getCreationCode(compiler, object)));
            }*/
        }

        // DataBinding
        for (DataBinding dataBinding : compiler.getDataBindings()) {
            if (dataBinding.compile(true)) {
                compiler.getInitDataBindings().append("applyDataBinding(").append(TypeManager.getJavaCode(dataBinding.getId())).append(");").append(JAXXCompiler.getLineSeparator());
            }
        }

        if (superclassIsJAXXObject) {
            boolean hasBind = compiler.getApplyDataBinding().length() > 0;
            if (hasBind) {
                compiler.appendApplyDataBinding(" else {");
                compiler.appendApplyDataBinding(JAXXCompiler.getLineSeparator());
                compiler.appendApplyDataBinding("    ");
            }
            compiler.appendApplyDataBinding("super.applyDataBinding($binding);");
            compiler.appendApplyDataBinding(JAXXCompiler.getLineSeparator());

            if (hasBind) {
                compiler.appendApplyDataBinding("    return;");
                compiler.appendApplyDataBinding(JAXXCompiler.getLineSeparator());
                compiler.appendApplyDataBinding("}");
            }


            hasBind = compiler.getRemoveDataBinding().length() > 0;
            if (hasBind) {
                compiler.appendRemoveDataBinding(" else {");
                compiler.appendRemoveDataBinding(JAXXCompiler.getLineSeparator());
                compiler.appendRemoveDataBinding("    ");
            }
            compiler.appendRemoveDataBinding("super.removeDataBinding($binding);");
            compiler.appendRemoveDataBinding(JAXXCompiler.getLineSeparator());

            if (hasBind) {
                compiler.appendRemoveDataBinding("}");
            }
        } else {
            javaFile.addInterface(JAXXCompiler.getCanonicalName(JAXXObject.class));
        }
    }

    @Override
    public void prepareJavaFile(CompiledObject root, JAXXCompiler compiler, JavaFile javaFile, String packageName, String className) throws ClassNotFoundException {

        String fullClassName = javaFile.getClassName();

        String jaxxContextImplementorClass = compiler.getOptions().getJaxxContextImplementorClass();
        boolean superclassIsJAXXObject = javaFile.isSuperclassIsJAXXObject();
        if (!superclassIsJAXXObject) {
            // add logger
            if (compiler.getOptions().isAddLogger()) {
                javaFile.addImport(Log.class);
                javaFile.addImport(LogFactory.class);
                javaFile.addField(JavaField.newField(Modifier.PUBLIC + Modifier.STATIC + FINAL, "Log", "log", "LogFactory.getLog(" + fullClassName + ".class)"));
            }

            // JAXXObject
            javaFile.addField(OBJECT_MAP_FIELD);
            javaFile.addMethod(GET_OBJECT_BY_ID_METHOD);
            javaFile.addField(BINDING_SOURCES_FIELD);
            javaFile.addField(ACTIVE_BINDINGS_FIELD);

            // JAXXContext
            javaFile.addField(JavaField.newField(PROTECTED | FINAL, JAXXContext.class.getName(), "delegateContext", "new " + jaxxContextImplementorClass + "(this);"));
            javaFile.addMethod(SET_CONTEXT_VALUE_METHOD);
            javaFile.addMethod(SET_CONTEXT_VALUE_NAMED_METHOD);
            javaFile.addMethod(GET_CONTEXT_VALUE_METHOD);
            javaFile.addMethod(GET_CONTEXT_VALUE_NAMED_METHOD);
            javaFile.addMethod(REMOVE_CONTEXT_VALUE_METHOD);
            javaFile.addMethod(REMOVE_CONTEXT_VALUE_NAMED_METHOD);
            javaFile.addMethod(GET_PARENT_CONTAINER_METHOD);
            javaFile.addMethod(GET_PARENT_CONTAINER_MORE_METHOD);

            // PropertyChangeSupport
            addPropertyChangeSupport(root, javaFile);

            // DataBinding
            javaFile.addMethod(PROCESS_DATA_BINDING_METHOD);
        }

        javaFile.addField(ALL_COMPONENTS_CREATED_FIELD);
        boolean overrideContextInitialized = false;
        FieldDescriptor[] scriptFields = compiler.getScriptFields();
        for (FieldDescriptor f : scriptFields) {
            if ("contextInitialized".equals(f.getName())) {
                overrideContextInitialized = true;
                break;
            }
        }
        if (!overrideContextInitialized) {
            javaFile.addField(CONTEXT_INITIALIZED);
        }
        javaFile.addField(createJAXXObjectDescriptorField(compiler, javaFile));

        if (compiler.getStylesheet() != null) {
            javaFile.addField(PREVIOUS_VALUES_FIELD);
        }
        /*for (CompiledObject object : compiler.getObjects().values()) {
        List<CompiledObject.ChildRef> refList = object.getChilds();
        if (refList==null || refList.isEmpty()) {
        continue;
        }
        for (ChildRef childRef : refList) {
        childRef.addToAdditionCode(buffer);
        }
        }*/
        //TC 20090228 - only generate constructors if not done in scripts
        boolean constructorDetected = false;
        MethodDescriptor[] methods = compiler.getScriptMethods();
        for (MethodDescriptor m : methods) {
            try {
                m.getReturnType();
                if (className.equals(m.getName())) {
                    constructorDetected = true;
                    break;
                }
            } catch (Exception e) {
                log.warn("could not find return type " + m);
            }
        }
        if (!constructorDetected) {
            javaFile.addMethod(createConstructor(compiler, className, superclassIsJAXXObject));
            javaFile.addMethod(createConstructorWithInitialContext(compiler, className, superclassIsJAXXObject));
        }

        javaFile.addMethod(createInitializer(compiler));
        javaFile.addMethod(GET_JAXX_OBJECT_DESCRIPTOR_METHOD);

        javaFile.addBodyCode(compiler.getBodyCode().toString());

        javaFile.addMethod(createCompleteSetupMethod(compiler, javaFile, compiler.getInitDataBindings()));


        javaFile.addMethod(JavaMethod.newMethod(Modifier.PUBLIC, "void", "applyDataBinding",
                compiler.getApplyDataBinding().toString() + JAXXCompiler.getLineSeparator() + "processDataBinding($binding);",
                new JavaArgument("String", "$binding")));

        javaFile.addMethod(JavaMethod.newMethod(Modifier.PUBLIC, "void", "removeDataBinding",
                compiler.getRemoveDataBinding().toString(), new JavaArgument("String", "$binding")));

        javaFile.addMethod(createProcessDataBindingMethod(compiler, superclassIsJAXXObject));

        addEventHandlers(compiler, javaFile);

    }


    /*---------------------------------------------------------------------------------*/
    /*-- Create fields ----------------------------------------------------------------*/
    /*---------------------------------------------------------------------------------*/
    protected JavaField createJAXXObjectDescriptorField(JAXXCompiler compiler, JavaFile javaFile) {
        try {
            JAXXObjectDescriptor descriptor = compiler.getJAXXObjectDescriptor();
            String data = Base64Coder.serialize(descriptor, true);
            /*ByteArrayOutputStream buffer = new ByteArrayOutputStream();
            ObjectOutputStream out = new ObjectOutputStream(new GZIPOutputStream(buffer));
            out.writeObject(descriptor);
            out.close();
            // the use of the weird deprecated constructor is deliberate -- we need to store the data as a String
            // in the compiled class file, since byte array initialization is horribly inefficient compared to
            // String initialization.  So we store the bytes in the String, and we quite explicitly want a 1:1
            // mapping between bytes and chars, with the high byte of the char set to zero.  We can then safely
            // reconstitute the original byte[] at a later date.  This is unquestionably an abuse of the String
            // type, but if we could efficiently store a byte[] we wouldn't have to do this.
            String data = new String(buffer.toByteArray(), 0);*/

            int sizeLimit = 65000; // constant strings are limited to 64K, and I'm not brave enough to push right up to the limit
            if (data.length() < sizeLimit) {
                return JavaField.newField(Modifier.PRIVATE | Modifier.STATIC, "java.lang.String", "$jaxxObjectDescriptor", TypeManager.getJavaCode(data));
            } else {
                StringBuffer initializer = new StringBuffer();
                for (int i = 0; i < data.length(); i += sizeLimit) {
                    String name = "$jaxxObjectDescriptor" + i;
                    javaFile.addField(new JavaField(Modifier.PRIVATE | Modifier.STATIC, "java.lang.String", name,
                            TypeManager.getJavaCode(data.substring(i, Math.min(i + sizeLimit, data.length())))));
                    if (initializer.length() > 0) {
                        initializer.append(" + ");
                    }
                    initializer.append("String.valueOf(").append(name).append(")");
                }
                return JavaField.newField(Modifier.PRIVATE | Modifier.STATIC, "java.lang.String", "$jaxxObjectDescriptor", initializer.toString());
            }
        } catch (IOException e) {
            throw new RuntimeException("Internal error: can't-happen error", e);
        }
    }

    /*---------------------------------------------------------------------------------*/
    /*-- Create methods ---------------------------------------------------------------*/
    /*---------------------------------------------------------------------------------*/
    protected void addPropertyChangeSupport(CompiledObject root, JavaFile javaFile) {
        ClassDescriptor currentClass = root.getObjectClass();
        MethodDescriptor firePropertyChange = null;
        while (firePropertyChange == null && currentClass != null) {
            try {
                firePropertyChange = currentClass.getDeclaredMethodDescriptor("firePropertyChange", ClassDescriptorLoader.getClassDescriptor(String.class),
                        ClassDescriptorLoader.getClassDescriptor(Object.class),
                        ClassDescriptorLoader.getClassDescriptor(Object.class));

            } catch (NoSuchMethodException e) {
                currentClass = currentClass.getSuperclass();
            }
        }

        int modifiers = firePropertyChange != null ? firePropertyChange.getModifiers() : 0;
        if (Modifier.isPublic(modifiers)) {
            // we have all the support we need
        }
        if (Modifier.isProtected(modifiers)) {
            // there is property change support but the firePropertyChange method is protected
            javaFile.addMethod(FIRE_PROPERTY_CHANGE_METHOD);
        } else {
            // either no support at all or firePropertyChange isn't accessible
            javaFile.addField(PROPERTY_CHANGE_SUPPORT_FIELD);
            javaFile.addMethod(GET_PROPERTY_CHANGE_SUPPORT_METHOD);
            javaFile.addMethod(ADD_PROPERTY_CHANGE_SUPPORT_METHOD);
            javaFile.addMethod(ADD_PROPERTY_CHANGE_SUPPORT_NAMED_METHOD);
            javaFile.addMethod(REMOVE_PROPERTY_CHANGE_SUPPORT_METHOD);
            javaFile.addMethod(REMOVE_PROPERTY_CHANGE_SUPPORT_NAMED_METHOD);
            javaFile.addMethod(FIRE_PROPERTY_CHANGE_NAMED_METHOD);
        }
    }

    protected void addEventHandlers(JAXXCompiler compiler, JavaFile javaFile) {
        for (Map.Entry<String, Map<ClassDescriptor, List<EventHandler>>> e1 : compiler.getEventHandlers().entrySet()) {
            // outer loop is iterating over different objects (well, technically, different Java expressions)
            for (Map.Entry<ClassDescriptor, List<EventHandler>> e2 : e1.getValue().entrySet()) {
                // iterate over different types of listeners for this particular object (MouseListener, ComponentListener, etc.)
                for (EventHandler handler : e2.getValue()) {
                    // iterate over individual event handlers of a single type
                    String methodName = compiler.getEventHandlerMethodName(handler);
                    MethodDescriptor listenerMethod = handler.getListenerMethod();
                    if (listenerMethod.getParameterTypes().length != 1) {
                        throw new CompilerException("Expected event handler " + listenerMethod.getName() + " of class " + handler.getListenerClass() + " to have exactly one argument");
                    }
                    javaFile.addMethod(JavaMethod.newMethod(Modifier.PUBLIC, "void", methodName, handler.getJavaCode(),
                            new JavaArgument(JAXXCompiler.getCanonicalName(listenerMethod.getParameterTypes()[0]), "event")));
                }
            }
        }
    }

    protected JavaMethod createConstructor(JAXXCompiler compiler, String className, boolean superclassIsJAXXObject) throws CompilerException {
        StringBuffer code = new StringBuffer();
        String constructorParams = compiler.getRootObject().getConstructorParams();
        if (constructorParams != null) {
            code.append("        super(").append(constructorParams).append(");").append(JAXXCompiler.getLineSeparator());
        } else {
            if (superclassIsJAXXObject) {
                code.append("        super();").append(JAXXCompiler.getLineSeparator());
            }
        }
        code.append("$initialize();");
        code.append(JAXXCompiler.getLineSeparator());
        return JavaMethod.newMethod(Modifier.PUBLIC, null, className, code.toString());
    }

    protected JavaMethod createConstructorWithInitialContext(JAXXCompiler compiler, String className, boolean superclassIsJAXXObject) throws CompilerException {
        StringBuffer code = new StringBuffer();
        String constructorParams = compiler.getRootObject().getConstructorParams();
        if (constructorParams != null) {
            code.append("        super(").append(constructorParams).append(");").append(JAXXCompiler.getLineSeparator());
        } else {
            if (superclassIsJAXXObject) {
                code.append("        super(parentContext);").append(JAXXCompiler.getLineSeparator());
            }
        }
        if (!superclassIsJAXXObject) {
            code.append("if (parentContext instanceof jaxx.runtime.JAXXInitialContext) {");
            code.append(JAXXCompiler.getLineSeparator());
            code.append("    ((jaxx.runtime.JAXXInitialContext)parentContext).to(this);");
            code.append(JAXXCompiler.getLineSeparator());
            code.append("} else {");
            code.append(JAXXCompiler.getLineSeparator());
            code.append("    setContextValue(parentContext);");
            code.append(JAXXCompiler.getLineSeparator());
            code.append("}");
            code.append(JAXXCompiler.getLineSeparator());
        }
        code.append("$initialize();");
        code.append(JAXXCompiler.getLineSeparator());
        return JavaMethod.newMethod(Modifier.PUBLIC, null, className, code.toString(), new JavaArgument("jaxx.runtime.JAXXContext", "parentContext"));
    }

    public JavaMethod createInitializer(JAXXCompiler compiler) throws CompilerException {
        StringBuffer code = new StringBuffer();
        CompiledObject root = compiler.getRootObject();
        code.append("if (allComponentsCreated || !contextInitialized) {");
        code.append(JAXXCompiler.getLineSeparator());
        code.append("    return;");
        code.append(JAXXCompiler.getLineSeparator());
        code.append("}");
        code.append(JAXXCompiler.getLineSeparator());
        code.append("$objectMap.put(").append(TypeManager.getJavaCode(root.getId())).append(", this);");
        code.append(JAXXCompiler.getLineSeparator());

        Iterator<CompiledObject> i = compiler.getObjectCreationOrder();
        boolean lastWasMethodCall = false;
        while (i.hasNext()) {
            CompiledObject object = i.next();
            if (object == root) {
                continue;
            }
            CompiledObjectDecorator decorator = object.getDecorator();
            lastWasMethodCall = decorator.createInitializer(compiler, root, object, code, lastWasMethodCall);
            /*if (object != root && !object.isOverride()) {
            if (compiler.inlineCreation(object)) {
            if (lastWasMethodCall) {
            lastWasMethodCall = false;
            code.append(JAXXCompiler.getLineSeparator());
            }
            code.append(getCreationCode(compiler, object));
            code.append(JAXXCompiler.getLineSeparator());
            } else {
            code.append(object.getCreationMethodName()).append("();");
            code.append(JAXXCompiler.getLineSeparator());
            lastWasMethodCall = true;
            }
            }*/
        }
        root.getDecorator().createInitializer(compiler, root, root, code, lastWasMethodCall);
        /*String rootCode = root.getInitializationCode(compiler);
        if (rootCode != null && rootCode.length() > 0) {
        code.append(rootCode);
        code.append(JAXXCompiler.getLineSeparator());
        }*/
        code.append(JAXXCompiler.getLineSeparator());
        if (compiler.getInitializer().length() > 0) {
            code.append(compiler.getInitializer());
            code.append(JAXXCompiler.getLineSeparator());
        }
        code.append("$completeSetup();");
        code.append(JAXXCompiler.getLineSeparator());
        return JavaMethod.newMethod(Modifier.PRIVATE, "void", "$initialize", code.toString());
    }

    protected JavaMethod createCompleteSetupMethod(JAXXCompiler compiler, JavaFile javaFile, StringBuffer initDataBindings) {
        StringBuffer code = new StringBuffer();
        code.append("allComponentsCreated = true;");
        code.append(JAXXCompiler.getLineSeparator());
        for (CompiledObject object : compiler.getObjects().values()) {
            CompiledObjectDecorator decorator = object.getDecorator();
            code.append(decorator.createCompleteSetupMethod(compiler, object, javaFile, initDataBindings));

            /*//TC - 20081017 only generate the method if not empty ?
            if (object.getId().startsWith("$")) {
            code.append(object.getAdditionCode()).append(JAXXCompiler.getLineSeparator());
            } else {
            String additionCode = object.getAdditionCode();
            if (additionCode.length() > 0) {
            code.append(object.getAdditionMethodName()).append("();").append(JAXXCompiler.getLineSeparator());
            additionCode = "if (!allComponentsCreated) {" + JAXXCompiler.getLineSeparator() + "    return;" + JAXXCompiler.getLineSeparator() + "}" + JAXXCompiler.getLineSeparator() + additionCode;
            javaFile.addMethod(JavaMethod.newMethod(Modifier.PROTECTED, "void", object.getAdditionMethodName(), additionCode));
            }
            }*/
            //code.append(getLineSeparator());
        }

        code.append(initDataBindings);

        if (compiler.getLateInitializer().length() > 0) {
            code.append(compiler.getLateInitializer());
            code.append(JAXXCompiler.getLineSeparator());
        }
        //TC-20090313 add an extra method after complete setup
        MethodDescriptor method = compiler.getScriptMethod("$afterCompleteSetup");
        if (method != null) {
            code.append("$afterCompleteSetup();").append(JAXXCompiler.getLineSeparator());
        }
        return JavaMethod.newMethod(Modifier.PRIVATE, "void", "$completeSetup", code.toString());
    }

    protected JavaMethod createProcessDataBindingMethod(JAXXCompiler compiler, boolean superclassIsJAXXObject) {
        StringBuffer code = new StringBuffer();
        //boolean superclassIsJAXXObject = ClassDescriptorLoader.getClassDescriptor(JAXXObject.class).isAssignableFrom(compiler.getRootObject().getObjectClass());
        // the force parameter forces the update to happen even if it is already in activeBindings.  This
        // is used on superclass invocations b/c by the time the call gets to the superclass, it is already
        // marked active and would otherwise be skipped
        if (compiler.getProcessDataBinding().length() > 0) {
            code.append("    if (!$force && $activeBindings.contains($dest)) { ");
            code.append(JAXXCompiler.getLineSeparator());
            code.append("    return;");
            code.append(JAXXCompiler.getLineSeparator());
            code.append("}");
            code.append(JAXXCompiler.getLineSeparator());
            code.append("$activeBindings.add($dest);");
            code.append(JAXXCompiler.getLineSeparator());
            code.append("try {");
            code.append(JAXXCompiler.getLineSeparator());
            if (compiler.getProcessDataBinding().length() > 0) {
                code.append(compiler.getProcessDataBinding().toString());
                //code.append(JAXXCompiler.getLineSeparator());
            }
            if (superclassIsJAXXObject) {
                code.append(" else {");
                code.append(JAXXCompiler.getLineSeparator());
                code.append("        super.processDataBinding($dest, true);");
                code.append(JAXXCompiler.getLineSeparator());
                code.append("    }");
                code.append(JAXXCompiler.getLineSeparator());
            }
            code.append("} finally {");
            code.append(JAXXCompiler.getLineSeparator());
            code.append("    $activeBindings.remove($dest);");
            code.append(JAXXCompiler.getLineSeparator());
            code.append("}");
            code.append(JAXXCompiler.getLineSeparator());
        } else if (superclassIsJAXXObject) {
            code.append("super.processDataBinding($dest, true);");
            code.append(JAXXCompiler.getLineSeparator());
        }
        return JavaMethod.newMethod(Modifier.PUBLIC, "void", "processDataBinding", code.toString(),
                new JavaArgument("String", "$dest"), new JavaArgument("boolean", "$force"));
    }

    /*---------------------------------------------------------------------------------*/
    /*-- Create methods code ----------------------------------------------------------*/
    /*---------------------------------------------------------------------------------*/

    /*    protected String getCreationCode(JAXXCompiler compiler, CompiledObject object) throws CompilerException {
    if (object instanceof ScriptInitializer) {
    return object.getInitializationCode(compiler);
    }
    CompiledObjectDecorator decorator = object.getDecorator();
    String result = decorator.getCreationCode(compiler, object);
    return result;*/
    /*StringBuffer result = new StringBuffer();
    result.append(object.getId());
    result.append(" = ");
    if (object.isJavaBean() && object.getJavaBeanInitCode() != null) {
    result.append(object.getJavaBeanInitCode()).append(";");
    } else {
    String constructorParams = object.getConstructorParams();
    if (constructorParams != null) {
    //TC - 20081017 compiledObject can have generics
    result.append(" new ").append(JAXXCompiler.getCanonicalName(object)).append("(").append(constructorParams).append(");");
    //result.append("(").append(getCanonicalName(object.getObjectClass())).append(") new ").append(getCanonicalName(object.getObjectClass())).append("(").append(constructorParams).append(");");
    } else {
    //TC - 20081017 compiledObject can have generics
    result.append("new ").append(JAXXCompiler.getCanonicalName(object)).append("();");
    }
    }
    result.append(JAXXCompiler.getLineSeparator());
    String initCode = object.getInitializationCode(compiler);
    if (initCode != null && initCode.length() > 0) {
    result.append(initCode);
    }
    result.append("$objectMap.put(").append(TypeManager.getJavaCode(object.getId())).append(", ").append(object.getId()).append(");");

    return result.toString();*/
//    }
}
