package jaxx.compiler;

import jaxx.tags.DefaultComponentHandler;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.Arrays;
import java.util.List;

/**
 * I18n methods to add {@link org.nuiton.i18n.I18n#_(String, Object[])} method on some attributes.
 * <p/>
 * Make sure to set an i18nable compiler to have his {@link CompilerOptions#isI18nable()} returning true.
 *
 * @author chemit
 */
public class I18nHelper {

    protected static final Log log = LogFactory.getLog(DefaultComponentHandler.class);

    public static final List<String> I18N_ATTRIBUTES = Arrays.asList("text", "title", "toolTipText");

    /**
     * Test if we have an active i18n attribute (says an i18n attribute on a i18neable compiler).
     *
     * @param attributeName name of attribute to test
     * @param compiler      current used compiler (contains options)
     * @return <code>true</code> if wa have an active i18n attribute, <code>false</code> otherwise.
     */
    public static boolean isI18nableAttribute(String attributeName, JAXXCompiler compiler) {
        return compiler.getOptions().isI18nable() && isI18nAttribute(attributeName);
    }

    /**
     * Test if we have an i18n attribute.
     *
     * @param attributeName name of attribute to test
     * @return <code>true</code> if wa have an active i18n attribute, <code>false</code> otherwise.
     */
    public static boolean isI18nAttribute(String attributeName) {
        return I18N_ATTRIBUTES.contains(attributeName);
    }
    /**
     * Add the i18n on a attribute.
     * <p/>
     * Note: <b>Be ware : </b> no test is done here to ensure we are on a i18neable attribute for an i18nable compiler.
     * <p/>
     * Make sure with the method {@link jaxx.compiler.I18nHelper#isI18nableAttribute(String, JAXXCompiler)} returns
     * <code>true</code< before using this method.
     *
     * @param widgetId           the id of the widget
     * @param attributeName      the name of the attribute
     * @param attributeValueCode the value code of the attribute value
     * @param compiler           the current used compile
     * @return the surrender i18n call if attribute name is matchrf the attributeValueCode otherwise
     */
    public static String addI18nInvocation(String widgetId, String attributeName, String attributeValueCode, JAXXCompiler compiler) {

        if (log.isDebugEnabled()) {
            log.debug(" try i18n support for [" + widgetId + ":" + attributeName + "] : " + attributeValueCode);
        }
        if (attributeValueCode.contains("_(") && attributeValueCode.contains(")")) {
            compiler.reportWarning("\n\tjaxx supports i18n, no need to add explicit call to I18n._ for attribute '" + attributeName + "' in component '" + widgetId + "' : ["+attributeValueCode+"]");
        } else {
            attributeValueCode = "_(" + attributeValueCode + ")";
        }

        return attributeValueCode;
    }
}
