package org.nuiton.helper.plugin;

/*
 * #%L
 * Helper Maven Plugin :: Mojos
 * %%
 * Copyright (C) 2009 - 2016 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.nuiton.plugin.AbstractPlugin;

import java.util.Properties;

/**
 * To compute distribution site url distinguising url when project version is a snapshot.
 *
 * Created on 27/02/16.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @since 2.3
 */
@Mojo(name = "compute-distribution-site-url",
        defaultPhase = LifecyclePhase.VALIDATE,
        requiresOnline = true,
        requiresProject = true)
public class ComputeDistributionSiteUrlMojo extends AbstractPlugin {

    public static final String SNAPSHOT_SUFFIX = "-SNAPSHOT";

    /**
     * Invoking project's version.
     */
    @Parameter(defaultValue = "${project}", required = true, readonly = true)
    private MavenProject project;

    /**
     * Site deploy classifier to use when project's version is a snapshot.
     */
    @Parameter(property = "helper.distributionSiteUrlPrefix", defaultValue = "scpexe://forge.nuiton.org/var/lib/doc/maven-site/${platform}/${projectId}", required = true)
    private String distributionSiteUrlPrefix;

    /**
     * Site deploy classifier to use when project's version is a snapshot.
     */
    @Parameter(property = "helper.snapshotSiteDeployClassifier", defaultValue = "develop", required = true)
    private String snapshotSiteDeployClassifier;

    /**
     * Site deploy classifier to use when project's version is not a snapshot.
     */
    @Parameter(property = "helper.releaseSiteDeployClassifier", defaultValue = "${project.version}", required = true)
    private String releaseSiteDeployClassifier;

    /**
     * A flag to activate verbose mode.
     */
    @Parameter(property = "helper.verbose", defaultValue = "${maven.verbose}")
    private boolean verbose;

    /**
     * A flag to execute only once the mojo.
     *
     * <b>Note:</b> By default, value is {@code true} since it is not necessary
     * to inject twice secrets in session.
     */
    @Parameter(property = "helper.runOnce", defaultValue = "true")
    private boolean runOnce;

    private String siteDeployClassifier;

    private String distributionSiteUrl;

    @Override
    public boolean checkSkip() {

        boolean result = true;
        if (runOnce) {

            // compute the unique key refering to parameters of plugin
            String key = "compute-distribution-site-url##" + project.getVersion();

            // check if plugin was already done.

            boolean shouldInvoke = needInvoke(true, false, key);

            result = shouldInvoke && super.checkSkip();
        }

        return result;

    }

    @Override
    protected void init() throws Exception {

        if (project.getVersion().endsWith(SNAPSHOT_SUFFIX)) {
            siteDeployClassifier = snapshotSiteDeployClassifier;
            if (isVerbose()) {
                getLog().info("Use snapshot siteDeployClassifier: " + siteDeployClassifier);
            }
        } else {
            siteDeployClassifier = releaseSiteDeployClassifier;
            if (isVerbose()) {
                getLog().info("Use release siteDeployClassifier: " + siteDeployClassifier);
            }
        }

        distributionSiteUrl = distributionSiteUrlPrefix.trim();
        if (!distributionSiteUrl.endsWith("/")) {
            distributionSiteUrl += "/";
        }
        distributionSiteUrl += siteDeployClassifier;
        if (isVerbose()) {
            getLog().info("Use release distributionSiteUrl: " + distributionSiteUrl);
        }

    }

    @Override
    protected void doAction() throws Exception {

        Properties properties = project.getProperties();

        getLog().info("export siteDeployClassifier [" + siteDeployClassifier + "]");

        project.getDistributionManagement().getSite().setUrl(distributionSiteUrl);
        properties.setProperty("siteDeployClassifier", siteDeployClassifier);

    }

    @Override
    public MavenProject getProject() {
        return project;
    }

    @Override
    public void setProject(MavenProject project) {
        this.project = project;
    }

    @Override
    public boolean isVerbose() {
        return verbose;
    }

    @Override
    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }
}
