package org.nuiton.eugene.java;

/*
 * #%L
 * EUGene :: Java templates
 * %%
 * Copyright (C) 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */





import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.models.extension.tagvalue.MissingStereoTypeException;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelInterface;
import org.nuiton.eugene.models.object.ObjectModelJavaModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelPackage;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Common class form javabean like templates.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @see SimpleJavaBeanTransformer
 * @see JavaBeanTransformer
 * @since 2.6
 */
public abstract class AbstractJavaBeanTransformer extends ObjectModelTransformerToJava {

    /** Logger. */
    private static final Log log = LogFactory.getLog(AbstractJavaBeanTransformer.class);

    protected final EugeneJavaTagValues javaTemplatesTagValues;

    public AbstractJavaBeanTransformer() {
        javaTemplatesTagValues = new EugeneJavaTagValues();
    }

    protected boolean notFoundInClassPath(ObjectModelClass input, String className) {
        String fqn = input.getPackageName() + "." + className;
        boolean inClassPath = getResourcesHelper().isJavaFileInClassPath(fqn);
        return !inClassPath;
    }

    protected void createPropertyConstant(ObjectModelClassifier output,
                                          ObjectModelAttribute attr,
                                          String prefix,
                                          Set<String> constantNames) {

        String attrName = getAttributeName(attr);

        String constantName = prefix + builder.getConstantName(attrName);

        if (!constantNames.contains(constantName)) {

            addConstant(output,
                        constantName,
                        String.class,
                        "\"" + attrName + "\"",
                        ObjectModelJavaModifier.PUBLIC
            );
        }
    }

    protected String getAttributeName(ObjectModelAttribute attr) {
        String attrName = attr.getName();
        if (attr.hasAssociationClass()) {
            String assocAttrName = JavaGeneratorUtil.getAssocAttrName(attr);
            attrName = JavaGeneratorUtil.toLowerCaseFirstLetter(assocAttrName);
        }
        return attrName;
    }

    protected String getAttributeType(ObjectModelAttribute attr) {
        String attrType = attr.getType();
        if (attr.hasAssociationClass()) {
            attrType = attr.getAssociationClass().getName();
        }
        return attrType;
    }

    protected String getAttributeTypeWithGeneric(ObjectModelAttribute attr) {
        String attrType = getAttributeType(attr);
        String generic = eugeneTagValues.getAttributeGenericTagValue(attr);
        if (generic != null) {
            attrType += "<" + getAttributeType(generic) + ">";
        }
        return attrType;
    }

    protected String getAttributeType(String attrType) {
        return attrType;
    }

    protected boolean containsMutiple(List<ObjectModelAttribute> attributes) {

        boolean result = false;

        for (ObjectModelAttribute attr : attributes) {

            if (JavaGeneratorUtil.isNMultiplicity(attr)) {
                result = true;

                break;
            }

        }
        return result;
    }

    protected void createProperty(ObjectModelClass output,
                                  ObjectModelAttribute attr,
                                  boolean usePCS,
                                  boolean generateBooleanGetMethods,
                                  boolean generateNotEmptyCollections) {

        String attrName = getAttributeName(attr);
        String attrType = getAttributeTypeWithGeneric(attr);

        boolean multiple = JavaGeneratorUtil.isNMultiplicity(attr);

        String constantName = getConstantName(attrName);
        String simpleType = JavaGeneratorUtil.getSimpleName(attrType);

        if (multiple) {

            createGetChildMethod(output,
                                 attrName,
                                 attrType,
                                 simpleType
            );

            createIsEmptyMethod(output, attrName);

            createSizeMethod(output, attrName);

            createAddChildMethod(output,
                                 attrName,
                                 attrType,
                                 constantName,
                                 usePCS
            );

            createAddAllChildrenMethod(output,
                                       attrName,
                                       attrType,
                                       constantName,
                                       usePCS
            );

            createRemoveChildMethod(output,
                                    attrName,
                                    attrType,
                                    constantName,
                                    usePCS
            );

            createRemoveAllChildrenMethod(output,
                                          attrName,
                                          attrType,
                                          constantName,
                                          usePCS
            );

            createContainsChildMethod(output,
                                      attrName,
                                      attrType,
                                      constantName,
                                      usePCS
            );

            createContainsAllChildrenMethod(output,
                                            attrName,
                                            attrType,
                                            constantName,
                                            usePCS
            );

            // Change type for Multiple attribute
            attrType = JavaGeneratorUtil.getAttributeInterfaceType(attr, getAttributeTypeWithGeneric(attr), true);
            simpleType = JavaGeneratorUtil.getSimpleName(attrType);
        }

        boolean booleanProperty = JavaGeneratorUtil.isBooleanPrimitive(attr);

        if (multiple) {

            String collectionImplementationType = JavaGeneratorUtil.getAttributeImplementationType(attr, getAttributeTypeWithGeneric(attr), true);

            // creates a getXXX (multiple) method
            createGetMethod(output,
                            attrName,
                            attrType,
                            JavaGeneratorUtil.OPERATION_GETTER_DEFAULT_PREFIX,
                            generateNotEmptyCollections,
                            collectionImplementationType
            );

        } else {

            if (booleanProperty) {

                // creates a isXXX method
                createGetMethod(output,
                                attrName,
                                attrType,
                                JavaGeneratorUtil.OPERATION_GETTER_BOOLEAN_PREFIX
                );
            }

            if (!booleanProperty || generateBooleanGetMethods) {

                // creates a getXXX method
                createGetMethod(output,
                                attrName,
                                attrType,
                                JavaGeneratorUtil.OPERATION_GETTER_DEFAULT_PREFIX
                );

            }


        }

        createSetMethod(output,
                        attrName,
                        attrType,
                        simpleType,
                        constantName,
                        usePCS
        );

        // Add attribute to the class
        addAttribute(output,
                     attrName,
                     attrType,
                     "",
                     ObjectModelJavaModifier.PROTECTED
        );

    }

    protected List<ObjectModelAttribute> getProperties(ObjectModelClass input) {
        List<ObjectModelAttribute> attributes =
                (List<ObjectModelAttribute>) input.getAttributes();

        List<ObjectModelAttribute> attrs =
                new ArrayList<>();
        for (ObjectModelAttribute attr : attributes) {
            if (attr.isNavigable()) {

                // only keep navigable attributes
                attrs.add(attr);
            }
        }
        return attrs;
    }

    protected void createGetMethod(ObjectModelClass output,
                                   String attrName,
                                   String attrType,
                                   String methodPrefix,
                                   boolean generateLayzCode,
                                   String collectionImplementationType) {

        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName(methodPrefix, attrName),
                attrType,
                ObjectModelJavaModifier.PUBLIC
        );
        if (generateLayzCode) {
            addImport(output, collectionImplementationType);
            String implementationSimpleType = JavaGeneratorUtil.getSimpleName(collectionImplementationType);
            setOperationBody(operation, ""
+"\n"
+"    if ("+attrName+" == null) {\n"
+"        "+attrName+" = new "+implementationSimpleType+"();\n"
+"    }\n"
+"    return "+attrName+";\n"
+""
            );
        } else {
            setOperationBody(operation, ""
+"\n"
+"    return "+attrName+";\n"
+""
            );
        }

    }

    protected void createGetMethod(ObjectModelClass output,
                                   String attrName,
                                   String attrType,
                                   String methodPrefix) {

        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName(methodPrefix, attrName),
                attrType,
                ObjectModelJavaModifier.PUBLIC
        );
        setOperationBody(operation, ""
    +"\n"
+"        return "+attrName+";\n"
+"    "
        );
    }

    protected void createGetChildMethod(ObjectModelClass output,
                                        String attrName,
                                        String attrType,
                                        String simpleType) {
        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName("get", attrName),
                attrType,
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, "int", "index");
        setOperationBody(operation, ""
    +"\n"
+"        "+simpleType+" o = getChild("+attrName+", index);\n"
+"        return o;\n"
+"    "
        );
    }

    protected void createIsEmptyMethod(ObjectModelClass output,
                                       String attrName) {
        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName("is", attrName) + "Empty",
                boolean.class,
                ObjectModelJavaModifier.PUBLIC
        );
        setOperationBody(operation, ""
    +"\n"
+"        return "+attrName+" == null || "+attrName+".isEmpty();\n"
+"    "
        );
    }

    protected void createSizeMethod(ObjectModelClass output,
                                    String attrName) {
        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName("size", attrName),
                int.class,
                ObjectModelJavaModifier.PUBLIC
        );
        setOperationBody(operation, ""
    +"\n"
+"        return "+attrName+" == null ? 0 : "+attrName+".size();\n"
+"    "
        );
    }

    protected void createAddChildMethod(ObjectModelClass output,
                                        String attrName,
                                        String attrType,
                                        String constantName,
                                        boolean usePCS) {
        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName("add", attrName),
                "void",
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, attrType, attrName);

        String methodName = getJavaBeanMethodName("get", attrName);
        StringBuilder buffer = new StringBuilder(""
    +"\n"
+"        "+methodName+"().add("+attrName+");\n"
+"    "
        );
        if (usePCS) {
            buffer.append(""
    +"    firePropertyChange("+constantName+", null, "+attrName+");\n"
+"    "
            );
        }
        setOperationBody(operation, buffer.toString());
    }

    protected void createAddAllChildrenMethod(ObjectModelClass output,
                                              String attrName,
                                              String attrType,
                                              String constantName,
                                              boolean usePCS) {
        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName("addAll", attrName),
                "void",
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, "java.util.Collection<" + attrType + ">", attrName);

        String methodName = getJavaBeanMethodName("get", attrName);
        StringBuilder buffer = new StringBuilder(""
    +"\n"
+"        "+methodName+"().addAll("+attrName+");\n"
+"    "
        );
        if (usePCS) {
            buffer.append(""
    +"    firePropertyChange("+constantName+", null, "+attrName+");\n"
+"    "
            );
        }
        setOperationBody(operation, buffer.toString());
    }

    protected void createRemoveChildMethod(ObjectModelClass output,
                                           String attrName,
                                           String attrType,
                                           String constantName,
                                           boolean usePCS) {
        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName("remove", attrName),
                "boolean",
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, attrType, attrName);
        String methodName = getJavaBeanMethodName("get", attrName);
        StringBuilder buffer = new StringBuilder();
        buffer.append(""
    +"\n"
+"        boolean removed = "+methodName+"().remove("+attrName+");"
        );

        if (usePCS) {
            buffer.append(""
    +"\n"
+"        if (removed) {\n"
+"            firePropertyChange("+constantName+", "+attrName+", null);\n"
+"        }"
            );
        }
        buffer.append(""
    +"\n"
+"        return removed;\n"
+"    "
        );
        setOperationBody(operation, buffer.toString());
    }

    protected void createRemoveAllChildrenMethod(ObjectModelClass output,
                                                 String attrName,
                                                 String attrType,
                                                 String constantName,
                                                 boolean usePCS) {

        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName("removeAll", attrName),
                "boolean",
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, "java.util.Collection<" + attrType + ">", attrName);
        StringBuilder buffer = new StringBuilder();
        String methodName = getJavaBeanMethodName("get", attrName);
        buffer.append(""
    +"\n"
+"        boolean  removed = "+methodName+"().removeAll("+attrName+");"
        );

        if (usePCS) {
            buffer.append(""
    +"\n"
+"        if (removed) {\n"
+"            firePropertyChange("+constantName+", "+attrName+", null);\n"
+"        }"
            );
        }
        buffer.append(""
    +"\n"
+"        return removed;\n"
+"    "
        );
        setOperationBody(operation, buffer.toString());
    }

    protected void createContainsChildMethod(ObjectModelClass output,
                                             String attrName,
                                             String attrType,
                                             String constantName,
                                             boolean usePCS) {

        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName("contains", attrName),
                "boolean",
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, attrType, attrName);
        StringBuilder buffer = new StringBuilder();
        String methodName = getJavaBeanMethodName("get", attrName);
        buffer.append(""
    +"\n"
+"        boolean contains = "+methodName+"().contains("+attrName+");\n"
+"        return contains;\n"
+"    "
        );
        setOperationBody(operation, buffer.toString());
    }

    protected void createContainsAllChildrenMethod(ObjectModelClass output,
                                                   String attrName,
                                                   String attrType,
                                                   String constantName,
                                                   boolean usePCS) {

        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName("containsAll", attrName),
                "boolean",
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, "java.util.Collection<" + attrType + ">", attrName);
        StringBuilder buffer = new StringBuilder();
        String methodName = getJavaBeanMethodName("get", attrName);
        buffer.append(""
    +"\n"
+"        boolean  contains = "+methodName+"().containsAll("+attrName+");\n"
+"        return contains;\n"
+"    "
        );
        setOperationBody(operation, buffer.toString());
    }

    protected void createSetMethod(ObjectModelClass output,
                                   String attrName,
                                   String attrType,
                                   String simpleType,
                                   String constantName,
                                   boolean usePCS) {
        boolean booleanProperty = GeneratorUtil.isBooleanPrimitive(simpleType);
        ObjectModelOperation operation = addOperation(
                output,
                getJavaBeanMethodName("set", attrName),
                "void",
                ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, attrType, attrName);

        if (usePCS) {
            String methodPrefix = JavaGeneratorUtil.OPERATION_GETTER_DEFAULT_PREFIX;
            if (booleanProperty) {
                methodPrefix = JavaGeneratorUtil.OPERATION_GETTER_BOOLEAN_PREFIX;
            }
            String methodName = getJavaBeanMethodName(methodPrefix, attrName);
            setOperationBody(operation, ""
    +"\n"
+"        "+simpleType+" oldValue = "+methodName+"();\n"
+"        this."+attrName+" = "+attrName+";\n"
+"        firePropertyChange("+constantName+", oldValue, "+attrName+");\n"
+"    "
            );
        } else {
            setOperationBody(operation, ""
    +"\n"
+"        this."+attrName+" = "+attrName+";\n"
+"    "
            );
        }
    }

    protected void addSerializable(ObjectModelClass input,
                                   ObjectModelClass output,
                                   boolean interfaceFound) {
        if (!interfaceFound) {
            addInterface(output, Serializable.class);
        }

        // Generate the serialVersionUID
        long serialVersionUID = JavaGeneratorUtil.generateSerialVersionUID(input);

        addConstant(output,
                    JavaGeneratorUtil.SERIAL_VERSION_UID,
                    "long",
                    serialVersionUID + "L",
                    ObjectModelJavaModifier.PRIVATE
        );
    }

    /**
     * Add all interfaces defines in input class and returns if
     * {@link Serializable} interface was found.
     *
     * @param input  the input model class to process
     * @param output the output generated class
     * @return {@code true} if {@link Serializable} was found from input,
     * {@code false} otherwise
     */
    protected boolean addInterfaces(ObjectModelClass input,
                                    ObjectModelClassifier output,
                                    String extraInterfaceName) {
        boolean foundSerializable = false;
        Set<String> added = new HashSet<>();
        for (ObjectModelInterface parentInterface : input.getInterfaces()) {
            String fqn = parentInterface.getQualifiedName();
            added.add(fqn);
            addInterface(output, fqn);
            if (Serializable.class.getName().equals(fqn)) {
                foundSerializable = true;
            }
        }
        if (extraInterfaceName != null && !added.contains(extraInterfaceName)) {
            addInterface(output, extraInterfaceName);
        }
        return foundSerializable;
    }

    protected void createPropertyChangeSupport(ObjectModelClass output) {

        addAttribute(output,
                     "pcs",
                     PropertyChangeSupport.class,
                     "new PropertyChangeSupport(this)",
                     ObjectModelJavaModifier.PROTECTED,
                     ObjectModelJavaModifier.FINAL,
                     ObjectModelJavaModifier.TRANSIENT
        );

        // Add PropertyListener

        ObjectModelOperation operation;

        operation = addOperation(output,
                                 "addPropertyChangeListener",
                                 "void",
                                 ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, PropertyChangeListener.class, "listener");
        setOperationBody(operation, ""
    +"\n"
+"        pcs.addPropertyChangeListener(listener);\n"
+"    "
        );

        operation = addOperation(output,
                                 "addPropertyChangeListener",
                                 "void",
                                 ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, String.class, "propertyName");
        addParameter(operation, PropertyChangeListener.class, "listener");
        setOperationBody(operation, ""
    +"\n"
+"        pcs.addPropertyChangeListener(propertyName, listener);\n"
+"    "
        );

        operation = addOperation(output,
                                 "removePropertyChangeListener",
                                 "void",
                                 ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, PropertyChangeListener.class, "listener");
        setOperationBody(operation, ""
    +"\n"
+"        pcs.removePropertyChangeListener(listener);\n"
+"    "
        );

        operation = addOperation(output,
                                 "removePropertyChangeListener",
                                 "void",
                                 ObjectModelJavaModifier.PUBLIC
        );
        addParameter(operation, String.class, "propertyName");
        addParameter(operation, PropertyChangeListener.class, "listener");
        setOperationBody(operation, ""
    +"\n"
+"        pcs.removePropertyChangeListener(propertyName, listener);\n"
+"    "
        );

        operation = addOperation(output,
                                 "firePropertyChange",
                                 "void",
                                 ObjectModelJavaModifier.PROTECTED
        );
        addParameter(operation, String.class, "propertyName");
        addParameter(operation, Object.class, "oldValue");
        addParameter(operation, Object.class, "newValue");
        setOperationBody(operation, ""
    +"\n"
+"        pcs.firePropertyChange(propertyName, oldValue, newValue);\n"
+"    "
        );

        operation = addOperation(output,
                                 "firePropertyChange",
                                 "void",
                                 ObjectModelJavaModifier.PROTECTED
        );
        addParameter(operation, String.class, "propertyName");
        addParameter(operation, Object.class, "newValue");
        setOperationBody(operation, ""
    +"\n"
+"        firePropertyChange(propertyName, null, newValue);\n"
+"    "
        );
    }

    protected void createGetChildMethod(ObjectModelClass output) {
        ObjectModelOperation getChild = addOperation(
                output,
                "getChild", "<T> T",
                ObjectModelJavaModifier.PROTECTED
        );
        addImport(output, List.class);

        addParameter(getChild, "java.util.Collection<T>", "childs");
        addParameter(getChild, "int", "index");
        setOperationBody(getChild, ""
+"\n"
+"        T result = null;\n"
+"        if (childs != null) {\n"
+"            if (childs instanceof List) {\n"
+"                if (index < childs.size()) {\n"
+"                    result = ((List<T>) childs).get(index);\n"
+"                }\n"
+"            } else {\n"
+"                int i = 0;\n"
+"                for (T o : childs) {\n"
+"                    if (index == i) {\n"
+"                        result = o;\n"
+"                        break;\n"
+"                    }\n"
+"                    i++;\n"
+"                }\n"
+"            }\n"
+"        }\n"
+"        return result;\n"
+""
        );
    }

    protected void generateI18nBlockAndConstants(ObjectModelPackage aPackage,
                                                 ObjectModelClass input,
                                                 ObjectModelClassifier output) {

        String i18nPrefix = eugeneTagValues.getI18nPrefixTagValue(input,
                                                                  aPackage,
                                                                  model);
        if (!StringUtils.isEmpty(i18nPrefix)) {
            generateI18nBlock(input, output, i18nPrefix);
        }

        String prefix = getConstantPrefix(input);

        setConstantPrefix(prefix);

        Set<String> constantNames = addConstantsFromDependency(input, output);

        // Add properties constant
        for (ObjectModelAttribute attr : getProperties(input)) {

            createPropertyConstant(output, attr, prefix, constantNames);
        }
    }

    protected void addDefaultMethodForNoneBeanSuperClass(ObjectModelClass output,
                                                         boolean usePCS,
                                                         List<ObjectModelAttribute> properties) {


        if (usePCS) {

            // Add property change support
            createPropertyChangeSupport(output);
        }

        boolean hasAMultipleProperty = containsMutiple(properties);

        // Add helper operations
        if (hasAMultipleProperty) {

            // add getChild methods
            createGetChildMethod(output);
        }
    }

    protected String wrapPrimitiveType(String attrType) {
        if (JavaGeneratorUtil.isPrimitiveType(attrType)) {
            attrType = JavaGeneratorUtil.getPrimitiveWrapType(attrType);
        }
        return attrType;
    }

    protected String getGetterName(ObjectModelAttribute attribute, String attrName) {
        boolean booleanProperty = JavaGeneratorUtil.isBooleanPrimitive(attribute);
        String methodPrefix = JavaGeneratorUtil.OPERATION_GETTER_DEFAULT_PREFIX;
        if (booleanProperty) {
            methodPrefix = JavaGeneratorUtil.OPERATION_GETTER_BOOLEAN_PREFIX;
        }
        return getJavaBeanMethodName(methodPrefix, attrName);
    }

    protected String generateName(String prefix, String name, String suffix) {
        StringBuilder sb = new StringBuilder();
        if (StringUtils.isNotEmpty(prefix)) {
            sb.append(prefix);
        }
        sb.append(name);
        if (StringUtils.isNotEmpty(suffix)) {
            sb.append(suffix);
        }
        return sb.toString();
    }

    protected boolean canGenerateAbstractClass(ObjectModelPackage aPackage, ObjectModelClass aclass, String abstractClassName) {

        boolean notInClassPath = notFoundInClassPath(aclass, abstractClassName);

        if (!notInClassPath) {

            boolean canOverride = javaTemplatesTagValues.isOverrideAbstractClasses(aclass, aPackage, model);
            String fqn = aclass.getPackageName() + "." + abstractClassName;
            if (!canOverride) {
                String message = "Can not override abstract class " + fqn + ", already found in classpath";
                message += "\n\nTo remove this check, add the stereotype *overrideAbstractClasses* on model, package or class.";
                MissingStereoTypeException exception = new MissingStereoTypeException("overrideAbstractClasses", message, aPackage, aclass);
                if (log.isErrorEnabled()) {
                    log.error(exception.toString());
                }
                throw exception;
            }
            if (log.isWarnEnabled()) {
                log.warn("Will not generate abstract class: " + fqn);
            }
        }

        return notInClassPath;

    }

    protected boolean canGenerateClassWithMethods(ObjectModelPackage aPackage, ObjectModelClass aclass, String className) {

        boolean noMethods = aclass.getOperations().isEmpty();

        if (!noMethods) {

            boolean canOverride = javaTemplatesTagValues.isAcceptBeanWithMethods(aclass, aPackage, model);
            String fqn = aclass.getPackageName() + "." + className;
            if (!canOverride) {
                String message = "Concrete class " + fqn + " contains some methods, the current generator does not accept it.";
                message += "\n\nSuch methods are not made to be designed, you should directly write them in the concrete class.";
                message += "\n\nTo remove this check, add the stereotype *acceptBeanWithMethods* on model, package or class.";
                MissingStereoTypeException exception = new MissingStereoTypeException("acceptBeanWithMethods", message, aPackage, aclass);
                if (log.isErrorEnabled()) {
                    log.error(exception.toString());
                }
                throw exception;

            }
            if (log.isWarnEnabled()) {
                log.warn("Will generate class (but designed methods won't be generated): " + fqn);
            }
        }

        return true;

    }

    protected boolean isUseJava8() {
        return javaTemplatesTagValues.isUseJava8(model);
    }

}
