/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.javassist.bytecode;

import java.lang.reflect.Method;

import javassist.CtClass;
import javassist.CtField;
import javassist.Modifier;
import javassist.NotFoundException;
import javassist.bytecode.Bytecode;
import javassist.bytecode.CodeAttribute;
import javassist.bytecode.ConstPool;
import javassist.bytecode.MethodInfo;
import javassist.bytecode.Opcode;
import javassist.bytecode.StackMapTable;
import javassist.util.proxy.RuntimeSupport;

import org.jboss.reflect.plugins.javassist.JavassistField;

/**
 * <p>Class to create implementations of the {{@link JavassistField} interface.</p>
 * 
 * <p>This implementation generates raw bytecode to avoid the overhead of compilation via javassist. If 
 * <code>sun.reflect.MagicAccessorImpl</code> is used as the <code>superClass</code> field the implementation
 * classes can access private and protected members of the target class.</p>
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
class JavassistFieldFactory extends JavassistMemberFactory
{
   /** The names of the interfaces we are implementing */ 
   protected static final String[] interfaceNames;
   
   /** The methods from the interface that are being implemented */ 
   protected static final Method[] methods;
   
   static
   {
      interfaceNames = new String[] {JavassistField.class.getName()};
      methods = new Method[2];
      try
      {
         methods[0] = SecurityActions.getDeclaredMethod(JavassistField.class, "get", Object.class);
         methods[1] = SecurityActions.getDeclaredMethod(JavassistField.class, "set", Object.class, Object.class);
      }
      catch (NoSuchMethodException e)
      {
         throw new RuntimeException(e);
      }
   }

   /** The field we are targeting */
   final CtField ctField;
   
   /** The name of the class being generated */
   final String className = JavassistField.class.getName() + counter.incrementAndGet();
   
   /**
    * Constructor
    * 
    * @param superClass the super class to use for the implementation
    * @param ctField the field we are generating a {@link JavassistField} for
    * @param debug true to cause the class bytes to be output to the file system so they can be inspected with a decompiler/javap
    */      
   JavassistFieldFactory(Class<?> superClass, CtField ctField, boolean debug)
   {
      super(superClass, debug);
      this.ctField = ctField;
   }

   @Override
   String getGeneratedClassName()
   {
      return className;
   }

   @Override
   String[] getInterfaceNames()
   {
      return interfaceNames;
   }
   
   /**
    * Gets the type of the field
    * 
    * @return the field type
    */
   private CtClass getFieldType()
   {
      try
      {
         return ctField.getType();
      }
      catch (NotFoundException e)
      {
         // AutoGenerated
         throw new RuntimeException(e);
      }
   }
   
   @Override
   String getAccessedMember()
   {
      return ctField.getDeclaringClass().getName() + "." + ctField.getName() + ctField.getSignature();
   }

   @Override
   String initAccessedMember()
   {
      return null;
   }

   @Override
   MethodInfo implementMethod(int index, ConstPool cp)
   {
      if (index >= methods.length)
         return null;

      String desc = RuntimeSupport.makeDescriptor(methods[index]);
      MethodInfo minfo = new MethodInfo(cp, methods[index].getName(), desc);
      minfo.setAccessFlags(Modifier.PUBLIC);
      setThrows(minfo, cp, methods[index].getExceptionTypes());
      Bytecode code = new Bytecode(cp, 0, 0);
      
      int pc = code.currentPc();
      boolean isStatic = Modifier.isStatic(ctField.getModifiers());

      int maxLocals = 0;
      if (index == 0)
      {
         //We need 2 local variable slots.
         //One for 'this' and one for the target reference
         //These are all object references and so take one slot each 
         maxLocals = 2;
         makeGetMethod(code, methods[index], cp, isStatic);
      }
      else
      {
         //We need 3 local variable slots.
         //One for 'this', one for the target reference and one for the argument array.
         //These are all object references and so take one slot each 
         maxLocals = 3;
         makeSetMethod(code, methods[index], cp, isStatic);
      }
      
      code.setMaxLocals(maxLocals);
      CodeAttribute ca = code.toCodeAttribute();
      minfo.setCodeAttribute(ca);

      StackMapTable.Writer writer = new StackMapTable.Writer(32);
      writer.sameFrame(pc);
      ca.setAttribute(writer.toStackMapTable(cp));
      return minfo;
   }
   
   private void makeGetMethod(Bytecode code, Method method, ConstPool cp, boolean isStatic)
   {
      if (isStatic)
      {
         code.addGetstatic(ctField.getDeclaringClass().getName(), ctField.getName(), ctField.getSignature());
      }
      else
      {
         //push and cast the target object
         code.addAload(1);
         code.addCheckcast(ctField.getDeclaringClass());
         code.addGetfield(ctField.getDeclaringClass(), ctField.getName(), ctField.getSignature());
      }
      boxReturnValue(code, getFieldType());
      code.addOpcode(Opcode.ARETURN);
         
   }
   
   private void makeSetMethod(Bytecode code, Method method, ConstPool cp, boolean isStatic)
   {
      if (!isStatic)
      {
         //push and cast the target object
         code.addAload(1);
         code.addCheckcast(ctField.getDeclaringClass());
      }

      //push and cast the value
      code.addAload(2);
      castAndUnbox(code, getFieldType());
      
      if (isStatic)
      {
         code.addPutstatic(ctField.getDeclaringClass().getName(), ctField.getName(), ctField.getSignature());
      }
      else
      {
         code.addPutfield(ctField.getDeclaringClass(), ctField.getName(), ctField.getSignature());
      }

      code.addOpcode(Opcode.RETURN);
   }
}