/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.svm.core.genscavenge;

import com.oracle.svm.core.config.ConfigurationValues;
import com.oracle.svm.core.genscavenge.CardTable;
import com.oracle.svm.core.genscavenge.HeapImpl;
import com.oracle.svm.core.hub.LayoutEncoding;
import com.oracle.svm.core.log.Log;
import com.oracle.svm.core.thread.VMOperation;
import com.oracle.svm.core.util.UnsignedUtils;
import org.graalvm.word.Pointer;
import org.graalvm.word.UnsignedWord;
import org.graalvm.word.WordBase;
import org.graalvm.word.WordFactory;

public final class FirstObjectTable {
    private static final int MEMORY_BYTES_PER_ENTRY = CardTable.getMemoryBytesPerEntry();
    private static final int ENTRY_BYTES = 1;
    private static final int ENTRY_MIN = -128;
    private static final int ENTRY_MAX = 127;
    private static final int MEMORY_OFFSET_MIN = -128;
    private static final int MEMORY_OFFSET_MAX = 0;
    private static final int LINEAR_OFFSET_MIN = 1;
    private static final int LINEAR_OFFSET_MAX = 63;
    private static final int EXPONENT_MIN = 6;
    private static final int EXPONENT_MAX = 55;
    private static final int EXPONENT_BIAS = 58;
    private static final int UNINITIALIZED_ENTRY = 127;

    private FirstObjectTable() {
    }

    protected static Pointer initializeTableToPointer(Pointer table, Pointer tableLimit) {
        UnsignedWord indexLimit = FirstObjectTable.tableOffsetToIndex((UnsignedWord)tableLimit.subtract((UnsignedWord)table));
        return FirstObjectTable.initializeTableToIndex(table, indexLimit);
    }

    protected static boolean isUninitializedIndex(Pointer table, UnsignedWord index) {
        int entry = FirstObjectTable.getEntryAtIndex(table, index);
        return FirstObjectTable.isUninitializedEntry(entry);
    }

    protected static void setTableForObject(Pointer table, Pointer memory, Pointer start, Pointer end) {
        VMOperation.guaranteeInProgress("Should only be called from the collector.");
        FirstObjectTable.setTableForObjectUnchecked(table, memory, start, end);
    }

    private static void setTableForObjectUnchecked(Pointer table, Pointer memory, Pointer start, Pointer end) {
        UnsignedWord memoryOffsetIndex;
        assert (memory.belowOrEqual((UnsignedWord)start)) : "memory.belowOrEqual(start)";
        assert (start.belowThan((UnsignedWord)end)) : "start.belowThan(end)";
        Pointer startOffset = start.subtract((UnsignedWord)memory);
        Pointer endOffset = end.subtract(1).subtract((UnsignedWord)memory);
        UnsignedWord startIndex = FirstObjectTable.memoryOffsetToIndex((UnsignedWord)startOffset);
        UnsignedWord endIndex = FirstObjectTable.memoryOffsetToIndex((UnsignedWord)endOffset);
        UnsignedWord startMemoryOffset = FirstObjectTable.indexToMemoryOffset(startIndex);
        if (startIndex.equal(endIndex) && startOffset.unsignedRemainder(MEMORY_BYTES_PER_ENTRY).notEqual(0)) {
            return;
        }
        if (startOffset.equal(startMemoryOffset)) {
            memoryOffsetIndex = startIndex;
            FirstObjectTable.setEntryAtIndex(table, memoryOffsetIndex, 0);
        } else {
            memoryOffsetIndex = startIndex.add(1);
            UnsignedWord memoryIndexOffset = FirstObjectTable.indexToMemoryOffset(memoryOffsetIndex);
            int entry = FirstObjectTable.memoryOffsetToEntry(memoryIndexOffset.subtract((UnsignedWord)startOffset));
            FirstObjectTable.setEntryAtIndex(table, memoryOffsetIndex, entry);
        }
        UnsignedWord linearIndexMax = UnsignedUtils.min(endIndex, memoryOffsetIndex.add(63));
        UnsignedWord entryIndex = memoryOffsetIndex.add(1);
        int entry = 1;
        while (entryIndex.belowOrEqual(linearIndexMax)) {
            FirstObjectTable.setEntryAtIndex(table, entryIndex, entry);
            entryIndex = entryIndex.add(1);
            ++entry;
        }
        int unbiasedExponent = 6;
        while (entryIndex.belowOrEqual(endIndex)) {
            for (int count = 0; count < 1 << unbiasedExponent; ++count) {
                int biasedEntry = FirstObjectTable.biasExponent(unbiasedExponent);
                FirstObjectTable.setEntryAtIndex(table, entryIndex, biasedEntry);
                entryIndex = entryIndex.add(1);
                if (!entryIndex.belowOrEqual(endIndex)) break;
            }
            ++unbiasedExponent;
        }
    }

    protected static Pointer getPreciseFirstObjectPointer(Pointer tableStart, Pointer memoryStart, Pointer memoryLimit, UnsignedWord index) {
        Log trace = Log.noopLog().string("[FirstObjectTable.getPreciseFirstObjectPointer:").string("  index: ").unsigned((WordBase)index);
        UnsignedWord currentIndex = index;
        int currentEntry = FirstObjectTable.getEntryAtIndex(tableStart, currentIndex);
        if (trace.isEnabled()) {
            trace.string("  entry: ");
            FirstObjectTable.entryToLog(trace, currentEntry);
        }
        assert (currentEntry != 127) : "uninitialized first object table entry";
        if (0 < currentEntry) {
            while (63 < currentEntry) {
                int exponent = FirstObjectTable.unbiasExponent(currentEntry);
                UnsignedWord deltaIndex = FirstObjectTable.exponentToOffset(exponent);
                currentIndex = FirstObjectTable.indexMinusDelta(currentIndex, deltaIndex);
                currentEntry = FirstObjectTable.getEntryAtIndex(tableStart, currentIndex);
            }
            if (0 < currentEntry) {
                currentIndex = currentIndex.subtract(currentEntry);
                currentEntry = FirstObjectTable.getEntryAtIndex(tableStart, currentIndex);
            }
        }
        Pointer result = FirstObjectTable.getPointerAtOffset(memoryStart, currentIndex, currentEntry);
        trace.string("  returns: ").hex((WordBase)result);
        assert (memoryStart.belowOrEqual((UnsignedWord)result)) : "memoryStart.belowOrEqual(result)";
        assert (result.belowThan((UnsignedWord)memoryLimit)) : "result.belowThan(memoryLimit)";
        trace.string("]").newline();
        return result;
    }

    protected static Pointer getImpreciseFirstObjectPointer(Pointer tableStart, Pointer memoryStart, Pointer memoryLimit, UnsignedWord index) {
        Pointer result;
        Log trace = Log.noopLog().string("[FirstObjectTable.getImpreciseFirstObjectPointer:");
        trace.string("  tableStart: ").hex((WordBase)tableStart).string("  memoryStart: ").hex((WordBase)memoryStart).string("  memoryLimit: ").hex((WordBase)memoryLimit).string("  index: ").unsigned((WordBase)index).newline();
        Pointer preciseFirstPointer = FirstObjectTable.getPreciseFirstObjectPointer(tableStart, memoryStart, memoryLimit, index);
        Pointer indexedMemoryStart = FirstObjectTable.indexToMemoryPointer(index, memoryStart);
        if (preciseFirstPointer.belowThan((UnsignedWord)indexedMemoryStart)) {
            Object crossingObject = preciseFirstPointer.toObject();
            result = LayoutEncoding.getObjectEnd(crossingObject);
        } else {
            assert (preciseFirstPointer.equal((UnsignedWord)indexedMemoryStart)) : "preciseFirstPointer.equal(indexedMemoryStart)";
            result = indexedMemoryStart;
        }
        trace.string("  returns: ").hex((WordBase)result);
        assert (memoryStart.belowOrEqual((UnsignedWord)result)) : "memoryStart.belowOrEqual(result)";
        assert (result.belowOrEqual((UnsignedWord)memoryLimit)) : "result.belowOrEqual(memoryLimit)";
        trace.string("]").newline();
        return result;
    }

    protected static boolean verify(Pointer tableStart, Pointer memoryStart, Pointer memoryLimit) {
        Log trace = HeapImpl.getHeapImpl().getHeapVerifierImpl().getTraceLog().string("[FirstObjectTable.verify:");
        trace.string("  tableStart: ").hex((WordBase)tableStart).string("  memoryStart: ").hex((WordBase)memoryStart).string("  memoryLimit: ").hex((WordBase)memoryLimit);
        UnsignedWord indexLimit = FirstObjectTable.getTableSizeForMemoryPointers(memoryStart, memoryLimit);
        UnsignedWord index = WordFactory.unsigned((int)0);
        while (index.belowThan(indexLimit)) {
            trace.newline().string("  FirstObjectTable.verify: index: ").unsigned((WordBase)index).newline();
            Pointer objStart = FirstObjectTable.getPreciseFirstObjectPointer(tableStart, memoryStart, memoryLimit, index);
            trace.string("  objStart: ").hex((WordBase)objStart).newline();
            if (objStart.belowThan((UnsignedWord)memoryStart)) {
                trace.string("  FirstObjectTable.verify: objStart.belowThan(memoryStart) => false]").newline();
                return false;
            }
            if (memoryLimit.belowOrEqual((UnsignedWord)objStart)) {
                trace.string("  FirstObjectTable.verify: memoryLimit.belowOrEqual(objStart) => false]").newline();
                return false;
            }
            if (!HeapImpl.getHeapImpl().getHeapVerifierImpl().verifyObjectAt(objStart)) {
                Log witness = HeapImpl.getHeapImpl().getHeapVerifierImpl().getWitnessLog().string("[FirstObjectTable.verify:");
                witness.string("  tableStart: ").hex((WordBase)tableStart).string("  memoryStart: ").hex((WordBase)memoryStart).string("  memoryLimit: ").hex((WordBase)memoryLimit);
                witness.string("  objStart: ").hex((WordBase)objStart).string("  fails to verify").string("]").newline();
                return false;
            }
            Pointer entryStart = memoryStart.add(FirstObjectTable.indexToMemoryOffset(index));
            trace.string("  entryStart: ").hex((WordBase)entryStart);
            if (!objStart.belowOrEqual((UnsignedWord)entryStart)) {
                Log witness = HeapImpl.getHeapImpl().getHeapVerifierImpl().getWitnessLog().string("[FirstObjectTable.verify:");
                witness.string("  tableStart: ").hex((WordBase)tableStart).string("  memoryStart: ").hex((WordBase)memoryStart).string("  memoryLimit: ").hex((WordBase)memoryLimit);
                witness.string("  objStart: ").hex((WordBase)objStart).string("  entryStart: ").hex((WordBase)entryStart).string("  !(objStart.belowOrEqual(entryStart))").string("]").newline();
                return false;
            }
            Object obj = objStart.toObject();
            Pointer objEnd = LayoutEncoding.getObjectEnd(obj);
            trace.string("  ");
            trace.string(obj.getClass().getName());
            trace.string("  objEnd: ").hex((WordBase)objEnd);
            if (!entryStart.belowThan((UnsignedWord)objEnd)) {
                Log witness = HeapImpl.getHeapImpl().getHeapVerifierImpl().getWitnessLog().string("[FirstObjectTable.verify:");
                witness.string("  tableStart: ").hex((WordBase)tableStart).string("  memoryStart: ").hex((WordBase)memoryStart).string("  memoryLimit: ").hex((WordBase)memoryLimit);
                witness.string("  objEnd: ").hex((WordBase)objEnd).string("  entryStart: ").hex((WordBase)entryStart).string("  !(entryStart.belowThan(objEnd))").string("]").newline();
                return false;
            }
            index = index.add(1);
        }
        trace.string("  => true]").newline();
        return true;
    }

    protected static UnsignedWord getTableSizeForMemoryPointers(Pointer memoryStart, Pointer memoryLimit) {
        assert (memoryStart.belowOrEqual((UnsignedWord)memoryLimit)) : "Pointers out of order";
        Pointer memorySize = memoryLimit.subtract((UnsignedWord)memoryStart);
        return FirstObjectTable.getTableSizeForMemorySize((UnsignedWord)memorySize);
    }

    private static UnsignedWord getTableSizeForMemorySize(UnsignedWord memorySize) {
        UnsignedWord roundedMemory = UnsignedUtils.roundUp(memorySize, WordFactory.unsigned((int)MEMORY_BYTES_PER_ENTRY));
        UnsignedWord index = FirstObjectTable.memoryOffsetToIndex(roundedMemory);
        return index.multiply(1);
    }

    private static int memoryOffsetScale() {
        return ConfigurationValues.getObjectLayout().getAlignment();
    }

    private static Pointer initializeTableToIndex(Pointer table, UnsignedWord indexLimit) {
        assert (FirstObjectTable.initializeTableToIndexForAsserts(table, indexLimit));
        return table;
    }

    private static boolean initializeTableToPointerForAsserts(Pointer table, Pointer tableLimit) {
        UnsignedWord indexLimit = FirstObjectTable.tableOffsetToIndex((UnsignedWord)tableLimit.subtract((UnsignedWord)table));
        FirstObjectTable.initializeTableToIndexForAsserts(table, indexLimit);
        return true;
    }

    private static boolean initializeTableToIndexForAsserts(Pointer table, UnsignedWord indexLimit) {
        UnsignedWord index = WordFactory.unsigned((int)0);
        while (index.belowThan(indexLimit)) {
            FirstObjectTable.initializeEntryAtIndex(table, index);
            index = index.add(1);
        }
        return true;
    }

    private static void initializeEntryAtIndex(Pointer table, UnsignedWord index) {
        table.writeByte((WordBase)FirstObjectTable.indexToTableOffset(index), (byte)127);
    }

    private static int getEntryAtIndex(Pointer table, UnsignedWord index) {
        return table.readByte((WordBase)FirstObjectTable.indexToTableOffset(index));
    }

    private static void setEntryAtIndex(Pointer table, UnsignedWord index, int value) {
        assert (FirstObjectTable.isValidEntry(value)) : "Invalid entry";
        assert (FirstObjectTable.isUninitializedIndex(table, index) || FirstObjectTable.getEntryAtIndex(table, index) == value) : "Overwriting!";
        table.writeByte((WordBase)FirstObjectTable.indexToTableOffset(index), (byte)value);
    }

    private static boolean isValidEntry(int entry) {
        return -128 <= entry && entry <= 127;
    }

    private static boolean isUninitializedEntry(int entry) {
        assert (FirstObjectTable.isValidEntry(entry)) : "Invalid entry";
        return entry == 127;
    }

    private static boolean isMemoryOffsetEntry(int entry) {
        assert (FirstObjectTable.isValidEntry(entry)) : "Invalid entry";
        return -128 <= entry && entry <= 0;
    }

    private static boolean isLinearOffsetEntry(int entry) {
        assert (FirstObjectTable.isValidEntry(entry)) : "Invalid entry";
        return 1 <= entry && entry <= 63;
    }

    private static boolean isExponentialOffsetEntry(int entry) {
        assert (FirstObjectTable.isValidEntry(entry)) : "Invalid entry";
        int unbiasedEntry = FirstObjectTable.unbiasExponent(entry);
        return 6 <= unbiasedEntry && unbiasedEntry <= 55;
    }

    private static int biasExponent(int exponent) {
        assert (6 <= exponent && exponent <= 55) : "Exponent out of bounds.";
        return exponent + 58;
    }

    private static int unbiasExponent(int entry) {
        int exponent = entry - 58;
        assert (6 <= exponent && exponent <= 55) : "Exponent out of bounds.";
        return exponent;
    }

    private static UnsignedWord exponentToOffset(int n) {
        assert (0 <= n && n <= 63) : "Exponent out of bounds.";
        return WordFactory.unsigned((long)(1L << n));
    }

    private static boolean memoryOffsetStartsCard(UnsignedWord offset) {
        return offset.unsignedRemainder(MEMORY_BYTES_PER_ENTRY).equal(0);
    }

    private static boolean memoryOffsetAndLengthCrossesCard(UnsignedWord offset, UnsignedWord length) {
        UnsignedWord startCard = FirstObjectTable.memoryOffsetToIndex(offset);
        UnsignedWord endCard = FirstObjectTable.memoryOffsetToIndex(offset.add(length).subtract(1));
        return startCard.belowThan(endCard);
    }

    private static Pointer getPointerAtOffset(Pointer memory, UnsignedWord currentIndex, int currentEntry) {
        assert (FirstObjectTable.isMemoryOffsetEntry(currentEntry)) : "Entry out of bounds.";
        UnsignedWord indexOffset = FirstObjectTable.indexToMemoryOffset(currentIndex);
        UnsignedWord entryOffset = FirstObjectTable.entryToMemoryOffset(currentEntry);
        return memory.add(indexOffset).subtract(entryOffset);
    }

    private static UnsignedWord indexToTableOffset(UnsignedWord index) {
        return index.multiply(1);
    }

    private static UnsignedWord tableOffsetToIndex(UnsignedWord offset) {
        return offset.unsignedDivide(1);
    }

    private static UnsignedWord indexToMemoryOffset(UnsignedWord index) {
        return index.multiply(MEMORY_BYTES_PER_ENTRY);
    }

    private static Pointer indexToMemoryPointer(UnsignedWord index, Pointer memory) {
        UnsignedWord offset = FirstObjectTable.indexToMemoryOffset(index);
        return memory.add(offset);
    }

    private static UnsignedWord memoryOffsetToIndex(UnsignedWord offset) {
        UnsignedWord result = offset.unsignedDivide(MEMORY_BYTES_PER_ENTRY);
        return result;
    }

    private static int memoryOffsetToEntry(UnsignedWord memoryOffset) {
        assert (memoryOffset.belowThan(MEMORY_BYTES_PER_ENTRY)) : "Offset out of bounds.";
        UnsignedWord scaledOffset = memoryOffset.unsignedDivide(FirstObjectTable.memoryOffsetScale());
        assert (scaledOffset.multiply(FirstObjectTable.memoryOffsetScale()).equal(memoryOffset)) : "Not a multiple.";
        long result = -scaledOffset.rawValue();
        assert (-128L <= result && result <= 0L) : "Scaled offset out of bounds.";
        return (int)result;
    }

    private static UnsignedWord entryToMemoryOffset(int entry) {
        assert (FirstObjectTable.isMemoryOffsetEntry(entry)) : "Entry out of bounds.";
        UnsignedWord entryUnsigned = WordFactory.unsigned((int)(-entry));
        UnsignedWord result = entryUnsigned.multiply(FirstObjectTable.memoryOffsetScale());
        assert (result.belowThan(MEMORY_BYTES_PER_ENTRY)) : "Entry out of bounds.";
        return result;
    }

    private static UnsignedWord indexMinusDelta(UnsignedWord index, UnsignedWord delta) {
        assert (delta.belowOrEqual(index)) : "Delta out of bounds.";
        return index.subtract(delta);
    }

    private static void entryToLog(Log log, int entry) {
        log.signed(entry).string(":");
        if (FirstObjectTable.isUninitializedEntry(entry)) {
            log.string("UninitializedEntry");
        } else if (FirstObjectTable.isMemoryOffsetEntry(entry)) {
            log.string("Memory:").signed((WordBase)FirstObjectTable.entryToMemoryOffset(entry));
        } else if (FirstObjectTable.isLinearOffsetEntry(entry)) {
            log.string("Linear:").signed(entry);
        } else if (FirstObjectTable.isExponentialOffsetEntry(entry)) {
            log.string("Exponent:").signed(FirstObjectTable.unbiasExponent(entry));
        } else {
            log.string("Unknown");
        }
    }

    public static final class TestingBackDoor {
        private TestingBackDoor() {
        }

        public static void initializeTableToPointerForAsserts(Pointer table, Pointer tableLimit) {
            FirstObjectTable.initializeTableToPointerForAsserts(table, tableLimit);
        }

        public static void initializeTableToIndexForAsserts(Pointer table, UnsignedWord indexLimit) {
            FirstObjectTable.initializeTableToIndexForAsserts(table, indexLimit);
        }

        public static void setTableForObject(Pointer table, Pointer memory, Pointer start, Pointer end) {
            FirstObjectTable.setTableForObjectUnchecked(table, memory, start, end);
        }

        public static int getEntryAtIndex(Pointer table, UnsignedWord index) {
            return FirstObjectTable.getEntryAtIndex(table, index);
        }

        public static Pointer getPreciseFirstObjectPointer(Pointer tableStart, Pointer memoryStart, Pointer memoryLimit, UnsignedWord index) {
            return FirstObjectTable.getPreciseFirstObjectPointer(tableStart, memoryStart, memoryLimit, index);
        }

        public static boolean memoryOffsetStartsCard(UnsignedWord offset) {
            return FirstObjectTable.memoryOffsetStartsCard(offset);
        }

        public static boolean memoryOffsetAndLengthCrossesCard(UnsignedWord offset, UnsignedWord length) {
            return FirstObjectTable.memoryOffsetAndLengthCrossesCard(offset, length);
        }

        static void indexToLog(Pointer tableStart, Log log, UnsignedWord index) {
            FirstObjectTable.entryToLog(log, TestingBackDoor.getEntryAtIndex(tableStart, index));
        }

        public static int getMemoryBytesPerEntry() {
            return MEMORY_BYTES_PER_ENTRY;
        }

        public static int getMemoryOffsetScale() {
            return FirstObjectTable.memoryOffsetScale();
        }

        public static int getMemoryOffsetMax() {
            return 0;
        }

        public static int getLinearOffsetMin() {
            return 1;
        }

        public static int getLinearOffsetMax() {
            return 63;
        }

        public static int getExponentBias() {
            return 58;
        }

        public static int getUninitializedEntry() {
            return 127;
        }

        public static UnsignedWord getTableSizeForMemorySize(UnsignedWord memorySize) {
            return FirstObjectTable.getTableSizeForMemorySize(memorySize);
        }

        public static UnsignedWord getTableSizeForMemoryPointers(Pointer memoryStart, Pointer memoryLimit) {
            return FirstObjectTable.getTableSizeForMemoryPointers(memoryStart, memoryLimit);
        }
    }
}

