/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: FormManager.java 21 2011-05-09 16:43:58Z sletellier $
 * $HeadURL: http://svn.chorem.org/svn/vradi/tags/vradi-0.6/vradi-services/src/main/java/org/chorem/vradi/services/managers/FormManager.java $
 * %%
 * Copyright (C) 2009 - 2010 Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.vradi.services.managers;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.vradi.VradiConstants;
import org.chorem.vradi.VradiConstants.SessionStatus;
import org.chorem.vradi.VradiServiceConfigurationHelper;
import org.chorem.vradi.beans.FormPagedResult;
import org.chorem.vradi.beans.QueryBean;
import org.chorem.vradi.beans.SendingHelper;
import org.chorem.vradi.entities.Form;
import org.chorem.vradi.entities.FormLink;
import org.chorem.vradi.entities.FormLinkTypeEnum;
import org.chorem.vradi.entities.Group;
import org.chorem.vradi.entities.GroupForms;
import org.chorem.vradi.entities.ModificationTag;
import org.chorem.vradi.entities.Sending;
import org.chorem.vradi.entities.SendingImpl;
import org.chorem.vradi.entities.Session;
import org.chorem.vradi.entities.SessionImpl;
import org.chorem.vradi.entities.Status;
import org.chorem.vradi.entities.StatusImpl;
import org.chorem.vradi.entities.Thesaurus;
import org.chorem.vradi.entities.User;
import org.chorem.vradi.services.VradiException;
import org.chorem.vradi.services.search.UnsupportedQueryException;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.DateUtil;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyUtil;
import org.nuiton.wikitty.entities.BusinessEntity;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.search.Search;
import org.nuiton.wikitty.search.operators.Element;

import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import static org.chorem.vradi.VradiConstants.FormStatus;
import static org.chorem.vradi.VradiConstants.SendingStatus;
import static org.nuiton.i18n.I18n._;

/**
 * Class containing the methods to manage the forms :
 * - creation, update, deletion, retrieving
 * - binding forms with clients whose queries are returning the forms
 *
 * @author schorlet
 * @version $Revision: 21 $ $Date: 2011-05-09 18:43:58 +0200 (lun., 09 mai 2011) $
 */
public class FormManager {

    private static final Log log = LogFactory.getLog(FormManager.class);

    protected ApplicationConfig config;

    protected WikittyProxy wikittyProxy;

    protected ThesaurusManager thesaurusManager;

    protected SearchManager searchManager;

    public FormManager(ApplicationConfig config,
                       WikittyProxy wikittyProxy,
                       ThesaurusManager thesaurusManager,
                       SearchManager searchManager) {

        this.config = config;
        this.wikittyProxy = wikittyProxy;
        this.thesaurusManager = thesaurusManager;
        this.searchManager = searchManager;
    }

    /**
     * Update forms and attach forms to specified thesaurus.
     *
     * @param form     form to modify
     * @param thesaurus thesaurus to attach form to (set empty set to clear thesaurus, null = no thesaurus management)
     * @return forms saved
     * @throws VradiException if exception occured
     */
    public Form updateForm(Form form, Collection<String> thesaurus) throws VradiException {
        return updateForms(Collections.singletonList(form), thesaurus).get(0);
    }

    /**
     * Update forms and attach forms to specified thesaurus.
     *
     * @param forms     forms to save
     * @param thesaurus thesaurus to attach form to (set empty set to clear thesaurus, null = no thesaurus management)
     * @return forms saved
     * @throws VradiException if exception occured
     */
    public List<Form> updateForms(List<Form> forms, Collection<String> thesaurus) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateForms(forms)");
        }

        if (forms == null || forms.isEmpty()) {
            return forms;
        }

        List<Thesaurus> entitiesToSave = new ArrayList<Thesaurus>();
        for (Form form : forms) {
            if (log.isDebugEnabled()) {
                log.debug("updating form: " + form.getId());
            }

            Date now = new Date();
            if (form.getCreationDate() == null) {
                form.setCreationDate(now);
            } else {
                ModificationTag modificationTag = wikittyProxy.cast(form, ModificationTag.class);
                modificationTag.setLastModified(now);
            }

            // add specified thesaurus to form
            if (thesaurus != null) {

                // search for thesaurus containing current form
                // to remove it
                List<Thesaurus> attachedThesauruses = thesaurusManager.getThesaurusAttachedToForm(form);
                for (Thesaurus attachedThesaurus : attachedThesauruses) {
                    attachedThesaurus.removeAttachment(form.getId());
                    if (!entitiesToSave.contains(attachedThesaurus)) {
                        entitiesToSave.add(attachedThesaurus);
                    }
                }

                for (String thesaurusId : thesaurus) {
                    Thesaurus node = thesaurusManager.getThesaurus(thesaurusId);

                    if (node.getAttachment() == null
                        || !node.getAttachment().contains(form.getWikittyId())) {

                        node.addAttachment(form.getWikittyId());

                        if (!entitiesToSave.contains(node)) {
                            entitiesToSave.add(node);
                        }
                    }
                }
            }
        }

        // also save form
        List<Form> result = wikittyProxy.store(forms);

        if (!entitiesToSave.isEmpty()) {
            wikittyProxy.store(entitiesToSave);
        }

        return result;
    }

    public Session getLastCloseSession() {

        // Get last close session to search after this one
        Search search = Search.query().eq(Element.ELT_EXTENSION, Session.EXT_SESSION);
        search.eq(Session.FQ_FIELD_SESSION_STATUS,
                  String.valueOf(VradiConstants.SessionStatus.SENT.getValue()))
                .criteria();

        Criteria criteria = search.criteria();
        criteria.addSortDescending(Session.FQ_FIELD_SESSION_SESSIONDATE);

        PagedResult<Session> result = wikittyProxy.findAllByCriteria(Session.class, criteria);
        if (result == null || result.size() == 0) {
            return null;
        }
        return result.getFirst();
    }

    public Session getLastOpenSession() throws VradiException {

        Session lastClose = getLastCloseSession();

        String formatedLastCloseDate = null;
        if (lastClose != null) {

            if (log.isDebugEnabled()) {
                log.debug("Last closed session : " + lastClose.getSessionDate());
            }
            try {
                formatedLastCloseDate = WikittyUtil.formatDate(lastClose.getSessionDate());
            } catch (ParseException e) {
                log.error("Cant parse date");
            }
        }
        Search search = Search.query().eq(Element.ELT_EXTENSION, Session.EXT_SESSION);

        if (formatedLastCloseDate != null) {
            search.ge(Session.FQ_FIELD_SESSION_SESSIONDATE,
                      formatedLastCloseDate);
        }

        search.or().eq(Session.FQ_FIELD_SESSION_STATUS,
                       String.valueOf(SessionStatus.ACTIVE.getValue()))
                .eq(Session.FQ_FIELD_SESSION_STATUS,
                    String.valueOf(SessionStatus.STOPPED.getValue()))
                .eq(Session.FQ_FIELD_SESSION_STATUS,
                    String.valueOf(SessionStatus.ERROR.getValue()));

        Criteria criteria = search.criteria();

        criteria.addSortDescending(Session.FQ_FIELD_SESSION_SESSIONDATE);

        PagedResult<Session> result = wikittyProxy.findAllByCriteria(Session.class, criteria);
        Session session = null;
        if (result != null && result.size() > 0) {
            session = result.getFirst();
        }

        if (log.isDebugEnabled() && session != null) {
            log.debug("Last sessions found : " + session.getWikittyId() + " - "
                      + session.getSessionDate() + " with status : "
                      + session.getStatus());
        }

        return session;
    }

    /**
     * Find all session for specific date.
     *
     * @param sessionDate session date
     * @return all session for date
     * @throws VradiException if exception occured
     */
    public List<Session> getSessions(Date sessionDate) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getSessions for date " + sessionDate.toString());
        }
        Search search = Search.query().eq(Element.ELT_EXTENSION, Session.EXT_SESSION);

        search.bw(Session.FQ_FIELD_SESSION_SESSIONDATE,
                  getBeginOfDaySolrDate(sessionDate),
                  getEndOfDaySolrDate(sessionDate));

        // Get result
        Criteria criteria = search.criteria();
        List<Session> allSession = wikittyProxy.findAllByCriteria(Session.class, criteria).getAll();

        if (log.isDebugEnabled()) {
            log.debug(allSession.size() + " sessions found");
        }

        return allSession;
    }

    /**
     * Retourne la date a minuit au format solr.
     *
     * @param date
     * @return
     * @throws VradiException
     */
    protected static String getBeginOfDaySolrDate(Date date) throws VradiException {
        Date localDate = DateUtil.setMinTimeOfDay(date);
        try {
            return WikittyUtil.formatDate(localDate);
        } catch (ParseException eee) {
            throw new VradiException("Cant format date " + date, eee);
        }
    }

    /**
     * Retourne la date a 23h59:59.999 au format solr.
     *
     * @param date
     * @return
     * @throws VradiException
     */
    protected static String getEndOfDaySolrDate(Date date) throws VradiException {
        Date localDate = DateUtil.setMaxTimeOfDay(date);
        try {
            return WikittyUtil.formatDate(localDate);
        } catch (ParseException eee) {
            throw new VradiException("Cant format date " + date, eee);
        }
    }

    /**
     * Crée une nouvelle session initialisée.
     *
     * @return la session créée
     * @throws VradiException
     */
    public Session createNewSession() throws VradiException {

        Session lastSession = getLastOpenSession();
        if (lastSession != null) {
            throw new VradiException(_("Already opened session on %tF", lastSession.getSessionDate()));
        }

        // Get all session for this date
        Date now = new Date();

        // TODO EC201009089 ca sert a rien de toutes les récuperer pour avoir le dernier numéro
        List<Session> sessionList = getSessions(now);

        // Calculate number af session for day
        int num = sessionList.size() + 1;

        // create new session
        if (log.isDebugEnabled()) {
            log.debug("No session found, creating a new one with num : " + num);
        }

        Session currentSession = new SessionImpl();
        currentSession.setSessionDate(now);
        currentSession.setNum(num);
        currentSession.setStatus(SessionStatus.ACTIVE.getValue());
        currentSession.setParagraph(VradiServiceConfigurationHelper.getSessionDefaultParagraph(config));

        currentSession = wikittyProxy.store(currentSession);

        return currentSession;
    }

    /**
     * Recherche tous les formulaires non lié et les lie à la session
     * demandée.
     *
     * @param currentSession session a remplir
     * @return la session en argument
     * @throws VradiException
     */
    public Session bindForms(Session currentSession) throws VradiException {

        if (log.isDebugEnabled()) {
            log.debug("bindForms " + currentSession.getNum() + " : " + currentSession.getSessionDate());
        }

        try {
            // Throw exception if session is not active
            if (!SessionStatus.isActive(currentSession)) {
                throw new VradiException("Session is not active, abording");
            }

            // un sending = un utilsateur et ses formulaires
            List<Sending> toSend = resetSendings(currentSession);
            if (toSend == null) {
                toSend = new ArrayList<Sending>();
            }

            // Restore updated session
            currentSession = wikittyProxy.restore(Session.class, currentSession.getWikittyId());

            List<GroupForms> groupFormToSave = new ArrayList<GroupForms>();

            // recherche de tous les executeurs de requette (client, groupes...)
            // was all query make before
            List<Group> groups = searchManager.findGroupsWithQueries();

            // iterate on the group
            for (Group group : groups) {

                // Only selected forms
                Set<String> formsIdsToBind = getFormsIdsFromRequest(group, getSelectionneStatus());

                // if some forms are to be bound, bind them
                if (!formsIdsToBind.isEmpty()) {

                    Set<String> userIds = group.getUser();
                    if (userIds != null) {
                        List<User> users = wikittyProxy.restore(User.class, new ArrayList<String>(userIds));

                        for (User user : users) {

                            // Create or find current sending for queryMaker
                            Sending sending = findExistingOrCreateSending(toSend, user);

                            sending.setUser(user.getWikittyId());
                            if (log.isDebugEnabled()) {
                                log.debug("Add user " + user.getWikittyId());
                            }

                            // Replace replace forms by previous
                            formsIdsToBind = repaceByPreviousReplacedForms(formsIdsToBind);

                            ArrayList<String> formsIdsAsList = new ArrayList<String>(formsIdsToBind);

                            Set<Form> formsToBind = new HashSet<Form>(wikittyProxy.restore(Form.class, formsIdsAsList));

                            // Remove already bound forms with queryMaker
                            formsToBind = removeAlreadyBound(currentSession, formsToBind, user);

                            formsToBind = removeNotConcernedForms(sending, user, formsToBind);
                            if (log.isDebugEnabled()) {
                                log.debug("Adding " + formsToBind.size() + " forms to sending");
                            }

                            // If not empty
                            if (!formsToBind.isEmpty()) {
                                GroupForms groupForms = SendingHelper.createGroupForms(group, formsToBind);
                                groupFormToSave.add(groupForms);
                                sending.addGroupForms(groupForms.getWikittyId());
                                if (log.isDebugEnabled()) {
                                    log.debug("GroupForm created");
                                }

                                toSend.add(sending);
                            }
                        }
                    }
                }
            }
            if (toSend.isEmpty()) {
                return currentSession;
            }

            for (Sending s : toSend) {
                currentSession.addSending(s.getWikittyId());
            }

            // Store the session and sending
            List<BusinessEntity> toSave = new ArrayList<BusinessEntity>();
            toSave.addAll(groupFormToSave);
            toSave.addAll(toSend);
            toSave.add(currentSession);

            wikittyProxy.store(toSave);

            // Restore session
            currentSession = wikittyProxy.restore(Session.class, currentSession.getWikittyId());

            if (log.isDebugEnabled()) {
                Set<String> sending = currentSession.getSending();
                int size = sending == null ? 0 : sending.size();

                log.debug("Saving session '" + currentSession.getSessionDate()
                          + "' number '" + currentSession.getNum()
                          + "' status '" + SessionStatus.getStatus(currentSession.getStatus()).getDescription()
                          + "' with '" + size
                          + "' sending");
            }

        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't bind form : ", e);
            }
            throw new VradiException("Can't bind form : ", e);
        }
        return currentSession;
    }

    protected Set<String> getFormsIdsFromRequest(Group group, Status status) throws VradiException {

        // List of the forms to bind ( = not yet bound)
        Set<String> formsIdsToBind = new HashSet<String>();

        // normalement c'est pas null, sinon la requette findGroupsWithQueries
        // marche pas
        Set<String> queries = group.getQueries();

        for (String queryEntry : queries) {

            try {
                // Get query
                QueryBean queryBean = new QueryBean(queryEntry);
                String query = queryBean.getQuery();

                // Create filter
                Search filter = Search.query();
                filter.eq(Form.FQ_FIELD_INFOGENE_STATUS, status.getWikittyId());

                // Limited on older inscription date
                Search filterDateModification = filter.or();
                Date olderInscriptionDate = getOlderInscriptionDate(group.getUser());

                // Limited on date modification
                filterDateModification.isNull(ModificationTag.FQ_FIELD_MODIFICATIONTAG_LASTMODIFIED);
                filterDateModification.ge(ModificationTag.FQ_FIELD_MODIFICATIONTAG_LASTMODIFIED, WikittyUtil.formatDate(olderInscriptionDate));

                // Limited on date peremption
                Search filterDatePeremption = filter.or();
                filterDatePeremption.isNull(Form.FQ_FIELD_FORM_DATEPEREMPTION);
                filterDatePeremption.ge(Form.FQ_FIELD_FORM_DATEPEREMPTION, WikittyUtil.formatDate(new Date()));

                Criteria criteria = filter.criteria();

                // create a new VradiFormPage to find all the forms corresponding to the query
                FormPagedResult vradiFormPage = new FormPagedResult();

                // Execute request
                vradiFormPage = searchManager.findForms(query, criteria, vradiFormPage);

                // Extract result
                Set<String> resultFormsIds = new HashSet<String>(vradiFormPage.getFormsIdsToShow());

                if (log.isDebugEnabled()) {
                    log.debug("Query " + query + " return " + resultFormsIds.size() + " forms");
                }

                formsIdsToBind.addAll(resultFormsIds);
            } catch (IOException eee) {
                throw new VradiException("Can't parse group query", eee);

            } catch (UnsupportedQueryException eee) {
                throw new VradiException("Can't execute group query", eee);

            } catch (ParseException eee) {
                throw new VradiException("Cant format date : ", eee);
            }
        }
        return formsIdsToBind;
    }

    /**
     * Use to replace forms by previous replace forms
     *
     * @param resultFormsIds to inspect
     * @return return forms with previous replace forms
     */
    protected Set<String> repaceByPreviousReplacedForms(Set<String> resultFormsIds) {
        List<String> copy = new ArrayList<String>(resultFormsIds);
        for (String formId : copy) {
            String previousReplaceFormId = getPreviousReplaceForm(formId);
            if (previousReplaceFormId != null) {
                if (log.isDebugEnabled()) {
                    Form form = wikittyProxy.restore(Form.class, formId);
                    Form previousReplaceForm = wikittyProxy.restore(Form.class, previousReplaceFormId);

                    log.debug("[Replace] for form : '" + form.getObjet() + "' id '" + form.getId() + "' by previous '" + previousReplaceForm.getId() + "'");
                }

                resultFormsIds.remove(formId);
                resultFormsIds.add(previousReplaceFormId);
            }
        }
        return resultFormsIds;
    }

    protected List<Sending> resetSendings(Session currentSession) throws VradiException {

        Set<String> sendingsIds = currentSession.getSending();
        if (sendingsIds == null) {
            return null;
        }

        List<BusinessEntity> toSave = new ArrayList<BusinessEntity>();

        // Reset sendings
        List<Sending> sendings = wikittyProxy.restore(Sending.class, new ArrayList<String>(sendingsIds));

        Set<String> sendingIdsToDelete = new HashSet<String>();
        Set<String> groupFormIdToDelete = new HashSet<String>();
        for (Sending sending : new ArrayList<Sending>(sendings)) {

            if (sending != null) {

                // If its not a manual add, restart with an empty sending
                // Remove all binded forms
                Set<String> concernedGroup = sending.getGroupForms();
                if (concernedGroup != null) {
                    List<String> concernedGroupList = new ArrayList<String>(concernedGroup);
                    groupFormIdToDelete.addAll(concernedGroupList);
                    sending.clearGroupForms();
                }

                // Restore all added forms manualy
                Set<String> groupFormsAddedIds = sending.getAddedGroupForms();
                if (groupFormsAddedIds != null) {
                    for (String groupFormsAddedId : groupFormsAddedIds) {

                        // Other that deleted
                        GroupForms groupFormsAdded = wikittyProxy.restore(GroupForms.class, groupFormsAddedId);
                        Set<String> addedForms = SendingHelper.extractForms(groupFormsAdded);
                        Set<String> deletedForms = SendingHelper.extractForms(wikittyProxy, sending.getDeletedGroupForms());

                        addedForms.removeAll(deletedForms);

                        GroupForms groupForms = SendingHelper.createGroupForms(groupFormsAdded.getGroup(), addedForms);
                        toSave.add(groupForms);
                        sending.addGroupForms(groupForms.getWikittyId());
                    }
                }

                // Delete empty sending
                Set<String> deletedGroupForm = sending.getDeletedGroupForms();
                if ((concernedGroup == null || concernedGroup.isEmpty()) &&
                    (groupFormsAddedIds == null || groupFormIdToDelete.isEmpty()) &&
                    (deletedGroupForm == null || deletedGroupForm.isEmpty())) {

                    String sendingId = sending.getWikittyId();
                    sendingIdsToDelete.add(sendingId);
                    currentSession.removeSending(sendingId);
                    sendings.remove(sending);
                }
            }
        }
        wikittyProxy.delete(groupFormIdToDelete);
        wikittyProxy.delete(sendingIdsToDelete);

        toSave.add(currentSession);
        toSave.addAll(sendings);
        wikittyProxy.store(toSave);

        sendingsIds = currentSession.getSending();
        if (sendingsIds == null) {
            return null;
        }
        return wikittyProxy.restore(Sending.class, new ArrayList<String>(sendingsIds));
    }

    protected Set<Form> removeNotConcernedForms(Sending sending, User user, Set<Form> finalFormsToBind) {

        // Get user begin search date
        Date beginSearchDate = user.getBeginSearchDate();

        // Check date
        for (Form f : new ArrayList<Form>(finalFormsToBind)) {

            // Using last modification date
            ModificationTag formTag = wikittyProxy.cast(f, ModificationTag.class);
            Date lastModifier = formTag.getLastModified();

            // Get peremption date
            Date peremptionDate = f.getDatePeremption();

            // If not enable, not attach forms
            if (!user.getEnable()) {
                if (log.isDebugEnabled()) {
                    log.debug("Removing form '" +
                              f.getObjet() +
                              "' because user (" +
                              user.getName() +
                              ") is not enable");
                }

                finalFormsToBind.remove(f);
            }

            // If date of peremption of form is pass
            if (lastModifier != null && lastModifier.before(beginSearchDate)) {
                if (log.isDebugEnabled()) {
                    log.debug("Removing form '" +
                              f.getObjet() +
                              "' because date of last modification (" +
                              peremptionDate +
                              ") is pass for user : " +
                              user.getName());
                }

                finalFormsToBind.remove(f);
            }

            // If date of peremption of form is pass
            if (peremptionDate != null && peremptionDate.before(new Date())) {
                if (log.isDebugEnabled()) {
                    log.debug("Removing form '" +
                              f.getObjet() +
                              "' because date of peremption (" +
                              peremptionDate +
                              ") is pass for user : " +
                              user.getName());
                }

                finalFormsToBind.remove(f);
            }

            // Remove deleted if are reassociated
            Set<String> deletedFormsIds = SendingHelper.extractForms(wikittyProxy, sending.getDeletedGroupForms());
            String fid = f.getWikittyId();
            if (deletedFormsIds.contains(fid)) {
                finalFormsToBind.remove(f);
                if (log.isDebugEnabled()) {
                    log.debug("Removing form '" +
                              f.getObjet() +
                              "' because form is already deleted" +
                              " is pass for user : " +
                              user.getName());
                }
            }
        }
        return finalFormsToBind;
    }

    /**
     * Retourne la liste {@code forms} d'origine en ayant filtré les formulaires
     * qui ont déja été lié.
     *
     * @param currentSession
     * @param forms
     * @param user
     * @return
     */
    protected Set<Form> removeAlreadyBound(Session currentSession, Set<Form> forms, User user) {
        Set<String> formIds = new HashSet<String>();

        // Extract formIds
        for (Form form : forms) {
            formIds.add(form.getWikittyId());
        }

        // Recherche d'un sending existant
        String userId = user.getWikittyId();

        // Search all groupForms concerned
        Search searchGroupForm = Search.query();

        // FIXME sletellier 24/10/10 : find a way to join tow requests
        searchGroupForm.contains(GroupForms.FQ_FIELD_GROUPFORMS_FORM, formIds);
        List<GroupForms> groupForms = wikittyProxy.findAllByCriteria(GroupForms.class, searchGroupForm.criteria()).getAll();
        List<String> groupFormsId = new ArrayList<String>();

        for (GroupForms gf : groupForms) {
            if (gf != null) {
                groupFormsId.add(gf.getWikittyId());
            }
        }

        Search searchSending = Search.query();

        Set<String> sendingIds = currentSession.getSending();

        // Except currents
        if (sendingIds != null) {
            for (String currentSendingId : sendingIds) {
                searchSending.neq(Element.ELT_ID, currentSendingId);
            }
        }

        // If its attached or deleted
        Search formSearch = searchSending.or();
        for (String groupFormId : groupFormsId) {
            formSearch.eq(Sending.FQ_FIELD_SENDING_GROUPFORMS, groupFormId);
            formSearch.eq(Sending.FQ_FIELD_SENDING_DELETEDGROUPFORMS, groupFormId);
        }

        // And attached to client / user / group
        searchSending.eq(Sending.FQ_FIELD_SENDING_USER, userId);

        // Find
        Criteria criteria = searchSending.criteria();

        PagedResult<Sending> pagedResult = wikittyProxy.findAllByCriteria(Sending.class, criteria);

        for (Sending send : pagedResult.getAll()) {
            Set<String> formsIds = SendingHelper.extractForms(wikittyProxy, send.getGroupForms());
            if (formsIds != null) {
                formIds.removeAll(formsIds);
            }
            Set<String> deletedIds = SendingHelper.extractForms(wikittyProxy, send.getDeletedGroupForms());
            if (deletedIds != null) {
                formIds.removeAll(deletedIds);
            }
        }
        if (log.isDebugEnabled()) {
            log.debug("Rest " + formIds.size() + " forms");
        }

        List<Form> result = wikittyProxy.restore(Form.class, new ArrayList<String>(formIds));
        return new HashSet<Form>(result);
    }

    /**
     * Retourne la plus ancienne date du query maker si c'est une date
     * ou des user gu groupe si c'est un groupe.
     *
     * @param usersIds
     * @return olderInscriptionDate
     */
    protected Date getOlderInscriptionDate(Set<String> usersIds) {
        Date result = new Date();

        if (usersIds != null) {
            for (String userId : usersIds) {
                User user = wikittyProxy.restore(User.class, userId);

                Date beginSearchDate = user.getBeginSearchDate();
                if (beginSearchDate != null && result.after(DateUtil.setMinTimeOfDay(beginSearchDate))) {
                    result = beginSearchDate;
                }
                if (log.isDebugEnabled()) {
                    log.debug("The older inscription date for user " + user.getName() + " is " + result);
                }
            }

            if (log.isDebugEnabled()) {
                log.debug("The older inscription date " + result);
            }
        }
        return result;
    }

    public List<Sending> createAllSending(String currentSessionId, Group group, List<String> formsIdsToBind) {
        return createAllSending(currentSessionId, group, new HashSet<String>(formsIdsToBind));
    }

    public List<Sending> createAllSending(String currentSessionId, Group group, Set<String> formsIdsToBind) {
        Session currentSession = wikittyProxy.restore(Session.class, currentSessionId);
        List<Sending> result = new ArrayList<Sending>();

        Set<String> userIds = group.getUser();
        if (userIds != null) {
            List<User> users = wikittyProxy.restore(User.class, new ArrayList<String>(userIds));

            for (User user : users) {
                Sending sending = createUserSending(currentSession, group, user, formsIdsToBind);
                if (sending != null) {
                    result.add(sending);
                }
            }
        }
        return result;
    }

    public Sending createUserSending(String currentSessionId, User user, List<String> formsIdsToBind) {
        return createUserSending(currentSessionId, user, new HashSet<String>(formsIdsToBind));
    }

    public Sending createUserSending(String currentSessionId, User user, Set<String> formsIdsToBind) {
        Session currentSession = wikittyProxy.restore(Session.class, currentSessionId);

        return createUserSending(currentSession, null, user, formsIdsToBind);
    }

    protected Sending createUserSending(Session currentSession, Group groupConcerned, User user, Set<String> formsIdsToBind) {

        // Create or find current sending for queryMaker
        Sending sending = findExistingOrCreateSending(currentSession, user);

        sending.setUser(user.getWikittyId());
        if (log.isDebugEnabled()) {
            log.debug("Add user " + user.getWikittyId());
        }

        // If not empty
        if (!formsIdsToBind.isEmpty()) {

            List<GroupForms> groupFormsList = new ArrayList<GroupForms>();

            GroupForms groupForms = SendingHelper.createGroupForms(groupConcerned, formsIdsToBind);
            groupFormsList.add(groupForms);
            sending.addGroupForms(groupForms.getWikittyId());
            if (log.isDebugEnabled()) {
                log.debug("GroupForm created");
            }

            // If is an manual add, keep it
            if (log.isDebugEnabled()) {
                log.debug("Adding added " + formsIdsToBind.size() + " forms for user : " +
                          user.getName());
            }

            // Create an other to keep it in state
            GroupForms groupFormsAdded = SendingHelper.createGroupForms(groupConcerned, formsIdsToBind);
            groupFormsList.add(groupFormsAdded);
            sending.addAddedGroupForms(groupFormsAdded.getWikittyId());

            currentSession.addSending(sending.getWikittyId());

            // store datas
            List<BusinessEntity> toSave = new ArrayList<BusinessEntity>();

            // update sending status if deleted
            if (SendingStatus.isDeleted(sending)) {
                sending.setStatus(SendingStatus.TO_SEND.getValue());
            }

            toSave.addAll(groupFormsList);
            toSave.add(sending);

            toSave.add(currentSession);
            wikittyProxy.store(toSave);

            return wikittyProxy.restore(Sending.class, sending.getWikittyId());
        }
        return null;
    }

    public List<Sending> removeAllSending(String sessionId, List<String> formsIds, Group group) throws VradiException {
        return removeAllSending(sessionId, new HashSet<String>(formsIds), group);
    }

    public List<Sending> removeAllSending(String sessionId, Set<String> formsIds, Group group) throws VradiException {
        List<Sending> result = new ArrayList<Sending>();

        Set<String> userIds = group.getUser();
        if (userIds != null) {
            List<User> users = wikittyProxy.restore(User.class, new ArrayList<String>(userIds));
            for (User user : users) {
                Sending sending = removeSending(sessionId, formsIds, group, user);

                if (sending != null) {
                    result.add(sending);
                }
            }
            return wikittyProxy.store(result);
        }
        return Collections.EMPTY_LIST;
    }

    public Sending removeSending(String sessionId, List<String> formsIds, User user) throws VradiException {
        return removeSending(sessionId, new HashSet<String>(formsIds), null, user);
    }

    protected Sending removeSending(String sessionId, Set<String> formsIds, Group group, User user) throws VradiException {
        try {
            Session session = wikittyProxy.restore(Session.class, sessionId);
            Set<String> sendingSet = session.getSending();
            List<String> sendingList = new ArrayList<String>();
            if (sendingSet != null) {
                sendingList = new ArrayList<String>(sendingSet);
            }

            List<Sending> sendings = wikittyProxy.restore(Sending.class, sendingList);

            for (Sending sending : sendings) {

                // Only for sendings to send
                if (sending != null && SendingStatus.isToSend(sending)) {
                    if (user.getWikittyId().equals(sending.getUser())) {
                        boolean formAreRemoved = false;
                        List<GroupForms> groupForms = wikittyProxy.restore(GroupForms.class, new ArrayList<String>(sending.getGroupForms()));

                        Set<String> formsIdsRestored = new HashSet<String>();
                        for (GroupForms gf : groupForms) {
                            if (gf != null) {

                                for (String formId : formsIds) {

                                    if (gf.getForm() != null && gf.getForm().contains(formId)) {
                                        // Remove
                                        gf.removeForm(formId);
                                        formAreRemoved = true;
                                    }
                                }
                                if (gf.getForm() != null) {
                                    formsIdsRestored.addAll(gf.getForm());
                                }
                            }
                        }
                        if (!formAreRemoved) {
                            return sending;
                        }

                        // Add group concerned
                        if (log.isDebugEnabled()) {
                            log.debug("Group concerned for sending of user '" + user.getName() + "' : " + (group == null ? "null" : group.getName()));
                        }

                        GroupForms groupFormsDeleted = SendingHelper.createGroupForms(group, formsIds);
                        sending.addDeletedGroupForms(groupFormsDeleted.getWikittyId());

                        // If is empty, set deleted
                        if (formsIdsRestored == null || formsIdsRestored.isEmpty()) {
                            sending.setStatus(SendingStatus.DELETED.getValue());
                        }

                        // Save all
                        List<BusinessEntity> toSave = new ArrayList<BusinessEntity>();
                        toSave.addAll(groupForms);
                        toSave.add(groupFormsDeleted);
                        toSave.add(sending);

                        wikittyProxy.store(toSave);

                        // Try to close session if it correct all errors
                        closeSessionIfNoMoreError(session);
                        return wikittyProxy.restore(Sending.class, sending.getWikittyId());
                    }
                }
            }
            return null;
        } catch (Exception eee) {
            throw new VradiException("Cant remove forms for user " + user.getName(), eee);
        }
    }

    /**
     * Check if template file form extension name in param
     *
     * @param formsIds to remove
     * @return false if no session is opened
     * @throws VradiException
     */
    public boolean removeFormsFromSession(String sessionId, List<String> formsIds) throws VradiException {
        return removeFormsFromSession(sessionId, new HashSet<String>(formsIds));
    }

    /**
     * Check if template file form extension name in param
     *
     * @param formsIds to remove
     * @return false if no session is opened
     * @throws VradiException
     */
    public boolean removeFormsFromSession(String sessionId, Set<String> formsIds) throws VradiException {
        try {
            Session session = wikittyProxy.restore(Session.class, sessionId);
            if (session == null) {
                return false;
            }

            List<BusinessEntity> toSave = new ArrayList<BusinessEntity>();
            Set<String> sendingsIds = session.getSending();
            if (sendingsIds == null) {
                return false;
            }
            List<Sending> sendings = wikittyProxy.restore(Sending.class, new ArrayList<String>(sendingsIds));
            for (Sending sending : sendings) {

                // Only for sendings to send
                if (sending != null && SendingStatus.isToSend(sending)) {

                    // Remove forms
                    List<GroupForms> groupsForms = wikittyProxy.restore(GroupForms.class, new ArrayList<String>(sending.getGroupForms()));
                    boolean isDeleted = false;
                    for (GroupForms gf : groupsForms) {
                        for (String formId : formsIds) {

                            if (gf != null) {

                                if (gf.getForm() != null && gf.getForm().contains(formId)) {
                                    gf.removeForm(formId);
                                    isDeleted = true;
                                }
                            }
                        }
                    }
                    if (isDeleted) {
                        toSave.addAll(groupsForms);
                        GroupForms groupFormsDeleted = SendingHelper.createGroupForms((String) null, formsIds);
                        sending.addDeletedGroupForms(groupFormsDeleted.getWikittyId());
                        toSave.add(groupFormsDeleted);

                        if (SendingHelper.extractForms(wikittyProxy, sending.getGroupForms()).isEmpty()) {
                            sending.setStatus(SendingStatus.DELETED.getValue());
                        }
                        toSave.add(sending);
                    }
                }
            }
            // Store
            wikittyProxy.store(toSave);

            // Try to close session if it correct all errors
            closeSessionIfNoMoreError(session);
            return true;
        } catch (Exception eee) {
            throw new VradiException("Cant remove all forms for session", eee);
        }
    }

    /**
     * Recherche dans la session l'object sending qui correspond au query maker
     * donné ou en crée une nouvelle s'il n'a pas été trouvé.
     * <p/>
     * Le sending retourné n'est pas affecter au query maker.
     *
     * @param currentSession
     * @param user
     * @return
     */
    protected Sending findExistingOrCreateSending(Session currentSession, User user) {

        // Restore existing sending in session
        Set<String> sendingsId = currentSession.getSending();

        // restore sendings
        List<Sending> sendings;
        if (sendingsId != null) {
            sendings = wikittyProxy.restore(Sending.class, new ArrayList<String>(sendingsId));
        } else {
            sendings = new ArrayList<Sending>();
        }

        return findExistingOrCreateSending(sendings, user);
    }

    /**
     * Recherche dans la session l'object sending qui correspond au query maker
     * donné ou en crée une nouvelle s'il n'a pas été trouvé.
     * <p/>
     * Le sending retourné n'est pas affecter au query maker.
     *
     * @param sendings
     * @param user
     * @return
     */
    protected Sending findExistingOrCreateSending(List<Sending> sendings, User user) {

        // chargement, si il existe, du sending concernant
        // ce client / user / group
        for (Sending sending : sendings) {
            if (sending != null) {
                String userId = user.getWikittyId();
                if (sending != null && userId.equals(sending.getUser())) {

                    if (log.isDebugEnabled()) {
                        log.debug("Sending is existing for queryMaker : " + user.getName());
                    }

                    return sending;
                }
            }
        }
        return createNewSending(user);
    }

    protected Sending createNewSending(User user) {

        // Set properties in default
        SendingImpl newSending = new SendingImpl();
        newSending.setSentDate(null);
        newSending.setReceptionDate(null);
        newSending.setParagraph(VradiConstants.DEFAULT_SENDING_PARAGRAPH);
        newSending.setReceptionProof(!user.getValidEmail());
        newSending.setStatus(SendingStatus.TO_SEND.getValue());
        newSending.setUser(user.getWikittyId());

        if (log.isDebugEnabled()) {
            log.debug("Sending dont exist for queryMaker : " + user + ", creating one");
        }
        return newSending;
    }

    public Sending revertDeletion(String sendingId, List<String> formsIdsToUpdate) {
        return revertDeletion(sendingId, null, new HashSet<String>(formsIdsToUpdate), true);
    }

    protected Sending revertDeletion(String sendingId, String groupId, Set<String> formsIdsToUpdate, boolean save) {
        Sending sending = wikittyProxy.restore(Sending.class, sendingId);

        Set<String> groupFormIds = sending.getDeletedGroupForms();
        if (groupFormIds != null) {
            List<GroupForms> groupForms = wikittyProxy.restore(GroupForms.class, new ArrayList<String>(groupFormIds));

            boolean isDeleted = false;
            List<String> toDeleteIds = new ArrayList<String>();
            for (GroupForms gf : groupForms) {
                if (gf != null) {
                    Set<String> formsInGroupForms = gf.getForm();
                    for (String formId : formsIdsToUpdate) {
                        if (formsInGroupForms != null &&
                            !formsInGroupForms.isEmpty() &&
                            formsInGroupForms.contains(formId)) {

                            gf.removeForm(formId);
                            isDeleted = true;

                            // If empty, delete it
                            if (gf.getForm().isEmpty()) {
                                String toDelete = gf.getWikittyId();
                                sending.removeDeletedGroupForms(toDelete);
                                toDeleteIds.add(toDelete);
                            }
                        }
                    }
                }
            }
            if (!isDeleted) {
                return sending;
            }
            GroupForms groupFormsCreated = SendingHelper.createGroupForms(groupId, formsIdsToUpdate);
            sending.addGroupForms(groupFormsCreated.getWikittyId());
            wikittyProxy.delete(toDeleteIds);
            wikittyProxy.store(groupForms);
            wikittyProxy.store(groupFormsCreated);

            // Update status if was deleted
            if (SendingStatus.isDeleted(sending)) {
                sending.setStatus(SendingStatus.TO_SEND.getValue());
            }
            if (save) {
                return wikittyProxy.store(sending);
            }
            return sending;
        }
        return sending;
    }

    public Sending revertUserDeletion(String sessionId, String userId, String formId) {
        return revertUserDeletion(sessionId, null, userId, formId, true);
    }

    protected Sending revertUserDeletion(String sessionId, String groupId, String userId, String formId, boolean save) {
        Session session = wikittyProxy.restore(Session.class, sessionId);

        Set<String> sendingIds = session.getSending();
        if (sendingIds == null && !SessionStatus.isToSend(session)) {
            return null;
        }

        List<Sending> sendings = wikittyProxy.restore(Sending.class, new ArrayList<String>(sendingIds));
        for (Sending sending : sendings) {

            if (userId.equals(sending.getUser())) {
                Set<String> formIds = new HashSet<String>();
                formIds.add(formId);
                return revertDeletion(sending.getWikittyId(), groupId, formIds, save);
            }
        }
        return null;
    }

    public List<Sending> revertGroupDeletion(String sessionId, String groupId, String formId) {
        List<Sending> result = new ArrayList<Sending>();

        Group group = wikittyProxy.restore(Group.class, groupId);

        Set<String> userIds = group.getUser();
        if (userIds == null) {
            return result;
        }

        for (String userId : userIds) {
            Sending sending = revertUserDeletion(sessionId, groupId, userId, formId, false);

            result.add(sending);
        }
        return wikittyProxy.store(result);
    }

    /**
     * Try to close session if rest no error and save
     *
     * @param session to close
     * @return session closed if rest no error
     */
    protected Session closeSessionIfNoMoreError(Session session) {
        session = wikittyProxy.restore(Session.class, session.getWikittyId());

        Set<String> sendingSet = session.getSending();
        List<String> sendingList = new ArrayList<String>();
        if (sendingSet != null) {
            sendingList = new ArrayList<String>(sendingSet);
        }

        List<Sending> sendings = wikittyProxy.restore(Sending.class, sendingList);

        session = closeSessionIfNoMoreError(session, sendings);

        return wikittyProxy.store(session);
    }

    /**
     * Try to close session if rest no error and save
     *
     * @param session to close
     * @return session closed if rest no error
     */
    protected Session closeSessionIfNoMoreError(Session session, List<Sending> sendings) {

        int sentCnt = 0;
        for (Sending sending : sendings) {
            if (sending != null && SendingStatus.isInError(sending)) {
                return session;
            }
            if (sending == null || SendingStatus.isSent(sending)) {
                sentCnt++;
            }
        }

        boolean allAreSent = sentCnt == sendings.size();
        boolean noOneAreSent = sentCnt == 0;

        if (allAreSent) {
            // Close if all are sent
            session.setStatus(SessionStatus.SENT.getValue());
            session.setSessionDate(new Date());
            session.setSessionLogs(StringUtils.EMPTY);
        } else if (noOneAreSent) {

            // Return to active if no one are sent
            session.setStatus(SessionStatus.ACTIVE.getValue());
            session.setSessionDate(new Date());
            session.setSessionLogs(StringUtils.EMPTY);
        }
        return session;
    }

    public void deleteSession(String sessionId) throws VradiException {

        if (sessionId == null) {
            return;
        }

        // Restore session
        Session sessionToDelete = wikittyProxy.restore(Session.class, sessionId);

        if (!SessionStatus.isActive(sessionToDelete)) {
            throw new VradiException("Session is not active, abording");
        }
        List<String> idToDelete = new ArrayList<String>();

        // Delete all sending asssociated
        Set<String> sendingsIds = sessionToDelete.getSending();
        if (sendingsIds != null) {
            List<Sending> sendings = wikittyProxy.restore(Sending.class, new ArrayList<String>(sendingsIds));
            for (Sending sending : sendings) {
                if (sending != null) {
                    Set<String> toDelete = sending.getAddedGroupForms();
                    if (toDelete != null) {
                        idToDelete.addAll(toDelete);
                    }
                    toDelete = sending.getDeletedGroupForms();
                    if (toDelete != null) {
                        idToDelete.addAll(toDelete);
                    }
                    toDelete = sending.getGroupForms();
                    if (toDelete != null) {
                        idToDelete.addAll(toDelete);
                    }
                }
            }
            idToDelete.addAll(sendingsIds);
        }
        idToDelete.add(sessionId);

        // Delete session
        wikittyProxy.delete(idToDelete);
    }

    public Session getSessionOfSending(String sendingId) throws VradiException {
        try {
            Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, Session.EXT_SESSION)
                    .contains(Session.FQ_FIELD_SESSION_SENDING, sendingId).criteria();

            return wikittyProxy.findByCriteria(Session.class, criteria);
        } catch (Exception eee) {
            log.error("Cant restore session for sending '" + sendingId + "' : ", eee);
            throw new VradiException("Cant restore session for sending '" + sendingId + "' : ", eee);
        }
    }

    /*
     * Status
     */
    public Status getStatus(String statusId) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getStatus(" + statusId + ")");
        }
        Status status = wikittyProxy.restore(Status.class, statusId);
        return status;
    }

    public List<Status> getStatuses(List<String> statusIds) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getStatuses(statusIds)");
        }
        List<Status> statuses = wikittyProxy.restore(Status.class, statusIds);
        return new ArrayList<Status>(statuses);
    }

    public List<Status> createDefaultStatuses() throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("createDefaultStatuses()");
        }

        List<Status> all = new ArrayList<Status>();
        all.add(getNonTraiteStatus());
        all.add(getNonSelectionneStatus());
        all.add(getPreselectionneStatus());
        all.add(getSelectionneStatus());

        return new ArrayList<Status>(all);
    }

    public Status getNonTraiteStatus() {
        return getOrCreateStatus(FormStatus.NON_TRAITE);
    }

    public Status getNonSelectionneStatus() {
        return getOrCreateStatus(FormStatus.NON_SELECTIONNE);
    }

    public Status getPreselectionneStatus() {
        return getOrCreateStatus(FormStatus.PRESELECTIONNE);
    }

    public Status getSelectionneStatus() {
        return getOrCreateStatus(FormStatus.SELECTIONNE);
    }

    protected Status getOrCreateStatus(VradiConstants.FormStatus type) {
        if (log.isDebugEnabled()) {
            log.debug("getOrCreateStatus(" + type.getName() + ")");
        }
        Search search = Search.query()
                .eq(Element.ELT_EXTENSION, Status.EXT_STATUS)
                .eq(Status.FQ_FIELD_STATUS_NAME, type.getName());

        Criteria criteria = search.criteria();

        Status status = wikittyProxy.findByCriteria(Status.class,
                                                    criteria);

        if (status == null) {
            log.warn("Status not found creating : " + type.getName());

            status = new StatusImpl();
            status.setName(type.getName());
            status.setValue(type.getValue());
            status.setDescription(type.getDescription());
            status.setModifiable(false);

            status = wikittyProxy.store(status);
        }
        return status;
    }

    public List<Status> updateStatuses(List<Status> statuses)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateStatuses(statuses)");
        }

        try {
            List<Status> result = wikittyProxy.store(statuses);
            return result;

        } catch (Exception eee) {
            log.error("Cant update statuses : ", eee);
            throw new VradiException("Cant update statuses : ", eee);
        }
    }

    public Status updateStatus(Status status) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateStatus(status)");
        }

        try {
            status = wikittyProxy.store(status);
            return status;

        } catch (Exception eee) {
            log.error("Cant update status : ", eee);
            throw new VradiException("Cant update status : ", eee);
        }
    }

    public boolean deleteStatuses(List<String> statusIds) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("deleteStatus(" + statusIds + ")");
        }
        boolean result = true;

        for (String id : statusIds) {
            // Check if status is in use
            Search search = Search.query().eq(Element.ELT_EXTENSION, Form.EXT_FORM)
                    .eq(Form.FQ_FIELD_INFOGENE_STATUS, id);
            Criteria criteria = search.criteria();
            PagedResult<Form> forms = wikittyProxy.findAllByCriteria(Form.class, criteria);
            if (forms.size() > 0) {
                result = false;
            } else {
                try {
                    wikittyProxy.delete(id);
                } catch (Exception e) {
                    log.error("Cant delete status : ", e);
                    throw new VradiException("Cant delete status : ", e);
                }
            }
        }
        return result;
    }

    protected List<User> getUsersOfClient(String clientId) {

        Search search = Search.query().eq(Element.ELT_EXTENSION, User.EXT_USER)
                .eq(User.FQ_FIELD_USER_CLIENT, clientId);
        Criteria criteria = search.criteria();

        PagedResult<User> usersPageResult = wikittyProxy.findAllByCriteria(User.class, criteria);
        return new ArrayList<User>(usersPageResult.getAll());

    }

    public List<String> getRelatedFormsLinkIds(String formId) {

        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, FormLink.EXT_FORMLINK)
                .eq(FormLink.FQ_FIELD_FORMLINK_FROMFORM, formId)
                .eq(FormLink.FQ_FIELD_FORMLINK_TYPE, FormLinkTypeEnum.RELATED.getValueAsString()).criteria();

        List<FormLink> formLinks = wikittyProxy.findAllByCriteria(FormLink.class, criteria).getAll();
        return extractWikittyId(formLinks);
    }

    public List<String> getReplaceFormsLinkIds(String formId) throws VradiException {
        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, FormLink.EXT_FORMLINK)
                .eq(FormLink.FQ_FIELD_FORMLINK_FROMFORM, formId)
                .eq(FormLink.FQ_FIELD_FORMLINK_TYPE, FormLinkTypeEnum.REPLACE.getValueAsString()).criteria();

        List<FormLink> formLinks = wikittyProxy.findAllByCriteria(FormLink.class, criteria).getAll();
        return extractWikittyId(formLinks);
    }

    public boolean isReplaceForm(String formId) {
        return getPreviousReplaceFormLink(formId) != null;
    }

    public String getPreviousReplaceForm(String formId) {
        FormLink replaceFormLink = getPreviousReplaceFormLink(formId);
        if (replaceFormLink == null) {
            return null;
        }
        return replaceFormLink.getFromForm();
    }

    protected FormLink getPreviousReplaceFormLink(String formId) {
        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, FormLink.EXT_FORMLINK)
                .eq(FormLink.FQ_FIELD_FORMLINK_TOFORM, formId)
                .eq(FormLink.FQ_FIELD_FORMLINK_TYPE, FormLinkTypeEnum.REPLACE.getValueAsString()).criteria();

        return wikittyProxy.findByCriteria(FormLink.class, criteria);
    }

    protected List<String> extractWikittyId(List<FormLink> formLinks) {
        List<String> formsLinkIds = new ArrayList<String>();
        for (FormLink formLink : formLinks) {
            formsLinkIds.add(formLink.getWikittyId());
        }
        return formsLinkIds;
    }
}
