/*
 * #%L
 * Pollen :: Vote Counting
 * $Id: Context.java 3423 2012-06-05 16:50:22Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4/pollen-votecounting/src/main/java/org/chorem/pollen/votecounting/business/Context.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.votecounting.business;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.votecounting.dto.ChoiceDTO;
import org.chorem.pollen.votecounting.dto.PollChoiceDTO;
import org.chorem.pollen.votecounting.utils.Utils;

/**
 * Contexte du dépouillement du sondage (PATTERN STRATEGY).
 *
 * @author fdesbois
 * @version $Id: Context.java 3423 2012-06-05 16:50:22Z tchemit $
 */
public class Context {

    /** log. */
    private static final Log log = LogFactory.getLog(Context.class);

    /**
     * Booléen permettant de savoir si la méthode doit calculer la valeur des
     * choix en fonction des groupes
     */
    private boolean groupCounting;
    /**
     * Liste des choix liés au contexte et résultats
     */
    private List<Choice> choices, hiddenChoices, results;
    /**
     * Méthode de dépouillement choisie pour ce contexte
     */
    private Method method;

    /**
     * Constructeur
     *
     * @param method : Méthode de dépouillement
     * @param groupCounting : Dépouillement pour les groupes ou non
     */
    public Context(Method method, boolean groupCounting) {
        this.groupCounting = groupCounting;
        this.choices = new ArrayList<Choice>();
        this.hiddenChoices = new ArrayList<Choice>();
        this.results = new ArrayList<Choice>();
        this.method = method;
    }

    /**
     * Renvoie un choix de la map.
     *
     * @param idChoice : identifiant du choix
     * @return choix
     */
    public Choice getChoice(String idChoice) {
        for (Choice choice : this.choices) {
            if (choice.getIdChoice().equals(idChoice)) {
                return choice;
            }
        }
        for (Choice choice : this.hiddenChoices) {
            if (choice.getIdChoice().equals(idChoice)) {
                return choice;
            }
        }
        return null;
    }

    /**
     * Renvoie la liste des choix
     *
     * @return liste des choix
     */
    public List<Choice> getChoices() {
        return this.choices;
    }

    /**
     * Renvoie la liste des choix cachés
     *
     * @return liste des choix cachés
     */
    public List<Choice> getHiddenChoices() {
        return this.hiddenChoices;
    }

    /**
     * Renvoie la liste des résultats
     *
     * @return liste des choix résultats
     */
    public List<Choice> getResults() {
        return this.results;
    }

    /**
     * Ajout d'un nouveau choix au contexte
     *
     * @param choice identifiant du choix
     */
    public void addChoice(PollChoiceDTO choice) {
        this.choices.add(new Choice(choice.getIdChoice(), choice.getName()));
    }

    /**
     * Ajout d'un nouveau choix au contexte
     *
     * @param choice identifiant du choix
     */
    public void addHiddenChoice(PollChoiceDTO choice) {
        this.hiddenChoices.add(new Choice(choice.getIdChoice(), choice.getName()));
    }

    /**
     * Ajout d'un groupe pour l'ensemble des choix du contexte
     *
     * @param idGroup identifiant du groupe
     * @param weight poids du groupe
     */
    public void addGroup(String idGroup, double weight) {
        for (Choice choice : this.choices) {
            choice.addGroup(idGroup, weight);
        }
        for (Choice choice : this.hiddenChoices) {
            choice.addGroup(idGroup, weight);
        }
    }

    /**
     * Test si le choix est un résultat d'après le contexte A utiliser après
     * dépouillement sinon aucun résultat
     *
     * @param choice choix à tester
     * @return {@code true} si le choix est un résultat, {@code false} sinon.
     */
    private boolean isChoiceResult(Choice choice) {
        boolean result = false;
        for (Choice res : results) {
            if (choice.getIdChoice().equals(res.getIdChoice())) {
                result = true;
                break;
            }
        }
        return result;
    }

    /**
     * Execution de la méthode de dépouillement et calcul des choix résultats
     *
     * @return la liste de choix résultats
     */
    public boolean executeCounting() {

        method.executeCounting(choices, groupCounting);
        method.executeCounting(hiddenChoices, groupCounting);

        this.results.clear();
        Choice result = new Choice("");

        for (Choice curr : this.choices) {
            if (curr.value() > result.value()) {
                result = curr;
                this.results.clear();
            } else if (curr.value() == result.value()) {
                this.results.add(curr);
            }
        }

        this.results.add(result);
        return true;
    }

    public List<ChoiceDTO> executeStats() {
        List<ChoiceDTO> resChoices = new ArrayList<ChoiceDTO>();

        try {
            for (Choice choice : choices) {
                ChoiceDTO choiceDTO = new ChoiceDTO();

                choiceDTO.setIdChoice(choice.getIdChoice());
                choiceDTO.setName(choice.getName());
                choiceDTO.setValue(choice.getValue());
                choiceDTO.setResult(isChoiceResult(choice));
                choiceDTO.setNbVotes(Utils.calculateNbVotes(choice.getGroups(), groupCounting));

                method.executeStats(choices, groupCounting, choice, choiceDTO);
                resChoices.add(choiceDTO);
            }

            for (Choice choice : hiddenChoices) {
                ChoiceDTO choiceDTO = new ChoiceDTO();
                choiceDTO.setIdChoice(choice.getIdChoice());
                choiceDTO.setName(choice.getName());
                choiceDTO.setValue(choice.getValue());
                resChoices.add(choiceDTO);
            }

        } catch (Exception e) {
            log.error("L'exécution des statistiques a échoué", e);
            resChoices = new ArrayList<ChoiceDTO>();
        }

        return resChoices;
    }

}
