/*
 * #%L
 * IsisFish
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 Ifremer, Code Lutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.util;

import java.io.File;
import java.io.FileFilter;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.FileUtil;

import com.sun.tools.doclets.standard.Standard;
import com.sun.tools.javadoc.Main;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.datastore.FormuleStorage;
import fr.ifremer.isisfish.datastore.JavaSourceStorage;
import fr.ifremer.isisfish.datastore.SimulationStorage;

/**
 * Helper for javadoc.
 * 
 * Use "javadoc" executable tools located in JAVA_HOME.
 *
 * @author chatellier
 * @version $Revision: 2087 $
 *
 * Last update: $Date: 2009-04-08 12:22:41 +0200 (mer. 08 avril 2009) $
 * by : $Author: chatellier $
 * 
 * @see CompileHelper for classpath purpose
 * @since 3.2.0.4
 */
public class JavadocHelper {

    /** Logger for this class. */
    private static final Log log = LogFactory.getLog(JavadocHelper.class);
    
    /** Nom du programme dans les outils JDK. */
    protected static final String JAVADOC_TOOL_NAME = "javadoc";

    /**
     * Generate javadoc of current.
     * 
     * @param source source storage
     * @param destDir javadoc destination directory
     * @param force force (if html file is newer than java file)
     * @param out out writer (javadoc output, can be {@code null})
     * @return javadoc tool result
     */
    public static int generateJavadoc(JavaSourceStorage source, File destDir,
            boolean force, PrintWriter out) {
        File src = source.getFile();
        File dst = new File(destDir, source.getFQN().replace('.', File.separatorChar) + ".html");
        
        int result = 0;
        if (force || FileUtil.isNewer(src, dst)) {
            result = generateJavadoc(source.getRoot(), src, destDir, out);
        }
        return result;
    }

    /**
     * Generate javadoc on all java file in a directory.
     * 
     * @param rootSrc le repertoire de toutes les sources
     * @param dest le répertoire de destination de la javadoc
     * @param out le flux de sortie (peut etre null)
     * @return le resulat de sortie de l'executable javadoc (0 si ok)
     */
    public static int generateJavadoc(File rootSrc, File dest, PrintWriter out) {
        // use specific filter to add more condition than FileUtil#find
        List<File> javaFiles = FileUtil.getFilteredElements(rootSrc, new FileFilter() {
            @Override
            public boolean accept(File f) {
                boolean result = true;
                // seulement sur les fichiers java
                result &= f.getAbsolutePath().endsWith(".java");
                // don't do javadoc in "formules" directory (fail)
                result &= !f.getAbsolutePath().startsWith(FormuleStorage.getFormuleDirectory().getAbsolutePath());
                // don't do javadoc for simulations too (useless)
                result &= !f.getAbsolutePath().startsWith(SimulationStorage.getSimulationDirectory().getAbsolutePath());
                return result;
            }
        }, true);

        int result = generateJavadoc(rootSrc, javaFiles, dest, out);
        return result;
    }
    
    /**
     * Generate single file javadoc.
     * 
     * @param rootSrc le repertoire de toutes les sources
     * @param fileSrc le fichier dont on veut la javadoc
     * @param dest le répertoire de destination de la javadoc
     * @param out le flux de sortie (peut etre null)
     * @return le resulat de sortie de l'executable javadoc (0 si ok)
     */
    public static int generateJavadoc(File rootSrc, File fileSrc, File dest, PrintWriter out) {
        int result = generateJavadoc(rootSrc, Collections.singletonList(fileSrc), dest, out);
        return result;
    }

    /**
     * Generate file collection javadoc.
     * 
     * @param rootSrc le repertoire de toutes les sources
     * @param filesSrc les fichiers dont on veut la javadoc
     * @param dest le répertoire de destination de la javadoc
     * @param out le flux de sortie (peut etre null)
     * @return le resulat de sortie de l'executable javadoc (0 si ok)
     */
    public static int generateJavadoc(File rootSrc, Collection<File> filesSrc, File dest,
            PrintWriter out) {
        int result = -1;
        try {
            List<File> classpath = new ArrayList<File>();
            classpath.add(rootSrc.getAbsoluteFile());
            result = generateJavadoc(classpath, filesSrc, dest, out);
        } catch (Exception eee) {
            if (log.isWarnEnabled()) {
                log.warn("Javadoc failed", eee);
            }
        }
        return result;
    }
    
    /**
     * Generate file collection javadoc.
     * 
     * @param classpath common classpath
     * @param filesSrc les fichiers dont on veut la javadoc
     * @param dest le répertoire de destination de la javadoc
     * @param out le flux de sortie (peut etre null)
     * @return le resulat de sortie de l'executable javadoc (0 si ok)
     */
    protected static int generateJavadoc(List<File> classpath, Collection<File> filesSrc,
            File dest, PrintWriter out) {

        int result = 0;
        
        // make destination directory
        dest.mkdirs();

        try {
            // Options de compilations
            String classpathAsString = CompileHelper.getClassPathAsString(classpath);
            
            List<String> args = new ArrayList<String>();
            args.add("-classpath");
            args.add(classpathAsString);
            args.add("-d");
            args.add(dest.getAbsolutePath());
            
            args.add("-link");
            args.add("http://java.sun.com/javase/6/docs/api/");
            args.add("-link");
            args.add(IsisFish.config.getJavadocURL());
            
            // also add author and version
            args.add("-author");
            args.add("-version");
            
            for (File srcFile : filesSrc) {
                args.add(srcFile.getAbsolutePath());
            }
            //result = Main.execute(JAVADOC_TOOL_NAME, args.toArray(new String[0]));
            result = Main.execute(JAVADOC_TOOL_NAME, out, out, out,
                    Standard.class.getName(), args.toArray(new String[0]));
            
        } catch (Exception eee) {
            if (log.isWarnEnabled()) {
                log.warn("Can't generate javadoc", eee);
            }
        }
        return result;
    }
}
