/*
 * #%L
 * IsisFish
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2006 - 2010 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.widget.editor;

import java.awt.Component;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.util.ArrayList;
import java.util.List;

import javax.swing.AbstractCellEditor;
import javax.swing.JComboBox;
import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.table.TableCellEditor;

import jaxx.runtime.swing.editor.EnumEditor;

import org.apache.commons.beanutils.ConvertUtilsBean;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.util.MonthEnum;

import fr.ifremer.isisfish.datastore.RegionStorage;
import fr.ifremer.isisfish.types.Date;
import fr.ifremer.isisfish.types.Month;
import fr.ifremer.isisfish.util.ConverterUtil;

/**
 * ParameterTableCellEditor.
 * 
 * Un editeur generic de champs qui se base sur le model de données
 * a editer.
 *
 * Created: 25 sept. 06 12:35:21
 *
 * @author poussin
 * @author chemit
 * @version $Revision: 2336 $
 *
 * Last update: $Date: 2009-06-08 14:47:41 +0200 (lun. 08 juin 2009) $
 * by : $Author: chatellier $
 */
public class ParameterTableCellEditor extends AbstractCellEditor implements TableCellEditor {

    private static final long serialVersionUID = 6860330126841984303L;

    /** Class logger. */
    private static Log log = LogFactory.getLog(ParameterTableCellEditor.class);

    protected RegionStorage region = null;

    protected Component editor = null;

    protected Class<?> type = null;

    private static enum TypeClassMapping {
        /** for simple int. */
        Int(int.class, Integer.class),
        /** for a topia entity. */
        Topia(TopiaEntity.class),
        /** for a mounth. */
        Mounth(Month.class),
        /** for a simple date. */
        Date(Date.class),
        /** for a simple boolean. */
        Double(double.class, Double.class),
        /** for a simple boolean. */
        Boolean(boolean.class, Boolean.class),
        /** for a file location. */
        File(java.io.File.class),
        /** for a simple String. */
        String(java.lang.String.class);

        private final Class<?>[] klazz;

        private TypeClassMapping(Class<?>... klazz) {
            this.klazz = klazz;
        }

        protected static TypeClassMapping getMapping(Class<?> klazz) {
            for (TypeClassMapping t : TypeClassMapping.values()) {
                for (Class<?> loopKlazz : t.klazz) {
                    if (loopKlazz.isAssignableFrom(klazz)) {
                        return t;
                    }
                }
            }
            throw new RuntimeException(
                    "could not found a TypeClassMapping for this class "
                            + klazz);
        }
    }

    /**
     * @return Returns the topiaContext.
     */
    public RegionStorage getRegion() {
        return this.region;
    }

    /**
     * @param region
     *            The topiaContext to set.
     */
    public void setRegion(RegionStorage region) {
        this.region = region;
    }

    public Component getTableCellEditorComponent(JTable table, Object value,
            boolean isSelected, int row, int column) {

        // get value type
        type = getType(value, row, column);
        JComboBox c;
        //if (!table.hasFocus()) return null;

        TypeClassMapping mapping = TypeClassMapping.getMapping(type);
        switch (mapping) {
        case Topia:
            // on a un bean comme parametre
            try {
                if (region == null) {
                    return null;
                }
                TopiaContext context = region.getStorage().beginTransaction();
                if (context != null) {
                    List list = context.find("from " + type.getName());
                    c = new JComboBox(list.toArray());
                    c.setSelectedItem(value);
                    editor = c;
                    context.closeContext();
                }

            } catch (Exception eee) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't get entity object for combobox", eee);
                }
            }
            break;
        case Boolean:
            List<Boolean> listB = new ArrayList<Boolean>();
            listB.add(Boolean.TRUE);
            listB.add(Boolean.FALSE);
            c = new JComboBox(listB.toArray());
            listB.clear();
            // TODO See how to convert
            c.setSelectedItem(value);
            editor = c;
            break;
        case Date:
            Date date = (Date) value;
            if (date == null) {
                date = new Date(0);
            }
            int month = date.getMonth().getMonthNumber();
            int yea = date.getYear();
            editor = new DateComponent(month, yea, region == null ? null : region
                    .getStorage());
            //editor = new JTextField(String.valueOf(date.getDate()));
            break;
        case Mounth:
            editor = DateComponent.createMounthCombo(((Month) value)
                    .getMonthNumber(), region == null ? null : region
                    .getStorage());
            break;
        case File:
            // break;
        default:
            editor = new JTextField(value.toString());
        }
        if (editor != null) {
            if (JComboBox.class.isInstance(editor)) {
                ((JComboBox) editor).addActionListener(getComboListener());
            } else if (JTextField.class.isInstance(editor)) {
                ((JTextField) editor).addFocusListener(new FocusAdapter() {
                    @Override
                    public void focusLost(FocusEvent e) {
                        if (log.isDebugEnabled()) {
                            log.debug("Stop cell editing");
                        }
                        stopCellEditing();
                    }
                });
            } else if (DateComponent.class.isInstance(editor)) {
                DateComponent date = ((DateComponent) editor);
                if (date.getMounthCombo() != null) {
                    date.getMounthCombo().addActionListener(getComboListener());
                }
                if (date.getYearCombo() != null) {
                    date.getYearCombo().addActionListener(getComboListener());
                }
            }
        }
        if (log.isDebugEnabled()) {
            log.debug("getCellEditorValue [" + type + "] [mapping:" + mapping + "="
                + editor);
        }
		
        return editor;
    }

    /**
     * Get type for value.
     * 
     * Must be overridden (value can be null).
     * 
     * @param value
     * @param row
     * @param column
     * @return type for value
     */
    protected Class<?> getType(Object value, int row, int column) {
        // default to value's class
        return value.getClass();
    }

    protected ActionListener getComboListener() {
        return new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent e) {
                stopCellEditing();
            }
        };
    }

    public Object getCellEditorValue() {

        Object result = null;
        TopiaContext context = null;
        if (editor == null)
            return "";
        TypeClassMapping mapping = TypeClassMapping.getMapping(type);
        switch (mapping) {
        case Topia:
            result = getComboBoxValue(editor);
            break;
        case Boolean:
            result = getComboBoxValue(editor);
            // TODO test
            break;
        case Mounth:
            result = getComboBoxIndex(editor) + "";
            break;
        case Date:
            result = ((DateComponent) editor).getSelectedValue() + "";
            break;
        case File:
            //TODO
            // result = getTextFieldValue(editor);break;
        default:
            if (editor instanceof JTextField) {
                result = ((JTextField) editor).getText();
            } else if (editor instanceof JComboBox) {
                result = ((JComboBox) editor).getSelectedItem();
            }
        }

        if (result instanceof String && !String.class.equals(type)) {
            if (region != null)
                context = region.getStorage();
            ConvertUtilsBean cub = ConverterUtil.getConverter(context);
            result = cub.convert((String) result, type);
        }

        if (log.isDebugEnabled()) {
            log.debug("getCellEditorValue [" + type + "] [mapping:" + mapping
                + "]= " + result);
        }
        
        return result;
    }

    protected Object getTextFieldValue(Component editor) {
        return ((JTextField) editor).getText();
    }

    protected Object getComboBoxValue(Component editor) {
        return ((JComboBox) editor).getSelectedItem();
    }

    protected int getComboBoxIndex(Component editor) {
        return (((JComboBox) editor)).getSelectedIndex();
    }

    public static class DateComponent extends JPanel {

        private static final long serialVersionUID = -6694461572642939712L;

        protected JComboBox mounth;

        protected JComboBox year;

        public DateComponent(int mounth, int year, TopiaContext context) {
            super(new GridLayout(0, 2));

            this.mounth = createMounthCombo(mounth, context);
            this.year = createYearCombo(200, year);

            this.add(this.mounth);
            this.add(this.year);
        }

        public JComboBox getYearCombo() {
            return year;
        }

        public JComboBox getMounthCombo() {
            return mounth;
        }

        public int getSelectedMounth() {
            return mounth.getSelectedIndex();
        }

        public int getSelectedYear() {
            return year.getSelectedIndex();
        }

        public void setSelectedValue(int year, int mounth) {
            this.mounth.setSelectedIndex(mounth);
            this.year.setSelectedIndex(year);
        }

        public int getSelectedValue() {
            int selectedYear = getSelectedYear();
            int selectedMounth = getSelectedMounth();
            if (log.isInfoEnabled()) {
                log.info("selected mounth " + selectedMounth);
                log.info("selected year   " + selectedYear);
            }
            return selectedMounth + selectedYear * 12;
        }

        public static JComboBox createMounthCombo(int mounth,
                TopiaContext context) {
            JComboBox combo = EnumEditor.newEditor(MonthEnum.class);
            combo.setSelectedIndex(mounth);
            return combo;
        }

        /*public static JComboBox createMounthCombo(int mounth, TopiaContext context) {
        	JComboBox combo = createMounthCombo0(context);
        	combo.setSelectedIndex(mounth);
        	return combo;
        }

        private static JComboBox createMounthCombo0(TopiaContext context) {

        	List<String> listM = new ArrayList<String>();
        	// TODO Should use database ? data instead of thoses awfull hardcore
        	// code...
        	listM.add("janvier");
        	listM.add("fevrier");
        	listM.add("mars");
        	listM.add("avril");
        	listM.add("mai");
        	listM.add("juin");
        	listM.add("juillet");
        	listM.add("aout");
        	listM.add("septembre");
        	listM.add("octobre");
        	listM.add("novembre");
        	listM.add("decembre");
        	JComboBox combo = new JComboBox(listM.toArray());
        	listM.clear();			
        	return combo;
        }*/

        public static JComboBox createYearCombo(int nb, int selectedYear) {
            List<Integer> listI = new ArrayList<Integer>();
            for (int i = 0; i < nb; i++)
                listI.add(i);
            JComboBox year = new JComboBox(listI.toArray());
            listI.clear();
            year.setSelectedIndex(selectedYear);
            return year;
        }
    }
}
