/*
 * #%L
 * IsisFish
 * 
 * $Id: GraphBeanModel.java 3124 2010-11-29 18:14:09Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2002 - 2010 Ifremer, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */
package fr.ifremer.isisfish.ui.result;

import java.util.HashSet;
import java.util.Iterator;

import org.jfree.chart.axis.CategoryAxis;
import org.jfree.chart.axis.CategoryLabelPositions;
import org.jfree.chart.axis.NumberAxis3D;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.labels.StandardCategoryToolTipGenerator;
import org.jfree.chart.plot.CategoryPlot;
import org.jfree.chart.renderer.category.BarRenderer3D;
import org.jfree.chart.renderer.category.CategoryItemRenderer;
import org.jfree.data.category.CategoryDataset;
import org.nuiton.math.matrix.MatrixND;

/**
 * GraphBeanModel.
 *
 * Created: Fri May 17 13:29:47 2002
 *
 * @author <a href="mailto:seb.regnier@free.fr"></a>
 * @version $Revision: 3124 $
 */
public class GraphBeanModel {

    // protected VerticalCategoryPlot verticalCategoryPlot;
    protected CategoryDataset categoryDataset;
    protected CategoryAxis horizontalAxis;
    protected ValueAxis verticalAxis;
    protected CategoryItemRenderer renderer;
    protected MatrixND matrix;

    // table de toutes les vues du modèle
    private HashSet listeners = new HashSet();

    public GraphBeanModel() {
        horizontalAxis = new CategoryAxis();
        horizontalAxis.setCategoryLabelPositions(CategoryLabelPositions.UP_90);
        renderer = new BarRenderer3D();
        renderer.setBaseToolTipGenerator(new StandardCategoryToolTipGenerator());
        verticalAxis = new NumberAxis3D();
    }

    /**
    * Méthode qui permet d'ajouter une vue au modèle
    * @param view la nouvelle vue
    */
    public void addGraphListener(GraphListener view) {
        listeners.add(view);
    }

    /**
    * Méthode qui retire une vue du modèle
    * @param view la vue à retirer
    */
    public void removeGraphListener(GraphListener view) {
        listeners.remove(view);
    }

    /**
    * Accesseur sur la table des listeners du modèle
    * @return la collection des listeners
    */
    public HashSet getGraphListeners() {
        return listeners;
    }

    /**
    * Accesseur en écriture sur la table des listeners du modèle
    * @param obs la collection des listeners
    */
    public void setGraphListeners(HashSet obs) {
        listeners = obs;
    }

    /**
    * Previent les vues que le model a change
    *
    */
    protected void fireGraphChanged() {
        // Iterateur pour parcourir les vues
        Iterator listenersIterator = listeners.iterator();
        GraphEvent ge = new GraphEvent(this);

        while (listenersIterator.hasNext()) {
            GraphListener listener = ((GraphListener) listenersIterator.next());
            listener.graphChanged(ge);
        }
    }

    /**
    * Get the value of title.
    * @return value of title.
    */
    public String getTitle() {
        return matrix.getName();
    }

    /**
    * Get the value of categoryDataset.
    * @return value of categoryDataset.
    */
    public CategoryDataset getCategoryDataset() {
        return categoryDataset;
    }

    /**
    * Set the value of categoryDataset.
    * @param v  Value to assign to categoryDataset.
    */
    public void setCategoryDataset(CategoryDataset v) {
        this.categoryDataset = v;
        fireGraphChanged();
    }

    public void setMatrix(MatrixND mi) {
        matrix = mi;
        setCategoryDataset(new GraphMatrixNDDataset(mi));
    }

    public CategoryPlot getCategoryPlot() {
        return new CategoryPlot(getCategoryDataset(), getHorizontalAxis(),
                getVerticalAxis(), getRenderer());
    }

    /**
    * Get the value of horizontalAxis.
    * @return value of horizontalAxis.
    */
    public CategoryAxis getHorizontalAxis() {
        horizontalAxis.setLabel(matrix.getDimensionName(0));
        return horizontalAxis;
    }

    /**
    * Get the value of renderer.
    * @return value of renderer.
    */
    public CategoryItemRenderer getRenderer() {
        return renderer;
    }

    /**
    * Set the value of renderer.
    * @param v  Value to assign to renderer.
    */
    public void setRenderer(CategoryItemRenderer v) {
        this.renderer = v;
        fireGraphChanged();
    }

    /**
    * Get the value of verticalAxis.
    * @return value of verticalAxis.
    */
    public ValueAxis getVerticalAxis() {
        return verticalAxis;
    }

    /**
    * Set the value of verticalAxis.
    * @param v  Value to assign to verticalAxis.
    */
    public void setVerticalAxis(ValueAxis v) {
        this.verticalAxis = v;
        fireGraphChanged();
    }

}// GraphBeanModel
