/*
 * #%L
 * IsisFish
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2005 - 2010 Ifremer, Code Lutin, Cedric Pineau, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.input;

import static org.nuiton.i18n.I18n._;

import java.awt.geom.Point2D;
import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.math.matrix.MatrixFactory;
import org.nuiton.math.matrix.MatrixND;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.util.FileUtil;
import org.nuiton.widget.SwingUtil;
import org.nuiton.widget.editor.Editor;

import fr.ifremer.isisfish.IsisFishDAOHelper;
import fr.ifremer.isisfish.datastore.FormuleStorage;
import fr.ifremer.isisfish.datastore.RegionStorage;
import fr.ifremer.isisfish.datastore.SimulationStorage;
import fr.ifremer.isisfish.datastore.StorageException;
import fr.ifremer.isisfish.datastore.update.ImportFromV2;
import fr.ifremer.isisfish.entities.Cell;
import fr.ifremer.isisfish.entities.CellDAO;
import fr.ifremer.isisfish.entities.EffortDescription;
import fr.ifremer.isisfish.entities.EffortDescriptionDAO;
import fr.ifremer.isisfish.entities.FisheryRegion;
import fr.ifremer.isisfish.entities.Formule;
import fr.ifremer.isisfish.entities.Gear;
import fr.ifremer.isisfish.entities.Metier;
import fr.ifremer.isisfish.entities.MetierSeasonInfo;
import fr.ifremer.isisfish.entities.MetierSeasonInfoDAO;
import fr.ifremer.isisfish.entities.Population;
import fr.ifremer.isisfish.entities.PopulationDAO;
import fr.ifremer.isisfish.entities.PopulationGroup;
import fr.ifremer.isisfish.entities.PopulationSeasonInfo;
import fr.ifremer.isisfish.entities.PopulationSeasonInfoDAO;
import fr.ifremer.isisfish.entities.PopulationSeasonInfoImpl;
import fr.ifremer.isisfish.entities.Selectivity;
import fr.ifremer.isisfish.entities.SelectivityDAO;
import fr.ifremer.isisfish.entities.SetOfVessels;
import fr.ifremer.isisfish.entities.Species;
import fr.ifremer.isisfish.entities.SpeciesDAO;
import fr.ifremer.isisfish.entities.TargetSpecies;
import fr.ifremer.isisfish.entities.TargetSpeciesDAO;
import fr.ifremer.isisfish.entities.Zone;
import fr.ifremer.isisfish.mexico.export.RegionExplorer;
import fr.ifremer.isisfish.mexico.export.RegionExport;
import fr.ifremer.isisfish.mexico.export.RegionExportFactorXML;
import fr.ifremer.isisfish.types.Month;
import fr.ifremer.isisfish.ui.input.check.CheckRegion;
import fr.ifremer.isisfish.ui.input.check.CheckResult;
import fr.ifremer.isisfish.ui.input.check.CheckResultFrame;
import fr.ifremer.isisfish.ui.simulator.filter.SimulationFilterUtil;
import fr.ifremer.isisfish.ui.util.ErrorHelper;
import fr.ifremer.isisfish.util.CellPointcomparator;

/**
 * All code used in input UI.
 *
 * Created: 1 aout 2005 18:37:25 CEST
 *
 * @author Benjamin POUSSIN <poussin@codelutin.com>
 * @version $Revision: 1312 $
 *
 * Last update: $Date: 2008-08-28 10:21:07 +0200 (jeu, 28 aoû 2008) $
 * by : $Author: sletellier $
 */
public class InputAction {

    /** Class logger. */
    private static Log log = LogFactory.getLog(InputAction.class);

    /**
     * Exporter la region dans un zip.
     *
     * @return le nom du ficher selectionné par l'utilisateur
     */
    public File importRegion() {
        File file = null;
        try {
            file = FileUtil.getFile(".*.zip$",
                    _("isisfish.message.import.region.zipped"));
            if (file != null) {
                RegionStorage.importZip(file);
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't import region", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.region.import"), eee);
        }
        return file;
    }

    /**
     * Exporter la region dans un zip.
     *
     * @return le nom du ficher selectionné par l'utilisateur
     */
    public File importRegionAndRename() {
        File file = null;
        try {
            file = FileUtil.getFile(".*.zip$",
                    _("isisfish.message.import.region.zipped"));
            if (file != null) {

                String newName = JOptionPane
                        .showInputDialog(_("isisfish.message.name.imported.region"));
                RegionStorage.importAndRenameZip(file, newName);
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't import region", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.region.import"), eee);
        }
        return file;
    }

    /**
     * Importer la region depuis un fichier XML de la version 2.
     *
     * @return le nom du ficher selectionné par l'utilisateur
     */
    public File importV2Region() {
        File file = null;
        try {
            file = FileUtil.getFile(".*.xml$",
                    _("isisfish.message.import.region.xml"));
            if (file != null) {
                new ImportFromV2(true).importXML(file);
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't import region", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.region.import"), eee);
        }
        return file;
    }

    /**
     * Extract from a simulation the region, and rename it with name given
     * by user.
     *
     * @return le nom de la region entré par l'utilsateur
     */
    public String importRegionFromSimulation() {

        // first step select a simulation and new region name
        String simulationName;
        try {
            //TODO Should use a shared model ?
            simulationName = SimulationFilterUtil
                    .selectSimulation(SimulationStorage.getSimulationNames());
            if (simulationName == null) {
                return null;
            }

        } catch (Exception eee) {
            String msg = _("isisfish.error.no.select.simulation");
            if (log.isWarnEnabled()) {
                log.warn(msg, eee);
            }
            ErrorHelper.showErrorDialog(
                    _("isisfish.error.no.select.simulation"), eee);
            return null;
        }

        if (log.isInfoEnabled()) {
            log.info("simulation used " + simulationName);
        }

        // ask new region name
        String regionName = JOptionPane.showInputDialog(
                _("isisfish.message.import.region.name"), "region from "
                        + simulationName);
        if (regionName == null || "".equals(regionName)) {
            return null;
        }
        if (RegionStorage.getRegionNames().contains(regionName)) {
            return null;
        }

        if (log.isInfoEnabled()) {
            log.info("new region name " + regionName);
        }

        try {
            SimulationStorage.getSimulation(simulationName).extractRegion(
                    regionName);
        } catch (StorageException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't extract region", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.region.import"), eee);
        }
        return regionName;
    }

    /**
     * Exporter la region dans un zip
     *
     * @param regionStorage storage to export
     * @return file were region were exported
     */
    public File exportRegion(RegionStorage regionStorage) {
        File file = null;
        try {
            file = FileUtil.getFile(".*.zip$",
                    _("isisfish.message.import.region.zipped"));
            
            // add .zip extension is not set
            if (!file.getAbsolutePath().endsWith(".zip")) {
                file = new File(file.getAbsolutePath() + ".zip");
            }

            if (file != null) {
                int resp = JOptionPane.YES_OPTION;
                if (file.exists()) {
                    resp = JOptionPane.showConfirmDialog(null,
                            _("isisfish.message.file.overwrite"));
                }
                if (resp == JOptionPane.YES_OPTION) {
                    regionStorage.createZip(file);
                }
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't export region", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.region.export"), eee);
        }
        return file;
    }

    /**
     * Copy la region avec un autre nom.
     *
     * @param regionStorage region to copy
     * @return le nom de la nouvelle region
     */
    public String copyRegion(RegionStorage regionStorage) {
        String newName = null;
        try {
            newName = JOptionPane
                    .showInputDialog(_("isisfish.message.new.region.name"));

            if (!StringUtils.isEmpty(newName)) {
                File zip = regionStorage.createZip();
                RegionStorage.importAndRenameZip(zip, newName);
            }

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't copy region", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.region.copy"), eee);
        }
        return newName;
    }

    /**
     * Remove region.
     *
     * @param regionStorage region storage
     * @param cvsDelete if true delete region in CVS too
     * @return removed region
     */
    public boolean removeRegion(RegionStorage regionStorage, boolean cvsDelete) {
        boolean result = false;
        try {
            int resp = JOptionPane.showConfirmDialog(null, _(
                    "isisfish.message.confirm.remove.region", regionStorage
                            .getName()));
            if (resp == JOptionPane.YES_OPTION) {
                regionStorage.delete(cvsDelete);
                result = true;
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't remove region", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.region.remove"), eee);
        }
        return result;
    }

    /**
     * Commit region.
     *
     * @param regionStorage region storage
     * @return status message
     */
    public String commitRegionInCVS(RegionStorage regionStorage) {
        String result = null;
        try {
            String msg = regionStorage.getCommentForNextCommit();
            JTextArea text = new JTextArea(msg);
            int resp = JOptionPane.showOptionDialog(null,
                    new JScrollPane(text), _("isisfish.commit.message"),
                    JOptionPane.OK_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE,
                    null, // icon
                    null, null);
            if (resp == JOptionPane.OK_OPTION) {
                regionStorage.commit(text.getText());
                regionStorage.clearCommentForNextCommit();
                result = _("isisfish.message.region.commited");
            } else {
                result = _("isisfish.message.commit.region.canceled");
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't export region", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.region.commit"), eee);
        }
        return result;
    }

    /**
     * Load region.
     * <p/>
     * RegionStorage, TopiaContext, FisheryRegion are put in uiContext data.
     * <p/>
     * tree ans
     *
     * @param ui TODO
     * @param name name of region
     * @return loaded region
     */
    public FisheryRegion loadRegion(InputUI ui, String name) {
        if (log.isDebugEnabled()) {
            log.debug("loadRegion : " + name);
        }

        FisheryRegion region = null;
        try {
            RegionStorage regionStorage = null;
            TopiaContext isisContext = null;
            if (name != null && !" ".equals(name)) {
                regionStorage = RegionStorage.getRegion(name);
                isisContext = regionStorage.getStorage().beginTransaction();
                region = RegionStorage.getFisheryRegion(isisContext);
                ui.setContextValue(regionStorage);
                ui.setContextValue(isisContext);
                // FIXME session shoul be closed
                // but make lazy init later
                //isisContext.rollbackTransaction();
                //isisContext.closeContext();
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't load region", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.region.load"), eee);
        }
        return region;
    }

    /**
     * Create new region, and select it in combo, and show input pane region in
     * input area.
     *
     * @param name name of the new region
     */
    public void newRegion(String name) {
        if (log.isTraceEnabled()) {
            log.trace("newRegion called");
        }
        try {
            if ("".equals(name)) {
                // showMsgBox("error " + _("isisfish.error.region.name.empty"));
            }
            if (RegionStorage.getRegionNames().contains(name)) {
                // showMsgBox("Error " + _("isisfish.error.region.already.exists"));
            }
            RegionStorage.create(name);
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't create region", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.region.create"), eee);
        }
    }

    /*
     * Permet de creer simplement un nouvelle objet portant un nom par defaut.
     * Le nouvel element est automatiquement selectionné dans l'arbre
     *
     * @param type le type de l'entite a creer, le type est compose seulement du
     *             nom de l'entite et pas du package.
     * @return null ou un fenetre d'erreur
     *
    public TopiaEntity create(TopiaContext isisContext, String type) {
        if (log.isTraceEnabled()) {
            log.trace("create called for " + type);
        }
        try {
            //TODO use the IsisFishEntityEnum for this purpose :
            //TopiaDAO<TopiaEntity> dao = IsisFishEntityEnum.getEntry(type).getDAO(isisContext);
            String name = type + "_new";

            Method method = MethodUtils.getAccessibleMethod(
                    IsisFishDAOHelper.class, "get" + type + "DAO",
                    TopiaContext.class);
            TopiaDAO<TopiaEntity> dao = (TopiaDAO<TopiaEntity>) method.invoke(
                    null, isisContext);

            TopiaEntity entity = dao.create("name", name);
            entity.update();
            isisContext.commitTransaction();
            return entity;

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't create entity", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.region.new"), eee);
        }
        return null;
    }*/

    /*
     * Save entity passed in argument, and commit
     *
     * @param selected entity to save
     * @return the saved entity
     *
    public TopiaEntity save(TopiaEntity selected) {
        if (log.isTraceEnabled()) {
            log.trace("save called");
        }
        try {
            if (selected == null) {
                if (log.isWarnEnabled()) {
                    log.warn("Try to save null entity");
                }
                return null;
            }
            // on est pas en autoUpdate donc il faut faire le update avant le
            // commit
            selected.update();

            selected.getTopiaContext().commitTransaction();
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't save region", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.save"), eee);
        }
        return selected;
    }*/

    /*
     * Cancel all modification on entity (rollback), and force reload it and
     * refresh all ui component that name match 'type'Tab
     *
     * @return the roolbacked entity
     *
    public TopiaEntity cancel(TopiaEntity selected) {
        if (log.isTraceEnabled()) {
            log.trace("cancel called");
        }
        try {
            TopiaContext isisContext = selected.getTopiaContext();
            isisContext.rollbackTransaction();

            // reload the object
            // can't be null here, or allready null before !!!
            //if (selected != null) {
            // FIXME don't modify parameters
            selected = isisContext.findByTopiaId(selected.getTopiaId());
            //}
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't cancel modification in region", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.region.cancel"), eee);
        }
        return selected;
    }*/

    /*
     * Delete one entity and commit the change, try to selected intelligently
     * other node in tree.
     * <p/>
     * Refresh all ui component where name match "input&lt;entity type without
     * package &gt;.*"
     *
     * @return if ok return null else OutView error
     *
    public String remove(TopiaEntity selected) {
        if (log.isTraceEnabled()) {
            log.trace("remove called");
        }
        String msg = "";
        try {
            boolean doDelete;
            List<TopiaEntity> allWillBeRemoved = selected.getComposite();
            if (allWillBeRemoved.size() > 0) {
                String text = _("isisfish.message.delete.object", selected
                        .toString());
                for (TopiaEntity e : allWillBeRemoved) {
                    text += ClassUtils.getShortClassName(e.getClass()) + " - "
                            + e.toString() + "\n";
                }
                int resp = JOptionPane.showConfirmDialog(null, text,
                        _("isisfish.message.delete.entities"),
                        JOptionPane.YES_NO_OPTION);
                doDelete = resp == JOptionPane.YES_OPTION;
            } else {
                String text = _("isisfish.message.confirm.delete.object",
                        selected.toString());
                int resp = JOptionPane.showConfirmDialog(null, text,
                        _("isisfish.message.delete.entity"),
                        JOptionPane.YES_NO_OPTION);
                doDelete = resp == JOptionPane.YES_OPTION;
            }

            if (doDelete) {
                selected.delete();
                selected.getTopiaContext().commitTransaction();
                msg = _("isisfish.message.remove.finished");
            } else {
                msg = _("isisfish.message.remove.canceled");
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't remove entity: " + selected, eee);
            }
            showMsgBox(eee);
        }
        return msg;
    }*/

    /*
     * Save an Equation as model, to reuse it for other equation
     *
     * @param eq equation to put in models
     * @return if ok return null else OutputView error message
     *
    public Object saveEquationAsModel(Equation eq) {
        try {
            if (eq != null) {
                String name = showInputDialog("");
                //if ("".equals(name)) {
                //  showMsgBox("Error " +_("isisfish.error.invalid.equation.name"));
                //}
                if (name != null) {
                    FormuleStorage storage = FormuleStorage.createFormule(eq
                            .getCategory(), name, eq.getLanguage());
                    storage.setContent(eq.getContent());
                }
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't save equation as model", eee);
            }
            showMsgBox(eee);
        }
        return null;
    }*/

    /**
     * Save an Equation as model, to reuse it for other equation
     *
     * @param category category for this equation
     * @param language equation to put in models
     * @param content  content ?
     * @return if ok return null else OutputView error message
     */
    public Object saveAsModel(String category, String language, String content) {
        try {
            String name = JOptionPane
                    .showInputDialog(_("isisfish.message.saveModel.dialog"));
            /*if ("".equals(name)) {
                showMsgBox("Error " +_("isisfish.error.invalid.equation.name"));
            }*/
            if (name != null) {
                FormuleStorage storage = FormuleStorage.createFormule(category,
                        name, language);
                storage.setContent(content);
            }

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't save equation as model", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.equation.savemodel"),
                    eee);
        }
        return null;
    }

    public Object openEditor(String category, String name, Class javaInterface,
            String content, Editor editor) {
        if (log.isTraceEnabled()) {
            log.trace("openEditor");
        }
        try {
            EquationEditorPaneUI pane = new EquationEditorPaneUI();
            pane.setEquation(category, name, javaInterface, content);
            pane.setVisible(true);
            if (pane.isOk() && editor != null) {
                editor.setText(pane.getEditor().getText());
            }
            pane.dispose();
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't open editor", eee);
            }
            ErrorHelper.showErrorDialog(
                    _("isisfish.error.equation.openeditor"), eee);
        }
        return null;
    }

    protected List<Formule> getFormules(TopiaContext isisContext, String name) {
        return FormuleStorage.getFormules(isisContext, name);
    }

    // InputGear

    public Object addSelectivity(Population pop, String equation, Gear gear) {
        try {

            SelectivityDAO dao = IsisFishDAOHelper.getSelectivityDAO(pop
                    .getTopiaContext());
            Selectivity selectivity = dao.create();

            selectivity.setGear(gear);
            selectivity.setPopulation(pop);
            selectivity.getEquation().setContent(equation);
            selectivity.update();

            gear.addPopulationSelectivity(selectivity);
            gear.update();

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't add selectivity", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.addentity",
                    "Selectivity"), eee);
        }
        return null;
    }

    public Object removeSelectivity(Gear gear, Selectivity selectivity) {
        if (log.isTraceEnabled()) {
            log.trace("removeSelectivity called: " + selectivity);
        }
        try {
            if (gear != null && selectivity != null) {
                gear.removePopulationSelectivity(selectivity);
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't remove TargetSpecies", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.removeentity",
                    "Selectivity"), eee);
        }
        return null;
    }

    // InputMetier

    /*public Object createSeasonInfo(Metier metier) {
        if (log.isTraceEnabled()) {
            log.trace("createSeasonInfo called");
        }
        try {
            MetierSeasonInfoDAO metierSeasonInfoPS = IsisFishDAOHelper
                    .getMetierSeasonInfoDAO(metier.getTopiaContext());
            MetierSeasonInfo metierSeasonInfo = metierSeasonInfoPS.create();
            metierSeasonInfo.setFirstMonth(Month.MONTH[0]);
            metierSeasonInfo.setLastMonth(Month.MONTH[3]);
            metier.addMetierSeasonInfo(metierSeasonInfo);
            metierSeasonInfo.update();
            metier.update();
            // isisContext.commitTransaction();

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't create MetierSeasonInfo", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.addentity", "SeasonInfo"), eee);
        }
        return null;
    }*/

    /*public Object removeSeasonInfo(Metier metier, MetierSeasonInfo info) {
        if (log.isTraceEnabled()) {
            log.trace("removeSeasonInfo called");
        }
        try {
            metier.removeMetierSeasonInfo(info);
            metier.update();
            metier.getTopiaContext().commitTransaction();
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't create MetierSeasonInfo", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.removeentity", "SeasonInfo"), eee);
        }
        return null;
    }*/

    public Object addTargetSpecies(Metier metier, MetierSeasonInfo m,
            Species species, String targetFactorEquationContent,
            boolean primaryCatch) {
        if (log.isDebugEnabled()) {
            log.debug("addTargetSpecies called: " + metier + " " + species
                    + " " + primaryCatch + " " + targetFactorEquationContent);
        }
        try {
            // build targetFactorEquation name
            String targetFactorEquationName = metier.getName() + "-"
                    + species.getName() + "(" + m.getFirstMonth() + "-"
                    + m.getLastMonth() + ")";

            TargetSpeciesDAO dao = IsisFishDAOHelper.getTargetSpeciesDAO(metier
                    .getTopiaContext());
            TargetSpecies targetSpecies = dao.create();

            targetSpecies.setSpecies(species);
            targetSpecies.getTargetFactorEquation().setName(
                    targetFactorEquationName);
            targetSpecies.getTargetFactorEquation().setContent(
                    targetFactorEquationContent);
            targetSpecies.setPrimaryCatch(primaryCatch);
            m.addSpeciesTargetSpecies(targetSpecies);
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't add TargetSpecies", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.addentity",
                    "TargetSpecies"), eee);
        }
        return null;
    }

    public Object removeTargetSpecies(MetierSeasonInfo m,
            TargetSpecies targetSpecies) {
        if (log.isDebugEnabled()) {
            log.debug("removeTargetSpecies called: " + targetSpecies);
        }
        try {
            if (targetSpecies != null) {
                m.removeSpeciesTargetSpecies(targetSpecies);
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't remove TargetSpecies", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.removeentity",
                    "TargetSpecies"), eee);
        }
        return null;
    }

    // SetOfVessels

    public void addEffortDescription(SetOfVessels setOfVessels, Metier metier) {
        if (log.isDebugEnabled()) {
            log.debug("addEffortDescription called: " + setOfVessels
                    + " metier: " + metier);
        }
        try {
            EffortDescriptionDAO effortDescriptionPS = IsisFishDAOHelper
                    .getEffortDescriptionDAO(metier.getTopiaContext());
            EffortDescription effortDescription = effortDescriptionPS.create();
            // EC20090715 : c'est la classe d'association
            // qui en étant sauvee, sauve les relations en base
            // il faut bien faire les set des deux cotes
            effortDescription.setSetOfVessels(setOfVessels);
            effortDescription.setPossibleMetiers(metier);
            effortDescription.update();
            setOfVessels.addPossibleMetiers(effortDescription);
            setOfVessels.update();
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't create EffortDescription", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.addentity",
                    "EffortDescription"), eee);
        }
    }

    public Object removeEffortDescription(SetOfVessels sov,
            EffortDescription effort) {
        if (log.isTraceEnabled()) {
            log.trace("removeEffortDescription called");
        }
        try {
            sov.removePossibleMetiers(effort);
            // EC-20091112 : commit() twice cause hibernate error:
            // Found two representations of same collection:
            //sov.getTopiaContext().commitTransaction();
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't remove entity: " + effort, eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.removeentity",
                    "EffortDescription"), eee);
        }
        return null;
    }

    // Population
    public Species getSpeciesByTopiaId(TopiaContext context, String topiaId) {
        Species result = null;
        try {
            SpeciesDAO dao = IsisFishDAOHelper.getSpeciesDAO(context);
            result = dao.findByTopiaId(topiaId);
        } catch (TopiaException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't get species", e);
            }
        }
        return result;
    }

    public Population createPopulation(TopiaContext context, Species species) {
        if (log.isTraceEnabled()) {
            log.trace("createPopulation called");
        }
        try {
            String name = "Population_new";

            PopulationDAO dao = IsisFishDAOHelper.getPopulationDAO(context);
            Population pop = dao.create();
            pop.setName(name);

            species.addPopulation(pop);
            pop.setSpecies(species);
            species.update();
            pop.update();
            // EC-20091112 : commit() twice cause hibernate error:
            // Found two representations of same collection:
            //context.commitTransaction();
            return pop;
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't create Population", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.addentity",
                    "Population"), eee);
        }
        return null;
    }

    public PopulationSeasonInfo createPopulationSeasonInfo(Population pop) {
        if (log.isDebugEnabled()) {
            log.debug("createSeasonInfo called");
        }
        PopulationSeasonInfo seasonInfo = null;
        try {
            PopulationSeasonInfoDAO dao = IsisFishDAOHelper
                    .getPopulationSeasonInfoDAO(pop.getTopiaContext());
            seasonInfo = dao.create();
            seasonInfo.setFirstMonth(Month.MONTH[0]);
            seasonInfo.setLastMonth(Month.MONTH[3]);
            pop.addPopulationSeasonInfo(seasonInfo);
            seasonInfo.setPopulation(pop);

            // EC-20090710 ajout du code d'initailisation
            // d'un populationseasoninfo
            // il a du se perdre au changement d'interface
            // swiat>jaxx
            // moved to entity
            //MatrixND matrix = ((PopulationSeasonInfoImpl) seasonInfo)
            //        .createNoSpacializedChangeGroupMatrix();
            //seasonInfo.setLengthChangeMatrix(matrix);

            seasonInfo.update();
            pop.update();
            // isisContext.commitTransaction();
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't create PopulationSeasonInfo", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.removeentity",
                    "PopulationSeasonInfo"), eee);
        }
        return seasonInfo;
    }

    public void removePopulationSeasonInfo(Population pop,
            PopulationSeasonInfo populationSeasonInfo) {
        if (log.isDebugEnabled()) {
            log.debug("removePopulationSeasonInfo called");
        }
        try {
            pop.removePopulationSeasonInfo(populationSeasonInfo);
            pop.update();
            //pop.getTopiaContext().commitTransaction();
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't remove PopulationSeasonInfo", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.removeentity",
                    "PopulationSeasonInfo"), eee);
        }
    }

    public void createMetierSeasonInfo(Metier metier) {
        if (log.isDebugEnabled()) {
            log.debug("createMetierSeasonInfo called");
        }
        try {
            MetierSeasonInfoDAO metierSeasonInfoPS = IsisFishDAOHelper
                    .getMetierSeasonInfoDAO(metier.getTopiaContext());
            MetierSeasonInfo metierSeasonInfo = metierSeasonInfoPS.create();
            metierSeasonInfo.setFirstMonth(Month.MONTH[0]);
            metierSeasonInfo.setLastMonth(Month.MONTH[3]);
            metier.addMetierSeasonInfo(metierSeasonInfo);
            metierSeasonInfo.update();
            metier.update();
            
            // EC-20091112 : commit() twice cause hibernate error:
            // Found two representations of same collection:
            // isisContext.commitTransaction();
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't create MetierSeasonInfo", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.addentity",
                    "MetierSeasonInfo"), eee);
        }
    }

    public void removeMetierSeasonInfo(Metier metier, MetierSeasonInfo info) {
        if (log.isDebugEnabled()) {
            log.debug("removeMetierSeasonInfo called");
        }
        try {
            metier.removeMetierSeasonInfo(info);
            metier.update();
            
            // EC-20091112 : commit() twice cause hibernate error:
            // Found two representations of same collection:
            //metier.getTopiaContext().commitTransaction();

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't remove MetierSeasonInfo", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.addentity",
                    "MetierSeasonInfo"), eee);
        }
    }

    public Object createRecruitmentDistribution(Population pop) {
        if (log.isTraceEnabled()) {
            log.trace("createRecruitmentDistributionon called: " + pop);
        }
        try {
            String val = JOptionPane
                    .showInputDialog(_("isisfish.message.recruitment.number.month"));
            int num = -1;
            if (val != null && !"".equals(val)) {
                try {
                    num = Integer.parseInt(val);
                } catch (RuntimeException eee) {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't parse val: " + val, eee);
                    }
                }
            }

            if (num > 0) {
                List<String> sem = new ArrayList<String>(num);
                for (int i = 0; i < num; i++) {
                    sem.add(_("isisfish.common.month", i));
                }
                MatrixND newMat = MatrixFactory.getInstance().create(
                        new List[] { sem });

                MatrixND mat = pop.getRecruitmentDistribution();
                if (mat != null) {
                    newMat.paste(mat);
                }
                pop.setRecruitmentDistribution(newMat);
                pop.update();
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't remove RecruitmentDistribution", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.removeentity",
                    "RecruitmentDistribution"), eee);
        }
        return null;
    }

    // Migration

    public Object addMigration(PopulationSeasonInfo info,
            PopulationGroup group, Zone departure, Zone arrival, double coeff) {
        if (log.isTraceEnabled()) {
            log.trace("addMigration called");
        }
        try {
            MatrixND mat = info.getMigrationMatrix().copy();
            mat.setValue(group, departure, arrival, coeff);
            info.setMigrationMatrix(mat);

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't add migration", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.addentity",
                    "Migration"), eee);
        }
        return null;
    }

    public Object addEmigration(PopulationSeasonInfo info,
            PopulationGroup group, Zone departure, double coeff) {
        if (log.isTraceEnabled()) {
            log.trace("addEmigration called");
        }
        try {
            MatrixND mat = info.getEmigrationMatrix().copy();
            mat.setValue(group, departure, coeff);
            info.setEmigrationMatrix(mat);

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't add emigration", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.addentity",
                    "Emigration"), eee);
        }
        return null;
    }

    public Object addImmigration(PopulationSeasonInfo info,
            PopulationGroup group, Zone arrival, double coeff) {
        if (log.isTraceEnabled()) {
            log.trace("addImmigration called");
        }
        try {
            MatrixND mat = info.getImmigrationMatrix().copy();
            mat.setValue(group, arrival, coeff);
            info.setImmigrationMatrix(mat);

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't add immigration", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.input.addentity",
                    "Immigration"), eee);
        }
        return null;
    }

    /**
     * Add new map in region.
     * 
     * Since isis-fsih 3.3.0.0, this method supports a multiple file format :
     *  - http://openmap.bbn.com/cgi-bin/faqw.py?req=all#9.1
     * 
     * @param fisheryRegion fishery region
     */
    public void addMap(FisheryRegion fisheryRegion) {
        if (log.isTraceEnabled()) {
            log.trace("AddMap called");
        }

        // Openmap suported fileformat
        String[] fileFormats = {
                // ESRI (http://www.esri.com) Shapefiles
                ".*\\.shp", "ESRI Shapefiles (.shp)",

                // NIMA (http://www.nima.mil)
                ".*\\.dcw", "Digital Chart of the World (.dcw)",
                ".*\\.vpf", "Vector Product Format (.vpf)",
                ".*\\.vmap", "Vector Map (.vmap)",
                ".*\\.cadrg", "Compressed ARC Digitized Raster Graphics (.cadrg)",
                ".*\\.cib", "Controlled Image Base (.cib)",
                ".*\\.rpf", "Raster Product Format (.rpf)",
                // seams to be a special format with multiples files
                //".*\\.dt[0-2]+", "Digital Terrain Elevation Data (levels 0, 1, 2) (.dt0, .dt0, .dt2)",

                // MapInfo (http://www.mapinfo.com) files (.mif)
                ".*\\.mif", "MapInfo (.mif)",

                // ArcInfo (.e00) files.
                ".*\\.e00", "ArcInfo (.e00)"
        };

        File inputMap = FileUtil.getFile(fileFormats);
        try {
            if (inputMap != null) {

                // copy inputMapFormat to region map directory
                // since isis-fsih-3.3.0.0, we store maps with extension
                String filename = inputMap.getName();

                RegionStorage regionStorage = RegionStorage.getRegion(fisheryRegion.getName());
                File destDir = regionStorage.getMapRepository();

                // copy map file
                File destMap = new File(destDir, filename);
                FileUtil.copy(inputMap, destMap);

                List<String> maps = fisheryRegion.getMapFileList();
                maps.add(filename);
                fisheryRegion.setMapFileList(maps);
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't copy map file for : " + inputMap, eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.region.addmap"), eee);
        }
    }

    /**
     * Remove map in a region.
     * 
     * @param fisheryRegion fishery region
     * @param selectedMaps la liste de map a retirer
     */
    public void removeMap(FisheryRegion fisheryRegion, Object[] selectedMaps) {
        if (log.isTraceEnabled()) {
            log.trace("removeMap called");
        }

        try {
            List<String> maps = fisheryRegion.getMapFileList();

            RegionStorage regionStorage = RegionStorage.getRegion(fisheryRegion.getName());
            File mapDir = regionStorage.getMapRepository();

            for (Object selectedMap : selectedMaps) {
                String mapName = (String)selectedMap;
                maps.remove(mapName);
                // remove on disk too, if possible
                File mapFile = new File(mapDir, mapName);
                mapFile.delete();
                if (log.isDebugEnabled()) {
                    log.debug("Removing map file : " + mapFile);
                }

                // special case, for some format, an index is created
                if (mapName.endsWith(".shp")) {
                    String indexName = mapName.replaceAll("\\.shp$", ".ssx");
                    File indexFile = new File(mapDir, indexName);
                    indexFile.delete();
                    if (log.isDebugEnabled()) {
                        log.debug("Removing index file : " + indexFile);
                    }
                }
            }

            fisheryRegion.setMapFileList(maps);

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't remove map", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.region.removemap"),
                    eee);
        }
    }

    /**
     * Affiche une boite de dialogue demandant a l'utilisateur de rentrer
     * un nom de fichier, et met ce nom dans la textField.
     * Passe le saveButton en enabled pour marque que la region a ete modifié
     *
     * @param cellFile file of cells
     */
    public void loadCellFile(String cellFile) {
        if (log.isWarnEnabled()) {
            log.warn("loadCell called but not implemented");
        }
        // FIXME todo loadCellFile
    }

    public void saveFisheryRegion(FisheryRegion fisheryRegion) {
        if (log.isTraceEnabled()) {
            log.trace("save called");
        }
        try {
            TopiaContext isisContext = fisheryRegion.getTopiaContext();

            //            frame.setInfoText(_("isisfish.message.checking.cell"));
            int latNumber = (int) Math
                    .round((fisheryRegion.getMaxLatitude() - fisheryRegion
                            .getMinLatitude())
                            / fisheryRegion.getCellLengthLatitude());
            int lonNumber = (int) Math
                    .round((fisheryRegion.getMaxLongitude() - fisheryRegion
                            .getMinLongitude())
                            / fisheryRegion.getCellLengthLongitude());

            //            frame.setProgressMin(0);
            //            frame.setProgressMax(latNumber * lonNumber);
            int progresscpt = 0;

            // il faut peut-etre creer ou supprimer des mailles
            CellPointcomparator cellPointcomparator = new CellPointcomparator();
            CellDAO cellPS = IsisFishDAOHelper.getCellDAO(isisContext);
            List<Cell> cells = cellPS.findAll();
            Collections.sort(cells, cellPointcomparator);
            Point2D.Float point = new Point2D.Float();

            for (float lati = fisheryRegion.getMinLatitude(); lati < fisheryRegion
                    .getMaxLatitude(); lati += fisheryRegion
                    .getCellLengthLatitude()) {
                lati = Math.round(lati * 1000f);
                lati = lati / 1000.0f;
                for (float longi = fisheryRegion.getMinLongitude(); longi < fisheryRegion
                        .getMaxLongitude(); longi += fisheryRegion
                        .getCellLengthLongitude()) {
                    longi = Math.round(longi * 1000f) / 1000.0f;
                    point.setLocation(lati, longi);
                    int position = Collections.binarySearch(cells, point,
                            cellPointcomparator);
                    if (position >= 0) {
                        // deja existant on l'enleve de la liste, et on ne cree rien
                        cells.remove(position);
                    } else {
                        // n'existe pas on la cree
                        Cell cell = cellPS.create();
                        cell.setName("La" + lati + "Lo" + longi);
                        cell.setLatitude(lati);
                        cell.setLongitude(longi);
                        cell.setLand(false);
                        cell.update();
                    }
                    // frame.setProgressValue(++progresscpt);
                }
            }

            // on est pas en autoUpdate donc il faut faire le update avant le commit
            fisheryRegion.update();

            // toutes les mailles restantes dans la liste sont des mailles en trop
            // on les supprimes

            progresscpt = 0;
            for (Cell cell : cells) {
                // FIXME il faudrait aussi rechercher les objets dependants
                // des mailles que l'on va supprimer et demander confirmation
                // a l'utilisateur qu'il souhaite reellement supprimer tous
                // ces objets. Si ce n'est pas le cas, on sort tout de suite
                // avant le commit
                cellPS.delete(cell);
            }
            isisContext.commitTransaction();
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error(_("isisfish.error.region.save"), eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.region.save"), eee);
        }
    }

    public void checkFisheryRegion(FisheryRegion fisheryRegion) {
        try {
            if (log.isTraceEnabled()) {
                log.trace("check called: ");
            }
            CheckResult result = new CheckResult();
            CheckRegion.check(fisheryRegion, result);
            CheckResultFrame dialog = new CheckResultFrame();
            dialog.setCheckResult(result);
            SwingUtil.center(dialog);
            dialog.setVisible(true);
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error(_("isisfish.error.region.check"), eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.region.check"), eee);
        }
    }
    
    /**
     * Explore region and exporte any enabled sensitivity factor name with value.
     * 
     * @param parentComponent parent component
     * @param fisheryRegion fishery region to export
     */
    public void exportRegionSensitivityFactors(InputUI parentComponent, FisheryRegion fisheryRegion) {

        try {
            File exportFile = FileUtil.getFile(_("isisfish.input.sensitivity.export.title"),
                    _("isisfish.common.ok"), parentComponent, ".*\\.xml", "XML Files");

            // make sur that filename ends with ".xml"
            if (!exportFile.getAbsolutePath().endsWith(".xml")) {
                exportFile = new File(exportFile.getAbsolutePath() + ".xml");
            }

            if (exportFile != null) {
                parentComponent.setStatusMessage(_("isisfish.input.sensitivity.export.running"), true);
                RegionExplorer explorer = new RegionExplorer();
                RegionExport exportXML = new RegionExportFactorXML(exportFile);
                explorer.explore(fisheryRegion, exportXML);
                parentComponent.setStatusMessage(_("isisfish.input.sensitivity.export.complete"));
            }
            else {
                if (log.isDebugEnabled()) {
                    log.debug("Export action canceled");
                }
                parentComponent.setStatusMessage(_("isisfish.input.sensitivity.export.cancel"));
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error(_("isisfish.error.sensitivity.export"), eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.sensitivity.export"), eee);
        }
    }
}
