/*
 * #%L
 * IsisFish
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 Ifremer, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.simulator.sensitivity.domain;

import org.nuiton.math.matrix.MapFunction;
import org.nuiton.math.matrix.MatrixND;

/**
 * Domaine continu portant sur une matrice.
 * 
 * Composé d'un borne min et max et d'une matrice.
 * 
 * @author chatellier
 * @version $Revision: 1.0 $
 * 
 * Last update : $Date: 24 févr. 2009 $ By : $Author: chatellier $
 */
public class MatrixContinuousDomain extends ContinuousDomain<MatrixND, Double> {

    /** serialVersionUID. */
    private static final long serialVersionUID = -2037768174807839046L;

    /** Matrix */
    protected MatrixND matrix;

    /** Coefficient. */
    protected double coefficient;

    /** Operation. (=,+,-,/) */
    protected String operator;

    /** Value used to create the matrix */
    protected double value;

    /**
     * Empty constructor.
     */
    public MatrixContinuousDomain() {

    }

    /**
     * Get matrix.
     * 
     * @return the matrix
     */
    public MatrixND getMatrix() {
        return matrix;
    }

    /**
     * Set matrix.
     * 
     * @param matrix
     *            the matrix to set
     */
    public void setMatrix(MatrixND matrix) {
        this.matrix = matrix;
    }

    /**
     * @return the coefficient
     */
    public double getCoefficient() {
        return coefficient;
    }

    /**
     * @param coefficient
     *            the coefficient to set
     */
    public void setCoefficient(double coefficient) {
        this.coefficient = coefficient;
    }

    /**
     * Get operator.
     * 
     * @return the operator
     */
    public String getOperator() {
        return operator;
    }

    /**
     * Set operator.
     * 
     * @param operator
     *            the operator to set
     */
    public void setOperator(String operator) {
        this.operator = operator;
    }

    /**
     * {@inheritDoc}.
     * 
     * In matrix continuous domain, always return 0
     */
    @Override
    public double getMinBound() {
        return 0;
    }

    /**
     * {@inheritDoc}.
     * 
     * In matrix continuous domain, always return 1
     */
    @Override
    public double getMaxBound() {
        return 1;
    }

    /**
     * {@inheritDoc}.
     * 
     * In continuous domain, just return identifier
     */
    public MatrixND getValueForIdentifier(final Double identifier) {
        this.value = identifier;
        MatrixND temp = this.matrix.copy();
        if (operator.equals("+")) {
            temp.map(new MapFunction() {
                public double apply(double val) {
                    return (val + ((val + coefficient) * ((2 * value - 1))));
                }
            });
        } else if (operator.equals("-")) {
            temp.map(new MapFunction() {
                public double apply(double val) {
                    return (val + ((val - coefficient) * ((2 * value - 1))));
                }
            });
        } else if (operator.equals("*")) {
            temp.map(new MapFunction() {
                public double apply(double val) {
                    return (val + ((val * coefficient) * ((2 * value - 1))));
                }
            });
        } else if (operator.equals("/")) {
            temp.map(new MapFunction() {
                public double apply(double val) {
                    return (val + ((val / coefficient) * ((2 * value - 1))));
                }
            });
        }
        return temp;
    }

    /**
     * Get value used to create the matrix.
     * 
     * @return the value
     */
    public Double getValue() {
        return this.value;
    }

    /**
     * Set value used to create the matrix.
     * 
     * @param value
     *            the value to set
     */
    public void setValue(Double value) {
        this.value = value;
    }

    public MatrixContinuousDomain clone() {
        MatrixContinuousDomain cloned = new MatrixContinuousDomain();
        cloned.setCardinality(this.cardinality);
        cloned.setMaxBound(this.maxBound);
        cloned.setMinBound(this.minBound);
        cloned.setCoefficient(this.coefficient);
        cloned.setMatrix(this.matrix);
        cloned.setOperator(this.operator);
        cloned.setValue(this.value);
        return cloned;
    }

    /**
     * Get the minimum bound of the coefficient used to create the matrix. As
     * the getMinBound for use in R always return 0.
     * 
     * @return the real minimum bound
     */
    public Double getRealMinBound() {
        if (operator.equals("+")) {
            return (1.0 + ((1.0 + coefficient) * ((2 * 0.0 - 1))));
        } else if (operator.equals("-")) {
            return (1.0 + ((1.0 - coefficient) * ((2 * 0.0 - 1))));
        } else if (operator.equals("*")) {
            return (1.0 + ((1.0 * coefficient) * ((2 * 0.0 - 1))));
        } else if (operator.equals("/")) {
            return (1.0 + ((1.0 / coefficient) * ((2 * 0.0 - 1))));
        } else {
            return null;
        }
    }

    /**
     * Get the maximal bound of the coefficient used to create the matrix. As
     * the getMaxBound for use in R always return 1.
     * 
     * @return the real maximum bound
     */
    public Double getRealMaxBound() {
        if (operator.equals("+")) {
            return (1.0 + ((1.0 + coefficient) * ((2 * 1.0 - 1))));
        } else if (operator.equals("-")) {
            return (1.0 + ((1.0 - coefficient) * ((2 * 1.0 - 1))));
        } else if (operator.equals("*")) {
            return (1.0 + ((1.0 * coefficient) * ((2 * 1.0 - 1))));
        } else if (operator.equals("/")) {
            return (1.0 + ((1.0 / coefficient) * ((2 * 1.0 - 1))));
        } else {
            return null;
        }
    }
}
