/*
 * #%L
 * IsisFish
 * 
 * $Id: SimulationServiceTableModel.java 3124 2010-11-29 18:14:09Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2002 - 2010 Ifremer, Code Lutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.simulator.launcher;

import static org.nuiton.i18n.I18n._;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.WeakHashMap;

import javax.swing.JProgressBar;
import javax.swing.table.AbstractTableModel;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import fr.ifremer.isisfish.simulator.SimulationControl;
import fr.ifremer.isisfish.simulator.SimulationParameter;
import fr.ifremer.isisfish.types.Date;

/**
 * Model de table pour suivre l'evolution des differentes simulations en cours.
 * 
 * <p>
 * <b>ATTENTION</b> Cette classe doit supporter les acces concurrents car
 * plusieurs threads peuvent etre simultanement en train de faire des
 * simulations
 *
 * @author poussin
 * @version $Revision: 3124 $
 * 
 * Last update : $Date: 2010-11-29 19:14:09 +0100 (lun., 29 nov. 2010) $
 * By : $Author: chatellier $
 */
public class SimulationServiceTableModel extends AbstractTableModel {

    /** serialVersionUID */
    private static final long serialVersionUID = 2414926794815727974L;

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory
            .getLog(SimulationServiceTableModel.class);

    /** Columns names. */
    protected final static String[] columnHeader = new String[] {
            _("isisfish.queue.id"),
            _("isisfish.queue.plan"),
            _("isisfish.queue.launcher"),
            _("isisfish.queue.status"),
            _("isisfish.queue.progression"), };

    /** Columns types. */
    protected Class<?>[] columnClass = new Class[] {
            String.class, // id
            String.class, // analyse plan number
            String.class, // local, remote, batch
            String.class, // text
            JProgressBar.class // progress
    };

    protected SimulationService simulationService;
    protected ArrayList<SimulationJob> jobs;
    protected Map<String, SimulationJob> jobIds;
    /** progress bar (one for each row) */
    protected Map<SimulationJob, JProgressBar> progress = new WeakHashMap<SimulationJob, JProgressBar>();

    protected AbstractJobListener jobListener;
    protected ControlListener controlListener;

    public SimulationServiceTableModel(SimulationService simulationService,
            boolean forDoToJobs) {
        this.simulationService = simulationService;
        if (forDoToJobs) {
            jobListener = new JobToDoListener(simulationService, this);
        } else {
            jobListener = new JobDoneListener(simulationService, this);
        }
        controlListener = new ControlListener(simulationService, this);
        simulationService.addSimulationServiceListener(jobListener);
        jobListener.setData();
    }

    public void addJob(SimulationJob job) {
        String id = job.getItem().getControl().getId();
        synchronized (jobs) {
            if (!contains(job)) {
                jobs.add(job);
                jobIds.put(id, job);
                fireTableRowsInserted(jobs.size() - 1, jobs.size() - 1);
            }
        }
    }

    public void removeJob(SimulationJob job) {
        String id = job.getItem().getControl().getId();
        synchronized (jobs) {
            int index = jobs.indexOf(job);
            if (index >= 0) {
                jobs.remove(index);
                jobIds.remove(id);
                fireTableRowsDeleted(index, index);
            }
        }
    }

    public void clearJob() {
        jobs.clear();
        fireTableDataChanged();
    }

    public ArrayList<SimulationJob> getJobs() {
        return jobs;
    }

    public void setJobs(ArrayList<SimulationJob> jobs) {
        this.jobs = jobs;
        synchronized (jobs) {
            jobIds = new HashMap<String, SimulationJob>(jobs.size());
            for (SimulationJob job : jobs) {
                jobIds.put(job.getItem().getControl().getId(), job);
            }
            fireTableDataChanged();
        }
    }

    public boolean contains(SimulationJob job) {
        String id = job.getItem().getControl().getId();
        synchronized (jobs) {
            boolean result = jobIds.containsKey(id);
            return result;
        }
    }

    protected JProgressBar getProgressBar(SimulationJob job) {
        JProgressBar result = progress.get(job);
        if (result == null) {
            result = new JProgressBar(JProgressBar.HORIZONTAL, 0, 100);
            result.setStringPainted(true);
            progress.put(job, result);
        }
        return result;
    }

    public SimulationJob getJob(int row) {
        SimulationJob result = jobs.get(row);
        return result;
    }

    /*
     * @see javax.swing.table.TableModel#getRowCount()
     */
    public int getRowCount() {
        int result = jobs.size();
        return result;
    }

    /*
     * @see javax.swing.table.TableModel#getColumnCount()
     */
    public int getColumnCount() {
        int result = columnHeader.length;
        return result;
    }

    /*
     * @see javax.swing.table.AbstractTableModel#getColumnClass(int)
     */
    @Override
    public Class<?> getColumnClass(int columnIndex) {
        return columnClass[columnIndex];
    }

    /*
     * @see javax.swing.table.AbstractTableModel#getColumnName(int)
     */
    @Override
    public String getColumnName(int column) {
        return columnHeader[column];
    }

    /*
     * @see javax.swing.table.TableModel#getValueAt(int, int)
     */
    public Object getValueAt(int rowIndex, int columnIndex) {

        // TODO IndexOutOfBoundException here
        // Concurrent problem ?
        if (rowIndex >= jobs.size()) {
            return null;
        }

        SimulationJob job = jobs.get(rowIndex);
        SimulationControl control = job.getItem().getControl();
        String id = control.getId();
        SimulationParameter param = job.getItem().getParameter();

        Object result = "";

        if (log.isTraceEnabled()) {
            log.trace("Update table model : " + "id = " + control.getId()
                    + ", " + "control.getProgress() = " + control.getProgress()
                    + ", " + "control.getProgressMax() = "
                    + control.getProgressMax() + ", " + "control.getDate() = "
                    + control.getDate());
        }

        switch (columnIndex) {
        case 0:
            result = id;
            break;
        case 1:
            if (param.getUseAnalysePlan()) {
                int number = param.getAnalysePlanNumber();
                if (number >= 0) {
                    result = number;
                }
            }
            break;
        case 2:
            if (job.getLauncher() == null) {
                if (param.getUseAnalysePlan()) {
                    result = _("isisfish.queue.masterplan");
                } else {
                    result = _("isisfish.queue.notstarted");
                }
            } else {
                result = job.getLauncher().toString();
            }
            break;
        case 3:
            if (control.isStopSimulationRequest()) {
                result = _("isisfish.launch.stop");
            } else {
                result = control.getText();
            }
            break;
        case 4:
            JProgressBar pb = getProgressBar(job);
            pb.setMaximum((int) control.getProgressMax());
            pb.setValue((int) control.getProgress());

            // progress can be used for other things
            Date date = control.getDate();
            if (date != null) {
                pb.setString(date.getMonth() + "/" + date.getYear());
            }
            else {
                pb.setString("");
            }
            result = pb;
            break;
        }
        return result;
    }

    protected interface AbstractJobListener extends SimulationServiceListener {
        public void setData();
    }

    class JobDoneListener implements AbstractJobListener,
            SimulationServiceListener {

        protected SimulationService simulationService;
        protected SimulationServiceTableModel model;

        public JobDoneListener(SimulationService simulationService,
                SimulationServiceTableModel model) {
            this.simulationService = simulationService;
            this.model = model;
            setData();
        }

        public void setData() {
            model.setJobs(new ArrayList<SimulationJob>(simulationService
                    .getJobDones()));
        }

        public void simulationStart(SimulationService simService,
                SimulationJob job) {
            // this happens when job is restarted
            // remove it from here
            model.removeJob(job);
        }

        public void simulationStop(SimulationService simService,
                SimulationJob job) {
            model.addJob(job);
        }

        public void clearJobDone(SimulationService simService) {
            model.clearJob();
        }
    }

    class JobToDoListener implements AbstractJobListener,
            SimulationServiceListener {

        protected SimulationService simulationService;
        protected SimulationServiceTableModel model;

        public JobToDoListener(SimulationService simulationService,
                SimulationServiceTableModel model) {
            this.simulationService = simulationService;
            this.model = model;
            setData();
        }

        public void setData() {
            model.setJobs(new ArrayList<SimulationJob>(simulationService
                    .getJobs()));
            for (SimulationJob job : model.getJobs()) {
                job.getItem().getControl().addPropertyChangeListener(
                        controlListener);
            }
        }

        public void simulationStart(SimulationService simService,
                SimulationJob job) {
            model.addJob(job);
            job.getItem().getControl().addPropertyChangeListener(
                    model.controlListener);
        }

        public void simulationStop(SimulationService simService,
                SimulationJob job) {
            model.removeJob(job);
            job.getItem().getControl().removePropertyChangeListener(
                    model.controlListener);
        }

        public void clearJobDone(SimulationService simService) {
            // nothing to do
        }

    }

    protected class ControlListener implements PropertyChangeListener {

        protected SimulationService simulationService;
        protected SimulationServiceTableModel model;

        public ControlListener(SimulationService simulationService,
                SimulationServiceTableModel model) {
            this.simulationService = simulationService;
            this.model = model;
        }

        public void propertyChange(PropertyChangeEvent evt) {
            SimulationControl control = (SimulationControl) evt.getSource();
            String id = control.getId();
            synchronized (model.jobs) {
                SimulationJob job = model.jobIds.get(id);
                int index = model.getJobs().indexOf(job);
                if (index >= 0) {
                    fireTableRowsUpdated(index, index);
                }
            }
        }

    }
}
