/*
 * #%L
 * IsisFish
 * 
 * $Id: StrategyImpl.java 3124 2010-11-29 18:14:09Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2006 - 2010 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.entities;

import static org.nuiton.i18n.I18n._;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaException;

import fr.ifremer.isisfish.IsisFishDAOHelper;
import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.datastore.RegionStorage;
import fr.ifremer.isisfish.datastore.StorageException;
import fr.ifremer.isisfish.equation.Language;
import fr.ifremer.isisfish.equation.StrategyInactivityEquation;
import fr.ifremer.isisfish.types.Month;

/**
 * Implantation des operations pour l'entité Strategy.
 * 
 * Created: 20 janv. 2006 03:29:14
 *
 * @author poussin
 * @version $Revision: 3124 $
 *
 * Last update: $Date: 2010-11-29 19:14:09 +0100 (lun., 29 nov. 2010) $
 * by : $Author: chatellier $
 */
public class StrategyImpl extends StrategyAbstract {

    /** Class logger. */
    private static Log log = LogFactory.getLog(StrategyImpl.class);

    /** serialVersionUID. */
    private static final long serialVersionUID = 1L;

    public FisheryRegion getFisheryRegion() {
        try {
            FisheryRegion result = RegionStorage
                    .getFisheryRegion(getTopiaContext());
            return result;
        } catch (StorageException eee) {
            throw new IsisFishRuntimeException(eee);
        }
    }

    /**
     * Override to change name of equation.
     * 
     * @see StrategyAbstract#setName(java.lang.String)
     */
    @Override
    public void setName(String value) {
        super.setName(value);
        Equation eq = getInactivityEquation();
        if (eq != null) {
            eq.setName(getName());
        }
    }

    /*
     * @see fr.ifremer.isisfish.entities.StrategyAbstract#getStrategyMonthInfo()
     */
    @Override
    public List<StrategyMonthInfo> getStrategyMonthInfo() {
        try {
            List<StrategyMonthInfo> result = super.getStrategyMonthInfo();
            if (result == null) { // Faire le test suivant pose de gros probleme, car il essai de faire un getStrategyMonthInfo et donc ca boucle || result.size() != Month.NUMBER_OF_MONTH) {
                if (result != null) {
                    log.warn(_("isisfish.error.strategy.order", getName(),
                            result.size()));
                    clearStrategyMonthInfo();
                } else {
                    result = new ArrayList<StrategyMonthInfo>(
                            Month.NUMBER_OF_MONTH);
                    setStrategyMonthInfo(result);
                }
                for (Month month : Month.MONTH) {
                    StrategyMonthInfoDAO dao = IsisFishDAOHelper
                            .getStrategyMonthInfoDAO(getTopiaContext());
                    StrategyMonthInfo smi = dao.create();
                    smi.setMonth(month);
                    smi.setStrategy(this);
                    smi.update();
                    addStrategyMonthInfo(smi);
                }
                this.update();
                result = super.getStrategyMonthInfo();
            }
            return result;
        } catch (TopiaException eee) {
            throw new IsisFishRuntimeException(
                    "Can't create StrategyMonthInfo", eee);
        }
    }

    /*
     * @see fr.ifremer.isisfish.entities.Strategy#getStrategyMonthInfo(fr.ifremer.isisfish.types.Month)
     */
    public StrategyMonthInfo getStrategyMonthInfo(Month month) {
        List<StrategyMonthInfo> smis = getStrategyMonthInfo();
        StrategyMonthInfo result = smis.get(month.getMonthNumber());
        if (!month.equals(result.getMonth())) {
            // normalement les StrategyMonthInfo sont dans l'ordre mais 
            // pour etre sur que ca fonctionne on fait un petit test
            log
                    .warn("StrategyMonthInfo not in month order, i take time to find good StrategyMonthInfo");
            result = null;
            for (StrategyMonthInfo smi : smis) {
                if (month.equals(smi.getMonth())) {
                    result = smi;
                    break;
                }
            }
        }
        return result;
    }

    /**
     * Compute or return inactivity days.
     * 
     * @param month month
     * @return age in month
     */
    public double getInactivityDays(Month month) {
        double result = 0;

        StrategyMonthInfo info = getStrategyMonthInfo(month);
        if (!getInactivityEquationUsed()) {
            result = info.getMinInactivityDays();
        } else {
            try {
                Equation eq = getInactivityEquation();
                result = eq.evaluate("month", month, "info", info);
            } catch (Exception eee) {
                if (log.isWarnEnabled()) {
                    log.warn("Error in inactivityDays equation");
                }
                if (log.isDebugEnabled()) {
                    log.debug("StackTrace", eee);
                }
            }
        }
        return result;
    }

    public void setInactivityEquationContent(String content) {
        try {
            Equation eq = getInactivityEquation();

            if (eq == null) {
                EquationDAO dao = IsisFishDAOHelper
                        .getEquationDAO(getTopiaContext());
                // create Growth equation
                eq = dao.create();
                eq.setCategory("Inactivity");
                eq.setJavaInterface(StrategyInactivityEquation.class);
                eq.setLanguage(Language.JAVA);
                setInactivityEquation(eq);
            }

            eq.setName(getName());

            // Fire
            String _oldValue = eq.getContent();
            fireOnPreWrite("content", _oldValue, content);

            eq.setContent(content);
            eq.update();

            fireOnPostWrite("content", _oldValue, content);

        } catch (TopiaException eee) {
            throw new IsisFishRuntimeException(_("isisfish.change.equation"),
                    eee);
        }
    }

    /*
     * @see fr.ifremer.isisfish.entities.StrategyAbstract#toString()
     */
    @Override
    public String toString() {
        String result = getName();
        return result;
    }

} //StrategyImpl
