/*
 * #%L
 * IsisFish
 * 
 * $Id: RuleStorage.java 3124 2010-11-29 18:14:09Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2005 - 2010 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.datastore;

import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.util.Doc;
import fr.ifremer.isisfish.util.DocHelper;
import fr.ifremer.isisfish.util.Docable;
import fr.ifremer.isisfish.rule.Rule;
import fr.ifremer.isisfish.vcs.VCSException;

import static org.nuiton.i18n.I18n._;

import org.apache.commons.collections.map.ReferenceMap;
import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.File;
import java.lang.reflect.Field;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * Class permettant de representer une regle de gestion.
 * Une regle de gestion est un fichier Java que l'on compile si besoin.
 * Il se trouve dans le package portant le nom de la region de cette maniere
 * plusieurs region peuvent avoir des regles avec le meme nom mais pas le
 * meme code.
 * <p/>
 * Gere les fichiers VCS de type {@link Rule} (package rules)
 *
 * Created: 17 août 2005 11:11:51 CEST
 *
 * @author Benjamin POUSSIN <poussin@codelutin.com>
 *
 * @version $Revision: 3124 $
 *
 * Last update: $Date: 2010-11-29 19:14:09 +0100 (lun., 29 nov. 2010) $ by : $Author: chatellier $
 */
public class RuleStorage extends JavaSourceStorage implements Docable { // RulesStorage

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(RuleStorage.class);
    
    public static final String RULE_PATH = "rules";

    /** Template freemarker pour les regles. */
    public static final String RULE_TEMPLATE = "templates/script/rule.ftl";

    @SuppressWarnings("unchecked")
    private static Map<String, RuleStorage> rulesCache =
            (Map<String, RuleStorage>) new ReferenceMap();

    /**
     * Contruit un nouveau rule storage
     *
     * @param rootSrc   La region auquelle est attaché la rule
     * @param directory le repertoire ou devrait se trouver la rule
     * @param name      le nom de la rule
     */
    protected RuleStorage(File rootSrc, File directory, String name) {
        super(rootSrc, directory, name);
    }

    static public File getRuleDirectory() {
        File result = new File(getContextDatabaseDirectory(), RULE_PATH);
        result.mkdirs();
        return result;
    }
    
    /**
     * Retourne le nom de toutes les regles existantes pour cette region
     *
     * @return all rule names found in local user database
     */
    static public List<String> getRuleNames() {
        File dir = getRuleDirectory();
        return RuleStorage.getStorageNames(dir);
    }

    /**
     * Retourne le storage pour la regle demandée
     *
     * @param name le nom de la regle souhaitée
     * @return Le storage pour la regle
     */
    static public RuleStorage getRule(String name) {
        String cacheName = getContextDatabaseCacheKey(name);
        RuleStorage result = rulesCache.get(cacheName);
        if (result == null) {
            result = new RuleStorage(getContextDatabaseDirectory(),
                    getRuleDirectory(), name);
            rulesCache.put(cacheName, result);
        }
        return result;
    }


    /**
     * Retourne une nouvelle instance de la regle. Compile le fichier si besoin
     *
     * @return a new rule instance
     * @throws IsisFishException if any exception while instanciation
     */
    public Rule getNewRuleInstance() throws IsisFishException {
        Object result = getNewInstance();
        return (Rule) result;
    }

    /**
     * Permet de recuperer les parametres de la regle.
     *
     * @return Une map contenant le nom du parametre et son type
     * @throws IsisFishException s'il est impossible de retourner une map
     *                           convenable
     */
    public Map<String, Class<?>> getParameterNames() throws IsisFishException {
        Map<String, Class<?>> result = null;
        // On essai de recuperer les paramètres depuis la classe compilé
        // si c possible        
        try {
            if (0 == compile(false, null)) {
                Rule rule = getNewRuleInstance();
                result = RuleStorage.getParameterNames(rule);
            }
        } catch (Exception eee) {
            log.info(_("isisfish.error.compiled.parameter"), eee);
        }
        // si on ne reussi pas avec la classe compilé on essai en parsant le
        // source
        if (result == null) {
            try {
                //String code = getContent();
                log.fatal("FIXME a faire recherche des parametre dans le source");
                throw new IsisFishException(_("isisfish.error.source.parameter"));
                // TODO implanter la recherche des parametres.
                // Se sont les attributs commencant par 'param_' et ils doivent 
                // etre public non static, non transient.
                // ex: public Date param_Date.
                // Lorsqu'on ajoute un champs dans la map il ne faut pas mettre
                // le prefix param.
            } catch (Exception eee) {
                throw new IsisFishException(_("isisfish.error.source.parameter"), eee);
            }
        }
        return result;
    }


    /**
     * Recherche par introspection tous les parametres de la classe
     * commencant par param_
     *
     * @param rule the rule to inspect
     * @return the list of parameters for a given rule
     */
    static public Map<String, Class<?>> getParameterNames(Rule rule) {
        Map<String, Class<?>> result = new LinkedHashMap<String, Class<?>>();
        for (Field field : rule.getClass().getFields()) {
            if (field.getName().startsWith(PARAM_PREFIX)) {
                result.put(field.getName().substring(PARAM_PREFIX.length()),
                        field.getType());
            }
        }
        return result;
    }

    /**
     * Donne la valeur d'un parametre par introspection
     *
     * @param name le nom du parametre
     * @param rule the rule to inspect
     * @return la valeur courante du parametre
     * @throws IsisFishException if any exception
     */
    static public Object getParameterValue(Rule rule, String name) throws IsisFishException {
        if (rule == null || name == null || "".equals(name)) {
            return null;
        }
        try {
            String fieldName = PARAM_PREFIX + name;
            Field field = rule.getClass().getDeclaredField(fieldName);
            return field.get(rule);
        } catch (IllegalAccessException eee) {
            throw new IsisFishException("Can't get rule parameter: " + name, eee);
        } catch (NoSuchFieldException eee) {
            throw new IsisFishException("Can't get rule parameter: " + name, eee);
        }
    }

    /**
     * Modifie la valeur d'un attribut par introspection.
     *
     * @param name  le nom de l'attribut
     * @param value la valeur de l'attribut
     * @param rule  the rule to inspect
     * @throws IsisFishException if any exception
     */
    static public void setParameterValue(Rule rule, String name, Object value) throws IsisFishException {
        try {
            String fieldName = PARAM_PREFIX + name;
            Field field = rule.getClass().getDeclaredField(fieldName);
            field.set(rule, value);
        } catch (IllegalAccessException eee) {
            throw new IsisFishException("Can't modify rule parameter: " + name + " with '" + value + "'(" + ObjectUtils.identityToString(value) + ")", eee);
        } catch (NoSuchFieldException eee) {
            throw new IsisFishException("Can't modify rule parameter: " + name + " with '" + value + "'(" + ObjectUtils.identityToString(value) + ")", eee);
        } catch (IllegalArgumentException eee) {
            throw new IsisFishException("Can't modify rule parameter: " + name + " with '" + value + "'(" + ObjectUtils.identityToString(value) + ")", eee);
        }
    }

    /**
     * Recupere les parametres et leur valeur pour les retourner sous forme
     * de chaine. Pour pouvoir par exemple les afficher a l'utilisateur.
     *
     * @param rule the rule to inspect
     * @return a string representation of parameters and their values for
     *         a given rule
     * @throws IsisFishException if any exception
     */
    public static String getParamAsString(Rule rule) throws IsisFishException {
        StringBuffer result = new StringBuffer();

        for (String name : getParameterNames(rule).keySet()) {
            Object value = getParameterValue(rule, name);

            result.append(name).append(" : ").append(value);
            result.append("\n");
        }
        return result.toString();
    }

    static public void checkout() throws VCSException {
        checkout(IsisFish.config.getDatabaseDirectory(), RULE_PATH);
    }

    /**
     * Retourne la liste des noms de toutes les régions disponible en local qui
     * ne sont pas encore sur le serveur VCS
     *
     * @return liste de noms de regions
     */
    static public List<String> getNewRuleNames() {
        List<String> result = getRuleNames();
        result.removeAll(getRemoteRuleNames());
        return result;
    }

    /**
     * Retourne la liste des noms de toutes les régions disponible sur le
     * serveur VCS
     *
     * @return la liste des noms de toutes les régions disponible sur le serveur
     *         VCS. Si le serveur n'est pas disponible la liste retournée est
     *         vide.
     */
    static public List<String> getRemoteRuleNames() {
        File dir = getRuleDirectory();
        return getRemoteStorageNames(dir);

    }

    /**
     * Retourne la liste des noms de toutes les régions disponible sur le
     * serveur VCS qui ne sont pas encore en local
     *
     * @return liste de noms de regions
     */
    static public List<String> getNewRemoteRuleNames() {
        List<String> result = getRemoteRuleNames();
        result.removeAll(getRuleNames());
        return result;
    }

    /**
     * @return the @Doc of the underlied rule class
     * @see DocHelper
     * @see Doc
     * @see Docable
     */
    public Doc getClassDoc() {
        Doc result=null;
        try {
            Class klazz = getCodeClass();
            result = DocHelper.getClassDoc(klazz);
        } catch (IsisFishException e) {
            log.warn(_("isisfish.error.not.found.code",this));
        }
        return result;
    }

    /**
     * @param fieldName name of the field to inspect
     * @return the @Doc of the given field of the underlied Rule class
     * @see DocHelper
     * @see Doc
     * @see Docable
     */
    public Doc getFieldDoc(String fieldName) {
        Doc result=null;
        try {
            Class klazz = getCodeClass();
            result = DocHelper.getFieldDoc(klazz,fieldName);
        } catch (Exception e) {
            log.warn(_("isisfish.error.not.found.field",fieldName,this));
        }
        return result;
    }

    /**
     * @param paramName the name of the param to inspect
     * @return the doc associated with the param,
     *         (says the field PARAM_PREFIX+fieldName)
     */
    public Doc getParamDoc(String paramName) {
        Doc doc;
        doc = getFieldDoc(PARAM_PREFIX + paramName);
        return doc;
    }

    /**
     * <b>Be ware this method require to instanciate a Rule, so
     * it would be better to call as often as possible.</b>
     *
     * @return the descript of the instanciate AnalysePlan
     * @see Doc
     * @see Docable
     */
    public String getDescription() {
        String result=null;
        try {
            Rule rule = getNewRuleInstance();
            result =rule == null ? null : rule.getDescription();
        } catch (Exception e) {
            log.warn(_("isisfish.error.not.found.description",this));
        }
        return result;
    }

} // RuleStorage

