/*
 * #%L
 * IsisFish
 * 
 * $Id: RuleAspect.java 3124 2010-11-29 18:14:09Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 1999 - 2010 Ifremer, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.aspect;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.codehaus.aspectwerkz.annotation.Around;
import org.codehaus.aspectwerkz.annotation.Aspect;
import org.codehaus.aspectwerkz.joinpoint.JoinPoint;

import fr.ifremer.isisfish.datastore.SimulationInformation;
import fr.ifremer.isisfish.datastore.SimulationStorage;
import fr.ifremer.isisfish.entities.Metier;
import fr.ifremer.isisfish.rule.Rule;
import fr.ifremer.isisfish.simulator.SimulationContext;
import fr.ifremer.isisfish.types.Date;

/**
 * Aspect utiliser pour intersepecter les appels a
 * {@link Rule#init(SimulationContext)}, {@link Rule#preAction(SimulationContext, Date, Metier)}, 
 * {@link Rule#postAction(SimulationContext, Date, Metier)} et memoriser
 * le temps mit.
 * 
 * @author chatellier
 * @version $Revision: 3124 $
 * 
 * Last update : $Date: 2010-11-29 19:14:09 +0100 (lun., 29 nov. 2010) $
 * By : $Author$
 */
@Aspect("perInstance")
public class RuleAspect {

    /**
     * State phase to register.
     */
    protected static enum State {
        INIT, PRE, POST
    }
    
    /** Log. */
    private static Log log = LogFactory.getLog(RuleAspect.class);

    /**
     * Aspect around {@link Rule#init(SimulationContext)} in packages "rules".
     * 
     * @param jp join point
     * @return init result
     * @throws Throwable
     */
    @Around("execution(* rules.*.init(..))")
    public Object initCall(final JoinPoint jp) throws Throwable {
        
        Object result = makeTimedCall(jp, State.INIT);
        return result;
        
    }

    /**
     * Aspect around {@link Rule#preAction(SimulationContext, Date, Metier)} in packages "rules".
     * 
     * @param jp join point
     * @return preAction result
     * @throws Throwable
     */
    @Around("execution(* rules.*.preAction(..))")
    public Object initPreCall(final JoinPoint jp) throws Throwable {
        
        Object result = makeTimedCall(jp, State.PRE);
        return result;
    }
    
    /**
     * Aspect around {@link Rule#postAction(SimulationContext, Date, Metier)} in packages "rules".
     * 
     * @param jp join point
     * @return preAction result
     * @throws Throwable
     */
    @Around("execution(* rules.*.postAction(..))")
    public Object initPostCall(final JoinPoint jp) throws Throwable {
        
        Object result = makeTimedCall(jp, State.POST);
        return result;
    }

    /**
     * Effectue l'appel reel en calculant le temps pris.
     * 
     * @param jp join point
     * @param state state
     * @throws Throwable
     */
    protected Object makeTimedCall(JoinPoint jp, State state) throws Throwable {
        
        if (log.isTraceEnabled()) {
            log.trace("Rule aspect called : " + jp.getCalleeClass().getSimpleName());
        }
        
        // get time before
        long timeBeforeCall = System.currentTimeMillis();
        
        // make real call
        Object result = jp.proceed();
        
        // get time after
        long timeAfterCall = System.currentTimeMillis();
        
        // get real time
        long timeTaken = timeAfterCall - timeBeforeCall;

        registerTime(jp, state, timeTaken);

        return result;
        
    }

    /**
     * Add time (init/pre/post) to simulation information in
     * current {@link SimulationContext}.
     * 
     * @param jp join point
     * @param state state 
     * @param timeTaken time taken by {@code state}
     */
    protected void registerTime(JoinPoint jp, State state, long timeTaken) {
        SimulationContext context = SimulationContext.get();
        
        SimulationStorage simulation = context.getSimulationStorage();
        
        // happen if called outside a simulation
        // so ... can't happen :)
        if (simulation != null) {
            
            // get rule information
            Class<?> calleeClass = jp.getCalleeClass();
            String ruleName = calleeClass.getSimpleName();
            
            SimulationInformation info = simulation.getInformation();
            
            switch (state) {
            case INIT:
                info.addRuleInitTime(ruleName, timeTaken);
                break;
            case PRE:
                info.addRulePreTime(ruleName, timeTaken);
                break;
            case POST:
                info.addRulePostTime(ruleName, timeTaken);
                break;
            default:
                break;
            }
        }
        else {
            if (log.isWarnEnabled()) {
                log.warn("Time register called outside a simulation context");
            }
        }
    }
}
