/*
 * #%L
 * IsisFish
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2002 - 2010 Ifremer, Code Lutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.lang.time.DurationFormatUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.ArgumentsParserException;
import org.nuiton.util.StringUtil;
import org.nuiton.util.Version;

import fr.ifremer.isisfish.actions.ExportAction;
import fr.ifremer.isisfish.actions.ImportAction;
import fr.ifremer.isisfish.actions.OtherAction;
import fr.ifremer.isisfish.actions.SimulationAction;
import fr.ifremer.isisfish.actions.VCSAction;
import fr.ifremer.isisfish.simulator.SimulationContext;
import fr.ifremer.isisfish.simulator.SimulationControl;
import fr.ifremer.isisfish.simulator.launcher.InProcessSimulatorLauncher;
import fr.ifremer.isisfish.simulator.launcher.SSHSimulatorLauncher;
import fr.ifremer.isisfish.simulator.launcher.SimulationService;
import fr.ifremer.isisfish.simulator.launcher.SubProcessSimulationLauncher;
import fr.ifremer.isisfish.vcs.VCS;

/**
 * Isis fish configuration.
 * 
 * @author poussin
 * @version $Revision: 1310 $
 *
 * Last update: $Date: 2008-08-23 00:46:00 +0200 (Sat, 23 Aug 2008) $
 * by : $Author: bpoussin $
 */
public class IsisConfig extends ApplicationConfig {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(IsisConfig.class);
    
    /**
     * Config step after init.
     *
     * @see org.nuiton.util.ApplicationConfig.Action.Step
     */
    public static final int STEP_AFTER_INIT = 0;
    
    /**
     * Config step after init vcs.
     * 
     * @see org.nuiton.util.ApplicationConfig.Action.Step
     */
    public static final int STEP_AFTER_INIT_VCS = 1;
    
    /**
     * Config step after ui.
     * 
     * @see org.nuiton.util.ApplicationConfig.Action.Step
     */
    public static final int STEP_AFTER_UI = 2;
    
    /**
     * Config step before exit.
     * 
     * @see org.nuiton.util.ApplicationConfig.Action.Step
     */
    public static final int STEP_BEFORE_EXIT = 3;
    
    /**
     * La version du logiciel constitue de l.d.a.r
     * <li>l: le numero de version du logiciel
     * <li>d: le numero de version du schema de la base de donnees
     * <li>a: le numero de version de l'api des scripts
     * <li>r: le numero de version de de l'interface graphique ou autre modif mineur
     * <p>
     * lors de l'increment de l, d, a et r sont remis a 0
     * lors de l'increment de d, a et r sont remis a 0
     * lors de l'increment de a, r est remis a 0
     * <p>
     * Un changement d'UI ne modifie jamais le numero de version de database
     * Un changement de schema de base pour lequel on ne peut pas faire de
     * migration de donnees demande automatiquement un changement de version
     * d'application.
     */
    protected final static Version version = new Version(3, 3, 0, 9);
    protected final static Version databaseVersion = new Version(
            version.getNumber(0), version.getNumber(1));
    protected final static Version apiVersion = new Version(
            version.getNumber(0), version.getNumber(1), version.getNumber(2));
            
    public static Version getVersionNumber() {
        return version;
    }
    
    /**
     * Le nombre global ex: 3.2.0.0
     * 
     * @return full version
     */
    static public String getVersion() {
        String result = version.toString();
        return result;
    }

    /**
     * La version de la base ex: 3.2
     * 
     * @return database version
     */
    public static Version getDatabaseVersion() {
        return databaseVersion;
    }

    /**
     * La version de l'api de programmation ex: 3.2.0
     * 
     * @return api version
     */
    public static Version getApiVersion() {
        return apiVersion;
    }
    
    static final public String COPYRIGHT_TEXT = "Version " + getVersion() + " IFREMER-MAERHA © 2000-2010";
    static final public String CONFIG_FILENAME = "isis-config-" + version.getNumber(0);
    
    /** separateur de liste */
    static final public String SEP = ",";
    static final public String REPORT_EMAIL = "isis-fish-bugreport@lists.labs.libre-entreprise.org";
            
    protected transient File backupSessionDirectory = null;
    
    /**
     * Mémorise une unité de temps en millisecondes.
     * 
     * Auparavant, c'etait {@link System#nanoTime()}, mais cette unité de temps
     * n'est pas lié a une date.
     * Par exemple, ca permet juste de mesuré avec une precision de 10-9 le temps
     * ecoulé, par l'heure qu'il est à cette unité de temps.
     */
    protected long startingTime = System.currentTimeMillis();
    
    public IsisConfig() {
        
        for (Option o : Option.values()) {
            setDefaultOption(o.key, o.defaultValue);
        }
        
        for (Action a : Action.values()) {
            for (String alias : a.aliases) {
                addActionAlias(alias, a.action);                
            }
        }

    }
    
    
    //////////////////////////////////////////////////
    // Methode d'acces aux options
    //////////////////////////////////////////////////

    /**
     * Retourne le repertoire racine de toutes les donnees (script, simulation
     * region, ...)
     * 
     * @return database directory
     */
    public File getDatabaseDirectory() {
        File result = getOptionAsFile(Option.DATABASE_DIRECTORY.key);
        return result;
    }

    /**
     * Get database directory to use for script.
     * 
     * Defaut to {@link #getDatabaseDirectory()} value, but to
     * {@link SimulationContext#getScriptDirectory()} during a simulation.
     * 
     * @return script database directory
     */
    public File getContextDatabaseDirectory() {

        // add a subdirectory for simulation context
        SimulationContext simContext = SimulationContext.get();
        File scriptDirectory = simContext.getScriptDirectory();

        if (scriptDirectory == null) {
            // if null, not in simulation 
            scriptDirectory = getDatabaseDirectory();
        }

        return scriptDirectory;
    }

    /**
     * Retourne le lock a utiliser pour la base h2.
     * 
     * Par defaut la base utilise 'file'.
     * 
     * Les valeurs acceptées sont :
     *  - file
     *  - socket
     *  - no
     * 
     * http://www.h2database.com/html/features.html#database_file_locking
     * 
     * @return h2 db lock mode
     */
    public String getDatabaseLockMode() {
        String result = getOption(Option.DATABASE_LOCK_MODE.key);
        return result;
    }

    /**
     * Retourne le repertoire ou sont stockes les scripts compiles.
     * 
     * Create directory if not exists.
     * 
     * Make a specific build directory for running simulation.
     * 
     * @return compilation directory
     */
    public File getCompileDirectory() {
        File result = getOptionAsFile(Option.COMPILATION_DIRECTORY.key);

        // add a subdirectory for simulation context
        SimulationContext simContext = SimulationContext.get();
        SimulationControl control = simContext.getSimulationControl();
        if (control != null) {
            result = new File(result, control.getId());
        }
        
        if (!result.exists()) {
            result.mkdirs();
        }

        return result;
    }

    /**
     * Retourne le repertoire ou sont stockes les informations relatives
     * aux simulations en cours.
     * 
     * @return monitoring directory
     */
    public File getMonitoringDirectory() {
        File result = getOptionAsFile(Option.MONITORING_DIRECTORY.key);

        if (!result.exists()) {
            result.mkdirs();
        }

        return result;
    }

    /**
     * Get javadoc directory.
     * 
     * Create directory if not exists.
     * 
     * @return javadoc directory
     */
    public File getJavadocDirectory() {
        File result = getOptionAsFile(Option.JAVADOC_DIRECTORY.key);
        if (!result.exists()) {
            result.mkdirs();
        }
        return result;
    }

    /**
     * Retourne l'objet {@link Locale} a utilise pour la langue.
     * 
     * @return application {@link Locale}
     */
    public Locale getLocale() {
        String value = getOption(Option.LOCALE.key);
        Locale result = (Locale)ConvertUtils.convert(value, Locale.class);
        return result;
    }

    /**
     * Change application locale.
     * (used during application config command line parse)
     * 
     * @param locale new locale
     */
    public void setLocale(String locale) {
        setOption(Option.LOCALE.key, locale);
    }

    /**
     * Retourne l'encoding a utiliser pour les fichiers textes.
     * 
     * @return encoding to use
     */
    public String getEncoding() {
        String result = getOption(Option.ENCODING.key);
        return result;
    }

    /**
     * Retourne le serveur SMTP a utiliser pour l'envoie de mail.
     * 
     * @return smtp server address
     */
    public String getSmtpServer() {
        String result = getOption(Option.SMTP_SERVER.key);
        return result;
    }

    /**
     * Retourne le nom usuel de l'utilisateur.
     * 
     * @return username
     */
    public String getUserName() {
        String result = getOption(Option.USER_NAME.key);
        return result;
    }
    
    /**
     * Retourne l'email de l'utilisateur.
     * 
     * @return user email
     */
    public String getUserMail() {
        String result = getOption(Option.USER_MAIL.key);
        return result;
    }
    
    public String getSimulatorClassfile() {
        String result = getOption(Option.SIMULATOR_CLASSFILE.key);
        return result;
    }
    
    public void setSimulatorClassfile(String value) {
        setOption(Option.SIMULATOR_CLASSFILE.key, value);
    }

    /**
     * Retourne l'url du serveur de simulation accessible via SSH.
     * 
     * @return simulator server
     */
    public String getSimulatorSshServer() {
        String result = getOption(Option.SIMULATOR_SSH_SERVER.key);
        return result;
    }

    /**
     * Change ssh server url value.
     * 
     * @param sshServer
     */
    public void setSimulatorSshServer(String sshServer) {
        setOption(Option.SIMULATOR_SSH_SERVER.key, sshServer);
    }

    /**
     * Retourne le login pour acceder au serveur de simulation accessible via SSH.
     * 
     * @return simulator username
     */
    public String getSimulatorSshUsername() {
        String result = getOption(Option.SIMULATOR_SSH_USER_NAME.key);
        return result;
    }

    /**
     * Change ssh username.
     * 
     * @param username username
     */
    public void setSimulatorSshUsername(String username) {
        setOption(Option.SIMULATOR_SSH_USER_NAME.key, username);
    }

    /**
     * Retourne le chemin relatif ou abs du repertoire root des données d'isis.
     * 
     * @return simulator data path
     */
    public String getSimulatorSshDataPath() {
        String result = getOption(Option.SIMULATOR_SSH_DATAPATH.key);
        return result;
    }

    /**
     * Change remote datapath.
     * 
     * @param datapath datapath
     */
    public void setSimulatorSshDataPath(String datapath) {
        setOption(Option.SIMULATOR_SSH_DATAPATH.key, datapath);
    }

    /**
     * Get remote user home directory.
     * 
     * @return remote user home
     */
    public String getSimulatorSshUserHome() {
        String result = getOption(Option.SIMULATOR_SSH_USER_HOME.key);
        return result;
    }

    /**
     * Change isis remote home directory.
     * 
     * @param userhome new remote home
     */
    public void setSimulatorSshUserHome(String userhome) {
        setOption(Option.SIMULATOR_SSH_USER_HOME.key, userhome);
    }

    /**
     * Retourne le chemin distant ou est installé isis.
     * 
     * @return remote isis home
     */
    public String getSimulatorSshIsisHome() {
        String result = getOption(Option.SIMULATOR_SSH_ISIS_HOME.key);
        return result;
    }

    /**
     * Change isis home on ssh server.
     * 
     * @param isishome isis home
     */
    public void setSimulatorSshIsisHome(String isishome) {
        setOption(Option.SIMULATOR_SSH_ISIS_HOME.key, isishome);
    }

    /**
     * Retourne l'emplacement du dossier temporaire distant.
     * 
     * Ce dossier doit être ABSOLUT.
     * 
     * @return tmp dir
     */
    public String getSimulatorSshTmpPath() {
        String result = getOption(Option.SIMULATOR_SSH_TMPPATH.key);
        return result;
    }

    /**
     * Change ssh temp path.
     * 
     * @param sshtemppath temp path
     */
    public void setSimulatorSshTmpPath(String sshtemppath) {
        setOption(Option.SIMULATOR_SSH_TMPPATH.key, sshtemppath);
    }

    /**
     * PBS bin path directory.
     * 
     * @return path
     */
    public String getSimulatorSshPbsBinPath() {
        String result = getOption(Option.SIMULATOR_SSH_PBSBINPATH.key);
        return result;
    }

    /**
     * Change PBS bin path.
     * 
     * @param path new path
     */
    public void setSimulatorSshPbsBinPath(String path) {
        setOption(Option.SIMULATOR_SSH_PBSBINPATH.key, path);
    }
    
    /**
     * PBS qsub options (command line).
     * 
     * See man qsub for available options.
     * 
     * @return options
     */
    public String getSimulatorSshPbsQsubOptions() {
        String result = getOption(Option.SIMULATOR_SSH_PBSQSUBOPTIONS.key);
        return result;
    }

    /**
     * Change PBS qsub options (command line).
     * 
     * @param options new options
     */
    public void setSimulatorSshPbsQsubOptions(String options) {
        setOption(Option.SIMULATOR_SSH_PBSQSUBOPTIONS.key, options);
    }
    
    /**
     * Java path directory.
     * 
     * @return path
     */
    public String getSimulatorSshJavaPath() {
        String result = getOption(Option.SIMULATOR_SSH_JAVAPATH.key);
        return result;
    }

    /**
     * Change Java bin path.
     * 
     * @param path new path
     */
    public void setSimulatorSshJavaPath(String path) {
        setOption(Option.SIMULATOR_SSH_JAVAPATH.key, path);
    }

    /**
     * Retourne l'interval de temps a utiliser
     * pour recuperer le fichier de control.
     * 
     * @return time (in seconds)
     */
    public int getSimulatorSshControlCheckInterval() {
        int result = getOptionAsInt(Option.SIMULATOR_SSH_CONTROLCHECKINTERVAL.key);
        return result;
    }

    /**
     * Change control check interval.
     * 
     * @param interval interval
     */
    public void setSimulatorSshControlCheckInterval(int interval) {
        setOption(Option.SIMULATOR_SSH_CONTROLCHECKINTERVAL.key, String.valueOf(interval));
    }
    
    /**
     * Retourne le nombre de thread simultane a utiliser par le pool de thread
     * du launcher SSH.
     * 
     * @return number of thread to use
     */
    public int getSimulatorSshMaxThreads() {
        int result = getOptionAsInt(Option.SIMULATOR_SSH_MAXTHREADS.key);
        return result;
    }

    /**
     * Set number of ssh simulation thread to use.
     * 
     * @param max max
     */
    public void setSimulatorSshMaxThreads(int max) {
        setOption(Option.SIMULATOR_SSH_MAXTHREADS.key, String.valueOf(max));
    }

    /**
     * Retourne la clé privée de l'utilisteur courant.
     * 
     * @return private ssh key path
     */
    public File getSSHPrivateKeyFilePath() {
        File result = getOptionAsFile(Option.SSH_KEY_FILE.key);
        return result;
    }

    /**
     * Set ssh private key file path.
     * 
     * @param sshFile new value
     */
    public void setSSHPrivateKeyFilePath(File sshFile) {
        setOption(Option.SSH_KEY_FILE.key, sshFile.getAbsolutePath());
    }

    /*public boolean isUseVCS() {
        String value = getOption(Option.VCS_TYPE.key);
        boolean result = !VCSNone.TYPE_NONE.equals(value);
        return result;
    }*/

    /**
     * Get launch.ui option value.
     * 
     * @return launch ui option value
     */
    public boolean isLaunchUI() {
        boolean result = getOptionAsBoolean(Option.LAUNCH_UI.key);
        return result;
    }

    /**
     * Get perform migration option value.
     * 
     * @return perform migration option value
     */
    public boolean isPerformMigration() {
        boolean result = getOptionAsBoolean(Option.PERFORM_MIGRATION.key);
        return result;
    }

    /**
     * Get perform vcs update option value.
     * 
     * @return perform vcs update option value
     */
    public boolean isPerformVcsUpdate() {
        boolean result = getOptionAsBoolean(Option.PERFORM_VCS_UPDATE.key);
        return result;
    }

    /**
     * Get perform cron option value.
     * 
     * @return perform cron option value
     */
    public boolean isPerformCron() {
        boolean result = getOptionAsBoolean(Option.PERFORM_CRON.key);
        return result;
    }

    public void setSimulatorLauncher(String value) {
        setOption(Option.SIMULATOR_LAUNCHER.key, value);
    }

    public boolean isSimulationShowOnlyError() {
        Boolean result = getOptionAsBoolean(Option.SIMULATION_SHOW_ONLY_ERROR.key);
        return result;
    }

    public String getDefaultExportNames() {
        String result = getOption(Option.DEFAULT_EXPORT_NAMES.key);
        return result;
    }

    public void setDefaultExportNames(List<String> exportNames) {
        StringBuilder sb = new StringBuilder();
        for (String exportName : exportNames) {
            sb.append(SEP).append(exportName);
        }
        String value = sb.toString();
        // remove first SEP
        if (value.length() > 0) {
            value = value.substring(1);
        }
        setOption(Option.DEFAULT_EXPORT_NAMES.key, value);
        saveForUser();
    }

    /**
     * @return la liste des noms d'exports par defaut sous forme de liste,
     *         a partir de la propriete {@link Option#DEFAULT_EXPORT_NAMES}
     *         ou null si ils n'ont jamais ete sauves par l'utilisateur.
     *         by user.
     * @see Option#DEFAULT_EXPORT_NAMES
     */
    public List<String> getDefaultExportNamesAsList() {
        List<String> result = null;
        String exportNamesList = getDefaultExportNames();
        if (exportNamesList != null) {
            result = new ArrayList<String>();
            String[] exportNames = StringUtil.split(exportNamesList, ",");
            result.addAll(Arrays.asList(exportNames));
        }
        return result;
    }

   public String getDefaultMapFilename() {
        String result = getOption(Option.DEFAULT_MAP_FILENAME.key);
        return result;
    }

   public String getDefaultResultNames() {
        String result = getOption(Option.DEFAULT_RESULT_NAMES.key);
        return result;
    }

    public void setDefaultResultNames(List<String> resultNames) {
        StringBuilder sb = new StringBuilder();
        for (String resultName : resultNames) {
            sb.append(SEP).append(resultName);
        }
        String value = sb.toString();
        // remove first SEP
        if (value.length() > 0) {
            value = value.substring(1);
        }
        setOption(Option.DEFAULT_RESULT_NAMES.key, value);
        saveForUser();
    }

    /**
     * @return les resultats par defaut d'une simulation sous forme de liste
     *         a partir de la propriete {@link Option#DEFAULT_RESULT_NAMES}
     * @see Option#DEFAULT_RESULT_NAMES
     */
    public List<String> getDefaultResultNamesAsList() {
        List<String> result = null;
        String resultNamesList = getDefaultResultNames();
        if (resultNamesList != null) {
            result = new ArrayList<String>();
            String[] resultNames = StringUtil.split(resultNamesList, ",");
            result.addAll(Arrays.asList(resultNames));
        }
        return result;
    }
    
    public String getDefaultTagValue() {
        String result = getOption(Option.DEFAULT_TAG_VALUE.key);
        return result;
    }
    
    public void setDefaultTagValues(Map<String, String> tagValues) {
        StringBuilder sb = new StringBuilder(" ");
        for (Map.Entry<String, String> entry : tagValues.entrySet()) {
            sb.append(SEP).append('"').append(entry.getKey()).append("\":\"").append(entry.getValue()).append('"');
        }
        String value = sb.toString().trim();
        setOption(Option.DEFAULT_TAG_VALUE.key, value.isEmpty() ? "" : value.substring(1));
        saveForUser();
    }


    
    /**
     * @return le dictionnaire des tags par defaut d'une simulation a partir
     *         de la propriete  {@link Option#DEFAULT_TAG_VALUE}
     * @see Option#DEFAULT_TAG_VALUE
     */
    public Map<String, String> getDefaultTagValueAsMap() {
        Map<String, String> result = new HashMap<String, String>();
        String tagValuesList = getDefaultTagValue();
        if (tagValuesList != null) {
            String[] tagValues = StringUtil.split(tagValuesList, ",");
            for (String tagValue : tagValues) {
                String[] tagAndValue = StringUtil.split(tagValue, ":");

                String tag = tagAndValue[0].trim();
                tag = tag.substring(1, tag.length() - 1); // remove "..."

                String value = tagAndValue[1].trim();
                value = value.substring(1, value.length() - 1); // remove "..."

                result.put(tag, value);
            }
        }
        return result;
    }
    
    public String getJavadocURL() {
        String result = getOption(Option.JAVADOC_URL.key);
        return result;
    }

    public File getBackupDirectory() {
        File result = getOptionAsFile(Option.BACKUP_DIRECTORY.key);
        return result;
    }

    /**
     * Retourne le temps de départ de l'application en millisecondes.
     *
     * @return application start millis time
     */
    public long getStartingTime() {
        return startingTime;
    }
    
    /**
     * Retourne le temps écoulé depuis de debut de l'application (en secondes).
     * 
     * @return le temps écoulé en seconde sous forme de chaine
     */
    public String getElapsedTimeAsString() {
        long diff = System.currentTimeMillis() - getStartingTime();
        String result = DurationFormatUtils.formatDuration(diff, "s'.'S");
        return result;
    }
            
    public File getBackupSessionDirectory() {
        if (backupSessionDirectory == null) {
            long time = getStartingTime();
            // creation de l'unique répertoire de backup pour la session
            String path = new SimpleDateFormat("yyyy-MM-dd-HH-mm-ss").format(new java.util.Date(time));
            backupSessionDirectory = new File(getBackupDirectory(), path);
            if (!backupSessionDirectory.exists()) {
                backupSessionDirectory.mkdirs();
            }
        }
        return backupSessionDirectory;
    }

    /**
     * Retourne un class loader contenant le repertoire de compilation
     * Il permet alors de charger des classes qui viennent d'etre compilees
     * dans isis
     * If current thread is in simulation then return specific simulation
     * compilation directory, else default compilation directory
     *
     * @return the class loader adequate
     */
    public ClassLoader getScriptClassLoader() {
        SimulationContext simContext = SimulationContext.get();
        ClassLoader result = simContext.getClassLoader();
        if (result == null) {
            // on est pas dans une simulation, il faut retourner un nouveau
            // a chaque fois. on force la creation d'un nouveau classloader
            // a chaque fois pour
            // que l'ancienne class compiler et charg<E9> ne soit pas presente
            File f = getCompileDirectory();
            try {
                URL[] cp = new URL[]{f.toURI().toURL()};
                // il faut prendre le ClassLoader du thread courant comme parent
                // car pour les simulations il a ete modifi<E9>, et il faut
                // que les classes de script soit recherch<E9> dedans avant
                // la recherche dans le getCompileDirectory().
                // ce qui est le cas avec les URLClassLoader
                ClassLoader parent = Thread.currentThread().getContextClassLoader();
                result = new URLClassLoader(cp, parent);
            } catch (MalformedURLException eee) {
                throw new IsisFishRuntimeException(_("isisfish.error.load.classloader", f, eee.getMessage()), eee);
            }
        }
        return result;
    }

    /**
     * Surcharge pour la migration des options de config qui ont change de nom
     * <p>
     * TODO: lors du passage en version 4.0 on pourrait supprimer cette methode
     * 
     * @param args
     * @throws org.nuiton.util.ArgumentsParserException
     */
    @Override
    public void parse(String[] args) throws ArgumentsParserException {
        super.parse(args);
        // dans la version precedente (3.1) seul le fichier user existe, il est
        // donc le seul a devoir etre modifie
        boolean mustSave = false;
        String[] keys = new String[]{
            "compileDirectory", Option.COMPILATION_DIRECTORY.key,
            "defaultBackupDirectory", Option.BACKUP_DIRECTORY.key,
            "defaultExportNames", Option.DEFAULT_EXPORT_NAMES.key,
            "defaultMapFile", Option.DEFAULT_MAP_FILENAME.key,
            "defaultResultNames", Option.DEFAULT_RESULT_NAMES.key,
            "defaultSimulator", Option.SIMULATOR_CLASSFILE.key,
            "defaultTagValue", Option.DEFAULT_TAG_VALUE.key,
            "javadocURL", Option.JAVADOC_URL.key,
            "locale", Option.LOCALE.key,
            "simulationShowOnlyError", Option.SIMULATION_SHOW_ONLY_ERROR.key,
            "simulationShowOnlyQueue", null, // Option.SIMULATION_SHOW_ONLY_QUEUE.key,
            "smtpServer", Option.SMTP_SERVER.key,
            "userMail", Option.USER_MAIL.key,
            "userName", Option.USER_NAME.key,
            "vcs.keyFile", Option.SSH_KEY_FILE.key,
            "vcs.keyFile", Option.SSH_KEY_FILE.key,
            "vcs.ssh.keyFile", Option.SSH_KEY_FILE.key,
            "vcs.localDatabasePath", Option.DATABASE_DIRECTORY.key,
            // on supprime car non compatible, les valeurs par defaut sont tres bien
            "localSimulator", null,
            "vcs.databaseVersion", null,
            "vcs.hostName", null,
            "vcs.noPassPhrase", null,
            "vcs.remoteDatabase", null,
            "vcs.remotePath", null, 
            "vcs.type", null,
            "vcs.typeRepo", null,
            "vcs.useSshConnexion", null,
            "vcs.userName", null,
            "version", null,
            "projectName", null,
            "simulationReportMail", null,
            
        };
        
        if (log.isInfoEnabled()) {
            log.info("Check configuration change");
        }
        
        for(int i=0; i<keys.length;) {
            String oldKey = keys[i++];
            String newKey = keys[i++];
            if (!oldKey.equals(newKey)) {
                String value = getOption(oldKey);
                if (value != null) {
                    mustSave = true;
                    // quoi qu'il arrive on enleve l'ancienne cle
                    homefile.remove(oldKey);    
                    if (newKey != null) {
                        setOption(newKey, value);
                    }
                }
            }
        }
        if (mustSave) {
            
            // FIXME EC-20100602 this save for user and configuration can be
            // lost when multiple instance are launched at same time !!!
            saveForUser();

            if (log.isInfoEnabled()) {
                log.info("Config file migration done");
            }
        }
        if (log.isDebugEnabled()) {
            printConfig();
        }
    }

    //////////////////////////////////////////////////
    // Toutes les options disponibles
    //////////////////////////////////////////////////

    /**
     * Options.
     * 
     * Set it protected to force getOption() call.
     */
    public static enum Option implements OptionDef {

        COMPILATION_DIRECTORY("compilation.directory", _("isisfish.config.main.compileDirectory.description"), getUserHome() + File.separator + "isis-build"),
        MONITORING_DIRECTORY("monitoring.directory", _("isisfish.config.main.monitoringDirectory.description"), getUserHome() + File.separator + "isis-monitoring"),
        JAVADOC_DIRECTORY("javadoc.directory", _("isisfish.config.main.javadocDirectory.description"), getUserHome() + File.separator + "isis-docs"),
        CONFIG_FILE(CONFIG_FILE_NAME, _("isisfish.config.main.configFileName.description"), CONFIG_FILENAME),
        BACKUP_DIRECTORY("backup.directory", _("isisfish.config.main.defaultBackupDirectory.description"), getUserHome() + File.separator + "isis-backup"),

        DEFAULT_EXPORT_NAMES("default.export.names", _("isisfish.config.main.defaultExportNames.description"), ""),
        DEFAULT_RESULT_NAMES("default.result.names", _("isisfish.config.main.defaultResultNames.description"), ""),
        DEFAULT_MAP_FILENAME("default.map.filename", _("isisfish.config.main.defaultMapFile.description"), "maps/vmap_area_thin.shp"),
        DEFAULT_TAG_VALUE("default.tagvalue", _("isisfish.config.main.defaultTagValue.description"), ""),
        ENCODING("encoding", _("isisfish.config.main.encoding.description"), "UTF-8"),
        JAVADOC_URL("javadoc.url", _("isisfish.config.main.javadocURL.description"), "http://isis-fish.labs.libre-entreprise.org/isis-fish/apidocs/"),
        
        SIMULATOR_CLASSFILE("simulator.classfile", _("isisfish.config.main.defaultSimulator.description"), "DefaultSimulator.java"),
        /** prevu pour l'architecture de lancement en plugin: local, isis-server, caparmor, ... */
        SIMULATOR_LAUNCHER(SimulationService.SIMULATION_LAUNCHER + ".3", _("isisfish.config.main.localSimulator.description"), InProcessSimulatorLauncher.class.getName()),
        SIMULATOR_LAUNCHER2(SimulationService.SIMULATION_LAUNCHER + ".1", _("isisfish.config.main.subSimulator.description"), SubProcessSimulationLauncher.class.getName()),
        SIMULATOR_LAUNCHER_REMOTE(SimulationService.SIMULATION_LAUNCHER + ".2", _("isisfish.config.main.remoteCaparmor.description"), SSHSimulatorLauncher.class.getName()),

        /** Serveur accessible par ssh : address */
        SIMULATOR_SSH_SERVER("simulation.ssh.server", _("isisfish.config.main.simulation.ssh.server.description"), "caparmor"),
        /** Serveur accessible par ssh : login */
        SIMULATOR_SSH_USER_NAME("simulation.ssh.username", _("isisfish.config.main.simulation.ssh.username.description"), ""),
        /** Serveur accessible par ssh : user home directory */
        SIMULATOR_SSH_USER_HOME("simulation.ssh.userhome", _("isisfish.config.main.simulation.ssh.userhome.description"), ""),
        /** Serveur accessible par ssh : remote data path */
        SIMULATOR_SSH_DATAPATH("simulation.ssh.datapath", _("isisfish.config.main.simulation.ssh.datapath.description"), "isis-database-3"),
        /** Serveur accessible par ssh : remote isis home install */
        SIMULATOR_SSH_ISIS_HOME("simulation.ssh.isis.home", _("isisfish.config.main.simulation.ssh.isis.home.description"), "/home3/caparmor/poussin/isis-fish-" + IsisConfig.getApiVersion()),
        /** Serveur accessible par ssh : remote tmp path */
        SIMULATOR_SSH_TMPPATH("simulation.ssh.tmppath", _("isisfish.config.main.simulation.ssh.tmppath.description"), "isis-tmp"),
        /** Serveur accessible par SSH : emplacement de Java (full path) */
        SIMULATOR_SSH_JAVAPATH("simulation.ssh.javapath", _("isisfish.config.main.simulation.ssh.javapath.description"), "/home3/caparmor/poussin/jdk/bin/java"),
        /** Serveur accessible par SSH : emplacement des executables pbs */
        SIMULATOR_SSH_PBSBINPATH("simulation.ssh.pbsbinpath", _("isisfish.config.main.simulation.ssh.pbsbinpath.description"), "/usr/pbs/bin"),
        /** Serveur accessible par SSH : option de l'executable qsub (defaut to -m n = no mail) */
        SIMULATOR_SSH_PBSQSUBOPTIONS("simulation.ssh.pbsqsuboptions", _("isisfish.config.main.simulation.ssh.pbsqsuboptions.description"), "-m n"),
        /** Serveur accessible par SSH : interval de check du fichier de control */
        SIMULATOR_SSH_CONTROLCHECKINTERVAL("simulation.ssh.control.check.interval", _("isisfish.config.main.simulation.ssh.control.check.interval.description"), "120"),
        /** Serveur accessible par SSH : nombre de thread au maximum a utilise simultanement */
        SIMULATOR_SSH_MAXTHREADS("simulation.ssh.max.threads", _("isisfish.config.main.simulation.ssh.max.threads.description"), "1"),

        /** Application locale (for i18n init). */
        LOCALE("locale", _("isisfish.config.main.locale.description"), "fr_FR"),
        SIMULATION_SHOW_ONLY_ERROR("simulationShowOnlyError", _("isisfish.config.main.simulationShowOnlyError.description"), "false"),
        
        /** Perform ui launch option. */
        LAUNCH_UI("launch.ui", _("isisfish.config.main.launchUI.description"), "true"),
        /** Perform data migration option. */
        PERFORM_MIGRATION("perform.migration", _("isisfish.config.main.performmigration.description"), "true"),
        /** Perform vcs update option. */
        PERFORM_VCS_UPDATE("perform.vcsupdate", _("isisfish.config.main.performvcsupdate.description"), "true"),
        /** Perform cron option. */
        PERFORM_CRON("perform.cron", _("isisfish.config.main.performcron.description"), "true"),
        
        /** Database directory. */
        DATABASE_DIRECTORY("database.directory", _("isisfish.config.vcs.localDatabasePath.description"), getUserHome() + File.separator + "isis-database-3"),
        /** Database lock mode (h2).*/
        DATABASE_LOCK_MODE("database.lockmode", _("isisfish.config.database.lockmode.description"), "file"),
        
        // SSH (global, for both VCS and caparmor)
        SSH_KEY_FILE("ssh.key.file", _("isisfish.config.ssh.key.file.description"), getUserHome() + File.separator + ".ssh" + File.separator + "isis_rsa"),
        //SSH_PASSPHRASE_ENABLED("ssh.passphrase.enabled", _("isisfish.config.vcs.noPassPhrase.description"), "false"),
        //SSH_PASSPHRASE("ssh.passphrase", _("isisfish.config.vcs.passphrase.description"), ""),

        // can be None, CVS or SVN. only None or SVN work
        VCS_TYPE(VCS.VCS_TYPE, _("isisfish.config.vcs.type.description"), VCS.TYPE_SVN),
        // depend of VCS_TYPE, for SVN can be svn, svn+ssh, http or file
        VCS_PROTOCOL(VCS.VCS_PROTOCOL, _("isisfish.config.vcs.useSshConnexion.description"), "svn"),
        
        // user login to access vcs
        VCS_USER_NAME(VCS.VCS_USER_NAME, _("isisfish.config.vcs.userName.description"), ""),
        VCS_USER_PASSWORD(VCS.VCS_USER_PASSWORD, _("isisfish.config.vcs.userPassword.description"), ""),
        VCS_HOST_NAME(VCS.VCS_HOST_NAME, _("isisfish.config.vcs.hostName.description"), "labs.libre-entreprise.org"),
        VCS_PATH(VCS.VCS_PATH, _("isisfish.config.vcs.remotePath.description"), "/svnroot/isis-fish-data/"),

        USER_NAME("user.name", _("isisfish.config.main.userName.description"), System.getProperty("user.name")),
        SMTP_SERVER("smtpServer", _("isisfish.config.main.smtpServer.description"), "smtp"),
        USER_MAIL("userMail", _("isisfish.config.main.userMail.description"), USER_NAME.key + "@" + VCS_HOST_NAME.key);
        
        public final String key;
        public final String description;
        public final String defaultValue;

        private Option(String key, String description, String defaultValue) {
            this.key = key;
            this.description = description;
            this.defaultValue = defaultValue;
        }

        /*
         * @see org.nuiton.util.ApplicationConfig.OptionDef#getDefaultValue()
         */
        @Override
        public String getDefaultValue() {
            return defaultValue;
        }

        /*
         * @see org.nuiton.util.ApplicationConfig.OptionDef#getDescription()
         */
        @Override
        public String getDescription() {
            return description;
        }

        /*
         * @see org.nuiton.util.ApplicationConfig.OptionDef#getKey()
         */
        @Override
        public String getKey() {
            return key;
        }

        /*
         * @see org.nuiton.util.ApplicationConfig.OptionDef#getType()
         */
        @Override
        public Class<?> getType() {
            return String.class;
        }

        /*
         * @see org.nuiton.util.ApplicationConfig.OptionDef#isFinal()
         */
        @Override
        public boolean isFinal() {
            return false;
        }

        /*
         * @see org.nuiton.util.ApplicationConfig.OptionDef#isTransient()
         */
        @Override
        public boolean isTransient() {
            return false;
        }
    }
    
    //////////////////////////////////////////////////
    // Toutes les actions disponibles
    //////////////////////////////////////////////////
    
    public static enum Action {
        HELP(_("Show help"), OtherAction.class.getName() + "#help", "-h", "--help"),
        VERSION(_("Show version"), OtherAction.class.getName() + "#version", "-v", "--version"),
        
        IMPORT_ANALYSE_PLAN(_(""), ImportAction.class.getName() + "#importAnalysePlan", "--importAnalysePlan"),
        IMPORT_EXPORT(_(""), ImportAction.class.getName() + "#importExport", "--importExport"),
        IMPORT_RULE(_(""), ImportAction.class.getName() + "#importRule", "--importRule"),
        IMPORT_SCRIPT(_("Import one java file script source"), ImportAction.class.getName() + "#importScript", "--importScript"),
        IMPORT_SIMULATOR(_(""), ImportAction.class.getName() + "#importSimulator", "--importSimulator"),
        IMPORT_FORMULA(_(""), ImportAction.class.getName() + "#importFormula", "--importFormula"),
        IMPORT_REGION(_(""), ImportAction.class.getName() + "#importRegion", "--importRegion"),
        IMPORT_REGION_AND_RENAME(_(""), ImportAction.class.getName() + "#importRegionAndRename", "--importRegionAndRename"),
        IMPORT_SIMULATION(_(""), ImportAction.class.getName() + "#importSimulation", "--importSimulation"),
        IMPORT_SCRIPT_MODULE(_("Import zipped file containing all scripts directory structure"), ImportAction.class.getName() + "#importScriptModule", "--importScriptModule"),

        LIST_ANALYSE_PLAN(_(""), ExportAction.class.getName() + "#listAnalysePlan", "--listAnalysePlan"),
        LIST_EXPORT(_(""), ExportAction.class.getName() + "#listExport", "--listExport"),
        LIST_RULE(_(""), ExportAction.class.getName() + "#listRule", "--listRule"),
        LIST_SCRIPT(_(""), ExportAction.class.getName() + "#listScript", "--listScript"),
        LIST_SIMULATOR(_(""), ExportAction.class.getName() + "#listSimulator", "--listSimulator"),
        LIST_FORMULA(_(""), ExportAction.class.getName() + "#listFormula", "--listFormula"),
        LIST_REGION(_(""), ExportAction.class.getName() + "#listRegion", "--listRegion"),
        LIST_SIMULATION(_(""), ExportAction.class.getName() + "#listSimulation", "--listSimulation"),

        EXPORT_ANALYSE_PLAN(_(""), ExportAction.class.getName() + "#exportAnalysePlan", "--exportAnalysePlan"),
        EXPORT_EXPORT(_(""), ExportAction.class.getName() + "#exportExport", "--exportExport"),
        EXPORT_RULE(_(""), ExportAction.class.getName() + "#exportRule", "--exportRule"),
        EXPORT_SCRIPT(_(""), ExportAction.class.getName() + "#exportScript", "--exportScript"),
        EXPORT_SIMULATOR(_(""), ExportAction.class.getName() + "#exportSimulator", "--exportSimulator"),
        EXPORT_FORMULA(_(""), ExportAction.class.getName() + "#exportFormula", "--exportFormula"),
        EXPORT_REGION(_(""), ExportAction.class.getName() + "#exportRegion", "--exportRegion"),
        EXPORT_SIMULATION(_(""), ExportAction.class.getName() + "#exportSimulation", "--exportSimulation"),
                
        VCS_SSH_CREATE_KEY(_(""), VCSAction.class.getName() + "#sshCreateKey", "--sshCreateKey"),
        VCS_UPDATE(_(""), VCSAction.class.getName() + "#vcsUpdate", "--vcsUpdate"),
        VCS_ADD(_(""), VCSAction.class.getName() + "#vcsAdd", "--vcsAdd"),
        VCS_REMOVE(_(""), VCSAction.class.getName() + "#vcsRemove", "--vcsRemove"),
        VCS_COMMIT(_(""), VCSAction.class.getName() + "#vcsCommit", "--vcsCommit"),

        SIMULATE_WITH_REGION(_(""), SimulationAction.class.getName() + "#simulateWithRegion", "--simulateWithRegion"),
        SIMULATE_WITH_SIMULATION(_(""), SimulationAction.class.getName() + "#simulateWithSimulation", "--simulateWithSimulation"),
        SIMULATE_WITH_SIMULATION_AND_SCRIPT(_(""), SimulationAction.class.getName() + "#simulateWithSimulationAndScript", "--simulateWithSimulationAndScript"),
        SIMULATE_REMOTELLY(_(""), SimulationAction.class.getName() + "#simulateRemotelly", "--simulateRemotelly"),
        SIMULATE_REMOTELLY_WITH_PRESCRIPT(_(""), SimulationAction.class.getName() + "#simulateRemotellyWithPreScript", "--simulateRemotellyWithPreScript");
        
        public String description;
        public String action;
        public String[] aliases;

        private Action(String description, String action, String ... aliases) {
            this.description = description;
            this.action = action;
            this.aliases = aliases;
        }   
    }
}
