/*
 * #%L
 * EchoBase :: UI
 * 
 * $Id: EchoBaseApplicationContext.java 589 2012-09-02 15:11:02Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/echobase/tags/echobase-2.0/echobase-ui/src/main/java/fr/ifremer/echobase/ui/EchoBaseApplicationContext.java $
 * %%
 * Copyright (C) 2011 Ifremer, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.ifremer.echobase.ui;

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import com.opensymphony.xwork2.ActionContext;
import fr.ifremer.echobase.EchoBaseConfiguration;
import fr.ifremer.echobase.persistence.EchoBaseDbMeta;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;

import javax.servlet.ServletContext;
import java.util.Map;
import java.util.Set;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class EchoBaseApplicationContext {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(EchoBaseApplicationContext.class);

    /** Key to store the single instance of the application context */
    private static final String APPLICATION_CONTEXT_PARAMETER =
            "echobaseApplicationContext";

    protected EchoBaseConfiguration configuration;

    protected EchoBaseDbMeta dbMeta;

    /** Root context for the internal database. */
    protected TopiaContext internalRootContext;

    /** Set of all loggued user sessions to be close at shutdown time. */
    protected Set<EchoBaseSession> sessions;

    /**
     * Flag setted to true when internal db was just created (should then
     * display in ui created user password).
     *
     * @since 1.1
     */
    protected boolean defaultUsersCreated;

    public static EchoBaseApplicationContext getApplicationContext(ActionContext actionContext) {
        Map<String, Object> application = actionContext.getApplication();
        EchoBaseApplicationContext result =
                (EchoBaseApplicationContext) application.get(
                        APPLICATION_CONTEXT_PARAMETER);
        return result;
    }

    public static EchoBaseApplicationContext getApplicationContext(ServletContext servletContext) {
        EchoBaseApplicationContext result =
                (EchoBaseApplicationContext) servletContext.getAttribute(
                        APPLICATION_CONTEXT_PARAMETER);
        return result;
    }

    public static void setApplicationContext(ServletContext servletContext,
                                             EchoBaseApplicationContext applicationContext) {
        servletContext.setAttribute(APPLICATION_CONTEXT_PARAMETER,
                                    applicationContext);
    }

    public static void removeApplicationContext(ServletContext servletContext) {
        servletContext.removeAttribute(APPLICATION_CONTEXT_PARAMETER);
    }

    public Set<EchoBaseSession> getEchoBaseSessions() {
        return sessions;
    }

    public synchronized void registerEchoBaseSession(EchoBaseSession session) {
        Preconditions.checkNotNull(session);
        Preconditions.checkNotNull(session.getUser());
        if (sessions == null) {
            sessions = Sets.newHashSet();
        }
        if (log.isInfoEnabled()) {
            log.info("Register user session for [" +
                     session.getUser().getEmail() + "]");
        }
        sessions.add(session);
    }

    public synchronized void destroyEchoBaseSession(EchoBaseSession session) {
        Preconditions.checkNotNull(session);
        Preconditions.checkNotNull(session.getUser());
        Preconditions.checkNotNull(sessions);
        if (log.isInfoEnabled()) {
            log.info("Destroy user session for [" +
                     session.getUser().getEmail() + "]");
        }
        // remove session from active ones
        sessions.remove(session);
        // close session
        session.close();
    }

    public EchoBaseConfiguration getConfiguration() {
        return configuration;
    }

    public void setConfiguration(EchoBaseConfiguration configuration) {
        this.configuration = configuration;
    }

    public TopiaContext getInternalRootContext() {
        return internalRootContext;
    }

    public void setInternalRootContext(TopiaContext internalRootContext) {
        this.internalRootContext = internalRootContext;
    }

    public EchoBaseDbMeta getDbMeta() {
        return dbMeta;
    }

    public void setDbMeta(EchoBaseDbMeta dbMeta) {
        this.dbMeta = dbMeta;
    }

    public boolean isDefaultUsersCreated() {
        return defaultUsersCreated;
    }

    public void setDefaultUsersCreated(boolean defaultUsersCreated) {
        this.defaultUsersCreated = defaultUsersCreated;
    }
}
