package fr.ifremer.coselmar.services.v1;

/*
 * #%L
 * Coselmar :: Rest Services
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2014 Ifremer, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import fr.ifremer.coselmar.beans.DocumentBean;
import fr.ifremer.coselmar.beans.QuestionBean;
import fr.ifremer.coselmar.beans.UserBean;
import fr.ifremer.coselmar.beans.UserWebToken;
import fr.ifremer.coselmar.converter.BeanEntityConverter;
import fr.ifremer.coselmar.exceptions.CoselmarTechnicalException;
import fr.ifremer.coselmar.persistence.entity.CoselmarUser;
import fr.ifremer.coselmar.persistence.entity.CoselmarUserGroup;
import fr.ifremer.coselmar.persistence.entity.CoselmarUserRole;
import fr.ifremer.coselmar.persistence.entity.Document;
import fr.ifremer.coselmar.persistence.entity.Privacy;
import fr.ifremer.coselmar.persistence.entity.Question;
import fr.ifremer.coselmar.services.CoselmarWebServiceSupport;
import fr.ifremer.coselmar.services.errors.InvalidCredentialException;
import fr.ifremer.coselmar.services.errors.NoResultException;
import fr.ifremer.coselmar.services.errors.UnauthorizedException;
import fr.ifremer.coselmar.services.indexation.DocumentsIndexationService;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.logging.Log;
import org.apache.lucene.queryparser.classic.ParseException;
import org.debux.webmotion.server.call.UploadFile;
import org.debux.webmotion.server.render.Render;
import org.nuiton.topia.persistence.TopiaNoResultException;
import org.nuiton.util.DateUtil;

import static org.apache.commons.logging.LogFactory.getLog;

/**
 * @author ymartel <martel@codelutin.com>
 */
public class DocumentsWebService extends CoselmarWebServiceSupport {

    private static final Log log = getLog(DocumentsWebService.class);

    public static final List<String> DOCUMENT_CREATE_ALLOWED_USER_ROLES =
        Lists.newArrayList(CoselmarUserRole.ADMIN.name(), CoselmarUserRole.SUPERVISOR.name(), CoselmarUserRole.EXPERT.name());

    /** Kind of admin role **/
    public static final List<String> DOCUMENT_SUPER_USER_ROLES =
        Lists.newArrayList(CoselmarUserRole.ADMIN.name(), CoselmarUserRole.SUPERVISOR.name());

    public static final List<String> DOCUMENT_VIEW_ALLOWED_USER_ROLES =
        Lists.newArrayList(
            CoselmarUserRole.ADMIN.name(),
            CoselmarUserRole.SUPERVISOR.name(),
            CoselmarUserRole.EXPERT.name()
        );

    public DocumentBean getDocument(String documentId) throws InvalidCredentialException, UnauthorizedException {

        // Check authentication
        String authorization = getContext().getHeader("Authorization");
        CoselmarUser currentUser = checkUserAuthentication(authorization);

        // reconstitute full id
        String fullId = getDocumentFullId(documentId);

        Document document = getDocumentDao().forTopiaIdEquals(fullId).findUnique();

        // If document if public, only admin, supervisor and expert could view it
        // If document if private, only admin, supervisor and owner could view it
        if (!isAllowedToAccessDocument(currentUser, document)) {

            String message = String.format("User %s %s ('%s') try to access to document '%s'",
                currentUser.getFirstname(), currentUser.getName(), getShortIdFromFull(currentUser.getTopiaId()), documentId);
            if (log.isWarnEnabled()) {
                log.warn(message);
            }
            throw new UnauthorizedException(message);

        }

        DocumentBean documentBean = BeanEntityConverter.toBean(getPersistenceContext().getTopiaIdFactory(), document);

        // Manage related Question
        List<Question> relatedQuestions = getQuestionDao().forRelatedDocumentsContains(document).findAll();
        for (Question relatedQuestion : relatedQuestions) {
            QuestionBean questionBean = BeanEntityConverter.toLightBean(getPersistenceContext().getTopiaIdFactory(), relatedQuestion);
            documentBean.addRelatedQuestion(questionBean);
        }

        // Manage restricted access
        if (document.getPrivacy() == Privacy.RESTRICTED) {
            Set<CoselmarUserGroup> restrictedList = document.getRestrictedList();
            for (CoselmarUserGroup usersGroup : restrictedList) {
                if (StringUtils.equalsIgnoreCase(fullId, usersGroup.getName())) {
                    Set<CoselmarUser> members = usersGroup.getMembers();
                    for (CoselmarUser member : members) {
                        String userLightId = getPersistenceContext().getTopiaIdFactory().getRandomPart(member.getTopiaId());
                        UserBean userBean = BeanEntityConverter.toBean(userLightId, member);
                        documentBean.addAuthorizedUser(userBean);
                    }
                }
            }
        }


        return documentBean;
    }

    public List<DocumentBean> getDocuments(List<String> searchKeywords) throws InvalidCredentialException {

        // Check authentication
        String authorization = getContext().getHeader("Authorization");
        UserWebToken userWebToken = checkAuthentication(authorization);

        // Retrieve current user
        String fullCurrentUserId = getFullUserIdFromShort(userWebToken.getUserId());
        CoselmarUser currentUser = getCoselmarUserDao().forTopiaIdEquals(fullCurrentUserId).findAnyOrNull();

        String currentUserRole = userWebToken.getRole().toUpperCase();

        List<Document> documentList;

        // Admin and Supervisor can see all documents (public, private and restricted)
        if (Lists.newArrayList(CoselmarUserRole.ADMIN.name(), CoselmarUserRole.SUPERVISOR.name()).contains(currentUserRole)) {
            if (searchKeywords != null && !searchKeywords.isEmpty()) {
                DocumentsIndexationService documentsIndexationService = getServicesContext().newService(DocumentsIndexationService.class);

                try {
                    List<String> documentIds = documentsIndexationService.searchDocuments(searchKeywords);
                    List<String> documentFullIds = getDocumentsFullId(documentIds);

                    documentList = getDocumentDao().forTopiaIdIn(documentFullIds).findAll();

                } catch (IOException | ParseException e) {
                    if (log.isErrorEnabled()) {
                        log.error("Unable to search by lucene, make search directly in database", e);
                    }
                    documentList = getDocumentDao().findAllContainingAllKeywords(searchKeywords);

                }
            } else {
                documentList = getDocumentDao().findAll();
            }

        } else {
            //Other can only see public, his own private and restricted for which he is allowed
            documentList = getDocumentDao().findAllFilterByUser(currentUser, searchKeywords);
        }

        List<DocumentBean> result = new ArrayList<>(documentList.size());

        for (Document document : documentList) {
            DocumentBean documentBean = BeanEntityConverter.toBean(getPersistenceContext().getTopiaIdFactory(), document);

            // Manage related Question
            long relatedQuestions = getQuestionDao().forRelatedDocumentsContains(document).count();
            documentBean.setNbRelatedQuestions((int) relatedQuestions);

            result.add(documentBean);
        }

        return result;
    }

    public DocumentBean addDocument(DocumentBean document, UploadFile uploadFile) throws InvalidCredentialException, UnauthorizedException {

        // Check authentication
        String authorization = getContext().getHeader("Authorization");
        UserWebToken userWebToken = checkAuthentication(authorization);

        // Only Expert or Supervisor can add document
        String userRole = userWebToken.getRole();
        if (!DOCUMENT_CREATE_ALLOWED_USER_ROLES.contains(userRole.toUpperCase())) {
            String message = String.format("User %s %s ('%s') is not allowed to add document",
                userWebToken.getFirstName(), userWebToken.getLastName(), userWebToken.getUserId());
            if (log.isWarnEnabled()) {
                log.warn(message);
            }
            throw new UnauthorizedException(message);
        }

        Preconditions.checkNotNull(document);

        // retrieve user who will be assigned as document owner
        String fullId = getFullUserIdFromShort(userWebToken.getUserId());

        CoselmarUser owner;
        try {
            owner = getCoselmarUserDao().forTopiaIdEquals(fullId).findUnique();
        } catch (TopiaNoResultException tnre) {
            // Should not happened, cause user are not really deleted
            String message = String.format("Seems that logged user ('%s') does not exist anymore.", fullId);
            if (log.isErrorEnabled()) {
                log.error(message);
            }
            throw new InvalidCredentialException(message);
        }

        String documentName = document.getName();
        String contentType = null;
        String filePath = null;

        // If document has a file, manager it !
        if (uploadFile != null) {
            Pair<String, String> pathAndContentType = managerDocumentFile(uploadFile, owner);
            filePath = pathAndContentType.getLeft();
            contentType = pathAndContentType.getRight();
        }

        // Document Metadata
        Document documentEntity = getDocumentDao().create();

        documentEntity.setOwner(owner);

        documentEntity.setName(documentName);

        String privacy = document.getPrivacy().toUpperCase();
        documentEntity.setPrivacy(Privacy.valueOf(privacy));

        // Manage privacy : if restricted, create an UserGroup with authorized users
        if (StringUtils.equals(Privacy.RESTRICTED.name(), privacy)) {
            Set<UserBean> authorizedUsers = document.getAuthorizedUsers();
            Set<CoselmarUser> coselmarUsers = retrieveUsers(authorizedUsers);
            CoselmarUserGroup restrictedUsers = getCoselmarUserGroupDao().create();
            restrictedUsers.setName(documentEntity.getTopiaId());

            restrictedUsers.addAllMembers(coselmarUsers);
            documentEntity.addRestrictedList(restrictedUsers);

        }

        documentEntity.addAllKeywords(document.getKeywords());

        Date depositDate = document.getDepositDate();
        if (depositDate != null) {
            documentEntity.setDepositDate(new Date(depositDate.getTime()));
        } else {
            documentEntity.setDepositDate(new Date());
        }

        documentEntity.setType(document.getType());
        documentEntity.setSummary(document.getSummary());
        documentEntity.setLanguage(document.getLanguage());
        documentEntity.setPublicationDate(document.getPublicationDate());


        // Legal / copyright part
        documentEntity.setAuthors(document.getAuthors());
        documentEntity.setCopyright(document.getCopyright());
        documentEntity.setLicense(document.getLicense());

        // Document resource part
        if (uploadFile != null) {
            documentEntity.setWithFile(true);
            documentEntity.setMimeType(contentType);
            documentEntity.setFilePath(filePath);
        } else {
            documentEntity.setWithFile(false);
        }
        documentEntity.setExternalUrl(document.getExternalUrl());

        documentEntity.setComment(document.getComment());

        commit();
        DocumentBean result = BeanEntityConverter.toBean(getPersistenceContext().getTopiaIdFactory(), documentEntity);

        DocumentsIndexationService documentsIndexationService = getServicesContext().newService(DocumentsIndexationService.class);
        try {
            documentsIndexationService.indexDocument(result);
            if (log.isDebugEnabled()) {
                String message = String.format("Document '%s' added to index", documentName);
                log.debug(message);
            }
        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("Unable to index new document", e);
            }
        }

        return result;

    }

    public void addDocumentFile(String documentId, UploadFile uploadFile) throws InvalidCredentialException, UnauthorizedException {


        Preconditions.checkNotNull(documentId);
        Preconditions.checkNotNull(uploadFile);

        // Check authentication
        String authorization = getContext().getHeader("Authorization");
        CoselmarUser currentUser = checkUserAuthentication(authorization);

        String documentFullId = getFullIdFromShort(Document.class, documentId);
        Document document = getDocumentDao().forTopiaIdEquals(documentFullId).findAny();

        // Only Owner Expert or Supervisor/Admin can add document file
        if (!DOCUMENT_SUPER_USER_ROLES.contains(currentUser.getRole().name())
            && document.getOwner() != currentUser) {
            String message = String.format("User %s %s ('%s') is not allowed to add document file",
                currentUser.getFirstname(), currentUser.getName(), getShortIdFromFull(currentUser.getTopiaId()));
            if (log.isWarnEnabled()) {
                log.warn(message);
            }
            throw new UnauthorizedException(message);
        }

        // Get owner to place correctly the file
        CoselmarUser owner = document.getOwner();
        Pair<String, String> pathAndContentType = managerDocumentFile(uploadFile, owner);
        String filePath = pathAndContentType.getLeft();
        String contentType = pathAndContentType.getRight();

        // If document has already a file, remove it
        if (StringUtils.isNotBlank(document.getFilePath())) {
            File documentFile = new File(document.getFilePath());
            FileUtils.deleteQuietly(documentFile);
        }

        document.setWithFile(true);
        document.setMimeType(contentType);
        document.setFilePath(filePath);
        document.setFileName(uploadFile.getName());

        commit();

    }


    public Render getDocumentFile(String documentId) throws NoResultException {

        // reconstitute full id
        String fullId =getDocumentFullId(documentId);

        Document document = getDocumentDao().forTopiaIdEquals(fullId).findUnique();
        //TODO ymartel 20141103 : manage file owner ?

        // Get file attached to document
        String filePath = document.getFilePath();
        if (StringUtils.isBlank(filePath)) {
            throw new NoResultException("No File");
        }
        File documentFile = new File(filePath);

        String fileName = StringUtils.isNotBlank(document.getFileName()) ? document.getFileName() : document.getName();
        String fileMimeType = document.getMimeType();
        try {
            InputStream fileStream = new FileInputStream(documentFile);
            return renderDownload(fileStream, fileName, fileMimeType);

        } catch (FileNotFoundException e) {
            if (log.isErrorEnabled()) {
                String message = String.format("Unable to retrieve file %s", fileName);
                log.error(message);
            }
            throw new NoResultException("File does not exist");
        }

    }

    public void saveDocument(DocumentBean document) throws InvalidCredentialException, UnauthorizedException {
        Preconditions.checkNotNull(document);
        Preconditions.checkNotNull(document.getId());

        // Check authentication
        String authorization = getContext().getHeader("Authorization");
        CoselmarUser currentUser = checkUserAuthentication(authorization);

        String documentId = getDocumentFullId(document.getId());

        Document documentEntity = getDocumentDao().forTopiaIdEquals(documentId).findUnique();

        if (!isAllowedToAccessDocument(currentUser, documentEntity)) {

            String message = String.format("User %s %s ('%s') try to modify document '%s'",
                currentUser.getFirstname(), currentUser.getName(), getShortIdFromFull(currentUser.getTopiaId()), documentId);
            if (log.isWarnEnabled()) {
                log.warn(message);
            }
            throw new UnauthorizedException(message);

        }

        boolean isUsedByQuestions = getQuestionDao().forRelatedDocumentsContains(documentEntity).exists();
        if (isUsedByQuestions) {
            String message = "Document is used by some questions, cannot be modified.";
            if (log.isWarnEnabled()) {
                log.warn(message);
            }
            throw new UnauthorizedException(message);
        }

        documentEntity.setName(document.getName());
        String oldPrivacy = documentEntity.getPrivacy().name();
        String newPrivacy = document.getPrivacy().toUpperCase();
        documentEntity.setPrivacy(Privacy.valueOf(newPrivacy));

        // Manage privacy : if restricted, create an UserGroup with authorized users
        if (StringUtils.equals(Privacy.RESTRICTED.name(), newPrivacy)) {
            Set<UserBean> authorizedUsers = document.getAuthorizedUsers();
            Set<CoselmarUser> coselmarUsers = retrieveUsers(authorizedUsers);

            Set<CoselmarUserGroup> restrictedList = documentEntity.getRestrictedList();
            CoselmarUserGroup restrictedUsers = null;
            if (restrictedList != null && !restrictedList.isEmpty()) {

                // Try to retrieve the UserGroup with documentId as name (corresponding to user choice)
                for (CoselmarUserGroup usersList : restrictedList) {
                    if (StringUtils.equalsIgnoreCase(documentId, usersList.getName())) {
                        restrictedUsers = usersList;
                        restrictedUsers.clearMembers();
                        break;
                    }
                }
            }
            if (restrictedUsers == null) {
                restrictedUsers = getCoselmarUserGroupDao().create();
                restrictedUsers.setName(documentEntity.getTopiaId());
            }

            restrictedUsers.addAllMembers(coselmarUsers);
            documentEntity.addRestrictedList(restrictedUsers);

        } else if (StringUtils.equals(Privacy.RESTRICTED.name(), oldPrivacy)) {
            // retrieve restrictive users list and remove it
            CoselmarUserGroup restrictedUsers = null;
            for (CoselmarUserGroup usersList : documentEntity.getRestrictedList()) {
                if (StringUtils.equalsIgnoreCase(documentId, usersList.getName())) {
                    restrictedUsers = usersList;
                    break;
                }
            }

            // Remove group from document, and delete it !
            if (restrictedUsers != null) {
                documentEntity.removeRestrictedList(restrictedUsers);
                getCoselmarUserGroupDao().delete(restrictedUsers);
            }
        }

        documentEntity.clearKeywords();
        documentEntity.addAllKeywords(document.getKeywords());

        documentEntity.setType(document.getType());
        documentEntity.setSummary(document.getSummary());
        documentEntity.setLanguage(document.getLanguage());
        documentEntity.setPublicationDate(document.getPublicationDate());

        // Legal / copyright part
        documentEntity.setAuthors(document.getAuthors());
        documentEntity.setCopyright(document.getCopyright());
        documentEntity.setLicense(document.getLicense());

        // Resource part
        documentEntity.setExternalUrl(document.getExternalUrl());
        // If had file info and now no file info : remove file
        boolean hadFile = documentEntity.isWithFile();
        if (hadFile && StringUtils.isBlank(document.getFileName())) {
            deleteAttachedFile(documentEntity);
        }


        documentEntity.setComment(document.getComment());

        commit();

        // Update index information for this document
        DocumentBean result = BeanEntityConverter.toBean(getPersistenceContext().getTopiaIdFactory(), documentEntity);

        DocumentsIndexationService documentsIndexationService = getServicesContext().newService(DocumentsIndexationService.class);
        try {
            documentsIndexationService.indexDocument(result);
            if (log.isDebugEnabled()) {
                String message = String.format("Document '%s' was updated in index", document.getName());
                log.debug(message);
            }
        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("Unable to update document index information", e);
            }
        }

    }

    public void deleteDocument(String documentId) throws InvalidCredentialException, UnauthorizedException {

        // Check authentication
        String authorization = getContext().getHeader("Authorization");
        CoselmarUser currentUser = checkUserAuthentication(authorization);

        // reconstitute full id
        String fullId = getDocumentFullId(documentId);

        Document document = getDocumentDao().forTopiaIdEquals(fullId).findUnique();

        if (!DOCUMENT_SUPER_USER_ROLES.contains(currentUser.getRole().name())
            && document.getOwner() != currentUser) {

            String message = String.format("User %s %s ('%s') try to delete document '%s'",
                currentUser.getFirstname(), currentUser.getName(), getShortIdFromFull(currentUser.getTopiaId()), documentId);
            if (log.isWarnEnabled()) {
                log.warn(message);
            }
            throw new UnauthorizedException(message);

        }

        // Delete physical file
        deleteAttachedFile(document);

        getDocumentDao().delete(document);

        commit();

        //Remove index entry
        DocumentsIndexationService documentsIndexationService = getServicesContext().newService(DocumentsIndexationService.class);
        try {
            documentsIndexationService.deleteDocument(documentId);
            if (log.isDebugEnabled()) {
                String message = String.format("Document '%s' removed from index", documentId);
                log.debug(message);
            }
        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("Unable to remove document entry from index", e);
            }
        }
    }

    public List<String> getKeywords() throws InvalidCredentialException, UnauthorizedException {

        List<String> themes = getDocumentDao().findAllKeywords();

        return themes;
    }

    public List<String> getTypes() throws InvalidCredentialException, UnauthorizedException {

        List<String> types = getDocumentDao().findAllTypes();

        return types;
    }


    ////////////////////////////////////////////////////////////////////////////
    ///////////////////////     Internal Parts     /////////////////////////////
    ////////////////////////////////////////////////////////////////////////////

    /**
     * When a Document is sent, it could have a File part : this manage the
     * upload file. The file is stored in the user specific directory, and the
     * contentType of document is returned, cause need in Document Metadata.
     *
     * @return the upload file Metadata
     */
    protected Pair<String, String> managerDocumentFile(UploadFile uploadFile, CoselmarUser owner) {
        Preconditions.checkNotNull(uploadFile);

        // Document File
        String fileName = uploadFile.getName();
        File uploadedFile = uploadFile.getFile();
        String contentType = uploadFile.getContentType();

        if (log.isInfoEnabled()) {
            String message = String.format("File name : %s, content-type : %s", fileName, contentType);
            log.info(message);
        }

        // put the document in the good directory
        String userPath = getUserDocumentPath(owner);

        Date now = getNow();
        String formattedDay = DateUtil.formatDate(now, "yyyyMMddHHmm");
        String prefix = formattedDay + "-";

        File destFile = new File(userPath + File.separator + prefix + fileName);
        try {
            FileUtils.moveFile(uploadedFile, destFile);
        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("error during File transfer", e);
            }
            throw new CoselmarTechnicalException("Internal error during file transfer");
        }
        return Pair.of(destFile.getAbsolutePath(), contentType);
    }

    protected String getUserDocumentPath(CoselmarUser user) {
        File dataDirectory = getCoselmarServicesConfig().getDataDirectory();
        String absolutePath = dataDirectory.getAbsolutePath();
        String userFolder = StringUtils.replaceChars(user.getFirstname() + "-" + user.getName(), " ", "_");
        String userPath = absolutePath + File.separator + userFolder;
        return userPath;
    }

    /**
     * Check if an user can access to a document metadata, according to its
     * privacy settings and the user grant.
     *
     * @param user      :   Current User,
     *                      containing {@link fr.ifremer.coselmar.persistence.entity.CoselmarUserRole} as String
     * @param document  :   the document user trying to access.
     *
     * @return true is user can access, false else.
     */
    protected boolean isAllowedToAccessDocument(CoselmarUser user, Document document) {
        boolean isAuthorized = false;

        String viewerRole = user.getRole().name().toUpperCase();

        if (DOCUMENT_SUPER_USER_ROLES.contains(viewerRole) || document.getOwner() == user) {
            return true;
        }

        // For public : only admin/supervisor/expert can access
        if (document.getPrivacy() == Privacy.PUBLIC) {
            isAuthorized = DOCUMENT_VIEW_ALLOWED_USER_ROLES.contains(viewerRole);

        // For Private : only admin/supervisor/owner can access
        } else if (document.getPrivacy() == Privacy.PRIVATE) {
            CoselmarUser documentOwner = document.getOwner();
            boolean isOwner = StringUtils.equals(documentOwner.getTopiaId(), user.getTopiaId());
            isAuthorized = isOwner || Lists.newArrayList(CoselmarUserRole.ADMIN.name(), CoselmarUserRole.SUPERVISOR.name()).contains(viewerRole);

        } else if (document.getPrivacy() == Privacy.RESTRICTED) {
            Set<CoselmarUserGroup> restrictedLists = document.getRestrictedList();
            for (CoselmarUserGroup restrictedList : restrictedLists) {
                if (restrictedList.containsMembers(user)) {
                    isAuthorized = true;
                    break;
                }
            }
        }

        return isAuthorized;
    }

    protected String getDocumentFullId(String documentId) {
        return Document.class.getCanonicalName() + getPersistenceContext().getTopiaIdFactory().getSeparator() + documentId;
    }

    protected List<String> getDocumentsFullId(List<String> documentShortIds) {

        Function<String, String> getFullIds = new Function<String, String>() {
            @Override
            public String apply(String shortId) {
                return getDocumentFullId(shortId);
            }
        };

        List<String> fullIds = Lists.transform(documentShortIds, getFullIds);
        return fullIds;
    }

    /**
     * Delete physically Document File, and update Document entity setting
     * filePath and fileName to null and withFile boolean to false
     */
    protected void deleteAttachedFile(Document document) {
        if (StringUtils.isNotBlank(document.getFilePath())) {
            File documentFile = new File(document.getFilePath());
            FileUtils.deleteQuietly(documentFile);

            document.setFilePath(null);
            document.setFileName(null);
            document.setWithFile(false);
        }
    }

    protected Set<CoselmarUser> retrieveUsers(Collection<UserBean> userBeans) {
        Function<UserBean, String> getIds = new Function<UserBean, String>() {
            @Override
            public String apply(UserBean userBean) {
                return getFullIdFromShort(CoselmarUser.class, userBean.getId());
            }
        };

        Collection<String> userIds = Collections2.transform(userBeans, getIds);
        List<CoselmarUser> coselmarUsers = getCoselmarUserDao().forTopiaIdIn(userIds).findAll();
        return new HashSet<>(coselmarUsers);

    }
}
