package fr.ifremer.coselmar.services.v1;

/*
 * #%L
 * Coselmar :: Rest Services
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2014 Ifremer, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import fr.ifremer.coselmar.beans.DocumentBean;
import fr.ifremer.coselmar.beans.QuestionBean;
import fr.ifremer.coselmar.beans.QuestionSearchBean;
import fr.ifremer.coselmar.beans.UserBean;
import fr.ifremer.coselmar.beans.UserWebToken;
import fr.ifremer.coselmar.converter.BeanEntityConverter;
import fr.ifremer.coselmar.persistence.entity.CoselmarUser;
import fr.ifremer.coselmar.persistence.entity.CoselmarUserGroup;
import fr.ifremer.coselmar.persistence.entity.CoselmarUserRole;
import fr.ifremer.coselmar.persistence.entity.Document;
import fr.ifremer.coselmar.persistence.entity.Privacy;
import fr.ifremer.coselmar.persistence.entity.Question;
import fr.ifremer.coselmar.persistence.entity.Status;
import fr.ifremer.coselmar.services.CoselmarWebServiceSupport;
import fr.ifremer.coselmar.services.errors.InvalidCredentialException;
import fr.ifremer.coselmar.services.errors.UnauthorizedException;
import fr.ifremer.coselmar.services.indexation.QuestionsIndexationService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.queryparser.classic.ParseException;
import org.nuiton.topia.persistence.TopiaIdFactory;
import org.nuiton.topia.persistence.TopiaNoResultException;

/**
 * @author ymartel <martel@codelutin.com>
 */
public class QuestionsWebService extends CoselmarWebServiceSupport {

    private static final Log log = LogFactory.getLog(QuestionsWebService.class);

    protected static final List<String> RESTRICTED_ACCESS_USERS = Lists.newArrayList(CoselmarUserRole.CLIENT.name(), CoselmarUserRole.MEMBER.name());

    public void addQuestion(QuestionBean question) throws InvalidCredentialException, UnauthorizedException {

        // Check authentication
        String authorization = getContext().getHeader("Authorization");
        UserWebToken userWebToken = checkAuthentication(authorization);

        // Only Supervisor can add question
        String userRole = userWebToken.getRole();

        if (!StringUtils.equalsIgnoreCase(CoselmarUserRole.SUPERVISOR.name(), userRole)) {
            String message = String.format("User %s %s ('%s') is not allowed to add question",
                userWebToken.getFirstName(), userWebToken.getLastName(), userWebToken.getUserId());
            if (log.isWarnEnabled()) {
                log.warn(message);
            }
            throw new UnauthorizedException(message);

        }

        // retrieve user who will be assigned as question supervisor
        String fullId = getFullUserIdFromShort(userWebToken.getUserId());

        CoselmarUser supervisor;
        try {
            supervisor = getCoselmarUserDao().forTopiaIdEquals(fullId).findUnique();
        } catch (TopiaNoResultException tnre) {
            // Should not happened, cause user are not really deleted
            String message = String.format("Logged user ('%s') does not exist.", fullId);
            if (log.isErrorEnabled()) {
                log.error(message);
            }
            throw new InvalidCredentialException(message);
        }

        Preconditions.checkNotNull(question);
        Preconditions.checkNotNull(question.getTitle());
        Preconditions.checkNotNull(question.getSummary());
        Preconditions.checkNotNull(question.getType());
        Preconditions.checkNotNull(question.getThemes());

        // let's go
        Question questionEntity = getQuestionDao().create();

        questionEntity.setUnavailable(false);

        // Question basics

        questionEntity.setTitle(question.getTitle());

        questionEntity.setSummary(question.getSummary());

        questionEntity.setType(question.getType());

        Set<String> themes = question.getThemes();
        if (themes != null) {
            questionEntity.setTheme(new HashSet(themes));
        }

        // By default, privacy is private
        String privacy = question.getPrivacy();
        Privacy realPrivacy = privacy  != null ? Privacy.valueOf(privacy.toUpperCase()) : Privacy.PRIVATE;
        questionEntity.setPrivacy(realPrivacy);

        // On creation, Status is Open
        questionEntity.setStatus(Status.OPEN);

        // Manage Dates : submission & deadline
        Date submissionDate = question.getSubmissionDate();
        if (submissionDate != null) {
            questionEntity.setSubmissionDate(new Date(submissionDate.getTime()));
        } else {
            questionEntity.setSubmissionDate(new Date());
        }

        Date deadline = question.getDeadline();
        if (deadline != null) {
            questionEntity.setDeadline(new Date(deadline.getTime()));
        }


        // Users around the question

        // First Supervisor is the one creating this question
        questionEntity.addSupervisors(supervisor);

        questionEntity.addAllExternalExperts(question.getExternalExperts());

        // Retrieve the clients
        Set<UserBean> clients = question.getClients();
        if (clients != null && !clients.isEmpty()) {
            Set<CoselmarUser> clientEntities = retrieveUsers(clients);
            questionEntity.addAllClients(clientEntities);
        }

        // For participants, should create a dedicated group,
        // that could be used to restrict documents access
        Set<UserBean> participants = question.getParticipants();
        CoselmarUserGroup participantGroup = getCoselmarUserGroupDao().create();
        if (participants != null && !participants.isEmpty()) {
            participantGroup.setName(question.getTitle());

            Set<CoselmarUser> expertEntities = retrieveUsers(participants);
            participantGroup.addAllMembers(expertEntities);

        }
        questionEntity.setParticipants(participantGroup);

        // Retrieve the supervisor
        Set<UserBean> supervisors = question.getSupervisors();
        if (supervisors != null && !supervisors.isEmpty()) {
            Set<CoselmarUser> supervisorEntities = retrieveUsers(supervisors);
            questionEntity.addAllSupervisors(supervisorEntities);
        }

        // Note : no contributors now, contributors are old participants,
        // excluded from process


        // Hierarchy of questions
        Set<QuestionBean> parents = question.getParents();
        if (parents != null && !parents.isEmpty()) {
            Set<Question> questions = retrieveQuestions(parents);
            questionEntity.addAllParents(questions);
        }


        // Documents on init
        Set<DocumentBean> relatedDocuments = question.getRelatedDocuments();
        if (relatedDocuments != null && !relatedDocuments.isEmpty()) {
            Set<Document> documents = retrieveDocuments(relatedDocuments);
            // Manage restriction list for document with Privacy.RESTRICTED
            for (Document document : documents) {
                if (document.getPrivacy() == Privacy.RESTRICTED) {
                    document.addRestrictedList(participantGroup);
                }
            }

            questionEntity.addAllRelatedDocuments(documents);
        }

        commit();

        QuestionBean result = BeanEntityConverter.toBean(getPersistenceContext().getTopiaIdFactory(), questionEntity);

        QuestionsIndexationService questionsIndexationService = getServicesContext().newService(QuestionsIndexationService.class);
        try {
            questionsIndexationService.indexQuestion(result);
            if (log.isDebugEnabled()) {
                String message = String.format("Question '%s' added to index", result.getTitle());
                log.debug(message);
            }
        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("Unable to index new question", e);
            }
        }
    }

    public List<QuestionBean> getQuestions(QuestionSearchBean searchOption) throws InvalidCredentialException, UnauthorizedException {

        // Check authentication
        String authorization = getContext().getHeader("Authorization");
        UserWebToken userWebToken = checkAuthentication(authorization);

        // Retrieve current user
        String fullCurrentUserId = getFullUserIdFromShort(userWebToken.getUserId());
        CoselmarUser currentUser = getCoselmarUserDao().forTopiaIdEquals(fullCurrentUserId).findAnyOrNull();

        String currentUserRole = userWebToken.getRole().toUpperCase();

        List<Question> questionList;

        if (searchOption != null) {
            questionList = getAllFilteredQuestions(currentUser, searchOption);

        } else {
            questionList = getAllQuestions(currentUser);
        }

        List<QuestionBean> result = new ArrayList<>(questionList.size());

        for (Question question : questionList) {
            TopiaIdFactory topiaIdFactory = getPersistenceContext().getTopiaIdFactory();

            QuestionBean questionBean;
            if (RESTRICTED_ACCESS_USERS.contains(currentUserRole)) {
                questionBean = BeanEntityConverter.toLightBean(topiaIdFactory, question);

            } else {
                questionBean = BeanEntityConverter.toBean(topiaIdFactory, question);
            }

            result.add(questionBean);
        }

        return result;
    }

    public void deleteQuestion(String questionId) throws InvalidCredentialException, UnauthorizedException {

        // Check authentication
        String authorization = getContext().getHeader("Authorization");
        UserWebToken userWebToken = checkAuthentication(authorization);

        // Only Supervisor can delete question
        String userRole = userWebToken.getRole();

        if (!StringUtils.equalsIgnoreCase(CoselmarUserRole.SUPERVISOR.name(), userRole)
            && StringUtils.equalsIgnoreCase(CoselmarUserRole.ADMIN.name(), userRole)) {
            String message = String.format("User %s %s ('%s') is not allowed to delete question",
                userWebToken.getFirstName(), userWebToken.getLastName(), userWebToken.getUserId());
            if (log.isWarnEnabled()) {
                log.warn(message);
            }
            throw new UnauthorizedException(message);

        }

        String fullUserId = getFullIdFromShort(CoselmarUser.class, userWebToken.getUserId());

        try {
            getCoselmarUserDao().forTopiaIdEquals(fullUserId).findUnique();
        } catch (TopiaNoResultException tnre) {
            // Should not happened, cause user are not really deleted
            String message = String.format("Logged user ('%s') does not exist.", fullUserId);
            if (log.isErrorEnabled()) {
                log.error(message);
            }
            throw new InvalidCredentialException(message);
        }

        // Retrieve Question
        String fullQuestionId = getFullIdFromShort(Question.class, questionId);
        Question question = getQuestionDao().forTopiaIdEquals(fullQuestionId).findUnique();

        // Participant group should be deleted, and so, we should remove it from Document using this group
        CoselmarUserGroup participantGroup = question.getParticipants();
        List<Document> documents = getDocumentDao().forRestrictedListContains(participantGroup).findAll();
        for (Document document : documents) {
            document.removeRestrictedList(participantGroup);
        }

        // Question become unavailable
        question.addAllContributors(participantGroup.getMembers());
        question.setUnavailable(true);

        getPersistenceContext().getCoselmarUserGroupDao().delete(participantGroup);

        commit();

        // Remove it from index too
        QuestionsIndexationService questionsIndexationService = getServicesContext().newService(QuestionsIndexationService.class);
        try {
            questionsIndexationService.deleteQuestion(questionId);
            if (log.isDebugEnabled()) {
                String message = String.format("Question '%s' deleted from index", questionId);
                log.debug(message);
            }
        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("Unable to remove question from index", e);
            }
        }
    }

    public QuestionBean getQuestion(String questionId) throws InvalidCredentialException, UnauthorizedException {

        // Check authentication
        String authorization = getContext().getHeader("Authorization");
        UserWebToken userWebToken = checkAuthentication(authorization);

        // Supervisor can get all the question elements
        // Expert can get all the question elements if public or if he is participant of the question
        // Client can get the question (not the documents) if he is client of the question.
        String userRole = userWebToken.getRole();

        // Member cannot access to question
        if (StringUtils.equalsIgnoreCase(CoselmarUserRole.MEMBER.name(), userRole)) {
            String message = String.format("User %s %s ('%s') is not allowed to view question",
                userWebToken.getFirstName(), userWebToken.getLastName(), userWebToken.getUserId());
            if (log.isWarnEnabled()) {
                log.warn(message);
            }
            throw new UnauthorizedException(message);

        }

        String fullUserId = getFullIdFromShort(CoselmarUser.class, userWebToken.getUserId());

        CoselmarUser currentUser;
        try {
            currentUser = getCoselmarUserDao().forTopiaIdEquals(fullUserId).findUnique();
        } catch (TopiaNoResultException tnre) {
            // Should not happened, cause user are not really deleted
            String message = String.format("Logged user ('%s') does not exist.", fullUserId);
            if (log.isErrorEnabled()) {
                log.error(message);
            }
            throw new InvalidCredentialException(message);
        }

        // Retrieve Question
        String fullQuestionId = getFullIdFromShort(Question.class, questionId);
        Question question = getQuestionDao().forTopiaIdEquals(fullQuestionId).findUnique();

        if (CoselmarUserRole.CLIENT == currentUser.getRole()) {
            // Client User can access if it is client of question
            checkIsClientAllowed(question, currentUser);

        }

        QuestionBean result = BeanEntityConverter.toBean(getPersistenceContext().getTopiaIdFactory(), question);
        //manager child
        List<Question> children = getQuestionDao().forParentsContains(question).findAll();
        for (Question child : children) {
            QuestionBean childBean = BeanEntityConverter.toLightBean(getPersistenceContext().getTopiaIdFactory(), child);
            result.addChild(childBean);
        }

        // Client is not allowed to see documents
        if (CoselmarUserRole.CLIENT == currentUser.getRole()
            || (CoselmarUserRole.SUPERVISOR != currentUser.getRole() && question.getClients() != null && question.getClients().contains(currentUser))) {

            // clients does not have to see all documents
            result.setRelatedDocuments(null);

        // If document is private, only participants could check it
        } else if (CoselmarUserRole.EXPERT == currentUser.getRole() && question.getPrivacy() == Privacy.PRIVATE) {

            CoselmarUserGroup participants = question.getParticipants();

            if (participants == null || !participants.getMembers().contains(currentUser)) {
                // Non participant only see title, privacy and hierarchy
                result = new QuestionBean();
                result.setTitle(question.getTitle());
                result.setPrivacy(question.getPrivacy().name());
                result.setRestricted(true);
                for (Question parent : question.getParents()) {
                    result.addParent(BeanEntityConverter.toLightBean(getPersistenceContext().getTopiaIdFactory(), parent));
                }
                for (Question child : children) {
                    QuestionBean childBean = BeanEntityConverter.toLightBean(getPersistenceContext().getTopiaIdFactory(), child);
                    result.addChild(childBean);
                }
            }
        }

        return result;
    }

    public void addDocuments(String questionId, DocumentBean[] documents) throws InvalidCredentialException, UnauthorizedException {

        // Check authentication
        String authorization = getContext().getHeader("Authorization");
        UserWebToken userWebToken = checkAuthentication(authorization);

        // Only Supervisor can add documents
        String userRole = userWebToken.getRole();

        if (!StringUtils.equalsIgnoreCase(CoselmarUserRole.SUPERVISOR.name(), userRole)
            && !StringUtils.equalsIgnoreCase(CoselmarUserRole.ADMIN.name(), userRole)
            && !StringUtils.equalsIgnoreCase(CoselmarUserRole.EXPERT.name(), userRole)) {

            String message = String.format("User %s %s ('%s') is not allowed to add document",
                userWebToken.getFirstName(), userWebToken.getLastName(), userWebToken.getUserId());
            if (log.isWarnEnabled()) {
                log.warn(message);
            }
            throw new UnauthorizedException(message);

        }

        String fullUserId = getFullIdFromShort(CoselmarUser.class, userWebToken.getUserId());

        CoselmarUser currentUser;
        try {
            currentUser = getCoselmarUserDao().forTopiaIdEquals(fullUserId).findUnique();
        } catch (TopiaNoResultException tnre) {
            // Should not happened, cause user are not really deleted
            String message = String.format("Logged user ('%s') does not exist.", fullUserId);
            if (log.isErrorEnabled()) {
                log.error(message);
            }
            throw new InvalidCredentialException(message);
        }

        // Retrieve Question
        String fullQuestionId = getFullIdFromShort(Question.class, questionId);
        Question question = getQuestionDao().forTopiaIdEquals(fullQuestionId).findUnique();

        // Check expert authorization on the document
        checkIsParticipant(question, currentUser);


        // Retrieve all documents
        Collection<Document> questionDocuments = question.getRelatedDocuments();
        if (documents != null && documents.length > 0) {
            // now with expert documents, question is in progress
            if (question.getStatus() == Status.OPEN) {
                question.setStatus(Status.IN_PROGRESS);
            }

            Set<Document> documentEntities = retrieveDocuments(Lists.newArrayList(documents));
            // Manage restriction list for document with Privacy.RESTRICTED
            for (Document document : documentEntities) {
                if (document.getPrivacy() == Privacy.RESTRICTED) {
                    document.addRestrictedList(question.getParticipants());
                }
                if (!questionDocuments.contains(document)) {
                    question.addRelatedDocuments(document);
                }
            }
        }

        commit();
    }

    public void saveQuestion(QuestionBean question) throws InvalidCredentialException, UnauthorizedException {

        // Check authentication
        String authorization = getContext().getHeader("Authorization");
        UserWebToken userWebToken = checkAuthentication(authorization);

        // Only Supervisor can save question
        String userRole = userWebToken.getRole();

        if (!StringUtils.equalsIgnoreCase(CoselmarUserRole.SUPERVISOR.name(), userRole)) {
            String message = String.format("User %s %s ('%s') is not allowed to save question",
                userWebToken.getFirstName(), userWebToken.getLastName(), userWebToken.getUserId());
            if (log.isWarnEnabled()) {
                log.warn(message);
            }
            throw new UnauthorizedException(message);

        }

        // retrieve user who will be assigned as question supervisor
        String fullUserId = getFullUserIdFromShort(userWebToken.getUserId());

        CoselmarUser supervisor;
        try {
            supervisor = getCoselmarUserDao().forTopiaIdEquals(fullUserId).findUnique();
        } catch (TopiaNoResultException tnre) {
            // Should not happened, cause user are not really deleted
            String message = String.format("Logged user ('%s') does not exist.", fullUserId);
            if (log.isErrorEnabled()) {
                log.error(message);
            }
            throw new InvalidCredentialException(message);
        }

        Preconditions.checkNotNull(question);
        Preconditions.checkNotNull(question.getTitle());
        Preconditions.checkNotNull(question.getSummary());
        Preconditions.checkNotNull(question.getType());
        Preconditions.checkNotNull(question.getThemes());


        // let's go
        Question questionEntity;
        // An update
        String questionId = question.getId();
        boolean inEdition = StringUtils.isNotBlank(questionId);
        if (inEdition) {
            String fullQuestionId = getFullIdFromShort(Question.class, questionId);
            questionEntity = getQuestionDao().forTopiaIdEquals(fullQuestionId).findUnique();

        } else {
            // or a create
            questionEntity = getQuestionDao().create();
        }

        // Question basics

        questionEntity.setTitle(question.getTitle());

        questionEntity.setSummary(question.getSummary());

        questionEntity.setType(question.getType());

        Set<String> themes = question.getThemes();
        if (themes != null) {
            questionEntity.setTheme(new HashSet(themes));
        }

        // By default, privacy is private
        String privacy = question.getPrivacy();
        Privacy realPrivacy = privacy  != null ? Privacy.valueOf(privacy.toUpperCase()) : Privacy.PRIVATE;
        questionEntity.setPrivacy(realPrivacy);

        // On creation, Status is Open
        if (inEdition) {
            String status = question.getStatus();
            questionEntity.setStatus(status != null ? Status.valueOf(status.toUpperCase()) : Status.OPEN);

            // If it is a close or adjourn update, put a closing date, if it is a reopen, remove closing date
            if (Lists.newArrayList(Status.CLOSED.name(), Status.ADJOURNED.name()).contains(status) && questionEntity.getClosingDate() == null) {
                questionEntity.setClosingDate(new Date());

            // it could be a reopen ...
            } else if (questionEntity.getClosingDate() != null && !Lists.newArrayList(Status.CLOSED.name(), Status.ADJOURNED.name()).contains(status)) {
                questionEntity.setClosingDate(null);
            }

        } else {
            questionEntity.setStatus(Status.OPEN);
        }

        // Manage Dates : submission & deadline
        Date submissionDate = question.getSubmissionDate();
        if (submissionDate != null) {
            questionEntity.setSubmissionDate(new Date(submissionDate.getTime()));
        } else {
            questionEntity.setSubmissionDate(new Date());
        }

        Date deadline = question.getDeadline();
        if (deadline != null) {
            questionEntity.setDeadline(new Date(deadline.getTime()));
        }


        // Users around the question

        // First Supervisor is the one creating this question
        questionEntity.addSupervisors(supervisor);

        questionEntity.addAllExternalExperts(question.getExternalExperts());

        // Retrieve the clients
        Set<UserBean> clients = question.getClients();
        questionEntity.clearClients();
        if (clients != null && !clients.isEmpty()) {
            Set<CoselmarUser> clientEntities = retrieveUsers(clients);
            questionEntity.addAllClients(clientEntities);
        }

        // For participants, should create a dedicated group,
        // that could be used to restrict documents access
        Set<UserBean> participants = question.getParticipants();
        CoselmarUserGroup participantGroup;

        //On update, get the existing group, make a diff : removed participants become contributor
        if (inEdition) {
            participantGroup = questionEntity.getParticipants();

            if (participants != null && !participants.isEmpty()) {
                Set<CoselmarUser> expertEntities = retrieveUsers(participants);

                // For each already assigned participants, if not in new list, assign them as contributors
                for (CoselmarUser participantBefore : participantGroup.getMembers()) {
                    if (!expertEntities.contains(participantBefore)) {
                        questionEntity.addContributors(participantBefore);
                    }
                }

                participantGroup.clearMembers();
                participantGroup.addAllMembers(expertEntities);
            } else {
                participantGroup.clearMembers();
            }

        } else {
            // If not update, create new group
            participantGroup = getCoselmarUserGroupDao().create();
            participantGroup.setName(question.getTitle());

            if (participants != null && !participants.isEmpty()) {
                Set<CoselmarUser> expertEntities = retrieveUsers(participants);
                participantGroup.addAllMembers(expertEntities);
            }
        }
        questionEntity.setParticipants(participantGroup);

        // Retrieve the supervisor
        Set<UserBean> supervisors = question.getSupervisors();
        questionEntity.clearSupervisors();
        if (supervisors != null && !supervisors.isEmpty()) {
            Set<CoselmarUser> supervisorEntities = retrieveUsers(supervisors);
            questionEntity.addAllSupervisors(supervisorEntities);
        }

        // Note : no contributors now, contributors are old participants,
        // excluded from process


        // Hierarchy of questions
        Set<QuestionBean> parents = question.getParents();
        if (parents != null && !parents.isEmpty()) {
            Set<Question> questions = retrieveQuestions(parents);

            questionEntity.addAllParents(questions);

        } else if(inEdition) {
            questionEntity.clearParents();
        }


        // Documents on init
        Set<DocumentBean> relatedDocuments = question.getRelatedDocuments();
        if (relatedDocuments != null && !relatedDocuments.isEmpty()) {
            Set<Document> documents = retrieveDocuments(relatedDocuments);
            // Manage restriction list for document with Privacy.RESTRICTED
            for (Document document : documents) {
                if (document.getPrivacy() == Privacy.RESTRICTED) {
                    document.addRestrictedList(participantGroup);
                }
            }

            questionEntity.clearRelatedDocuments();
            questionEntity.addAllRelatedDocuments(documents);
        } else if (inEdition) {
            questionEntity.clearRelatedDocuments();
        }

        questionEntity.setConclusion(question.getConclusion());

        // Documents on init
        Set<DocumentBean> closingDocuments = question.getClosingDocuments();
        if (closingDocuments != null && !closingDocuments.isEmpty()) {
            Set<Document> documents = retrieveDocuments(closingDocuments);
            // Manage restriction list for document with Privacy.RESTRICTED
            for (Document document : documents) {
                if (document.getPrivacy() == Privacy.RESTRICTED) {
                    document.addRestrictedList(participantGroup);
                }
            }

            questionEntity.clearClosingDocuments();
            questionEntity.addAllClosingDocuments(documents);
        } else if (inEdition) {
            questionEntity.clearClosingDocuments();
        }

        commit();

        QuestionBean result = BeanEntityConverter.toBean(getPersistenceContext().getTopiaIdFactory(), questionEntity);

        QuestionsIndexationService questionsIndexationService = getServicesContext().newService(QuestionsIndexationService.class);
        try {
            questionsIndexationService.indexQuestion(result);
            if (log.isDebugEnabled()) {
                String message = String.format("Question '%s' added to index", result.getTitle());
                log.debug(message);
            }
        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("Unable to index new question", e);
            }
        }
    }

    public List<String> getThemes() throws InvalidCredentialException, UnauthorizedException {

        //XXX ymartel 20141211 : do we need authentication check for that ?
//        // Check authentication
//        String authorization = getContext().getHeader("Authorization");
//        UserWebToken userWebToken = checkAuthentication(authorization);
//
//        // Check current user
//        String fullCurrentUserId = getFullUserIdFromShort(userWebToken.getUserId());
//        getCoselmarUserDao().forTopiaIdEquals(fullCurrentUserId).findAny();

        List<String> themes = getQuestionDao().findAllThemes();

        return themes;
    }

    public List<String> getTypes() throws InvalidCredentialException, UnauthorizedException {

        //XXX ymartel 20141211 : do we need authentication check for that ?
//        // Check authentication
//        String authorization = getContext().getHeader("Authorization");
//        UserWebToken userWebToken = checkAuthentication(authorization);
//
//        // Check current user
//        String fullCurrentUserId = getFullUserIdFromShort(userWebToken.getUserId());
//        getCoselmarUserDao().forTopiaIdEquals(fullCurrentUserId).findAny();

        List<String> types = getQuestionDao().findAllTypes();

        return types;
    }


    ////////////////////////////////////////////////////////////////////////////
    ///////////////////////     Internal Parts     /////////////////////////////
    ////////////////////////////////////////////////////////////////////////////


    protected void checkIsParticipant(Question question, CoselmarUser currentUser) throws UnauthorizedException {
        String userRole = currentUser.getRole().name();
        Set<CoselmarUser> questionsParticipants = question.getParticipants().getMembers();

        if (StringUtils.equalsIgnoreCase(CoselmarUserRole.EXPERT.name(), userRole)
            && !questionsParticipants.contains(currentUser)) {

            String message = String.format("Expert %s %s ('%s') is not allowed to add document",
                currentUser.getFirstname(), currentUser.getName(), currentUser.getTopiaId());
            if (log.isWarnEnabled()) {
                log.warn(message);
            }
            throw new UnauthorizedException(message);

        }
    }

    protected void checkIsClientAllowed(Question question, CoselmarUser currentUser) throws UnauthorizedException {
        String userRole = currentUser.getRole().name();
        Set<CoselmarUser> questionsClients = question.getClients();

        if (StringUtils.equalsIgnoreCase(CoselmarUserRole.CLIENT.name(), userRole)
            && questionsClients != null
            && !questionsClients.contains(currentUser)) {

            String message = String.format("Client %s %s ('%s') is not allowed to access question %s",
                currentUser.getFirstname(), currentUser.getName(), currentUser.getTopiaId(), question.getTopiaId());
            if (log.isWarnEnabled()) {
                log.warn(message);
            }
            throw new UnauthorizedException(message);

        }
    }

    protected Set<CoselmarUser> retrieveUsers(Collection<UserBean> userBeans) {
        Function<UserBean, String> getIds = new Function<UserBean, String>() {
            @Override
            public String apply(UserBean userBean) {
                return getFullIdFromShort(CoselmarUser.class, userBean.getId());
            }
        };

        Collection<String> userIds = Collections2.transform(userBeans, getIds);
        List<CoselmarUser> coselmarUsers = getCoselmarUserDao().forTopiaIdIn(userIds).findAll();
        return new HashSet<>(coselmarUsers);

    }

    protected Set<Question> retrieveQuestions(Collection<QuestionBean> questionBeans) {
        Function<QuestionBean, String> getIds = new Function<QuestionBean, String>() {
            @Override
            public String apply(QuestionBean questionBean) {
                return getFullIdFromShort(Question.class, questionBean.getId());
            }
        };

        Collection<String> questionIds = Collections2.transform(questionBeans, getIds);
        List<Question> questions = getQuestionDao().forTopiaIdIn(questionIds).findAll();
        return new HashSet<>(questions);

    }

    protected Set<Document> retrieveDocuments(Collection<DocumentBean> documentBeans) {
        Function<DocumentBean, String> getIds = new Function<DocumentBean, String>() {
            @Override
            public String apply(DocumentBean documentBean) {
                return getFullIdFromShort(Document.class, documentBean.getId());
            }
        };

        Collection<String> documentIds = Collections2.transform(documentBeans, getIds);
        List<Document> documents = getDocumentDao().forTopiaIdIn(documentIds).findAll();
        return new HashSet<>(documents);

    }

    /**
     * Methods that retrieve all questions depending of current user role permission :
     * <ul>
     *   <li>{@code ADMIN} and {@code SUPERVISOR} can access to all questions ;</li>
     *   <li>{@code MEMBER} can access to all public questions ;</li>
     *   <li>{@code EXPERT} can access to all private questions where he is participant or client and all public questions ;</li>
     *   <li>{@code CLIENT} can access to all questions where he is client.</li>
     * </ul>
     *
     * @param currentUser   :   current user that make request
     *
     * @return list of all question user is authorized to access.
     * @throws UnauthorizedException
     */
    protected List<Question> getAllQuestions(CoselmarUser currentUser) throws UnauthorizedException {
        String currentUserRole = currentUser.getRole().name();

        List<Question> questionList;
        if (StringUtils.equalsIgnoreCase(CoselmarUserRole.ADMIN.name(), currentUserRole)
            || StringUtils.equalsIgnoreCase(CoselmarUserRole.SUPERVISOR.name(), currentUserRole)) {
            questionList = getQuestionDao().findAll();

        } else if (StringUtils.equalsIgnoreCase(CoselmarUserRole.MEMBER.name(), currentUserRole)) {
            questionList = getQuestionDao().forPrivacyEquals(Privacy.PUBLIC).findAll();

        } else if (StringUtils.equalsIgnoreCase(CoselmarUserRole.EXPERT.name(), currentUserRole)) {
            questionList = getQuestionDao().findForExpert(currentUser);

        } else if (StringUtils.equalsIgnoreCase(CoselmarUserRole.CLIENT.name(), currentUserRole)) {
            questionList = getQuestionDao().forClientsContains(currentUser).findAll();
        } else {
            String message = "Not allowed to access this page";
            if (log.isWarnEnabled()) {
                log.warn("Unknown user type try to access questions list.");
            }
            throw new UnauthorizedException(message);
        }
        return questionList;
    }

    /**
     * Methods that retrieve all questions matching search filter (based on
     * title, summary, themes, status, privacy), also depending of current user role permission :
     * <ul>
     *   <li>{@code ADMIN} and {@code SUPERVISOR} can access to all questions ;</li>
     *   <li>{@code MEMBER} can only access to all public questions (so, if filter is on private, it has no result) ;</li>
     *   <li>{@code EXPERT} can access to all private questions where he is participant or client and all public questions ;</li>
     *   <li>{@code CLIENT} can access to all questions where he is client.</li>
     * </ul>
     *
     * @param currentUser   :   current user that make request
     * @param searchBean    :   bean containing search param to filter result
     *
     * @return list of all question user is authorized to access.
     * @throws UnauthorizedException
     */
    protected List<Question> getAllFilteredQuestions(CoselmarUser currentUser, QuestionSearchBean searchBean) throws UnauthorizedException {
        String currentUserRole = currentUser.getRole().name();
        QuestionsIndexationService questionsIndexationService = getServicesContext().newService(QuestionsIndexationService.class);

        //Try to retrieve corresponding questionIds from the index if searchBean given
        List<String> fromIndexQuestionIds = null;
        if (searchBean != null) {
            try {
                List<String> questionIds = questionsIndexationService.searchQuestion(searchBean);
                fromIndexQuestionIds = getQuestionsFullId(questionIds);
                if (log.isInfoEnabled()) {
                    log.info("Found " + fromIndexQuestionIds.size() + " questions from index.");
                }

            } catch (IOException |ParseException e) {
                if (log.isErrorEnabled()) {
                    log.error("Unable to search by lucene, make search directly in database", e);
                }
            }

        // classical search with DAO
        }

        List<Question> questionList;

        // Supervisor or Admin : access to all questions, no restriction
        if (StringUtils.equalsIgnoreCase(CoselmarUserRole.ADMIN.name(), currentUserRole)
            || StringUtils.equalsIgnoreCase(CoselmarUserRole.SUPERVISOR.name(), currentUserRole)) {

            if (fromIndexQuestionIds != null && !fromIndexQuestionIds.isEmpty()) {
                questionList = getQuestionDao().forTopiaIdIn(fromIndexQuestionIds).findAll();

            // classical search with DAO
            } else {
                questionList = getQuestionDao().findWithSearchBean(searchBean);

            }

        // Member : access to public question only
        } else if (StringUtils.equalsIgnoreCase(CoselmarUserRole.MEMBER.name(), currentUserRole)) {

            if (StringUtils.equals(searchBean.getPrivacy(), Privacy.PRIVATE.name())) {
                // Asking for private question ? directly no result !
                questionList = new ArrayList<>(0);

            } else {

                if (fromIndexQuestionIds != null && !fromIndexQuestionIds.isEmpty()) {
                    questionList = getQuestionDao().forTopiaIdIn(fromIndexQuestionIds).findAll();

                // classical search with DAO
                } else {
                    questionList = getQuestionDao().findWithSearchBean(searchBean);
                }
            }

        // Expert : access to all public question and private question if he is participant or client
        } else if (StringUtils.equalsIgnoreCase(CoselmarUserRole.EXPERT.name(), currentUserRole)) {
            if (fromIndexQuestionIds != null && !fromIndexQuestionIds.isEmpty()) {
                questionList = getQuestionDao().findForExpert(currentUser, fromIndexQuestionIds);

            } else {
                questionList = getQuestionDao().findForExpert(currentUser, searchBean);
            }

        // Client : access to question he is client
        } else if (StringUtils.equalsIgnoreCase(CoselmarUserRole.CLIENT.name(), currentUserRole)) {
            if (fromIndexQuestionIds != null && !fromIndexQuestionIds.isEmpty()) {
                questionList = getQuestionDao().findForClient(currentUser, fromIndexQuestionIds);

            } else {
                questionList = getQuestionDao().findForClient(currentUser, searchBean);
            }

        } else {
            String message = "Not allowed to access this page";
            if (log.isWarnEnabled()) {
                log.warn("Unknown user type try to access questions list.");
            }
            throw new UnauthorizedException(message);
        }
        return questionList;
    }

    protected List<String> getQuestionsFullId(List<String> questionShortIds) {

        Function<String, String> getFullIds = new Function<String, String>() {
            @Override
            public String apply(String shortId) {
                return Question.class.getCanonicalName() + getPersistenceContext().getTopiaIdFactory().getSeparator() + shortId;
            }
        };

        List<String> fullIds = Lists.transform(questionShortIds, getFullIds);
        return fullIds;
    }

}
