/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id: VradiService.java 1835 2010-12-21 09:28:19Z sletellier $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.5.0/vradi-swing/src/main/java/com/jurismarches/vradi/services/VradiService.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.services;

import com.caucho.hessian.client.HessianProxyFactory;
import com.jurismarches.vradi.VradiConfig;
import com.jurismarches.vradi.VradiConfigHelper;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyService;
import org.nuiton.wikitty.WikittyServiceFactory;

import java.net.MalformedURLException;

/**
 * Vradi service management.
 *
 * Can handle embedded server or remote hessian service.
 *
 * @author $Author: sletellier $
 * @version $Revision: 1835 $ $Date: 2010-12-21 10:28:19 +0100 (mar., 21 déc. 2010) $
 * @since 18 févr. 2010 17:33:03
 */
public class VradiService {

    /** log. */
    private static final Log log = LogFactory.getLog(VradiService.class);

    /** Wikitty service, used to add cache listener. */
    protected static WikittyService wikittyService;

    /** Wikitty proxy on cached service. */
    protected static WikittyProxy wikittyProxy;

    /** Vradi management service. */
    protected static VradiStorageService vradiStorageService;

    /** Local version to use local {{@link #wikittyProxy}. */
    protected static VradiDataService vradiDataService;

    /** File service. */
    protected static FileService fileService;

    /**
     * Init all service proxies.
     * 
     * @param configuration configuration
     */
    public static void initServiceProxies(VradiConfig configuration) {

        String serviceEndpoint = VradiConfigHelper.getRemoteEndpoint(configuration);

        // check is it's valid url
        if (serviceEndpoint != null && serviceEndpoint.startsWith("http://")) {
            if (log.isInfoEnabled()) {
                log.info("Use remote mode with url : " + serviceEndpoint);
            }

            HessianProxyFactory factory = new HessianProxyFactory();
            // Fix : com.caucho.hessian.io.HessianProtocolException: '���' is an unknown code
            factory.setHessian2Request(true);
            // pour que les méthodes aux noms dupliquées fonctionnent (arguments different)
            factory.setOverloadEnabled(true);

            try {

                // vradi service
                vradiStorageService = (VradiStorageService)factory.create(VradiStorageService.class,
                        getServiceURL(serviceEndpoint, "vradiservice"));

                wikittyService = initLocalWikittyService(configuration);

                // file service
                fileService = new FileServiceProxy(getServiceURL(serviceEndpoint, "file"));

            } catch (MalformedURLException eee) {
                if (log.isErrorEnabled()) {
                    log.error("Can't init remote proxy", eee);
                }
            }
        }
        else {
            if (log.isWarnEnabled()) {
                log.warn("Can't parse remote serveur endpoint as valid url (" + serviceEndpoint + ")");
                log.warn("Use local embedded mode");
            }

            vradiStorageService = new VradiStorageServiceImpl();
            wikittyService = ServiceFactory.getWikittyService(configuration);
            fileService = new FileServiceImpl();
        }

        // wrap service local/remote to a proxy
        wikittyProxy = new WikittyProxy();
        wikittyProxy.setWikittyService(wikittyService);

        // init data service on local cached wikitty proxy
        vradiDataService = new VradiDataServiceImpl(wikittyProxy);
    }

    /**
     * Create a new wikitty service with cache support that use
     * provided WikittyService for non cached data.
     * 
     * @param wikittyServiceRemote remote wikitty service
     * @param configuration vradi server configuration
     * @return local wikitty service
     */
    protected static WikittyService initLocalWikittyService(ApplicationConfig configuration) {

        log.info("Propagation event is set to : " + VradiConfigHelper.getWikittyEventPropagate(configuration));
        log.info("XMPP server is set to : " + VradiConfigHelper.getWikittyEventTransporterXMPPServer(configuration));
        log.info("XMPP room is set to : " + VradiConfigHelper.getWikittyEventTransporterXMPPRoom(configuration));

        WikittyService wikittyServiceLocal = WikittyServiceFactory.buildWikittyService(configuration);
        return wikittyServiceLocal;
    }

    /**
     * Get service full url.
     * 
     * @param serviceEndpoint end point (can't be null)
     * @param serviceEndUrl service small name
     * @return service full url
     */
    protected static String getServiceURL(String serviceEndpoint, String serviceEndUrl) {
        String fullUrl = serviceEndpoint;
        
        // Warn hessian won't work with double /
        if (!fullUrl.endsWith("/")) {
            fullUrl += "/";
        }
        fullUrl += serviceEndUrl;
        return fullUrl;
    }

    /**
     * Get remote vradi service (execute code on server).s
     * 
     * @return vradi storage service proxy
     */
    public static VradiStorageService getVradiStorageService() {
        return vradiStorageService;
    }
    
    /**
     * Get local vradi service (use local chached wikitty proxy).
     * 
     * @return local vradi data service proxy
     */
    public static VradiDataService getVradiDataService() {
        return vradiDataService;
    }

    /**
     * Get wikitty service proxy.
     * 
     * This wikitty service instance is cached and use a remote
     * wikitty service (also cached) for non available data.
     * 
     * @return wikitty service proxy
     */
    public static WikittyProxy getWikittyProxy() {
        return wikittyProxy;
    }
    
    /**
     * Get wikitty service (only use it to add listener on cache).
     * 
     * @return wikitty service
     */
    public static WikittyService getWikittyService() {
        return wikittyService;
    }

    /**
     * Get file service proxy.
     * 
     * @return file service
     */
    public static FileService getFileService() {
        return fileService;
    }
}
