/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id: VradiNotifier.java 1835 2010-12-21 09:28:19Z sletellier $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.5.0/vradi-swing/src/main/java/com/jurismarches/vradi/VradiNotifier.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi;

import com.jurismarches.vradi.entities.*;
import com.jurismarches.vradi.services.VradiService;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ListenerSet;
import org.nuiton.wikitty.WikittyService;
import org.nuiton.wikitty.entities.BusinessEntity;
import org.nuiton.wikitty.entities.BusinessEntityImpl;
import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.entities.WikittyExtension;
import org.nuiton.wikitty.services.WikittyEvent;
import org.nuiton.wikitty.services.WikittyListener;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * @author sletellier
 */
public class VradiNotifier implements WikittyListener {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(VradiNotifier.class);

    protected enum BusinessEntitiesEnum {
        CLIENT(Client.class, ClientImpl.class, Client.EXT_CLIENT),
        GROUP(Group.class, GroupImpl.class, Group.EXT_GROUP),
        USER(User.class, UserImpl.class, User.EXT_USER),
        STATUS(Status.class, StatusImpl.class, Status.EXT_STATUS),
        WEB_HARVEST_STREAM(WebHarvestStream.class, WebHarvestStreamImpl.class, WebHarvestStream.EXT_WEBHARVESTSTREAM),
        XML_STREAM(XmlStream.class, XmlStreamImpl.class, XmlStream.EXT_XMLSTREAM),
        FORM(Form.class, FormImpl.class, Form.EXT_FORM),
        INFOGENE(Infogene.class, InfogeneImpl.class, Infogene.EXT_INFOGENE),
        THESAURUS(Thesaurus.class, ThesaurusImpl.class, Thesaurus.EXT_THESAURUS),
        ROOT_THESAURUS(RootThesaurus.class, RootThesaurusImpl.class, RootThesaurus.EXT_ROOTTHESAURUS),
        SESSION(Session.class, SessionImpl.class, Session.EXT_SESSION),
        SENDING(Sending.class, SendingImpl.class, Sending.EXT_SENDING),
        GROUP_FORMS(GroupForms.class, GroupFormsImpl.class, GroupForms.EXT_GROUPFORMS);

        private Class<? extends BusinessEntity> beanClass;

        private Class<? extends BusinessEntityImpl> beanImplClass;
        private String ext;
        private BusinessEntitiesEnum(Class<? extends BusinessEntity> beanClass, Class<? extends BusinessEntityImpl> beanImplClass, String ext) {
            this.ext = ext;
            this.beanClass = beanClass;
            this.beanImplClass = beanImplClass;
        }

        public Class<? extends BusinessEntity> getBeanClass() {
            return beanClass;
        }

        private Class<? extends BusinessEntityImpl> getBeanImplClass() {
            return beanImplClass;
        }

        public String getExt() {
            return ext;
        }

        public static BusinessEntitiesEnum getValue(Class<? extends BusinessEntity> beanClass) {
            for (BusinessEntitiesEnum constant : values()) {
                if (beanClass.equals(constant.getBeanClass())){
                    return constant;
                }
            }
            return null;
        }

        public static BusinessEntitiesEnum getValue(Wikitty wikitty) {
            for (BusinessEntitiesEnum constant : values()) {
                if (wikitty.hasExtension(constant.getExt())){
                    return constant;
                }
            }
            return null;
        }

        public <C extends BusinessEntity> C getBusinessEntity(Class<C> classToConvert, Wikitty wikitty) {

            // try to find constructor with wikitty argument
            Class<? extends BusinessEntityImpl> beanClass = getBeanImplClass();

            C result = null;
            try {
                Constructor<?> cons = beanClass
                        .getConstructor(Wikitty.class);
                result = (C) cons.newInstance(wikitty);
            } catch (NoSuchMethodException eee) {
                log.error("Failed to find constructor of '" + beanClass + "' : ", eee);
            } catch (InstantiationException eee) {
                log.error("Failed to instanciate '" + beanClass + "' : ", eee);
            } catch (IllegalAccessException eee) {
                log.error(eee);
            } catch (InvocationTargetException eee) {
                log.error(eee);
            }
            return result;
        }

        public <C extends BusinessEntity> void addToMap(
                Class<C> classToConvert, Map<BusinessEntitiesEnum,
                        Set<? extends BusinessEntity>> mapToAdd, Wikitty wikitty) {

            Set<C> listToAdd = (Set<C>)mapToAdd.get(this);
            if (listToAdd == null) {
                listToAdd = new HashSet<C>();
                mapToAdd.put(this, listToAdd);
            }

            listToAdd.add(getBusinessEntity(classToConvert, wikitty));
        }

    }
    protected Map<BusinessEntitiesEnum,
            ListenerSet<VradiEntitiesListener<? extends BusinessEntity>>> entitiesListeners;
    protected ListenerSet<VradiExtensionsListener> extentionsListeners;

    public VradiNotifier() {

        entitiesListeners = new HashMap<BusinessEntitiesEnum,
                ListenerSet<VradiEntitiesListener<? extends BusinessEntity>>>();
        extentionsListeners = new ListenerSet<VradiExtensionsListener>();

        // register each tree on wikitty service
        VradiService.getWikittyService().addWikittyServiceListener(this,
                WikittyService.ServiceListenerType.REMOTE);
    }

    public <C extends BusinessEntity> void addVradiListener(Class<C> beanClass, VradiEntitiesListener<C> listener) {
        BusinessEntitiesEnum businessEntitiesEnum = BusinessEntitiesEnum.getValue(beanClass);
        ListenerSet<VradiEntitiesListener<? extends BusinessEntity>> listenerList = entitiesListeners.get(businessEntitiesEnum);
        if (listenerList == null) {
            listenerList = new ListenerSet<VradiEntitiesListener<? extends BusinessEntity>>();

            log.info("Add listener for enties " + businessEntitiesEnum + " class " + beanClass);
            entitiesListeners.put(businessEntitiesEnum, listenerList);
        }
        listenerList.add(listener);
    }

    public void addVradiListener(VradiExtensionsListener listener) {
        extentionsListeners.add(listener);
    }

    public <C extends BusinessEntity> void removeVradiEntitiesListener(Class<C> beanClass, VradiEntitiesListener<C> listener) {
        BusinessEntitiesEnum businessEntitiesEnum = BusinessEntitiesEnum.getValue(beanClass);
        ListenerSet<VradiEntitiesListener<? extends BusinessEntity>> listenerList = entitiesListeners.get(businessEntitiesEnum);
        if (listenerList == null) {
            return;
        }
        listenerList.remove(listener);
    }

    public void removeVradiExtensionsListener(VradiExtensionsListener listener) {
        extentionsListeners.remove(listener);
    }

    protected Map<BusinessEntitiesEnum, Set<? extends BusinessEntity>> extractEntities(WikittyEvent event) {

        Map<BusinessEntitiesEnum, Set<? extends BusinessEntity>> toNotify =
                new HashMap<BusinessEntitiesEnum, Set<? extends BusinessEntity>>();

        Map<String, Wikitty> wikittiesMap = event.getWikitties();

        // Extract entities
        if (wikittiesMap != null) {
            for (Wikitty w : wikittiesMap.values()) {
                BusinessEntitiesEnum businessEntitiesEnum = BusinessEntitiesEnum.getValue(w);
                if (businessEntitiesEnum != null) {
                    businessEntitiesEnum.addToMap(businessEntitiesEnum.getBeanClass(), toNotify, w);
                }
            }
        }
        return toNotify;
    }

    protected Set<WikittyExtension> extractExtensions(Map<String, WikittyExtension> extensionsMap) {
        Set extensions = new HashSet<WikittyExtension>();
        if (extensionsMap != null) {
            for (WikittyExtension extension : extensionsMap.values()) {
                if (extension.getFieldNames().contains(Form.EXT_FORM)) {
                    extensions.add(extension);
                }
            }
        }
        return extensions;
    }


    protected Set<String> extractDeletedIds(WikittyEvent event) {
        Map<String, Date> removeDate = event.getRemoveDate();
        if (removeDate != null) {
            return removeDate.keySet();
        }
        return Collections.EMPTY_SET;
    }

    public void notifyAllAddEntities(
            Map<BusinessEntitiesEnum, ListenerSet<VradiEntitiesListener<? extends BusinessEntity>>> listeners,
            Map<BusinessEntitiesEnum, Set<? extends BusinessEntity>> toNotify) {

        for (BusinessEntitiesEnum businessEntitiesEnum : toNotify.keySet()) {

            Set<? extends BusinessEntity> entitiesToNotify = toNotify.get(businessEntitiesEnum);
            ListenerSet<VradiEntitiesListener<? extends BusinessEntity>> listernersToNotify = listeners.get(businessEntitiesEnum);

            notifyAllAddEntities(businessEntitiesEnum.getBeanClass(), listernersToNotify, entitiesToNotify);

        }
    }

    protected <C extends BusinessEntity> void notifyAllAddEntities(Class<C> beanClass,
                                                                   ListenerSet<VradiEntitiesListener<? extends BusinessEntity>> listernersToNotify,
                                                                   Set<? extends BusinessEntity> entitiesToNotify) {

        if (log.isDebugEnabled()) {
            log.debug("[VradiNotifier] notifyAllAddEntities of type : " + beanClass.getName());
        }

        if (listernersToNotify != null) {
            for (VradiEntitiesListener<? extends BusinessEntity> listener : listernersToNotify) {
                notifyAddEntities((VradiEntitiesListener<C>)listener, (Set<C>)entitiesToNotify);
            }
        }
    }

    protected <C extends BusinessEntity> void notifyAddEntities(VradiEntitiesListener<C> listener, Set<C> entities) {
        listener.entitiesAdded(entities);
    }

    public void notifyAllRemoveEntities(
            Map<BusinessEntitiesEnum, ListenerSet<VradiEntitiesListener<? extends BusinessEntity>>> listeners,
            Set<String> toNotify) {

        for (ListenerSet<VradiEntitiesListener<? extends BusinessEntity>> listenerSet : listeners.values()) {

            for (VradiEntitiesListener<? extends BusinessEntity> listener : listenerSet) {
                notifyRemoveEntities(listener, toNotify);
            }

        }
    }

    protected <C extends BusinessEntity> void notifyRemoveEntities(VradiEntitiesListener<C> listener, Set<String> removedIds) {
        listener.entitiesRemoved(removedIds);
    }

    public void notifyAllAddExtensions(ListenerSet<VradiExtensionsListener> listernersToNotify, Set<WikittyExtension> extensions) {
        if (listernersToNotify != null) {
            for (VradiExtensionsListener listener : listernersToNotify) {
                notifyAddExtensions(listener, extensions);
            }
        }
    }

    protected void notifyAddExtensions(VradiExtensionsListener listener, Set<WikittyExtension> extensions) {
        listener.extensionsAdded(extensions);
    }

    public void notifyAllRemoveExtensions(ListenerSet<VradiExtensionsListener> listernersToNotify, Set<String> extensionsIds) {
        if (listernersToNotify != null) {
            for (VradiExtensionsListener listener : listernersToNotify) {
                notifyRemoveExtensions(listener, extensionsIds);
            }
        }
    }

    protected void notifyRemoveExtensions(VradiExtensionsListener listener, Set<String> extensionsIds) {
        listener.extensionsRemoved(extensionsIds);
    }

    @Override
    public void putWikitty(WikittyEvent event) {

        if (log.isDebugEnabled()) {
            log.debug("[VradiNotifier] Put wikitty");
        }

        Map<BusinessEntitiesEnum, Set<? extends BusinessEntity>> toNotify = extractEntities(event);

        notifyAllAddEntities(entitiesListeners, toNotify);
    }

    @Override
    public void removeWikitty(WikittyEvent event) {

        if (log.isDebugEnabled()) {
            log.debug("[VradiNotifier] Remove wikitty");
        }

        Set<String> toNotify = extractDeletedIds(event);

        notifyAllRemoveEntities(entitiesListeners, toNotify);
    }

    @Override
    public void clearWikitty(WikittyEvent event) {
        // Never append in vradi
    }

    @Override
    public void putExtension(WikittyEvent event) {
        Set<WikittyExtension> extensions = extractExtensions(event.getExtensions());
        notifyAllAddExtensions(extentionsListeners, extensions);
    }

    @Override
    public void removeExtension(WikittyEvent event) {
        Set<String> extensionsIds = event.getDeletedExtensions();
        notifyAllRemoveExtensions(extentionsListeners, extensionsIds);
    }

    @Override
    public void clearExtension(WikittyEvent event) {
        // Never append in vradi
    }
}
