/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id: SearchHandler.java 1715 2010-10-27 19:21:28Z tchemit $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.3.1/vradi-swing/src/main/java/com/jurismarches/vradi/ui/search/SearchHandler.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.ui.search;

import static org.nuiton.i18n.I18n._;

import java.awt.Component;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Set;

import com.jurismarches.vradi.beans.QueryBean;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.ui.models.RequestFieldMenu;
import javax.swing.Action;
import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JComboBox;
import javax.swing.JList;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JTextArea;
import javax.swing.ListCellRenderer;
import javax.swing.SwingUtilities;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.context.JAXXInitialContext;

import jaxx.runtime.swing.ErrorDialogUI;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTreeTable;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.decorator.HighlighterFactory;
import org.nuiton.wikitty.FieldType;
import org.nuiton.wikitty.WikittyExtension;

import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.beans.FormPagedResult;
import com.jurismarches.vradi.beans.QueryParameters;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Infogene;
import com.jurismarches.vradi.entities.RootThesaurus;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.ui.admin.content.AdminThesaurusHandler;
import com.jurismarches.vradi.ui.helpers.ThesaurusDataHelper;
import com.jurismarches.vradi.ui.helpers.UIHelper;
import com.jurismarches.vradi.ui.models.EntityModel;
import com.jurismarches.vradi.ui.offer.OfferListUI;
import com.jurismarches.vradi.ui.offer.models.OfferListTableModel;
import com.jurismarches.vradi.ui.offer.thesaurus.ThesaurusCartographyUI;
import com.jurismarches.vradi.ui.offer.thesaurus.models.ThesaurusCartographyTreeTableModel;
import com.jurismarches.vradi.ui.task.VradiTask;

/**
 * @author letellier
 */
public class SearchHandler {

    static private final Log log = LogFactory.getLog(SearchHandler.class);

    static public final String CSV_FILE_EXTENSION = ".csv";

    /** Prefix i18n des valeurs dynamiques suivant l'infogene. */
    static public final String DATE_CRITERIA_I18N_PREFIX = "vradi.criteria.typeDate.";

    /**
     * Methode pour initialiser l'ui principale sans l'afficher.
     *
     * @param rootContext le context applicatif
     * @return l'ui instancie et initialisee mais non visible encore
     */
    public SearchUI initUI(JAXXContext rootContext, OfferListTableModel resultTableModel) {

        JAXXInitialContext context = new JAXXInitialContext().add(rootContext);

        // show main ui
        context.add(this);
        context.add(resultTableModel);

        SearchUI ui = new SearchUI(context);

        initCriteria(ui);
        
        return ui;
    }

    public SearchUI initCloneUI(JAXXContext rootContext, OfferListTableModel resultTableModel) {
        return initCloneUI(rootContext, resultTableModel, null);
    }

    public SearchUI initCloneUI(JAXXContext rootContext, OfferListTableModel resultTableModel, Thesaurus thesaurus) {

        JAXXInitialContext context = new JAXXInitialContext().add(rootContext);

        // show main ui
        context.add(this);
        context.add(resultTableModel);

        SearchUI ui = new SearchUI(context);

        if (thesaurus != null) {
            List<Thesaurus> criteriaList = new ArrayList<Thesaurus>();
            criteriaList.add(thesaurus);
            initCriteria(ui, criteriaList);
        }

        return ui;
    }

    SearchUI getUI(JAXXContext context) {
        if (context instanceof SearchUI) {
            return (SearchUI) context;
        }
        SearchUI ui = null;
        if (context instanceof JAXXObject) {
            JAXXObject o = (JAXXObject) context;
            ui = o.getParentContainer(SearchUI.class);
        }

        if (ui != null) {
            ui = context.getContextValue(SearchUI.class);
        }
        return ui;
    }

    // From Admin Handler
    public void executeQuery(final SearchUI searchUI) {

        final String query = searchUI.getRequest();
        final JComboBox combo = searchUI.isOfferListUI() ? searchUI.getRequests() : null;
        final CriteriaUI criterias = searchUI.getCriterias().isVisible() ? searchUI.getCriterias() : null;

        final OfferListTableModel listTableModel = searchUI.getResultTableModel();
        final FormPagedResult formPageDTO = listTableModel.getFormPagedResult();

        VradiTask<FormPagedResult> task = new VradiTask<FormPagedResult>(searchUI, false) {
            @Override
            public FormPagedResult doAction() throws Exception {
                return executeQuery(searchUI, criterias, formPageDTO, combo, query);
            }

            @Override
            public void doWhenDone() throws Exception {
                searchUI.getRootPane().setCursor(null);
                FormPagedResult formPageDTO2 = get();
                listTableModel.setFormPagedResult(formPageDTO2);
            }
        };

        task.execute();
    }

    public void executeQuery(SearchUI searchUI, int pageToShow) {
        //executeQueries(context);
        OfferListTableModel listTableModel = searchUI.getResultTableModel();
        listTableModel.setPageToShow(pageToShow);
    }

    public java.util.List<String> loadRequests() {
        return VradiHelper.loadRequests();
    }

    public JPanel getCriteriaPanel(SearchUI searchUI) {
        CriteriaUI criterias = searchUI.getCriterias();
        return criterias.getCriteriaPanel();
    }

    public CriteriaUI getCriterias(JAXXContext context) {
        return getUI(context).getCriterias();
    }

    public void initCriteria(SearchUI searchUI) {
        initCriteria(searchUI, null);
    }

    public void initCriteria(SearchUI searchUI, List<Thesaurus> thesaurus) {
        // Remove existing
        getCriteriaPanel(searchUI).removeAll();

        // Adding saved root thesaurus criteria
        List<String> rootThesaurusListCriteria = VradiHelper.getVradiListRootCriteria();
        for (String value : rootThesaurusListCriteria) {
            addCriteria(searchUI, value, true);
        }

        // Adding saved thesaurus criteria
        List<String> thesaurusListCriteria = VradiHelper.getVradiListCriteria();
        for (String value : thesaurusListCriteria) {
            addCriteria(searchUI, value, true);
        }

        // Adding thesaurus suppl
        if (thesaurus != null){
            for (Thesaurus t : thesaurus){
                addUnmodifiableCriteria(searchUI, t);
            }
        }
    }

    public void removeCriteria(SearchUI searchUI, CriteriaField criteriaField) {
        getCriteriaPanel(searchUI).remove(criteriaField);
        searchUI.validate();
    }

    public void removeCriteriaAndPersist(CriteriaField criteriaField, Thesaurus bean) {
        removeCriteriaAndPersist(getUI(criteriaField), criteriaField, bean);
    }

    public void removeCriteriaAndPersist(SearchUI searchUI, CriteriaField criteriaField, Thesaurus bean) {
        VradiHelper.removeVradiListCriteria(bean);
        removeCriteria(searchUI, criteriaField);
        searchUI.validate();
    }

    public void removeRootCriteriaAndPersist(CriteriaField criteriaField, RootThesaurus bean) {
        removeRootCriteriaAndPersist(getUI(criteriaField), criteriaField, bean);
    }

    public void removeRootCriteriaAndPersist(SearchUI searchUI, CriteriaField criteriaField, RootThesaurus bean) {
        VradiHelper.removeVradiListRootCriteria(bean);
        removeCriteria(searchUI, criteriaField);
        searchUI.validate();
    }

    public void addUnmodifiableCriteria(SearchUI searchUI, Thesaurus thesaurus) {

        CriteriaField criteriaField;
        if (ThesaurusDataHelper.isFirstChild(thesaurus)) {
            RootThesaurus root = ThesaurusDataHelper.restoreRootThesaurus(thesaurus.getRootThesaurus());
            criteriaField = addCriteria(searchUI, root, false, false);
        } else {
            Thesaurus parent = ThesaurusDataHelper.restoreThesaurus(thesaurus.getParent());
            criteriaField = addCriteria(searchUI, parent, false, false);
        }

        // Get combo
        JComboBox combo = criteriaField.getCombo();

        // Select thesaurus
        combo.getModel().setSelectedItem(thesaurus);
        combo.setSelectedItem(thesaurus);
    }

    public void addAndPersistCriteria(JAXXContext context, Thesaurus thesaurus) {
        addAndPersistCriteria(getUI(context), thesaurus);
    }

    public void addAndPersistCriteria(SearchUI searchUI, Thesaurus thesaurus) {
        VradiHelper.addVradiListCriteria(thesaurus);
        addCriteria(searchUI, thesaurus, true, true);
    }

    protected void addRootCriteria(SearchUI searchUI, String wikittyId, boolean deletable) {

        if (wikittyId == null || wikittyId.isEmpty()){
             return;
        }

        RootThesaurus criteriaThe = ThesaurusDataHelper.restoreRootThesaurus(wikittyId);

        if (criteriaThe != null) {
             addCriteria(searchUI, criteriaThe, deletable, true);
        }
    }

    protected void addCriteria(SearchUI searchUI, String wikittyId, boolean deletable) {

        if (wikittyId == null || wikittyId.isEmpty()){
             return;
        }

        Thesaurus criteriaThe = ThesaurusDataHelper.restoreThesaurus(wikittyId);

        if (criteriaThe != null) {
             addCriteria(searchUI, criteriaThe, deletable, true);
        }
    }

    protected CriteriaField addCriteria(SearchUI searchUI, RootThesaurus thesCriteria, boolean deletable, boolean enabled) {
        // New criteria field
        CriteriaField criteriaField = new CriteriaField(searchUI);

        // Set Name
        criteriaField.setLabelName(thesCriteria.getName());

        // Set bean
        criteriaField.setRootThesaurus(thesCriteria);

        // Get children treeNode
        try {
            List<Thesaurus> children = VradiService.getVradiDataService().getChildrenThesaurus(thesCriteria.getWikittyId());
            initCriteriaField(searchUI, criteriaField, children, deletable, enabled);
        }
        catch (VradiException ex) {
            throw new RuntimeException("Can't get children thesaurus", ex);
        }

        return criteriaField;
    }

    protected CriteriaField addCriteria(SearchUI searchUI, Thesaurus thesCriteria, boolean deletable, boolean enabled) {
        // New criteria field
        CriteriaField criteriaField = new CriteriaField(searchUI);

        // Set Name
        criteriaField.setLabelName(thesCriteria.getName());

        // Set bean
        criteriaField.setThesaurus(thesCriteria);

        // Get children treeNode
        List<Thesaurus> children = null;
        String thesaurusId = thesCriteria.getWikittyId();
        try {
            children = VradiService.getVradiDataService().getChildrenThesaurus(thesaurusId);
        } catch (VradiException eee) {
            log.error("Can't restore children of thesaurus '" + thesaurusId + "' : ", eee);
            ErrorDialogUI.showError(eee);
        }

        initCriteriaField(searchUI, criteriaField, children, deletable, enabled);

        return criteriaField;
    }

    protected void initCriteriaField(SearchUI searchUI, CriteriaField criteriaField, List<Thesaurus> children, boolean deletable, boolean enabled) {
        // Deletable
        criteriaField.setDeletable(deletable);
        criteriaField.setActif(enabled);

        // Get combo
        JComboBox combo = criteriaField.getCombo();

        // Apply model
        combo.setModel(new EntityModel<Thesaurus>(Thesaurus.class, Thesaurus.EXT_THESAURUS, children, true));
        combo.setMaximumRowCount(30);

        // Apply renderer
        combo.setRenderer(VradiContext.get().getDecoratorProviderListCellRenderer());

        // Adding to ui
        criteriaField.init();

        JPanel criteriaPanel = getCriteriaPanel(searchUI);
        criteriaPanel.add(criteriaField);

        searchUI.validate();
    }

    protected List<String> getDateFormType(WikittyExtension formType) {
        List<String> result = new ArrayList<String>();
        result.add(Form.FQ_FIELD_INFOGENE_CREATIONDATE);
        result.add(Form.FQ_FIELD_FORM_DATEPUB);
        result.add(Form.FQ_FIELD_FORM_DATEPEREMPTION);
        if (formType != null) {
            for (String name : formType.getFieldNames()) {
                FieldType type = formType.getFieldType(name);
                if (type.getType() == FieldType.TYPE.DATE) {
                    result.add(formType.getName() + "." + name);
                }
            }
        }
        return result;
    }

    public DefaultComboBoxModel updateDateSelectionModel(WikittyExtension selected) {
        List<String> datas = getDateFormType(selected);
        return new DefaultComboBoxModel(datas.toArray(new String[datas.size()]));
    }

    /**
     * Execute query.
     *
     * @param context
     * @param criterias
     * @param formPagedResult
     * @param combo
     * @param request
     * @return paged result
     */
    public FormPagedResult executeQuery(final SearchUI searchUI, CriteriaUI criterias, FormPagedResult formPagedResult, final JComboBox combo, final String request) {
        if (criterias == null) {
            // Criteria panel is hidden
            QueryParameters queryParameters = new QueryParameters(request);
            formPagedResult = VradiHelper.executeQuery(queryParameters, formPagedResult);
        } else {
            // Apply criterian search
            Component[] thesaurusCriteria = getCriteriaPanel(searchUI).getComponents();
            List<Thesaurus> thesaurus = new ArrayList<Thesaurus>();
            for (int i = 0; i < thesaurusCriteria.length; i++) {

                // Get selected
                Thesaurus tn = (Thesaurus) ((CriteriaField) thesaurusCriteria[i])
                        .getCombo().getSelectedItem();

                // If is not empty
                if (tn != null && StringUtils.isNotBlank(tn.getName())) {

                    // Add parent
                    String wikittyId = tn.getWikittyId();

                    // Find thesaurus corresponding in context
                    Thesaurus parent = ThesaurusDataHelper.restoreThesaurus(wikittyId);

                    // Get children recursivly
                    thesaurus.add(parent);
                }
            }
            formPagedResult = executeQuery(request, criterias, thesaurus, formPagedResult);
        }

        if (log.isInfoEnabled()) {
            log.info("Total forms found        : " + formPagedResult.getTotalFoundFormNb());
            log.info("Number of forms per page : " + formPagedResult.getNbFormsToShow());
            log.info("Actual page              : " + formPagedResult.getPageToShow());
        }

        // TODO SLE 06082010 : Refactor search to have a better way
        // If it's offer edit it's calling search
        if (searchUI.getParentContainer(OfferListUI.class) != null) {

            // Store request in properties
            VradiHelper.addRequestToProperties(request);
        }

        // Update models
        if (combo != null) {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    DefaultComboBoxModel model = searchUI.getRequestModel();
                    model.removeElement(request);
                    model.insertElementAt(request, 0);
                    combo.setSelectedItem(request);
                }
            });
        }

        return formPagedResult;
    }

    public DefaultComboBoxModel getRequestModel() {
        java.util.List<String> queries = loadRequests();
        DefaultComboBoxModel model = new DefaultComboBoxModel(queries.toArray());
        return model;
    }

    public void updateRequests(SearchUI searchUI) {
        JComboBox groupsCombo = searchUI.getGroups();
        Group selected = (Group)groupsCombo.getSelectedItem();
        JComboBox requestCombo = searchUI.getRequests();
        if (selected == null) {

            // Use saved request model
            requestCombo.setModel(searchUI.getRequestModel());
            return;
        }

        // Use group request
        Set<String> queries = selected.getQueries();

        // Extract queries
        List<QueryBean> queriesBean = new ArrayList<QueryBean>();
        for (String query : queries) {
            try {
                QueryBean queryBean = new QueryBean(query);
                queriesBean.add(queryBean);
            } catch (IOException eee) {
                log.error("Cant parse query : " + query, eee);
                ErrorDialogUI.showError(eee);
            }
        }

        // Use group requests model
        requestCombo.setModel(new DefaultComboBoxModel(queriesBean.toArray()));

        // Try to select updated
        requestCombo.setSelectedItem(searchUI.getRequest());
    }

    /**
     * Affiche une fenetre contenant la catographie pour une requete en cours.
     * @param searchUI
     */
    public void executeCartography(SearchUI searchUI) {

        String query = searchUI.getRequest();
        JComboBox combo = searchUI.isOfferListUI() ? searchUI.getRequests() : null;
        CriteriaUI criterias = searchUI.getCriterias().isVisible() ? searchUI.getCriterias() : null;
        
        ThesaurusCartographyUI cartographyUI = new ThesaurusCartographyUI(searchUI);
        QueryParameters queryParameters = getQueryParameters(searchUI, criterias, combo, query);
        ThesaurusCartographyTreeTableModel cartographyModel = new ThesaurusCartographyTreeTableModel(queryParameters);
        // cause NPE in jaxx ?
        cartographyUI.getCartographyTable().setTreeTableModel(cartographyModel);
        cartographyUI.getCartographyTable().setTreeCellRenderer(cartographyModel.new CartographieTreeCellRenderer());
        configureCartographyUI(searchUI, cartographyUI);
        cartographyUI.setLocationRelativeTo(searchUI);
        cartographyUI.setVisible(true);
    }

    protected void configureCartographyUI(final SearchUI searchUI, ThesaurusCartographyUI cartographyUI) {
        
        final JXTreeTable cartographyTable = cartographyUI.getCartographyTable();
        final Action copyToClipboard = new AdminThesaurusHandler.CopyToClipboard(cartographyUI);

        Highlighter highlighter = HighlighterFactory.createAlternateStriping(
        HighlighterFactory.NOTEPAD, HighlighterFactory.GENERIC_GRAY);
        cartographyTable.addHighlighter(highlighter);

        TableColumnModel columnModel = cartographyTable.getColumnModel();
        TableColumn column = columnModel.getColumn(0);
        column.setWidth(350);
        column.setPreferredWidth(350);

        column = columnModel.getColumn(1);
        column.setWidth(80);
        column.setPreferredWidth(80);

        column = columnModel.getColumn(2);
        column.setWidth(150);
        column.setPreferredWidth(150);

        cartographyTable.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent mouseEvent) {
                if (mouseEvent.getButton() == MouseEvent.BUTTON1 &&
                        mouseEvent.getClickCount() == 2) {

                    int selectedRow = cartographyTable.getSelectedRow();
                    Object selected = cartographyTable.getValueAt(selectedRow, 0);
                    if (!(selected instanceof Thesaurus)) {
                        return;
                    }
                    Thesaurus selectedThesaurus = (Thesaurus)selected;

                    List<Thesaurus> thesaurus = new ArrayList<Thesaurus>();
                    thesaurus.add(selectedThesaurus);

                    CriteriaUI criterias = searchUI.getCriterias().isVisible() ? searchUI.getCriterias() : null;
                    FormPagedResult formPagedResult = new FormPagedResult();
                    formPagedResult.setNbFormsToShow(10);
                    formPagedResult = searchUI.getHandler().executeQuery(searchUI.getRequest(), criterias, thesaurus, formPagedResult);

                    searchUI.getResultTableModel().setFormPagedResult(formPagedResult);

                    if (log.isDebugEnabled()) {
                        log.debug("Found " + formPagedResult.getTotalFoundFormNb() + " form to show");
                    }
                }
                
                else {
                    // Add mouse listener to enable menus
                    if (mouseEvent.getButton() == MouseEvent.BUTTON3) {
                        JPopupMenu popup = new JPopupMenu();
                        popup.add(copyToClipboard);
                        popup.show(cartographyTable, mouseEvent.getX(), mouseEvent.getY());
                    }
                }
            }
        });
    }

    protected QueryParameters getQueryParameters(SearchUI searchUI,
            CriteriaUI criterias, JComboBox combo, String request) {

        QueryParameters queryParameters = new QueryParameters(request);

        // Criteria panel not hidden
        if (criterias != null) {
            // Apply criteria search
            Component[] thesaurusCriteria = getCriteriaPanel(searchUI).getComponents();
            List<Thesaurus> thesaurus = new ArrayList<Thesaurus>();
            for (int i = 0; i < thesaurusCriteria.length; i++) {
                // Get selected
                Thesaurus tn = (Thesaurus) ((CriteriaField) thesaurusCriteria[i])
                        .getCombo().getSelectedItem();
                thesaurus.add(tn);
            }
            queryParameters.setThesaurus(thesaurus);

            // begin date
            Calendar beginDate = Calendar.getInstance();
            beginDate.setTime(criterias.getFromDate().getDate());
            beginDate.set(Calendar.HOUR_OF_DAY, (Integer) criterias
                    .getFromHour().getSelectedItem());
            beginDate.set(Calendar.MINUTE, (Integer) criterias
                    .getFromMinute().getSelectedItem());
            queryParameters.setBeginDate(beginDate.getTime());

            // end date
            Calendar endDate = Calendar.getInstance();
            endDate.setTime(criterias.getToDate().getDate());
            endDate.set(Calendar.HOUR_OF_DAY, (Integer) criterias.getToHour()
                    .getSelectedItem());
            endDate.set(Calendar.MINUTE, (Integer) criterias.getToMinute()
                    .getSelectedItem());
            queryParameters.setEndDate(endDate.getTime());

            // form type
            WikittyExtension extension = (WikittyExtension)criterias
                    .getFormTypeSelection().getSelectedItem();
            queryParameters.setExtension(extension);

            // date field name
            String dateFieldName = (String)criterias.getDateTypeSelection()
                    .getSelectedItem();
            queryParameters.setDateFieldName(dateFieldName);

            // status
            Object[] statuses = criterias.getStatusSelection()
                    .getSelectedValues();
            List<String> statusIds = new ArrayList<String>(statuses.length);
            for (int i = 0 ; i < statuses.length ; i++) {
                String statusId = ((Status)statuses[i]).getWikittyId();
                statusIds.add(statusId);
            }
            queryParameters.setStatusIds(statusIds);

            // xml streams id
            XmlStream xmlStream = (XmlStream)criterias.getXmlStreamSelection()
                    .getSelectedItem();
            if (xmlStream != null) {
                String streamId = xmlStream.getWikittyId();
                queryParameters.setStreamId(streamId);
            }
        }

        return queryParameters;
    }

    /**
     * Execute query methods (called by double clic on thesaurus catography
     * treetable).
     * 
     * FIXME EC20100920 code duplicated with getQueryParameters(SearchUI, CriteriaUI, JComboBox, String )
     * 
     * @param request
     * @param criterias
     * @param thesauruses
     * @param formPagedResult
     * @return
     */
    public FormPagedResult executeQuery(String request, CriteriaUI criterias, List<Thesaurus> thesauruses, FormPagedResult formPagedResult) {

        QueryParameters queryParameters = new QueryParameters(request);

        // Criteria panel not hidden
        if (criterias != null) {
            // begin date
            Calendar beginDate = Calendar.getInstance();
            beginDate.setTime(criterias.getFromDate().getDate());
            beginDate.set(Calendar.HOUR_OF_DAY, (Integer) criterias
                    .getFromHour().getSelectedItem());
            beginDate.set(Calendar.MINUTE, (Integer) criterias
                    .getFromMinute().getSelectedItem());
            queryParameters.setBeginDate(beginDate.getTime());
    
            // end date
            Calendar endDate = new GregorianCalendar();
            endDate.setTime(criterias.getToDate().getDate());
            endDate.set(Calendar.HOUR_OF_DAY, (Integer) criterias.getToHour()
                    .getSelectedItem());
            endDate.set(Calendar.MINUTE, (Integer) criterias.getToMinute()
                    .getSelectedItem());
            queryParameters.setEndDate(endDate.getTime());
    
            // form type
            WikittyExtension extension = (WikittyExtension)criterias
                    .getFormTypeSelection().getSelectedItem();
            queryParameters.setExtension(extension);
    
            // date field
            String dateFieldName = (String)criterias.getDateTypeSelection()
                    .getSelectedItem();
            queryParameters.setDateFieldName(dateFieldName);
    
            // status (traité/non traité)
            Object[] statuses = criterias.getStatusSelection().getSelectedValues();
            List<String> statusIds = new ArrayList<String>(statuses.length);
            for (int i = 0 ; i < statuses.length ; i++) {
                statusIds.add(((Status)statuses[i]).getWikittyId());
            }
            queryParameters.setStatusIds(statusIds);
    
            // xml stream id
            XmlStream xmlStream = (XmlStream)criterias.getXmlStreamSelection().getSelectedItem();
            if (xmlStream != null) {
                String streamId = xmlStream.getWikittyId();
                queryParameters.setStreamId(streamId);
            }
        }

        // thesaurus is
        queryParameters.setThesaurus(thesauruses);

        // execute query
        FormPagedResult result = VradiHelper.executeQuery(queryParameters, formPagedResult);
        return result;
    }

    /**
     * Criteria UI date sort renderer.
     *
     * Prefix i18n translation with {@link #DATE_CRITERIA_I18N_PREFIX}.
     *
     * @return ListCellRenderer
     */
    public ListCellRenderer getDateTypeComboBoxRenderer() {

        // List of translated criteria : (don't remove this comment) :
        //
        // _("vradi.criteria.typeDate.Form.datePeremption");
        // _("vradi.criteria.typeDate.Form.datePub");
        // _("vradi.criteria.typeDate.Infogene.creationDate");
        //

        return new DefaultListCellRenderer() {
            /** serialVersionUID. */
            private static final long serialVersionUID = -1967943679031854092L;

            @Override
            public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
                String stringValue = null;
                if (value != null) {
                    stringValue = value.toString();
                    int dot = stringValue.lastIndexOf(".");
                    String extName = stringValue.substring(0, dot);
                    if(Infogene.EXT_INFOGENE.equals(extName)
                            || Form.EXT_FORM.equals(extName)) {
                        stringValue = _(DATE_CRITERIA_I18N_PREFIX + stringValue);
                    } else {
                        stringValue = stringValue.substring(dot + 1);
                    }
                }
                return super.getListCellRendererComponent(list, stringValue, index, isSelected, cellHasFocus);
            }
        };
    }

    public void exportToCSV(SearchUI searchUI) {
        File file = UIHelper.openFileChooser(CSV_FILE_EXTENSION, _("vradi.export.filter.csv"),
                searchUI, _("vradi.export.validate.label"));

        if (file != null) {
            FormPagedResult formPageDTO = new FormPagedResult();
            // export all results
            formPageDTO.setNbFormsToShow(-1);
            formPageDTO.setPageToShow(1);

            // FIXME EC20100607 : Just init an UI tree model to get column names !!!
            OfferListTableModel tableModel = new OfferListTableModel(formPageDTO);

            if (searchUI != null) {
                CriteriaUI criterias = searchUI.getCriterias();
                String request = searchUI.getRequest();
                executeQuery(searchUI, criterias, formPageDTO, null, request);

            } else {
                executeQuery(searchUI, null, formPageDTO, null, null);
            }

            UIHelper.exportToCSV(file, tableModel);
        }
    }

    /**
     * Show context menu on query text area to paste clibboard content.
     * 
     * @param searchUI parent UI
     * @param event mouse event
     */
    public void showRequestAreaContextMenu(final SearchUI searchUI, MouseEvent event) {
        
        // clic droit
        if (event.getButton() == MouseEvent.BUTTON3) {
            JPopupMenu popupMenu = new JPopupMenu();
            
            JMenuItem pasteMenuItem = new JMenuItem(_("vradi.common.paste"));
            pasteMenuItem.addActionListener(new ActionListener() {

                @Override
                public void actionPerformed(ActionEvent e) {

                    if (e.getActionCommand().equals("paste")) {

                        // get clipboard content
                        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
                        Transferable transferable = clipboard.getContents(null);
                        try {
                            String stringContent = (String)transferable.getTransferData(DataFlavor.stringFlavor);

                            JTextArea requestArea = searchUI.getRequestArea();

                            // Supprimer le text selectionné
                            String text = requestArea.getText();
                            text = text.substring(requestArea.getSelectionStart(), requestArea.getSelectionEnd());
                            requestArea.setText(text);

                            // A la position du pointeur
                            int position = requestArea.getCaretPosition();
                            requestArea.insert(stringContent, position);
                            searchUI.setRequest(requestArea.getText());
                        } catch (Exception e1) {
                            if (log.isDebugEnabled()) {
                                log.debug("Can't get clipboard content");
                            }
                        }
                    }
                }
            });
            pasteMenuItem.setActionCommand("paste");
            popupMenu.add(pasteMenuItem);

            popupMenu.add(new RequestFieldMenu(searchUI));
            
            popupMenu.show(searchUI, event.getX(), event.getY());
        }
    }
}

