/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package com.jurismarches.vradi.ui.renderers;

import java.awt.Color;
import java.awt.Component;

import javax.swing.DefaultListCellRenderer;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JTree;
import javax.swing.ListCellRenderer;
import javax.swing.tree.TreeCellRenderer;

import com.jurismarches.vradi.entities.RootThesaurus;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.ui.helpers.ThesaurusDataHelper;
import com.jurismarches.vradi.ui.helpers.ToolTipHelper;
import com.jurismarches.vradi.ui.offer.thesaurus.helpers.ThesaurusTreeHelper;
import com.jurismarches.vradi.ui.tree.VradiTreeNode;
import com.jurismarches.vradi.ui.tree.VradiTreeTableNode;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.decorator.AbstractHighlighter;
import org.jdesktop.swingx.decorator.ComponentAdapter;


/**
 * @author letellier
 */
public class ThesaurusHighlighter extends AbstractHighlighter implements TreeCellRenderer, ListCellRenderer {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(ThesaurusHighlighter.class);

    protected DefaultListCellRenderer delegate = null;
    protected HighlighterType type;

    public enum HighlighterType {
        DEFAULTS,
        EDIT,
        PROPOSITION,
        MOVE
    }

    public ThesaurusHighlighter(){
        this(HighlighterType.DEFAULTS);
    }

    public ThesaurusHighlighter(HighlighterType type){
        super();
        this.type = type;
        this.delegate = new DefaultListCellRenderer();
    }

    protected boolean isDefault() {
        return type.equals(HighlighterType.DEFAULTS);
    }

    protected boolean isEdit() {
        return type.equals(HighlighterType.EDIT);
    }

    protected boolean isMove() {
        return type.equals(HighlighterType.MOVE);
    }

    protected boolean isProposition() {
        return type.equals(HighlighterType.PROPOSITION);
    }

    @Override
    protected Component doHighlight(Component component, ComponentAdapter adapter) {
        int depth = adapter.getDepth();
        Color color = getDepthColor(depth);
        if (color != null){
            component.setForeground(color);
        }
        return component;
    }

    @Override
    public Component getTreeCellRendererComponent(JTree tree, Object value, boolean selected, boolean expanded, boolean leaf, int row, boolean hasFocus) {

        int depth = 0;
        String text = null;
        Thesaurus thesaurus = null;
        RootThesaurus rootThesaurus;

        // Get real data
        if (value instanceof VradiTreeNode){
            VradiTreeNode node = (VradiTreeNode) value;

            rootThesaurus = ThesaurusTreeHelper.getRootThesaurus(node);
            if (rootThesaurus == null) {
                thesaurus = ThesaurusTreeHelper.getThesaurus(node);
            }

        } else if (value instanceof VradiTreeTableNode) {
            VradiTreeTableNode node = (VradiTreeTableNode) value;

            rootThesaurus = ThesaurusTreeHelper.getRootThesaurus(node);
            if (rootThesaurus == null) {
                thesaurus = ThesaurusTreeHelper.getThesaurus(node);
            }
        } else {
            return new JLabel(value.toString());
        }

        if (rootThesaurus != null) {
            text = rootThesaurus.getName();
        }
        if (thesaurus != null) {
            depth = ThesaurusDataHelper.getDepth(thesaurus);
            text = ThesaurusDataHelper.concateOrderName(thesaurus);
        }
        if (text == null){
            return new JLabel(value.toString());
        }
        Color color = getDepthColor(depth);
        JLabel jLabel = new JLabel(text);
        if (color != null){
            jLabel.setForeground(color);
        }

        // Show tooltip for thesaurus selection
        if (thesaurus != null) {
            String tooltip;
            if (isMove() || isEdit()) {
                tooltip = ToolTipHelper.getThesaurusDescriptionToolTip(thesaurus);
            } else {
                tooltip = ToolTipHelper.getThesaurusToolTip(thesaurus);
            }
            if (tooltip != null && !tooltip.trim().isEmpty()) {
                jLabel.setToolTipText(tooltip);
            }
        }

        return jLabel;
    }

    @Override
    public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
        int depth = 0;
        String text;
        String namePath = null;
        Thesaurus thesaurus = null;
        if (value instanceof VradiTreeNode){
            VradiTreeNode node = (VradiTreeNode) value;

            RootThesaurus rootThesaurus = ThesaurusTreeHelper.getRootThesaurus(node);
            if (rootThesaurus != null) {
                text = rootThesaurus.getName();
            } else {
                thesaurus = ThesaurusTreeHelper.getThesaurus(node);
                if (thesaurus == null) {
                    return new JLabel(value.toString());
                }
                depth = ThesaurusDataHelper.getDepth(thesaurus);
                text = ThesaurusDataHelper.concateOrderName(thesaurus);
            }

        } else if (value instanceof Thesaurus){
            thesaurus = (Thesaurus)value;
            depth = ThesaurusDataHelper.getDepth(thesaurus);
            text = ThesaurusDataHelper.concateOrderName(thesaurus);
            namePath = ThesaurusDataHelper.getNamePath(thesaurus, "/");
        } else {
            return new JLabel(value.toString());
        }
        if (text == null){
            return new JLabel(value.toString());
        }
        Color color = getDepthColor(depth);
        if (isProposition()){
            text = namePath;
        }
        JLabel jLabel = (JLabel)delegate.getListCellRendererComponent(list, text, index, isSelected, cellHasFocus);
        if (color != null){
            jLabel.setForeground(color);
        }

        // Show tooltip for thesaurus proposition and thesaurus selected list
        if (thesaurus != null) {
            jLabel.setToolTipText(ToolTipHelper.getThesaurusToolTip(thesaurus));
        }
        return jLabel;
    }
    
    /**
     * Get colors for depth in map.
     * 
     * Default to black.
     * 
     * @param depth depth to get color
     * @return color at depth
     */
    public Color getDepthColor(int depth) {
        Color color = ColorsMap.getInstance().get(depth);
        
        // default color
        if (color == null) {
            color = Color.BLACK;
        }
        return color;
    }
}
