/* *##%
 * Copyright (c) 2009 poussin. All rights reserved.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.sharengo.wikitty;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.FieldType.TYPE;
import org.sharengo.wikitty.search.And;
import org.sharengo.wikitty.search.BinaryOperator;
import org.sharengo.wikitty.search.Element;
import org.sharengo.wikitty.search.Restriction;

/**
 * In memory implementation of WikittyService, currently used for test only
 *
 * @author poussin
 * @version $Revision$
 *
 * Last update: $Date$
 * by : $Author$
 */
public class WikittyServiceInMemory extends AbstractWikittyService {

	//FIXME InMemory implementation is not usable for production. Must be reviewed.
	//FIXME The version increment must be done in 'prepare' method

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyServiceInMemory.class);

    @Override
    public List<WikittyExtension> getAllExtensions(boolean lastVersion) {
        return null;  //To change body of implemented methods use File | Settings | File Templates.
    }

    class WikittyStorageInMemory implements WikittyStorage {
        protected Map<String, Wikitty> wikitties = new LinkedHashMap<String, Wikitty>();

        public List<Command> prepare(WikittyTransaction transaction,
                Collection<Wikitty> wikitties,
        		boolean disableAutoVersionIncrement) throws WikittyException {

            List<Command> result = new ArrayList<Command>(wikitties.size());
            for (Wikitty w : wikitties) {
                result.add(new CommandInMemory(w, true));
            }
            return result;
        }

        public UpdateResponse commit(WikittyTransaction transaction,
                List<Command> wikittyStorageCommandList) {
            UpdateResponse result = new UpdateResponse();
            Date now = new Date();
            for (Command c : wikittyStorageCommandList) {
                CommandInMemory cim = (CommandInMemory)c;
                if (cim.toStore) {
                    cim.wikitty.version = WikittyUtil.incrementMajorRevision(
                            cim.wikitty.version);
                    cim.wikitty.fieldDirty.clear();
                    result.addVersionUpdate(cim.wikitty.getId(), cim.wikitty.version);
                } else {
                    cim.wikitty.setDeleteDate(now);
                    result.addDeletionDateUpdate(cim.wikitty.getId(), now);
                }
                wikitties.put(cim.wikitty.getId(), cim.wikitty);
            }
            return result;
        }

        public boolean exists(String id) {
            boolean result = wikitties.containsKey(id);
            return result;
        }

        public boolean isDeleted(String id) {
            boolean result = false;
            Wikitty w = wikitties.get(id);
            if (w == null) {
                throw new WikittyException(String.format("No wikitty with id '%s'", id));
            } else {
                result = w.isDeleted();
            }
            return result;
        }

        public Wikitty restore(String id, String ... fqFieldName) throws WikittyException {
            Wikitty result = wikitties.get(id);
            if (result.isDeleted()) {
                result = null;
            }
            if (result == null) {
                throw new WikittyException(String.format("No wikitty with id '%s'", id));
            }
            return result;
        }

        public List<Wikitty> restore(Collection<String> ids,
                String ... fqFieldName) throws WikittyException {
            List<Wikitty> result = new ArrayList<Wikitty>(ids.size());
            for (String id : ids) {
                Wikitty w = restore(id, fqFieldName);
                result.add(w);
            }
            return result;
        }

        public List<Command> delete(List<String> idList) throws WikittyException {
            List<Command> result = new ArrayList<Command>(idList.size());
            for (String id : idList) {
                Wikitty w = restore(id);
                result.add(new CommandInMemory(w, false));
            }
            return result;
        }

        class CommandInMemory implements Command {
            /** if true command is for storage, if false for deletion */
            boolean toStore;
            Wikitty wikitty;
            public CommandInMemory(Wikitty wikitties, boolean toStore) {
                this.wikitty = wikitties;
                this.toStore = toStore;
            }
        }
    }

    class WikittyExtensionStorageInMemory implements WikittyExtensionStorage {
        protected Map<String, WikittyExtension> extensions;

        public WikittyExtensionStorageInMemory() {
            this.extensions = new HashMap<String, WikittyExtension>();
        }

        public List<Command> prepare(WikittyTransaction transaction,
                Collection<WikittyExtension> extensions) throws WikittyException {
            List<Command> result = new ArrayList<Command>(extensions.size());
            for (WikittyExtension ext : extensions) {
                CommandInMemory command = new CommandInMemory(ext);
                result.add(command);
            }
            return result;
        }

        public UpdateResponse commit(WikittyTransaction transaction,
                List<Command> extensionStorageCommandList) {
            for (Command c : extensionStorageCommandList) {
                CommandInMemory cim = (CommandInMemory)c;
                WikittyExtension ext = cim.ext;
                extensions.put(ext.getId(), ext);
            }

            // nothing to do in UpdateResponse
            UpdateResponse result = new UpdateResponse();
            return result;
        }

        public boolean exists(String id) {
            boolean result = extensions.containsKey(id);
            return result;
        }

        public List<String> getAllExtensionIds() {
            List<String> result = new ArrayList<String>(extensions.keySet());
            return result;
        }

        @Override
        public List<WikittyExtension> getAllExtensions(boolean lastVersion) {
            return null;  //To change body of implemented methods use File | Settings | File Templates.
        }

        public WikittyExtension restore(String id) throws WikittyException {
            WikittyExtension result = extensions.get(id);
            if (result == null) {
                throw new WikittyException(String.format("No extension with id '%s'", id));
            }
            return result;
        }

        public String getLastVersion(String extName) {
            throw new UnsupportedOperationException("Not supported yet.");
        }

        class CommandInMemory implements Command {
            WikittyExtension ext;
            public CommandInMemory(WikittyExtension ext) {
                this.ext = ext;
            }
        }
    }

    class WikittySearchEngineInMemory implements WikittySearchEngine {

        WikittyStorageInMemory wikittyStorage;

        public WikittySearchEngineInMemory(WikittyStorageInMemory wikittyStorage) {
            this.wikittyStorage = wikittyStorage;
        }

        public void clear() {
            // do nothing
        }

        @Override
        public void changeDataDir(String newDataDir, String oldDataDir) {
            //Do nothing
        }

        public List<Command> prepare(WikittyTransaction transaction,
                Collection<Wikitty> wikitties) {
            // do nothing
            return new ArrayList<Command>();
        }

        public UpdateResponse commit(WikittyTransaction transaction,
                List<Command> wikittyIndexationCommandList) {
            // do nothing
            return new UpdateResponse();
        }

        public List<Command> delete(WikittyTransaction transaction,
                List<String> idList) throws WikittyException {
            // do nothing
            return new ArrayList<Command>();
        }

        public boolean checkRestriction( Restriction restriction, Wikitty w ) {
        	if ( restriction instanceof BinaryOperator ) {
        		BinaryOperator binOp = (BinaryOperator) restriction;

        		if (  binOp.getElement().equals(Element.ELT_EXTENSION) ) {
        			return w.hasExtension( (String) binOp.getValue() );
        		}

        		String fqfieldName = binOp.getElement().getName();
        		Object o = w.getFqField( fqfieldName );
        		FieldType t = w.getFieldType(fqfieldName);
        		Object value = t.getValidValue( binOp.getValue() );
        		boolean checked = false;
            	switch( restriction.getName() ) {
            	case EQUALS:
            		checked = value.equals(o);
            		break;
            	case LESS:
            		checked = ((Comparable)o).compareTo( value ) < 0;
            		break;
            	case LESS_OR_EQUAL:
            		checked = ((Comparable)o).compareTo( value ) <= 0;
            		break;
            	case GREATER:
                	checked = ((Comparable)o).compareTo( value ) > 0;
            		break;
            	case GREATER_OR_EQUAL:
            		checked = ((Comparable)o).compareTo( value ) >= 0;
            		break;
            	case NOT_EQUALS:
            		checked = !value.equals(o);
            		break;
            	case ENDS_WITH:
            		if ( t.getType() != TYPE.STRING )
            			throw new WikittyException("Can't search for contents that 'ends with' on attribute type different of String. " +
            					"Attribute " + fqfieldName + " is " + t.getType().name() );
            		checked = ((String) o).endsWith( (String) value );
            		break;
            	case STARTS_WITH:
            		if ( t.getType() != TYPE.STRING )
            			throw new WikittyException("Can't search for contents that 'starts with' on attribute type different of String. " +
            					"Attribute " + fqfieldName + " is " + t.getType().name() );
            		checked = ((String) o).startsWith( (String) value );
            		break;
            	}
            	return checked;
        	} else if ( restriction instanceof And ) {
        		And and = (And) restriction;
        		for ( Restriction sub : and.getRestrictions() ) {
        			if ( !checkRestriction(sub, w) ) return false;
        		}
        		return true;
        	} else {
        		throw new UnsupportedOperationException( restriction.getName() + " Search Not yet implemented");
        	}
        }

        public PagedResult<String> findAllByCriteria(
                Criteria criteria) {
            // throw new UnsupportedOperationException("Not supported yet.");

        	int firstIndex = criteria.getFirstIndex();
        	int endIndex = criteria.getEndIndex();

        	List<String> ids = new LinkedList<String>();
        	int currentIndex = 0;

            for( Entry<String, Wikitty> entry : wikittyStorage.wikitties.entrySet() ) {
            	Wikitty w = entry.getValue();
            	String id = entry.getKey();
            	Restriction dto = criteria.getRestriction();

            	if ( checkRestriction(dto, w) ) {
            		currentIndex++;
            		if ( currentIndex > firstIndex ) ids.add( id );
            		if ( endIndex >= 0 && currentIndex >= endIndex ) break;
            	}

            }
            return  new PagedResult<String>(firstIndex, endIndex, criteria.getRestriction().toString(), null, ids );
        }

        public PagedResult<String> findAllByCriteria(
                Criteria criteria, int firstIndex, int endIndex, String... fieldFacet) {
            throw new UnsupportedOperationException("Not supported yet.");
        }

        public PagedResult<String> findAllByCriteria(
                Criteria criteria, int firstIndex, int endIndex, Criteria... criteriaFacet) {
            throw new UnsupportedOperationException("Not supported yet.");
        }

        public Map<String, Integer> findAllChildren(Wikitty w) {
            throw new UnsupportedOperationException("Not supported yet.");
        }

    }

    /**
     * if persitenceFile is not null, serialize all data to disk during store
     * operation and the file is reloaded during init
     */
    protected File persitenceFile = null;

    protected WikittyStorage wikittyStorage;
    protected WikittyExtensionStorage extensionStorage;
    protected WikittySearchEngine searchEngine;

    public WikittyServiceInMemory() {
        extensionStorage = new WikittyExtensionStorageInMemory();
        wikittyStorage = new WikittyStorageInMemory();
        searchEngine = new WikittySearchEngineInMemory((WikittyStorageInMemory)wikittyStorage);
    }

    public WikittyServiceInMemory(File persitenceFile) {
        this();
        this.persitenceFile = persitenceFile;
        if (persitenceFile != null && persitenceFile.exists()) {
            try {
                ObjectInputStream in = new ObjectInputStream(new FileInputStream(
                        persitenceFile));
                ((WikittyExtensionStorageInMemory)extensionStorage).extensions = (Map) in.readObject();
                ((WikittyStorageInMemory)wikittyStorage).wikitties = (Map) in.readObject();
                in.close();
            } catch (Exception eee) {
                log.error("Can't read data file " + persitenceFile, eee);
            }
        }
    }

    @Override
    protected WikittyStorage getWikittyStorage() {
        return wikittyStorage;
    }

    @Override
    protected WikittyExtensionStorage getExtensionStorage() {
        return extensionStorage;
    }

    @Override
    protected WikittySearchEngine getSearchEngine() {
        return searchEngine;
    }

    @Override
    protected void finalize() throws Throwable {
        saveToPersistenceFile();

        super.finalize();
    }

    public void saveToPersistenceFile() {
        if (persitenceFile != null) {
            try {
                ObjectOutputStream out = new ObjectOutputStream(new FileOutputStream(
                        persitenceFile));
                out.writeObject(((WikittyExtensionStorageInMemory)extensionStorage).extensions);
                out.writeObject(((WikittyStorageInMemory)wikittyStorage).wikitties);
                out.close();
            } catch (IOException eee) {
                log.error("Can't write data file " + persitenceFile, eee);
            }
        }
    }

}