/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: SearchManager.java 1857 2010-12-23 16:51:08Z sletellier $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.5.2/vradi-services/src/main/java/com/jurismarches/vradi/services/managers/SearchManager.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.services.managers;

import com.jurismarches.vradi.beans.FormPagedResult;
import com.jurismarches.vradi.beans.QueryBean;
import com.jurismarches.vradi.beans.QueryParameters;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.ModificationTag;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.entities.RootThesaurus;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.search.CompareFilter;
import com.jurismarches.vradi.services.search.Filter;
import com.jurismarches.vradi.services.search.FilterList;
import com.jurismarches.vradi.services.search.RangeFilter;
import com.jurismarches.vradi.services.search.UnsupportedQueryException;
import com.jurismarches.vradi.services.search.VradiQueryParser;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.DateUtil;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyUtil;
import org.nuiton.wikitty.entities.WikittyTreeNode;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.search.Search;
import org.nuiton.wikitty.search.operators.Element;
import org.nuiton.wikitty.search.operators.Like;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;
import org.nuiton.wikitty.storage.solr.WikittySolrConstant;

/**
 * Class containing the methods to manage the form research
 *
 * @author schorlet
 * @date 2010-01-29 12:40:26
 * @version $Revision: 1857 $ $Date: 2010-12-23 17:51:08 +0100 (jeu., 23 déc. 2010) $
 */
public class SearchManager {
    private static final Log log = LogFactory.getLog(SearchManager.class);

    protected WikittyProxy wikittyProxy;
    protected ThesaurusManager thesaurusManager;

    private static final DateFormat frenchDateFormat = DateFormat.getDateInstance(
            DateFormat.SHORT, Locale.FRANCE);
    
    private static final DateFormat numericDateFormat = new SimpleDateFormat("yyyyMMdd", Locale.FRANCE);

    /**
     * Match dd/mm/yyyy, requiring leading zeros.
     * 
     * Match the pattern but not the validity of the date, so that <code>01/01/0000</code> will match.
     */
    private static final Pattern frenchDateFormatPattern =
            Pattern.compile("^(3[01]|[12][0-9]|0[1-9])/(1[0-2]|0[1-9])/[0-9]{4}$");

    /**
     * Match yyyymmdd, requiring leading zeros.
     */
    private static final Pattern numericDateFormatPattern =
            Pattern.compile("^[0-9]{4}(1[0-2]|0[1-9])(3[01]|[12][0-9]|0[1-9])$");

    /**
     * Match UUID pattern, eg. 32 hexadecimal digits, displayed in 5 groups separated by hyphens,
     * in the form 8-4-4-4-12 for a total of 36 characters.
     */
    private static final Pattern uuidPattern =
            Pattern.compile("^[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}$");

    /**
     * Match yyyy-mm-ddUUID pattern.
     */
    private static final Pattern formIdPattern =
            Pattern.compile("^[0-9]{4}-(1[0-2]|0[1-9])-(3[01]|[12][0-9]|0[1-9])([a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12})$");

    /** Alias pour {@link ModificationTag#FIELD_MODIFICATIONTAG_LASTMODIFIER}. */
    public static final String ALIAS_LAST_MODIFIER = "modificateur";

    /** Alias pour {@link ModificationTag#FIELD_MODIFICATIONTAG_LASTSTATUSMODIFIER}. */
    public static final String ALIAS_LAST_STATUS_MODIFIER = "modificateur.status";

    /**
     * Search manager constructor.
     * 
     * @param wikittyProxy wikitty proxy
     * @param thesaurusManager
     */
    public SearchManager(WikittyProxy wikittyProxy, ThesaurusManager thesaurusManager) {
        this.wikittyProxy = wikittyProxy;
        this.thesaurusManager = thesaurusManager;

        // FIXME sletellier 23/12/10 : hack for maven analyse-depency, constant are not concidered like reference,
        // So wikitty-solr dependency is concidred like unused
        new WikittySolrConstant();
    }
    
    protected FormPagedResult findForms(String query, Criteria filter, FormPagedResult formPagedResult)
            throws UnsupportedQueryException, VradiException {

        // string query to FilterList structure
        FilterList queryParsed = VradiQueryParser.parse(query);

        // FilterList to wikitty Search
        Search search = Search.query(filter);
        buildSearch(queryParsed, search);
        Criteria criteria = search.criteria();

        // add index restriction
        // last index can be -1 (all value in UI)
        int firstIndex = (formPagedResult.getPageToShow() - 1) * formPagedResult.getNbFormsToShow();
        int lastIndex = formPagedResult.getPageToShow() * formPagedResult.getNbFormsToShow() - 1;
        if (firstIndex >= 0 && lastIndex > 0 && lastIndex > firstIndex) {
            criteria.setFirstIndex(firstIndex).setEndIndex(lastIndex);
        }

        // Add field restriction
        if (formPagedResult.getFieldToSort() != null) {
            if (!formPagedResult.isAscending()) {
                criteria.addSortDescending(formPagedResult.getFieldToSort());
            } else {
                criteria.addSortAscending(formPagedResult.getFieldToSort());
            }
        }
        criteria.addSortAscending(Form.FQ_FIELD_INFOGENE_ID);

        //finds the forms
        PagedResult<Form> queryResult = wikittyProxy.findAllByCriteria(Form.class, criteria);
        if (log.isDebugEnabled()) {
            log.debug("[findForms] found: " + queryResult.size() + " forms");
        }

        List<Form> result = new ArrayList<Form>(queryResult.getAll());
        FormPagedResult formPageResult = new FormPagedResult(result,
                queryResult.getNumFound(), formPagedResult.getPageToShow(),
                formPagedResult.getNbFormsToShow());

        return formPageResult;
    }

    public FormPagedResult findForms(QueryParameters queryParameters, FormPagedResult formPagedResult)
        throws UnsupportedQueryException, VradiException {

        if (log.isDebugEnabled()) {
            log.debug(String.format(
                    "findForms(query:%s, extension:%s, dateFieldName:%s, beginDate:%tc, endDate:%tc, status:%s)",
                    queryParameters.getQuery(), queryParameters.getExtension(),
                    queryParameters.getDateFieldName(), queryParameters.getBeginDate(),
                    queryParameters.getEndDate(), queryParameters.getStatusIds()));
        }

        Criteria filter = createFilter(queryParameters);
        FormPagedResult formPageResult = findForms(queryParameters.getQuery(), filter, formPagedResult);
        return formPageResult;
    }

    public Map<Thesaurus, Integer> getChildrenCartography(String thesaurusId, QueryParameters queryParameters)
            throws VradiException, UnsupportedQueryException {

        // create search
        Criteria filter = createFilter(queryParameters);

        // string query to FilterList structure
        FilterList queryParsed = VradiQueryParser.parse(queryParameters.getQuery());

        // FilterList to wikitty Search
        Search search = Search.query(filter);
        buildSearch(queryParsed, search);
        Criteria criteria = search.criteria();

        Map<Thesaurus, Integer> results = wikittyProxy.restoreChildren(Thesaurus.class, thesaurusId, criteria);
        return results;
    }

    public Map<Group, List<QueryBean>> findQueriesReturningForm(String formId)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("findQueriesReturningForm(formId)");
        }

        Map<Group, List<QueryBean>> results = new HashMap<Group, List<QueryBean>>();

        if (formId == null) {
            return results;
        }

        // find QueryMaker which do have queries defined
        List<Group> groups = findGroupsWithQueries();

        results = findQueriesReturningForm(groups, formId);

        if (log.isDebugEnabled()) {
             log.debug("Group found : " + results.size());
        }

        return results;
    }

    public Map<Group, List<QueryBean>> findQueriesReturningForm(List<Group> groups, String formId)
            throws VradiException {

        Map<Group, List<QueryBean>> results = new HashMap<Group, List<QueryBean>>();

        for (Group group : groups) {
            Set<String> queries = group.getQueries();

            for (String query : queries) {
                if (log.isDebugEnabled()) {
                    log.debug("Searching with query : " + query);
                }
                try {
                    QueryBean bean = new QueryBean(query, group.getWikittyId());
                    String realQuery = bean.getQuery();
                    FilterList filter = VradiQueryParser.parse(realQuery);

                    Search search = Search.query();
                    search.eq(Element.ELT_ID, formId);
                    buildSearch(filter, search);

                    Criteria criteria = search.criteria();

                    PagedResult<Form> forms = wikittyProxy.findAllByCriteria(Form.class, criteria);

                    if (log.isDebugEnabled()) {
                        log.debug("Search for group " + group.getName() +
                                " return " + forms.getNumFound() + " results : \n\n" + criteria);
                    }

                    if (forms.getNumFound() > 0) {
                        if (results.containsKey(group)) {
                            results.get(group).add(bean);

                        } else {
                            List<QueryBean> list = new ArrayList<QueryBean>();
                            list.add(bean);
                            results.put(group, list);
                        }
                    }

                } catch (Exception eee) {
                    throw new VradiException("Cant get groups", eee);
                }
            }
        }
        return results;
    }

    /**
     * Find all groups with non null queries.
     * 
     * Warning : do not return {@code QueryMaker} instance.
     * 
     * @return groups
     */
    public List<Group> findGroupsWithQueries() {
        // find QueryMaker which do have queries defined
        Criteria criteria = Search.query().bw(QueryMaker.FQ_FIELD_QUERYMAKER_QUERIES, "*", "*").criteria();
        PagedResult<Group> pagedResult = wikittyProxy.findAllByCriteria(Group.class, criteria);
        List<Group> all = pagedResult.getAll();
        return all;
    }

    /**
     * Build wikitty {@link Search} criteria with all {@link QueryParameters}
     * attribute (excepted {@link QueryParameters#query}).
     * 
     * All parameters can be {@code null}.
     * 
     * Transform first query to {@link FilterList} and transform this
     * {@link FilterList} into wikitty {@link Search}. Then add all parameter
     * to {@link Search} query.
     * 
     * @param queryParameters query parameters
     * @return wikitty search query
     * @throws UnsupportedQueryException
     * @throws VradiException 
     */
    protected Criteria createFilter(QueryParameters queryParameters)
            throws UnsupportedQueryException, VradiException {

        // FilterList to wikitty Search
        Search search = Search.query();
        boolean addedFilter = false;

        // add the extension in the criteria
        if (queryParameters.getExtension() != null) {
            search.eq(Element.ELT_EXTENSION, queryParameters.getExtension().getName());
            addedFilter = true;
        }

        // add the date in the criteria
        if (queryParameters.getDateFieldName() != null && queryParameters.getBeginDate()
                != null && queryParameters.getEndDate() != null) {
            String beginString = null;
            String endString = null;
            try {
                beginString = WikittyUtil.formatDate(queryParameters.getBeginDate());
                endString = WikittyUtil.formatDate(queryParameters.getEndDate());
            } catch (ParseException eee) {
                throw new VradiException("Cant format date : ", eee);
            }
            search.bw(queryParameters.getDateFieldName(), beginString, endString);
            addedFilter = true;
        }

        //add the thesaurus in the criteria
        if (queryParameters.getThesaurus() != null) {
            for (Thesaurus thesaurus : queryParameters.getThesaurus()) {
                Search subSearch = search.or();
                subSearch.eq(WikittySolrConstant.TREENODE_PREFIX + thesaurus.getParent(), thesaurus.getWikittyId());
                addedFilter = true;
            }
        }

        // add the status in the criteria
        if (CollectionUtils.isNotEmpty(queryParameters.getStatusIds())) {
            Search subSearch = search.or();
            for (String statusId : queryParameters.getStatusIds()) {
                subSearch.eq(Form.FQ_FIELD_INFOGENE_STATUS, statusId);
                addedFilter = true;
            }
        }

        // add the xmlStreams in the criteria
        if (CollectionUtils.isNotEmpty(queryParameters.getStreamIds())) {
            addedFilter = true;
            Search subSearch = search.or();
            for (String streamId : queryParameters.getStreamIds()) {
                subSearch.eq(Form.FQ_FIELD_FORM_XMLSTREAM, streamId);
                addedFilter = true;
            }
        }

        Criteria criteria = null;
        if (addedFilter) {
            criteria = search.criteria();
        }
        return criteria;
    }

    /**
     * Transform FilterList part into wikitty sub Search.
     * 
     * This method recursively explore FilterList, and recursively call
     * himself.
     * 
     * @param list current filter list
     * @param search parent wikitty search to add new created search
     * @throws VradiException 
     */
    protected void buildSearch(FilterList list, Search search) throws VradiException {

        // create new sub search
        FilterList.Operator operator = list.getOperator();
        Search subSearch = null;
        switch (operator) {
            case FILTER_OR :
                subSearch = search.or();
                break;
            case FILTER_AND :
                subSearch = search.and();
                break;
            case FILTER_NOT :
                subSearch = search.not();
                break;
        }

        // transform each filters into wikitty
        List<Filter> filters = list.getFilters();
        for (Filter filter : filters) {
            if (filter instanceof FilterList) {
                buildSearch((FilterList) filter, subSearch);
        
            } else if (filter instanceof RangeFilter) {
                buildRangeSearch((RangeFilter) filter, subSearch);
                
            } else if (filter instanceof CompareFilter) {
                buildCompareSearch(operator, (CompareFilter) filter, subSearch);
            }
            
        }
    }

    /**
     * Build wikitty range search.
     * 
     * @param rangeFilter vradi range filter
     * @param search wikitty range search
     */
    protected void buildRangeSearch(RangeFilter rangeFilter, Search search) {

        String name = rangeFilter.getName();
        String lowerValue = rangeFilter.getLowerValue();
        String upperValue = rangeFilter.getUpperValue();

        Search subSearch = search.or();

        // date search
        if (rangeFilter.match(numericDateFormatPattern)) {
            try {
                Date lowerTime = numericDateFormat.parse(lowerValue);
                lowerTime = DateUtil.setMinTimeOfDay(lowerTime);
                String lowerTimeString = WikittyUtil.formatDate(lowerTime);

                Date upperTime = numericDateFormat.parse(upperValue);
                upperTime = DateUtil.setMaxTimeOfDay(upperTime);
                String upperTimeString = WikittyUtil.formatDate(upperTime);

                subSearch.bw(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                        + name + Criteria.SEPARATOR
                        + Element.ElementType.DATE, lowerTimeString, upperTimeString);

            } catch (ParseException e) {
                if (log.isWarnEnabled()) {
                    log.warn(lowerValue + " OR " + upperValue + " cannot be a date.");
                }
            }

        // number search
        } else if (rangeFilter.isNumber()) {
            subSearch.bw(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                    + name + Criteria.SEPARATOR
                    + Element.ElementType.NUMERIC, lowerValue, upperValue);
        }

        // TODO EC 20100622 il manque pas un else ici ?
        // default to string search
        subSearch.bw(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                + name + Criteria.SEPARATOR
                + Element.ElementType.STRING, lowerValue, upperValue);
    }

    /**
     * Convertit les termes simple (name:value).
     * 
     * @param operator
     * @param compareFilter
     * @param search
     * @throws VradiException 
     */
    protected void buildCompareSearch(FilterList.Operator operator, CompareFilter compareFilter,
            Search search) throws VradiException {
        String name = compareFilter.getName();
        String value = compareFilter.getValue();

        if (VradiQueryParser.DEFAULT_FIELD.equals(name)) {
            search.keyword(value);
            return;

        } else {

            // loop on each root thesaurus
            List<RootThesaurus> rootThesauruses = thesaurusManager.getRootThesaurus();
            for (RootThesaurus rootThesaurus : rootThesauruses) {
                if (name.equalsIgnoreCase(rootThesaurus.getName())) {
                    buildDescripteurSearch(search, rootThesaurus.getWikittyId(), value);
                    return;
                }
            }
        }
            
        if (ALIAS_LAST_MODIFIER.equals(name)) {
            name = ModificationTag.FIELD_MODIFICATIONTAG_LASTMODIFIER;

        } else if (ALIAS_LAST_STATUS_MODIFIER.equals(name)) {
            name = ModificationTag.FIELD_MODIFICATIONTAG_LASTSTATUSMODIFIER;
        }

        Search subSearch = search.or();

        // a confirmer, mais l'explication doit être :
        // - si c'est not, ca doit ne pas être la date en question
        // - sinon, ca doit être compris entre le début et la fin du jour
        //   en question
        if (operator == FilterList.Operator.FILTER_NOT) {
            // date search
            if (compareFilter.match(frenchDateFormatPattern)) {
                try {
                    Date time = frenchDateFormat.parse(value);
                    String timeString = WikittyUtil.formatDate(time);

                    subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                            + name + Criteria.SEPARATOR
                            + Element.ElementType.DATE, timeString);

                } catch (ParseException e) {
                    if (log.isWarnEnabled()) {
                        log.warn(value + " cannot be a date.");
                    }
                }
            }

        } else {
            // date search
            if (compareFilter.match(frenchDateFormatPattern)) {
                try {
                    Date time = frenchDateFormat.parse(value);

                    Date beginDate = DateUtil.setMinTimeOfDay(time);
                    Date endDate = DateUtil.setMaxTimeOfDay(time);

                    String beginString = WikittyUtil.formatDate(beginDate);
                    String endString = WikittyUtil.formatDate(endDate);

                    subSearch.bw(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                            + name + Criteria.SEPARATOR
                            + Element.ElementType.DATE, beginString, endString);

                } catch (ParseException e) {
                    if (log.isWarnEnabled()) {
                        log.warn(value + " cannot be a date.");
                    }
                }
            }
        }

        // text search
        subSearch.like(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                + name + Criteria.SEPARATOR
                + Element.ElementType.STRING, value, Like.SearchAs.AsText);

        // boolean search
        if (compareFilter.isBoolean()) {
            subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                    + name + Criteria.SEPARATOR
                    + Element.ElementType.BOOLEAN, value);

        // number search
        } else if (compareFilter.isNumber()) {
            subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                    + name + Criteria.SEPARATOR
                    + Element.ElementType.NUMERIC, value);

        // extension id search (eq)
        } else if (compareFilter.match(formIdPattern)) {
            subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                + name + Criteria.SEPARATOR
                + Element.ElementType.STRING, value);

        // extension id search (ew)
        } else if (compareFilter.match(uuidPattern)) {
            subSearch.ew(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                + name + Criteria.SEPARATOR
                + Element.ElementType.STRING, value);
        }
    }

    /**
     * Cherche tous les thesaurus en base qui correspondent à {@code value}
     * et ajoute un critère {@link Form#FQ_FIELD_THESAURUS} a la Search query
     * (ou entre tous les résultats s'il y en a plusieurs).
     * 
     * Les formulaires sont taggués avec tous les noeuds de thesaurus
     * auquel ils appartiennent depuis le noeud jusqu'a sa racine.
     * 
     * recherche : TreeNode.wikittyidparent : TreeNode.wikittyid
     *
     * FIXME sletellier 20/12/10 : doit etre independant de l'indexer solr choisi
     * 
     * @param rootThesaurusId filtering on thesaurus node of root thesaurus
     * @param search parent search
     * @param value thesaurus value (can be solr pattern with * by expemple)
     */
    protected void buildDescripteurSearch(Search search, String rootThesaurusId, String value) {

        Criteria criteria = Search.query()
            .eq(WikittyTreeNode.FQ_FIELD_WIKITTYTREENODE_NAME, value)
            .eq(Thesaurus.FQ_FIELD_THESAURUS_ROOTTHESAURUS, rootThesaurusId)
            .criteria();

        PagedResult<Thesaurus> queryResult = wikittyProxy.findAllByCriteria(Thesaurus.class, criteria);

        if (queryResult.getNumFound() > 0) {
            Search subSearch = search.or();

            List<Thesaurus> result = queryResult.getAll();
            for (Thesaurus thesaurus : result) {
                if (thesaurus.getParent() == null) {
                    subSearch.eq(WikittySolrConstant.TREENODE_ROOT, thesaurus.getWikittyId());
                }
                else {
                    subSearch.eq(WikittySolrConstant.TREENODE_PREFIX + thesaurus.getParent(), thesaurus.getWikittyId());
                }
            }
        }
    }
}
