/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: VradiDataServiceImpl.java 1844 2010-12-22 14:30:09Z sletellier $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.5.1/vradi-services/src/main/java/com/jurismarches/vradi/services/VradiDataServiceImpl.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package com.jurismarches.vradi.services;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.VradiServiceConfiguration;
import com.jurismarches.vradi.VradiServiceConfigurationHelper;
import com.jurismarches.vradi.beans.QueryBean;
import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.Session;
import com.jurismarches.vradi.services.managers.BindingManager;
import com.jurismarches.vradi.services.managers.ClientManager;
import com.jurismarches.vradi.services.managers.FormManager;
import com.jurismarches.vradi.services.managers.FormTypeManager;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.entities.BusinessEntity;
import org.nuiton.wikitty.entities.FieldType;
import org.nuiton.wikitty.entities.WikittyExtension;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.search.Search;

import com.jurismarches.vradi.beans.FormPagedResult;
import com.jurismarches.vradi.beans.QueryParameters;
import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.RootThesaurus;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.entities.WebHarvestStream;
import com.jurismarches.vradi.entities.XmlFieldBinding;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.managers.SearchManager;
import com.jurismarches.vradi.services.managers.ThesaurusManager;
import com.jurismarches.vradi.services.search.UnsupportedQueryException;
import org.nuiton.wikitty.search.operators.Element;

/**
 * Vradi data service implementation (on a local wikiity proxy).
 * 
 * @author chatellier
 * @version $Revision: 1844 $
 * 
 * Last update : $Date: 2010-12-22 15:30:09 +0100 (mer., 22 déc. 2010) $
 * By : $Author: sletellier $
 */
public class VradiDataServiceImpl implements VradiDataService {

    /** Log.*/
    private static final Log log = LogFactory.getLog(VradiDataServiceImpl.class);

    protected ApplicationConfig config;

    /** Wikiity proxy. */
    protected WikittyProxy wikittyProxy;

    protected ThesaurusManager thesaurusManager;
    protected SearchManager searchManager;
    protected FormManager formManager;
    protected FormTypeManager formTypeManager;
    protected BindingManager bindingManager;
    protected ClientManager clientManager;

    /**
     * Constructor with a wikitty proxy.
     * 
     * Make sure that wikitty proxy doesn't access remote service.
     * 
     * @param wikittyProxy wikitty proxy
     */
    public VradiDataServiceImpl(WikittyProxy wikittyProxy) {
        this(VradiServiceConfiguration.getInstance(), wikittyProxy);
    }
    
    /**
     * Constructor with a wikitty proxy.
     *
     * Make sure that wikitty proxy doesn't access remote service.
     *
     * @param config vradi configuration
     * @param wikittyProxy wikitty proxy
     */
    public VradiDataServiceImpl(ApplicationConfig config, WikittyProxy wikittyProxy) {
        this.config = config;
        this.wikittyProxy = wikittyProxy;
        initManagers();
    }

    /**
     * Init managers.
     */
    protected void initManagers() {

        thesaurusManager = new ThesaurusManager(wikittyProxy);
        searchManager = new SearchManager(wikittyProxy, thesaurusManager);
        formManager = new FormManager(config, wikittyProxy, thesaurusManager, searchManager);
        formTypeManager = new FormTypeManager(config, wikittyProxy);
        clientManager = new ClientManager(config, wikittyProxy);
        bindingManager = new BindingManager(config, wikittyProxy, formTypeManager, formManager);
    }

    /**
     * Do a find all by applying a criteria to WikittyProxy.
     * 
     * Example :
     * List<Client> client = VradiServiceHelper.findAllByExtension(Client.class, Client.EXT_CLIENT)
     * 
     * @param type type to search
     * @param extension extension to search
     * @param sortFields sort fields (ascending to sort on)
     * @param <T> type of result
     * 
     * @return all of entities with extension
     */
    protected <T extends BusinessEntity> List<T> findAllByExtension(Class<T> type, String extension, String... sortFields) {
        
        Search search = Search.query().eq(Element.ELT_EXTENSION, extension);
        Criteria criteria = search.criteria();

        // set sort order
        for (String sortField : sortFields) {
            criteria.addSortAscending(sortField);
        }

        PagedResult<T> findAllByCriteria = wikittyProxy.findAllByCriteria(type, criteria);
        List<T> results = findAllByCriteria.getAll();
        
        return new ArrayList<T>(results);
    }

    /**
     * Find all clients.
     *
     * @return all clients
     */
    @Override
    public List<Client> findAllClients() {
        return findAllByExtension(Client.class, Client.EXT_CLIENT, Client.FQ_FIELD_CLIENT_NAME);
    }

    /**
     * Find all users.
     *
     * @return all users
     */
    @Override
    public List<User> findAllUsers() {

        //FIXME JC 25/08/2010 Should also not return user that have id of a non existing client

        //Create query to return only User that have clients. Users with no
        //clients should not exist and are corrupted.
        Search search = Search.query().eq(User.FQ_FIELD_USER_CLIENT, "*");
        Criteria criteria = search.criteria();

        PagedResult<User> users = wikittyProxy.findAllByCriteria(User.class,
                criteria);

        return new ArrayList<User>(users.getAll());
    }

    /**
     * Find all groups.
     *
     * @return all groups
     */
    @Override
    public List<Group> findAllGroups() {
        return findAllByExtension(Group.class, Group.EXT_GROUP, Group.FQ_FIELD_GROUP_NAME);
    }

    /**
     * Find all status.
     *
     * @return all status
     */
    @Override
    public List<Status> findAllStatus() {
        return findAllByExtension(Status.class, Status.EXT_STATUS, Status.FQ_FIELD_STATUS_NAME);
    }

    @Override
    public List<Group> getGroupsForUser(String userId) throws VradiException {
        Search search = Search.query().eq(Group.FQ_FIELD_GROUP_USER, userId);
        PagedResult<Group> groups = wikittyProxy.findAllByCriteria(Group.class, search.criteria());
        return groups.getAll();
    }

    @Override
    public boolean deleteStatus(List<String> statusIds) throws VradiException {
        return formManager.deleteStatuses(statusIds);
    }
    
    /**
     * Get the status passed in parameter. Init status if it have not been done
     * already
     *
     * @param formStatus the status from enum
     * @return the Status object
     * @throws VradiException if an error occur during statuses initialisation
     */
    @Override
    public Status getStatus(VradiConstants.FormStatus formStatus) throws VradiException {

        List<Status> statuses = createDefaultStatuses();

        for (Status status : statuses) {
            if (status.getName().equals(formStatus.getName())) {
                return status;
            }
        }
        return null;
    }

    @Override
    public Session getSessionOfSending(String sendingId) throws VradiException {
        return formManager.getSessionOfSending(sendingId);
    }
    
    @Override
    public void deleteClient(String clientId) throws VradiException  {
        clientManager.deleteClient(clientId);
    }

    @Override
    public void deleteUser(String userId) throws VradiException  {
        clientManager.deleteUser(userId);
    }

    @Override
    public void deleteGroup(String groupId) throws VradiException {
        clientManager.deleteGroup(groupId);
    }

    @Override
    public void deleteXmlStream(String id) throws VradiException {
        wikittyProxy.delete(id);
    }

    @Override
    public Map<Group, List<QueryBean>> findQueriesReturningForm(String formId)
            throws VradiException {
        return searchManager.findQueriesReturningForm(formId);
    }

    @Override
    public Map<Group, List<QueryBean>> findQueriesReturningForm(List<Group> groups, String formId)
            throws VradiException {
        return searchManager.findQueriesReturningForm(groups, formId);
    }

    @Override
    public Map<String, FieldType> getFormTypeFields(String name) throws VradiException {
        return formTypeManager.getFormTypeFields(name);
    }

    /**
     * Create all default status.
     *
     * @return all status created
     */
    @Override
    public List<Status> createDefaultStatuses() throws VradiException {
        return formManager.createDefaultStatuses();
    }
    
    /**
     * Find all webHarvest stream.
     *
     * @return all stream webHarvest
     */
    @Override
    public List<WebHarvestStream> findAllWebHarvest() {
        return findAllByExtension(WebHarvestStream.class, WebHarvestStream.EXT_WEBHARVESTSTREAM, WebHarvestStream.FQ_FIELD_XMLSTREAM_NAME);
    }

    /**
     * Find all xml streams.
     *
     * @return all xml streams
     */
    @Override
    public List<XmlStream> findAllXmlStreams() {
        return findAllByExtension(XmlStream.class, XmlStream.EXT_XMLSTREAM, XmlStream.FQ_FIELD_XMLSTREAM_NAME);
    }

    /**
     * Find all users for a client.
     *
     * @param clientId client wikitty id
     * @return all users for client
     */
    @Override
    public List<User> findClientUsers(String clientId) {

        // Get user of client
        Search search = Search.query().eq(User.FQ_FIELD_USER_CLIENT, clientId);
        Criteria criteria = search.criteria();
        criteria.addSortAscending(User.FQ_FIELD_USER_NAME);

        PagedResult<User> usersResult = wikittyProxy.findAllByCriteria(User.class, criteria);
        List<User> users = usersResult.getAll();
        return users;
    }

    /**
     * Find all groups for a user.
     *
     * @param userId user wikitty id
     * @return groups for a user
     */
    @Override
    public List<Group> findUserGroups(String userId) {

        // Get user of client
        Search search = Search.query().eq(Group.FQ_FIELD_GROUP_USER, userId);
        Criteria criteria = search.criteria();

        PagedResult<Group> groupsResult = wikittyProxy.findAllByCriteria(Group.class, criteria);
        List<Group> groups = groupsResult.getAll();

        if (log.isDebugEnabled()) {
            log.debug(groups.size() + " groups found for user " + userId);
        }

        return groups;
    }

    /**
     * Retrieves the form type whose name is the parameter 'name'
     *
     * @param name the name of the form type we want to retrieve
     * @return the form type whose name is the parameter
     * @throws VradiException 
     */
    @Override
    public WikittyExtension getFormType(String name) throws VradiException {

        return formTypeManager.getFormType(name);
    }

    /**
     * Updates the form type whose name is 'name'.
     *
     * @param name the name of the form type to update
     * @param fields the new fields of the form type
     * @param requires the new requires of the form type
     * @param tagValues the new tag values of the form type
     * @return the form type up to date
     */
    @Override
    public WikittyExtension updateFormType(String name, Map<String, FieldType> fields,
           String requires, Map<String, String> tagValues) throws VradiException {

        return formTypeManager.updateFormType(name, fields, requires, tagValues);
    }

    /**
     * Update the form type given in parameter
     *
     * @param extension the form type to update
     * @return the form type up to date
     */
    @Override
    public WikittyExtension updateFormType(WikittyExtension extension) throws VradiException {
        return formTypeManager.updateFormType(extension, null);
    }
               
    /**
     * Update the form type given in parameter
     *
     * @param extension the form type to update
     * @return the form type up to date
     */
    public WikittyExtension updateFormType(WikittyExtension extension, String templateName) throws VradiException {

        return formTypeManager.updateFormType(extension, templateName);
    }
    
    @Override
    public FormPagedResult findForms(QueryParameters queryParameters,
            final FormPagedResult formPagedResult) throws UnsupportedQueryException, VradiException {

        FormPagedResult result = searchManager.findForms(queryParameters, formPagedResult);
        return result;
    }

    @Override
    public Map<Thesaurus, Integer> getChildrenCartography(String thesaurusId, QueryParameters queryParameters)
            throws VradiException, UnsupportedQueryException {
        
        Map<Thesaurus, Integer> thesaurusCartography = searchManager.getChildrenCartography(
                thesaurusId, queryParameters);
        
        return thesaurusCartography;
    }

    @Override
    public Map<Group, List<QueryBean>> getQueriesToModifyAfterThesaurusModification(
            String rootThesaurusName, String thesaurusName) {
        return thesaurusManager.getQueriesToModifyAfterThesaurusModification(rootThesaurusName, thesaurusName);
    }
           
    /**
     * Updates the xml field bindings given in parameters.
     *
     * @param bindings the list of the xml field bindings to update
     * @return the list of the xml field bindings up to date
     */
    @Override
    public List<XmlFieldBinding> updateXmlFieldBindings(List<XmlFieldBinding> bindings)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateXmlFieldBindings(bindings)");
        }
        
        try {
            List<XmlFieldBinding> list = new ArrayList<XmlFieldBinding>();
        
            if (bindings != null) {
                bindings = wikittyProxy.store(bindings);
                list.addAll(bindings);
            }
            
            return list;
            
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't update xml field bindings", eee);
            }
            throw new VradiException("Can't update xml field bindings", eee);
        }
    }
    
    /**
     * Retrieves all the form types
     *
     * @return a list containing all the form types
     */
    @Override
    public List<WikittyExtension> getAllFormTypes() throws VradiException {

        List<WikittyExtension> extensions = new ArrayList<WikittyExtension>();

        List<String> allExtensionIds = wikittyProxy.getAllExtensionsRequires(Form.EXT_FORM);
        Map<String, WikittyExtension> lastVersions =
                new HashMap<String, WikittyExtension>();

        for (String extensionId : allExtensionIds) {
            String extensionName  = WikittyExtension.computeName(extensionId);
            if(lastVersions.get(extensionName) == null) {
                WikittyExtension extension = wikittyProxy.restoreExtensionLastVersion(extensionName);
                extension = computeExtension(extension);
                lastVersions.put(extensionName, extension);
            }
        }
        extensions.addAll(lastVersions.values());

        return extensions;
    }

    /**
     * Compute the specified extension and return a new one with ordered fields.
     * Fields order is specified by their tag values.
     *
     * @param extension specified
     * @return extention ordered
     */
    protected WikittyExtension computeExtension(WikittyExtension extension) {

        Collection<String> fieldNames = extension.getFieldNames();
        List<String> fieldOrder = new ArrayList<String>();
        List<String> orderlessFields = new ArrayList<String>();

        for (String fieldName : fieldNames) {
            FieldType fieldType = extension.getFieldType(fieldName);

            try {
                Integer.valueOf(fieldType.getTagValue("rank"));
                fieldOrder.add(fieldName);

            } catch(Exception eee) {
                orderlessFields.add(fieldName);
            }
        }

        LinkedHashMap<String, FieldType> fields = new LinkedHashMap<String, FieldType>();

        for (String key : fieldOrder) {
            fields.put(key, extension.getFieldType(key));
        }

        for (String key : orderlessFields) {
            fields.put(key, extension.getFieldType(key));
        }

        WikittyExtension fieldOrderedExtension =
                new WikittyExtension(extension.getName(),
                        extension.getVersion(),
                        extension.getRequires(),
                        fields);
        fieldOrderedExtension.setTagValues(extension.getTagValues());

        return fieldOrderedExtension;
    }

    @Override
    public List<Thesaurus> proposeThesaurus(String formId) throws VradiException {
        return thesaurusManager.proposeThesaurus(formId);
    }

    /**
     * Return root thesaurus ids (ie thesaurus nodes without parent) sorted
     * by root thesaurus name.
     * 
     * @return a {@link RootThesaurus} collection
     * @throws VradiException
     */
    @Override
    public List<RootThesaurus> getRootThesaurus() throws VradiException {
        return thesaurusManager.getRootThesaurus();
    }

    @Override
    public int getNbFormsForThesaurus(String thesaurusId)
            throws VradiException {
        return thesaurusManager.getNbFormsForThesaurus(thesaurusId);
    }

    /**
     * Retourne l'ensemble des thesaurus qui sont attachés a un formulaire.
     * 
     * @param form form dont on veut les thesaurus
     * @return thesaurus attachés au formulaire
     */
    public List<Thesaurus> getThesaurusAttachedToForm(Form form) {
        return thesaurusManager.getThesaurusAttachedToForm(form);
    }

    /**
     * Returns all children of the specified {@code thesaurusId} sorted by name.
     * 
     * @param thesaurusId thesaurus wikitty id
     * @return all list of <code>TreeNodeImpl</code>
     * @throws VradiException
     */
    @Override
    public List<Thesaurus> getChildrenThesaurus(String thesaurusId) throws VradiException {
        return thesaurusManager.getChildrenThesaurus(thesaurusId);
    }

    @Override
    public RootThesaurus createRootThesaurus(String rootThesaurusName) throws VradiException {
        return thesaurusManager.createRootThesaurus(rootThesaurusName);
    }

    /**
     * Delete all thesaurus and sub thesaurus.
     * 
     * @param thesaurusId thesaurusId to delete
     * @throws VradiException
     */
    public Collection<String> deleteThesaurus(String thesaurusId) throws VradiException {
        return thesaurusManager.deleteThesaurus(thesaurusId);
    }
    
    /**
     * Delete all thesauruses recursively.
     * 
     * @throws VradiException
     */
    public void deleteAllThesaurus() throws VradiException {
        thesaurusManager.deleteAllThesaurus();
    }

    /**
     * Return true if thesaurusName exists in rootThesaurus children.
     * 
     * @param rootThesaurus root thesaurus
     * @param thesaurusName thesaurus name to test
     * @return true if thesaurusName exists in rootThesaurus children
     */
    public boolean isThesaurusNameExistsInRootThesaurus(RootThesaurus rootThesaurus, String thesaurusName) {
        return thesaurusManager.isThesaurusNameExistsInRootThesaurus(rootThesaurus, thesaurusName);
    }
    
    /**
     * Check if name of thesaurus is existing in rootThesaurus.
     *
     * @param rootThesaurus to check
     * @param thesaurusName name to check
     * @param exceptedThesaurusId in case of rename, exclude exceptedThesaurusId from check
     * @return true if thesaurus already exist
     */
    public boolean isThesaurusNameExistsInRootThesaurus(RootThesaurus rootThesaurus, String thesaurusName, String exceptedThesaurusId) {
        return thesaurusManager.isThesaurusNameExistsInRootThesaurus(rootThesaurus, thesaurusName, exceptedThesaurusId);
    }

    /**
     * Return true if rootThesaurusName already exists.
     * 
     * @param rootThesaurusName rootThesaurus name to test
     * @return true if rootThesaurusName already exists
     */
    public boolean isRootThesaurusNameExists(String rootThesaurusName) {
        return thesaurusManager.isRootThesaurusNameExists(rootThesaurusName);
    }
    
    /**
     * Return true if rootThesaurusName already exists.
     * 
     * @param rootThesaurusName rootThesaurus name to test
     * @param exceptedRootThesaurusId in case of rename, exclude exceptedRootThesaurusId from check
     * @return true if rootThesaurusName already exists
     */
    public boolean isRootThesaurusNameExists(String rootThesaurusName, String exceptedRootThesaurusId) {
        return thesaurusManager.isRootThesaurusNameExists(rootThesaurusName, exceptedRootThesaurusId);
    }

    /**
     * Check if a form type name already exists.
     * 
     * @param formTypeName form type name
     * @return new created wikitty extension
     * @throws VradiException if name is not valid
     */
    public boolean isFormTypeExists(String formTypeName) throws VradiException {
        return formTypeManager.isFormTypeExists(formTypeName);
    }

    /**
     * Create new form type.
     * 
     * @param formTypeName form type name
     * @return new created wikitty extension
     * @throws VradiException if name is not valid
     */
    public WikittyExtension createFormType(String formTypeName) throws VradiException {
        return formTypeManager.createFormType(formTypeName);
    }

    @Override
    public boolean checkUserEmail(Sending sending) {

        // Check users email
        String userId = sending.getUser();

        User user = wikittyProxy.restore(User.class, userId);
        if (user != null){
            String emailUser = user.getEmail();
            if (StringUtils.isBlank(emailUser)) {
                 return true;
            }
        }
        return false;
    }

    @Override
    public Session getSessionBySending(String sendingId) {

        Search query = Search.query();
        query.eq(Element.ELT_EXTENSION, Session.EXT_SESSION);

        query.contains(Session.FQ_FIELD_SESSION_SENDING, sendingId);

        Criteria criteria = query.criteria();
        Session session = wikittyProxy.findByCriteria(Session.class, criteria);
        return session;
    }

    @Override
    public List<Session> getSessions(Date sessionDate) throws VradiException {
        return formManager.getSessions(sessionDate);
    }

    @Override
    public Session getLastCloseSession() throws VradiException {
        return formManager.getLastCloseSession();
    }

    @Override
    public Session getLastOpenSession() throws VradiException {
        return formManager.getLastOpenSession();
    }

    @Override
    public List<String> getRelatedFormsLinkIds(String formId) throws VradiException {
        return formManager.getRelatedFormsLinkIds(formId);
    }

    @Override
    public List<String> getReplaceFormsLinkIds(String formId) throws VradiException {
        return formManager.getReplaceFormsLinkIds(formId);
    }

    @Override
    public boolean isReplaceForm(String formId) {
        return formManager.isReplaceForm(formId);
    }

    @Override
    public String getPreviousReplaceForm(String formId) {
        return formManager.getPreviousReplaceForm(formId);
    }

    @Override
    public List<String> getAllWebHarvestScripts() {
        File scriptDir = VradiServiceConfigurationHelper.getWebHarvestScriptDir(config);

        List<String> files = new ArrayList<String>();
        for (File file : scriptDir.listFiles()) {
            if (file.isFile()) {
                files.add(file.getName());
            }
        }

        return files;
    }
    
    @Override
    public String getWebHarvestPreviewURL(WebHarvestStream stream) throws VradiException {
        return bindingManager.getWebHarvestPreviewUrlAsString(stream);
    }

    @Override
    public String getFormTypeTemplateName(WikittyExtension extension) {
        return formTypeManager.getFormTypeTemplateName(extension);
    }
}
