/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: VradiDataService.java 1515 2010-09-28 09:41:33Z sletellier $
 * $HeadURL: svn+ssh://chatellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.2.0/vradi-services/src/main/java/com/jurismarches/vradi/services/VradiDataService.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package com.jurismarches.vradi.services;

import java.util.List;
import java.util.Map;

import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.Session;
import org.nuiton.wikitty.FieldType;
import org.nuiton.wikitty.WikittyExtension;
import org.nuiton.wikitty.WikittyProxy;

import com.jurismarches.vradi.beans.FormPagedResult;
import com.jurismarches.vradi.beans.QueryParameters;
import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.RootThesaurus;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.entities.WebHarvestStream;
import com.jurismarches.vradi.entities.XmlFieldBinding;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.search.UnsupportedQueryException;

/**
 * This service contains business code to do on data before calling
 * {@link WikittyProxy}.
 * 
 * It's intended to be used on a local {@link WikittyProxy}.
 * 
 * @author chatellier
 * @version $Revision: 1515 $
 * 
 * Last update : $Date: 2010-09-28 11:41:33 +0200 (mar., 28 sept. 2010) $
 * By : $Author: sletellier $
 */
public interface VradiDataService {

    /**
     * Find all clients ordered by client name.
     *
     * @return all clients
     */
    public List<Client> findAllClients();

    /**
     * Find all users ordered by user name.
     *
     * @return all users
     */
    public List<User> findAllUsers();
    
    /**
     * Find all users for a client ordered by user name.
     *
     * @param clientId client wikitty id
     * @return all users for client
     */
    public List<User> findClientUsers(String clientId);

    /**
     * Find user group.
     *
     * @param userId user wikitty id
     * @return all groups for user
     */
    public List<Group> findUserGroups(String userId);

    /**
     * Find all groups ordered by group name.
     *
     * @return all groups
     */
    public List<Group> findAllGroups();

    /**
     * Find all status ordered by status name.
     *
     * @return all status
     */
    public List<Status> findAllStatus();
    /**
     * Create all default status.
     *
     * @return all status
     * @throws VradiException
     */
    public List<Status> createDefaultStatuses() throws VradiException;
    
    /**
     * Find all webHarvest stream ordered by stream name.
     *
     * @return all stream webHarvest
     */
    public List<WebHarvestStream> findAllWebHarvest();

    /**
     * Find all xml streams ordered by stream name.
     *
     * @return all xml streams
     */
    public List<XmlStream> findAllXmlStreams();

    /**
     * Return root thesaurus ids (ie thesaurus nodes without parent) sorted
     * by root thesaurus name.
     * 
     * @return a {@link RootThesaurus} collection
     * @throws VradiException
     */
    List<RootThesaurus> getRootThesaurus() throws VradiException;

    /**
     * Retrieves the form type whose name is the parameter 'name'
     *
     * @param name the name of the form type we want to retrieve
     * @return the form type whose name is the parameter
     * @throws VradiException 
     */
    WikittyExtension getFormType(String name) throws VradiException;

    /**
     * Creates or Update a form type.
     *
     * @param extension the extension designing the form type
     * @return a <code>WikittyExtension</code>
     * @throws VradiException
     */
    WikittyExtension updateFormType(WikittyExtension extension) throws VradiException;

    /**
     * Creates or Update a form type.
     *
     * @param extension the extension designing the form type
     * @param template name
     * @return a <code>WikittyExtension</code>
     * @throws VradiException
     */
    WikittyExtension updateFormType(WikittyExtension extension, String templateName) throws VradiException;

    /**
     * Creates or Update a form type.
     *
     * @param name      the name of the form type
     * @param fields    the fields of the form type
     * @param requires  the extension required by the extension to update
     * @param tagValues the map containing the tags and their values
     * @return a <code>WikittyExtension</code>
     * @throws VradiException 
     */
    WikittyExtension updateFormType(String name, Map<String, FieldType> fields, String requires, Map<String, String> tagValues) throws VradiException;

    /**
     * Retourne l'ensemble des thesaurus qui sont attachés a un formulaire.
     * 
     * @param form form dont on veut les thesaurus
     * @return thesaurus attachés au formulaire
     */
    List<Thesaurus> getThesaurusAttachedToForm(Form form) throws VradiException;

    /**
     * Searches for forms with the specified filters.
     * 
     * The <code>beginDate</code> and <code>endDate</code> specifies a
     * date range filter which is applied on <code>dateType</code> field.
     * The date range filter is applied only if the three parameters are not null.
     * 
     * @param queryParameters query parameters
     * @return FormPagedResult
     * 
     * @throws UnsupportedQueryException
     * @throws VradiException
     */
    FormPagedResult findForms(QueryParameters queryParameters, FormPagedResult formPagedResult)
            throws VradiException, UnsupportedQueryException;
    
    /*
     * Computes the number of forms associated with every thesaurus node.
     * 
     * @param query     the query filter
     * @param extension the extension filter
     * @param dateType  the date field. Must be fully qualified (extension.field)
     * @param beginDate the begin date filter
     * @param endDate   the end date filter
     * @param streamId  id of stream
     * @param thesaurus the thesaurus filter
     * @param statusIds  the ids of the status filters
     * @return a <code>VradiCartographyDTO</code> containing forms result and a map of form count indexed by thesaurus nodes
     * @throws VradiException
     * @throws UnsupportedQueryException
     *
    ThesaurusCartography getThesaurusCartography(String query,
            WikittyExtension extension, String dateType, Date beginDate,
            Date endDate, String streamId, java.util.List<String> thesaurus, String[] statusIds)
            throws VradiException, UnsupportedQueryException;*/

    /**
     * Computes the number of forms associated with every thesaurus node.
     * 
     * @param thesaurusId thesaurusId
     * @param queryParameters query parameters
     * @return a map with children thesaurus and associated form count
     * @throws VradiException
     * @throws UnsupportedQueryException
     */
    Map<Thesaurus, Integer> getChildrenCartography(String thesaurusId, QueryParameters queryParameters)
            throws VradiException, UnsupportedQueryException;

    /**
     * Update a list of <code>XmlFieldBinding</code>.
     *
     * @param bindings the list of XmlFieldBinding to update
     * @return the list of updated XmlFieldBinding
     * @throws VradiException 
     */
    List<XmlFieldBinding> updateXmlFieldBindings(List<XmlFieldBinding> bindings) throws VradiException;
    
    /**
     * Retrieves all the form types
     *
     * @return a list containing all the form types
     * @throws VradiException 
     */
    public List<WikittyExtension> getAllFormTypes() throws VradiException;
    
    /**
     * Returns all children of the specified {@code thesaurusId} sorted by name.
     * 
     * @param thesaurusId thesaurus wikitty id
     * @return all list of <code>TreeNodeImpl</code>
     * @throws VradiException
     */
    List<Thesaurus> getChildrenThesaurus(String thesaurusId) throws VradiException;
    
    /**
     * Computes the number of forms associated with the specified {@code thesaurusId}.
     *
     * @param thesaurusId thesaurus id
     * @return the number of forms associated
     * @throws VradiException
     */
    int getNbFormsForThesaurus(String thesaurusId) throws VradiException;

    /**
     * Create new thesaurus.
     * 
     * @param rootThesaurusName root thesaurus name (must contains only alphnum characters)
     * @return new created thesaurus
     * @throws VradiException if name is not valid
     */
    RootThesaurus createRootThesaurus(String rootThesaurusName) throws VradiException;

    /**
     * Delete all thesaurus and sub thesaurus.
     * 
     * @param thesaurusId thesaurusId to delete
     * @throws VradiException
     */
    List<String> deleteThesaurus(String thesaurusId) throws VradiException;

    /**
     * Check if a form type name already exists.
     * 
     * @param formTypeName form type name
     * @return new created wikitty extension
     * @throws VradiException if name is not valid
     */
    boolean isFormTypeExists(String formTypeName) throws VradiException;

    /**
     * Create new form type.
     * 
     * @param formTypeName form type name
     * @return new created wikitty extension
     * @throws VradiException if name is not valid
     */
    WikittyExtension createFormType(String formTypeName) throws VradiException;
    
    /**
     * Return true if thesaurusName exists in rootThesaurus children.
     * 
     * @param rootThesaurus root thesaurus
     * @param thesaurusName thesaurus name to test
     * @return true if thesaurusName exists in rootThesaurus children
     */
    boolean isThesaurusNameExistsInRootThesaurus(RootThesaurus rootThesaurus, String thesaurusName);

    /**
     * Check if name of thesaurus is existing in rootThesaurus.
     *
     * @param rootThesaurus to check
     * @param thesaurusName name to check
     * @param exceptedThesaurusId in case of rename, exclude exceptedThesaurusId from check
     * @return true if thesaurus already exist
     */
    boolean isThesaurusNameExistsInRootThesaurus(RootThesaurus rootThesaurus, String thesaurusName, String exceptedThesaurusId);

    /**
     * Return true if rootThesaurusName already exists.
     * 
     * @param rootThesaurusName rootThesaurus name to test
     * @return true if rootThesaurusName already exists
     */
    boolean isRootThesaurusNameExists(String rootThesaurusName);
    
    /**
     * Return true if rootThesaurusName already exists.
     * 
     * @param rootThesaurusName rootThesaurus name to test
     * @param exceptedRootThesaurusId in case of rename, exclude exceptedRootThesaurusId from check
     * @return true if rootThesaurusName already exists
     */
    boolean isRootThesaurusNameExists(String rootThesaurusName, String exceptedRootThesaurusId);

    /**
     * Check if user have email non empyt
     *
     * @param sending to check
     * @return false if no valid email found
     */
    boolean checkUserEmail(Sending sending);

    /**
     * Return session corresponding of sending id in param
     *
     * @param sendingId corresponding to session to find
     * @return session found
     */
    Session getSessionBySending(String sendingId);
       

    /**
     * Delete all thesauruses recursively.
     * 
     * @throws VradiException 
     */
    public void deleteAllThesaurus() throws VradiException;
}
