/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services.managers;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.DateUtils;
import org.nuiton.wikitty.Criteria;
import org.nuiton.wikitty.FacetTopic;
import org.nuiton.wikitty.PagedResult;
import org.nuiton.wikitty.TreeNode;
import org.nuiton.wikitty.WikittyExtension;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyUtil;
import org.nuiton.wikitty.search.Element;
import org.nuiton.wikitty.search.Like.SearchAs;
import org.nuiton.wikitty.search.Search;
import org.nuiton.wikitty.solr.WikittySearchEnginSolr;

import com.jurismarches.vradi.beans.FormPagedResult;
import com.jurismarches.vradi.beans.QueryBean;
import com.jurismarches.vradi.beans.ThesaurusCartography;
import com.jurismarches.vradi.entities.EntityHelper;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.ModificationTag;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.entities.RootThesaurus;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.search.CompareFilter;
import com.jurismarches.vradi.services.search.Filter;
import com.jurismarches.vradi.services.search.FilterList;
import com.jurismarches.vradi.services.search.RangeFilter;
import com.jurismarches.vradi.services.search.UnsupportedQueryException;
import com.jurismarches.vradi.services.search.VradiQueryParser;

/**
 * Class containing the methods to manage the form research
 *
 * @author schorlet
 * @date 2010-01-29 12:40:26
 * @version $Revision: 1308 $ $Date: 2010-09-09 17:35:18 +0200 (jeu., 09 sept. 2010) $
 */
public class SearchManager {
    private static final Log log = LogFactory.getLog(SearchManager.class);

    private final WikittyProxy proxy;
    private final ThesaurusManager thesaurusManager;

    private static final DateFormat frenchDateFormat = DateFormat.getDateInstance(
            DateFormat.SHORT, Locale.FRANCE);
    
    private static final DateFormat numericDateFormat = new SimpleDateFormat("yyyyMMdd", Locale.FRANCE);

    /**
     * Match dd/mm/yyyy, requiring leading zeros.
     * 
     * Match the pattern but not the validity of the date, so that <code>01/01/0000</code> will match.
     */
    private static final Pattern frenchDateFormatPattern =
            Pattern.compile("^(3[01]|[12][0-9]|0[1-9])/(1[0-2]|0[1-9])/[0-9]{4}$");

    /**
     * Match yyyymmdd, requiring leading zeros.
     */
    private static final Pattern numericDateFormatPattern =
            Pattern.compile("^[0-9]{4}(1[0-2]|0[1-9])(3[01]|[12][0-9]|0[1-9])$");

    /**
     * Match UUID pattern, eg. 32 hexadecimal digits, displayed in 5 groups separated by hyphens,
     * in the form 8-4-4-4-12 for a total of 36 characters.
     */
    private static final Pattern uuidPattern =
            Pattern.compile("^[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}$");

    /**
     * Match yyyy-mm-ddUUID pattern.
     */
    private static final Pattern formIdPattern =
            Pattern.compile("^[0-9]{4}-(1[0-2]|0[1-9])-(3[01]|[12][0-9]|0[1-9])([a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12})$");

    /** Alias pour {@link ModificationTag#FIELD_LASTMODIFIER}. */
    public static final String ALIAS_LAST_MODIFIER = "modificateur";

    /** Alias pour {@link ModificationTag#FIELD_LASTSTATUSMODIFIER}. */
    public static final String ALIAS_LAST_STATUS_MODIFIER = "modificateur.status";

    public SearchManager(WikittyProxy proxy, ThesaurusManager thesaurusManager) {
        this.proxy = proxy;
        this.thesaurusManager = thesaurusManager;
    }

    /*public FormPagedResult findForms(String query, FormPagedResult formPagedResult)
            throws UnsupportedQueryException {
        return findForms(query, null, null, null, null, null, null, formPagedResult);
    }*/

    public FormPagedResult findForms(String query, FormPagedResult formPagedResult, String dateType, Date fromDate, String statusId)
            throws UnsupportedQueryException, VradiException {

        return findForms(query, null, dateType, fromDate, new Date(), null, null, new String[]{statusId}, formPagedResult);
    }
    
    public FormPagedResult findForms(String query, WikittyExtension extension,
            String dateType, Date beginDate, Date endDate, String streamId,
            List<String> thesaurus, String[] statusIds,
            FormPagedResult formPagedResult) throws UnsupportedQueryException, VradiException {

        if (log.isDebugEnabled()) {
            log.debug(String.format(
                    "findForms(query:%s, extension:%s, dateType:%s, beginDate:%tc, endDate:%tc)",
                    query, extension, dateType, beginDate, endDate) + " status : " + statusIds);
        }

        Search search = createSearch(query, extension, dateType, beginDate, endDate,
                streamId, thesaurus, statusIds);
        
        Criteria criteria = search.criteria();

        // add index restriction
        // last index can be -1 (all value in UI)
        int firstIndex = (formPagedResult.getPageToShow() - 1) * formPagedResult.getNbFormsToShow();
        int lastIndex = formPagedResult.getPageToShow() * formPagedResult.getNbFormsToShow() - 1;
        if (firstIndex >= 0 && lastIndex > 0 && lastIndex > firstIndex) {
            criteria.setFirstIndex(firstIndex).setEndIndex(lastIndex);
        }

        // Add field restriction
        if (formPagedResult.getFieldToSort() != null) {
            if (!formPagedResult.isAscending()) {
                criteria.addSortDescending(formPagedResult.getFieldToSort());

            } else {
                criteria.addSortAscending(formPagedResult.getFieldToSort());
            }
        }

        criteria.addSortAscending(Form.FQ_FIELD_INFOGENE_ID);

        //finds the forms
        PagedResult<Form> queryResult = proxy.findAllByCriteria(Form.class, criteria);
        List<Form> result = new ArrayList<Form>(queryResult.getAll());

        if (log.isDebugEnabled()) {
            log.debug("[findForms] found: " + result.size() + " forms");
        }

        FormPagedResult formPageResult = new FormPagedResult(result,
                queryResult.getNumFound(), formPagedResult.getPageToShow(),
                formPagedResult.getNbFormsToShow());
        
        return formPageResult;
    }

    public ThesaurusCartography getThesaurusCartography(String query,
            WikittyExtension extension, String dateType, Date beginDate,
            Date endDate, String streamId, List<String> thesaurusIds, String[] statusIds)
            throws VradiException, UnsupportedQueryException {
        
        if (log.isDebugEnabled()) {
            log.debug("getThesaurusCartography()");
        }

        // count forms indexed by thesaurus nodes
        Map<String, Integer> cartography = new HashMap<String, Integer>();

        // getting all thesaurus nodes
        List<Thesaurus> thesauruses = thesaurusManager.getAllThesaurus();

        //rootThesaurus is now a tree node
        //List<RootThesaurus> rootThesaurus = thesaurusManager.getRootThesaurus();
        
        // create search
        Search search = createSearch(query, extension, dateType, beginDate, endDate, streamId, thesaurusIds, statusIds);
        Criteria criteria = search.criteria();
        
        // add facet fields on search
//        criteria.addFacetField(TreeNode.EXT_TREENODE + "." + rootThesaurus.getWikittyId());
        for (Thesaurus thesaurus : thesauruses) {
            String facetField = WikittySearchEnginSolr.TREENODE_PREFIX + thesaurus.getWikittyId();

            criteria.addFacetField(facetField);
            
            cartography.put(thesaurus.getWikittyId(), 0);
        }
        
        // execute search
        PagedResult<Form> results = proxy.findAllByCriteria(Form.class, criteria);
        
        // compute facets
        Collection<String> facetNames = results.getFacetNames();

        if (log.isDebugEnabled()) {
            log.debug("FacetName size " + facetNames.size());
        }
        for (String facetName : facetNames) {
            List<FacetTopic> topics = results.getTopic(facetName);

            if (log.isDebugEnabled()) {
                if (topics.size() > 0) {
                    log.debug("FacetName " + facetName + " topic size " + topics.size());
                }
            }
            for (FacetTopic topic : topics) {

                if (log.isDebugEnabled()) {
                    if (topic.getCount() > 0) {
                        log.debug("FacetTopic name " + topic.getTopicName() + " count " + topic.getCount());
                    }
                }
                Thesaurus thesaurus = proxy.restore(Thesaurus.class, topic.getTopicName());

                if (thesaurus == null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Thesaurus not found for id : " + topic.getTopicName());
                    }
                    continue;
                }
                cartography.put(thesaurus.getWikittyId(), topic.getCount());
            }
        }

        List<Form> formsFounds = results.getAll();
        if (log.isDebugEnabled()) {
            log.debug("Creating cartography with : " + formsFounds.size() + " forms founds and " + cartography.size() + " thesaurus concerned");
        }
        ThesaurusCartography thesaurusCartography = new ThesaurusCartography(cartography, formsFounds);
        return thesaurusCartography;
    }

    public Map<QueryMaker, List<QueryBean>> findQueriesReturningForm(Form form)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("findQueriesReturningForm(form)");
        }
        
        Map<QueryMaker, List<QueryBean>> results = new HashMap<QueryMaker, List<QueryBean>>();
        
        if (form == null) {
            return results;
        }
        
        // find QueryMaker which do have queries defined
        List<QueryMaker> queryMakers = findQueryMakersWithQueries();

        for (QueryMaker queryMaker : queryMakers) {
            Set<String> queries = queryMaker.getQueries();
            // just to make sure ...
            if (queries == null || queries.isEmpty()) {
                continue;
            }
            
            QueryMaker realQueryMaker = EntityHelper.castAsRealQueryMaker(queryMaker);
            if (realQueryMaker == null) {
                continue;
            }
            
            for (String query : queries) {
                try {
                    QueryBean bean = new QueryBean(query, queryMaker.getWikittyId());
                    String realQuery = bean.getQuery();
                    FilterList filter = VradiQueryParser.parse(realQuery);
                    
                    Search search = Search.query();
                    search.eq(Form.FQ_FIELD_INFOGENE_ID, form.getId());
                    buildSearch(filter, search);
    
                    Criteria criteria = search.criteria();
                    PagedResult<Form> forms = proxy.findAllByCriteria(Form.class, criteria);
                    
                    if (forms.getNumFound() > 0) {
                        if (results.containsKey(realQueryMaker)) {
                            results.get(realQueryMaker).add(bean);
                            
                        } else {
                            List<QueryBean> list = new ArrayList<QueryBean>();
                            list.add(bean);
                            results.put(realQueryMaker, list);
                        }
                    }
                    
                } catch (Exception e) {
                    log.warn(e.getMessage(), e);
                }
            }
        }

        return results;
    }

    /**
     * Find all query maker with non null queries.
     * 
     * @return query maker
     */
    public List<QueryMaker> findQueryMakersWithQueries() {
        // find QueryMaker which do have queries defined
        Criteria criteria = Search.query().bw(QueryMaker.FQ_FIELD_QUERYMAKER_QUERIES, "*", "*").criteria();
        PagedResult<QueryMaker> pagedResult = proxy.findAllByCriteria(QueryMaker.class, criteria);
        List<QueryMaker> all = pagedResult.getAll();
        return all;
    }

    /**
     * Find all groups with non null queries.
     * 
     * @return group maker
     */
    public List<Group> findGroupsWithQueries() {
        // find QueryMaker which do have queries defined
        Criteria criteria = Search.query().bw(QueryMaker.FQ_FIELD_QUERYMAKER_QUERIES, "*", "*").criteria();
        PagedResult<Group> pagedResult = proxy.findAllByCriteria(Group.class, criteria);
        List<Group> all = pagedResult.getAll();
        return all;
    }

    /**
     * Build wikitty {@link Search} criteria with arguments parameters.
     * 
     * All parameters can be {@code null}.
     * 
     * Transform first query to {@link FilterList} and transform this
     * {@link FilterList} into wikitty {@link Search}. Then add all parameter
     * to {@link Search} query.
     * 
     * @param query query
     * @param extension extension
     * @param dateType date type
     * @param beginDate begin date
     * @param endDate end date
     * @param streamId id of stream
     * @param thesaurusIds or between each collection, and between all element in collection
     * @param statusIds or parameters
     * @return wikitty search query
     * @throws UnsupportedQueryException
     * @throws VradiException 
     */
    protected Search createSearch(String query, WikittyExtension extension,
            String dateType, Date beginDate, Date endDate, String streamId,
            Collection<String> thesaurusIds, String[] statusIds)
            throws UnsupportedQueryException, VradiException {

        // string query to FilterList structure
        FilterList filter = VradiQueryParser.parse(query);

        // FilterList to wikitty Search
        Search search = Search.query();
        buildSearch(filter, search);

        // add the extension in the criteria
        if (extension != null) {
            search.eq(Element.ELT_EXTENSION, extension.getName());
        }

        // add the date in the criteria
        if (dateType != null && beginDate != null && endDate != null) {
            String beginString = null;
            String endString = null;
            try {
                beginString = WikittyUtil.formatDate(beginDate);
                endString = WikittyUtil.formatDate(endDate);
            } catch (ParseException eee) {
                throw new VradiException("Cant format date : ", eee);
            }
            search.bw(dateType, beginString, endString);
        }

        //add the thesaurus in the criteria
        if (thesaurusIds != null) {
            for (String thesaurusId : thesaurusIds) {
                Search subSearch = search.or();
                subSearch.eq(Form.FQ_FIELD_FORM_THESAURUS, thesaurusId);
            }
        }

        // add the status in the criteria
        if (!ArrayUtils.isEmpty(statusIds)) {
            Search subSearch = search.or();
            for (String statusId : statusIds) {
                subSearch.eq(Form.FQ_FIELD_INFOGENE_STATUS, statusId);
            }
        }

        // add the xmlStream in the criteria
        if (streamId != null) {
            search.eq(Form.FQ_FIELD_FORM_XMLSTREAM, streamId);
        }

        return search;
    }

    /**
     * Transform FilterList part into wikitty sub Search.
     * 
     * This method recursively explore FilterList, and recursively call
     * himself.
     * 
     * @param list current filter list
     * @param search parent wikitty search to add new created search
     * @throws VradiException 
     */
    protected void buildSearch(FilterList list, Search search) throws VradiException {

        // create new sub search
        FilterList.Operator operator = list.getOperator();
        Search subSearch = null;
        switch (operator) {
            case FILTER_OR :
                subSearch = search.or();
                break;
            case FILTER_AND :
                subSearch = search.and();
                break;
            case FILTER_NOT :
                subSearch = search.not();
                break;
        }

        // transform each filters into wikitty
        List<Filter> filters = list.getFilters();
        for (Filter filter : filters) {
            if (filter instanceof FilterList) {
                buildSearch((FilterList) filter, subSearch);
        
            } else if (filter instanceof RangeFilter) {
                buildRangeSearch((RangeFilter) filter, subSearch);
                
            } else if (filter instanceof CompareFilter) {
                buildCompareSearch(operator, (CompareFilter) filter, subSearch);
            }
            
        }
    }

    /**
     * Build wikitty range search.
     * 
     * @param rangeFilter vradi range filter
     * @param search wikitty range search
     */
    protected void buildRangeSearch(RangeFilter rangeFilter, Search search) {

        String name = rangeFilter.getName();
        String lowerValue = rangeFilter.getLowerValue();
        String upperValue = rangeFilter.getUpperValue();

        Search subSearch = search.or();

        // date search
        if (rangeFilter.match(numericDateFormatPattern)) {
            try {
                Date lowerTime = numericDateFormat.parse(lowerValue);
                lowerTime = DateUtils.setMinTimeOfDay(lowerTime);
                String lowerTimeString = WikittyUtil.formatDate(lowerTime);

                Date upperTime = numericDateFormat.parse(upperValue);
                upperTime = DateUtils.setMaxTimeOfDay(upperTime);
                String upperTimeString = WikittyUtil.formatDate(upperTime);

                subSearch.bw(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                        + name + Criteria.SEPARATOR
                        + Element.ElementType.DATE, lowerTimeString, upperTimeString);

            } catch (ParseException e) {
                log.warn(lowerValue + " OR " + upperValue + " cannot be a date.");
            }

        // number search
        } else if (rangeFilter.isNumber()) {
            subSearch.bw(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                    + name + Criteria.SEPARATOR
                    + Element.ElementType.NUMERIC, lowerValue, upperValue);
        }

        // TODO EC 20100622 il manque pas un else ici ?
        // default to string search
        subSearch.bw(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                + name + Criteria.SEPARATOR
                + Element.ElementType.STRING, lowerValue, upperValue);
    }

    /**
     * Convertit les termes simple (name:value).
     * 
     * @param operator
     * @param compareFilter
     * @param search
     * @throws VradiException 
     */
    protected void buildCompareSearch(FilterList.Operator operator, CompareFilter compareFilter,
            Search search) throws VradiException {
        String name = compareFilter.getName();
        String value = compareFilter.getValue();

        if (VradiQueryParser.DEFAULT_FIELD.equals(name)) {
            search.keyword(value);
            return;

        } else {

            // loop on each root thesaurus
            List<RootThesaurus> rootThesauruses = thesaurusManager.getRootThesaurus();
            for (RootThesaurus rootThesaurus : rootThesauruses) {
                if (name.equalsIgnoreCase(rootThesaurus.getName())) {
                    buildDescripteurSearch(search, rootThesaurus.getWikittyId(), value);
                    return;
                }
            }
        }
            
        if (ALIAS_LAST_MODIFIER.equals(name)) {
            name = ModificationTag.FIELD_MODIFICATIONTAG_LASTMODIFIER;

        } else if (ALIAS_LAST_STATUS_MODIFIER.equals(name)) {
            name = ModificationTag.FIELD_MODIFICATIONTAG_LASTSTATUSMODIFIER;
        }

        Search subSearch = search.or();

        // a confirmer, mais l'explication doit être :
        // - si c'est not, ca doit ne pas être la date en question
        // - sinon, ca doit être compris entre le début et la fin du jour
        //   en question
        if (operator == FilterList.Operator.FILTER_NOT) {
            // date search
            if (compareFilter.match(frenchDateFormatPattern)) {
                try {
                    Date time = frenchDateFormat.parse(value);
                    String timeString = WikittyUtil.formatDate(time);

                    subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                            + name + Criteria.SEPARATOR
                            + Element.ElementType.DATE, timeString);

                } catch (ParseException e) {
                    if (log.isWarnEnabled()) {
                        log.warn(value + " cannot be a date.");
                    }
                }
            }

        } else {
            // date search
            if (compareFilter.match(frenchDateFormatPattern)) {
                try {
                    Date time = frenchDateFormat.parse(value);

                    Date beginDate = DateUtils.setMinTimeOfDay(time);
                    Date endDate = DateUtils.setMaxTimeOfDay(time);

                    String beginString = WikittyUtil.formatDate(beginDate);
                    String endString = WikittyUtil.formatDate(endDate);

                    subSearch.bw(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                            + name + Criteria.SEPARATOR
                            + Element.ElementType.DATE, beginString, endString);

                } catch (ParseException e) {
                    if (log.isWarnEnabled()) {
                        log.warn(value + " cannot be a date.");
                    }
                }
            }
        }

        // string search
        subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                + name + Criteria.SEPARATOR
                + Element.ElementType.STRING, value);

        // text search
        if (!compareFilter.isPhrase()) {
            subSearch.like(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                    + name + Criteria.SEPARATOR + Element.ElementType.STRING,
                    value, SearchAs.AsText);
        }

        // boolean search
        if (compareFilter.isBoolean()) {
            subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                    + name + Criteria.SEPARATOR
                    + Element.ElementType.BOOLEAN, value);

        // number search
        } else if (compareFilter.isNumber()) {
            subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                    + name + Criteria.SEPARATOR
                    + Element.ElementType.NUMERIC, value);

        // extension id search (eq)
        } else if (compareFilter.match(formIdPattern)) {
            subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                + name + Criteria.SEPARATOR
                + Element.ElementType.STRING, value);

        // extension id search (ew)
        } else if (compareFilter.match(uuidPattern)) {
            subSearch.ew(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                + name + Criteria.SEPARATOR
                + Element.ElementType.STRING, value);
        }
    }

    /**
     * Cherche tous les thesaurus en base qui correspondent à {@code value}
     * et ajoute un critère {@link Form#FQ_FIELD_THESAURUS} a la Search query
     * (ou entre tous les résultats s'il y en a plusieurs).
     * 
     * Les formulaires sont taggués avec tous les noeuds de thesaurus
     * auquel ils appartiennent depuis le noeud jusqu'a sa racine.
     * 
     * recherche : TreeNode.wikittyidparent : TreeNode.wikittyid
     * 
     * @param rootThesaurusId filtering on thesaurus node of root thesaurus
     * @param search parent search
     * @param value thesaurus value
     */
    protected void buildDescripteurSearch(Search search, String rootThesaurusId, String value) {

        Criteria criteria = Search.query()
            .eq(TreeNode.FQ_FIELD_TREENODE_NAME, value)
            .eq(Thesaurus.FQ_FIELD_THESAURUS_ROOTTHESAURUS, rootThesaurusId)
            .eq(Element.ELT_EXTENSION, Thesaurus.EXT_THESAURUS)
            .criteria();

        PagedResult<Thesaurus> queryResult = proxy.findAllByCriteria(Thesaurus.class, criteria);

        if (queryResult.getNumFound() > 0) {
            Search subSearch = search.or();

            List<Thesaurus> result = queryResult.getAll();
            for (Thesaurus thesaurus : result) {
                if (thesaurus.getParent() == null) {
                    subSearch.eq(WikittySearchEnginSolr.TREENODE_ROOT, thesaurus.getWikittyId());
                }
                else {
                    subSearch.eq(WikittySearchEnginSolr.TREENODE_PREFIX + thesaurus.getParent(), thesaurus.getWikittyId());
                }
            }
        }
    }
}
