/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services.managers;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.net.URL;
import java.security.NoSuchAlgorithmException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdom.Document;
import org.jdom.Namespace;
import org.jdom.input.SAXBuilder;
import org.nuiton.util.StringUtil;
import org.nuiton.wikitty.Criteria;
import org.nuiton.wikitty.FieldType;
import org.nuiton.wikitty.PagedResult;
import org.nuiton.wikitty.Wikitty;
import org.nuiton.wikitty.WikittyExtension;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyUtil;
import org.nuiton.wikitty.search.Element;
import org.nuiton.wikitty.search.Search;
import org.webharvest.definition.ScraperConfiguration;
import org.webharvest.runtime.Scraper;
import org.webharvest.runtime.variables.Variable;

import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.VradiConstants.XmlStreamConfig;
import com.jurismarches.vradi.entities.EntityHelper;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.FormImpl;
import com.jurismarches.vradi.entities.WebHarvestStream;
import com.jurismarches.vradi.entities.XmlFieldBinding;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.Configuration;
import com.jurismarches.vradi.services.ServiceFactory;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.util.SSLUtils;

/**
 * Class containing the methods to manage the binding of the xml streams fields
 * with the form fields :
 * - xml field bindings creation, update and retrieving
 * - xml streams retrieving
 * - form creation with the data from an xml stream
 *
 * @author schorlet
 * @date 2010-01-22 20:18:29
 * @version $Revision: 1317 $ $Date: 2010-09-10 10:29:46 +0200 (ven., 10 sept. 2010) $
 */
public class BindingManager {
    private static final Log log = LogFactory.getLog(BindingManager.class);

    private final WikittyProxy proxy;
    private final FormTypeManager formTypeManager;
    private final FormManager formManager;
    private Timer timer = null;
    private TimerTask xmlStreamTask = null;

    static {
        // used for xml stream on https/ssl
        SSLUtils.intallCertificateTruster();
    }

    public BindingManager(WikittyProxy proxy, FormTypeManager formTypeManager,
            FormManager formManager) {
        this.proxy = proxy;
        this.formTypeManager = formTypeManager;
        this.formManager = formManager;
    }

    public BindingManager(FormTypeManager formTypeManager, FormManager formManager) {
        this.proxy = ServiceFactory.getWikittyProxy();
        this.formTypeManager = formTypeManager;
        this.formManager = formManager;
    }

    /**
     * Retrieves the xml field bindings whose xml stream is xmlStream
     *
     * @param xmlStream the xml stream associated with the xml field bindings
     * we want to retrieve
     * @return a list containing the xml field bindings associated with
     * the xml stream xmlStream
     */
    public List<XmlFieldBinding> getXmlFieldBindings(XmlStream xmlStream) {
        List<XmlFieldBinding> list = new ArrayList<XmlFieldBinding>();

        if (xmlStream != null && xmlStream.getXmlFieldBinding() != null) {
            if (log.isDebugEnabled()) {
                log.debug("getXmlFieldBindings(" + xmlStream.getName() + ")");
                log.debug(xmlStream.getXmlFieldBinding());
            }

            List<String> bindings = new ArrayList<String>();
            bindings.addAll(xmlStream.getXmlFieldBinding());
            
            List<XmlFieldBinding> restore = proxy.restore(XmlFieldBinding.class, bindings);
            list.addAll(restore);
        }

        return list;
    }

    /**
     * Retrieves the xml field binding whose id is xmlFieldBindingId
     *
     * @param xmlFieldBindingId the id of the xml field binding we want to retrieve
     * @return the xml field binding whose id is xmlFieldBindingId
     */
    public XmlFieldBinding getXmlFieldBinding(String xmlFieldBindingId) {
        if (log.isDebugEnabled()) {
            log.debug("getXmlFieldBinding(" + xmlFieldBindingId + ")");
        }
        
        XmlFieldBinding xmlFieldBinding = proxy.restore(XmlFieldBinding.class, xmlFieldBindingId);
        return xmlFieldBinding;
    }

    /*
     * Updates the xml field bindings given in parameters
     *
     * @param bindings the list of the xml field bindings to update
     * @return the list of the xml field bindings up to date
     *
    public List<XmlFieldBinding> updateXmlFieldBindings(List<XmlFieldBinding> bindings)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateXmlFieldBindings(bindings)");
        }
        
        try {
            List<XmlFieldBinding> list = new ArrayList<XmlFieldBinding>();
        
            if (bindings != null) {
                bindings = proxy.store(bindings);
                list.addAll(bindings);
            }
            
            return list;
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new VradiException(e);
        }
    }*/
    
    /*public <E extends XmlStream> E updateStream(E stream, List<XmlFieldBinding> bindings)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateXmlStream(xmlStream, bindings)");
        }
        
        try {
            bindings = updateXmlFieldBindings(bindings);
            
            stream.clearXmlFieldBinding();
            for (XmlFieldBinding binding : bindings) {
                stream.addXmlFieldBinding(binding.getWikittyId());
            }
            
            stream = proxy.store(stream);
            return stream;
            
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new VradiException("Cant save stream : ", e);
        }
    }*/
    
    public List<XmlStream> getAllXmlStreams() {
        if (log.isDebugEnabled()) {
            log.debug("getAllXmlStreams()");
        }
        
        Criteria criteria = Search.query()
                .eq(Element.ELT_EXTENSION, XmlStream.EXT_XMLSTREAM)
                .criteria();
        
        PagedResult<XmlStream> xmlStreams = proxy.findAllByCriteria(XmlStream.class, criteria);
        List<XmlStream> all = xmlStreams.getAll();
        
        List<XmlStream> list = new ArrayList<XmlStream>();
        list.addAll(all);
        
        return EntityHelper.castAsRealStreams(list);
    }

    public List<WebHarvestStream> getAllWebHarvestStreams() {
        if (log.isDebugEnabled()) {
            log.debug("getAllWebHarvestStreams()");
        }

        Criteria criteria = Search.query()
                .eq(Element.ELT_EXTENSION, WebHarvestStream.EXT_WEBHARVESTSTREAM)
                .criteria();

        PagedResult<WebHarvestStream> xmlStreams = proxy.findAllByCriteria(WebHarvestStream.class, criteria);
        List<WebHarvestStream> all = xmlStreams.getAll();

        List<WebHarvestStream> list = new ArrayList<WebHarvestStream>();
        list.addAll(all);

        return list;
    }

    public URI getWebHarvestPreviewUrl(WebHarvestStream stream) throws VradiException {
        String path = getWebHarvestPreviewPath(stream);
        File previewFile = new File(path);

        // If doesnt exist, creating one
        if (!previewFile.exists()) {
            if (log.isDebugEnabled()) {
                log.debug("Preview file doesnt exist, creating one for script path : " + path);
            }
            previewFile = createPreviewFile(stream);
        }

        URI uri = previewFile.toURI();
        return uri;
    }

    public String getWebHarvestPreviewUrlAsString(WebHarvestStream stream) throws VradiException {
        String uri = getWebHarvestPreviewUrl(stream).toString();

        if (log.isDebugEnabled()) {
            log.debug("Previews url is : " + uri);
        }

        return uri;
    }

    public String getWebHarvestPreviewPath(WebHarvestStream stream) {
        File scriptFile = new File(Configuration.getInstance().getWebHarvestScriptDir() +
                File.separator + stream.getScriptUrl());

        String previewPath = Configuration.getInstance().getWebHarvestPreviewDir() +
                File.separator +
                "Preview" +
                scriptFile.getName();

        if (log.isDebugEnabled()) {
            log.debug("Previews path is : " + previewPath);
        }

        return previewPath;
    }

    public XmlStream getXmlStream(String xmlStreamId) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getXmlStream(" + xmlStreamId + ")");
        }

        XmlStream xmlStream = proxy.restore(XmlStream.class, xmlStreamId);
        return xmlStream;
    }

    private static class BindingContext {
        int dateParsingError = 0;
        int numberParsingError = 0;
        int nbCreated = 0;
    }

    /**
     * Create from from feed element.
     * 
     * @param formType
     * @param bindings
     * @param feed
     * @param namespace namespace for getting correct field with fieldnames
     * @param bindingContext
     * @return
     */
    protected FormImpl createForm(WikittyExtension formType, List<XmlFieldBinding> bindings,
             org.jdom.Element feed, Namespace namespace, BindingContext bindingContext) throws VradiException {
        FormImpl form = new FormImpl();
        Wikitty wikitty = form.getWikitty();
        wikitty.addExtension(formType);
        
        for (XmlFieldBinding binding : bindings) {
            String fqFormField = binding.getFormField();
            FieldType fieldType;
            try {
                fieldType = wikitty.getFieldType(fqFormField);
            } catch (Exception e) {
                continue;
            }
            
            fillFormField(wikitty, fieldType, binding, feed, namespace, bindingContext);
        }
        
        return form;
    }

    /**
     * Fill form with feed element.
     * 
     * @param wikitty
     * @param fieldType
     * @param binding
     * @param feed
     * @param namespace namespace to get field from feed element
     * @param bindingContext
     */
    protected void fillFormField(Wikitty wikitty, FieldType fieldType, XmlFieldBinding binding,
            org.jdom.Element feed, Namespace namespace, BindingContext bindingContext) throws VradiException {
        
        String fqFormField = binding.getFormField();
        Set<String> xmlFields = binding.getXmlField();

        if (xmlFields == null || xmlFields.isEmpty()) {
            // no mapping
            String defaultValue = binding.getDefaultValue();
            fillFormField2(wikitty, fieldType, fqFormField, defaultValue, bindingContext);
            return;
        }
            
        for (String xmlField : xmlFields) {
            org.jdom.Element child = feed.getChild(xmlField, namespace);
            String feedValue = null;
            
            // get feed field text
            if (child != null) {
                feedValue = child.getTextTrim();
            }
            
            // get default value
            if (feedValue == null || feedValue.isEmpty()) {
                feedValue = binding.getDefaultValue();
            }
            
            fillFormField2(wikitty, fieldType, fqFormField, feedValue, bindingContext);
        }
    }
    
    private void fillFormField2(Wikitty wikitty, FieldType fieldType, String fqFormField,
            String feedValue, BindingContext bindingContext) throws VradiException {
        // if no value then return
        if (feedValue == null || feedValue.isEmpty()) {
            return;
        }

        switch (fieldType.getType()) {
        case DATE:
            Date date = DateParser.parse(feedValue);
            if (date != null) {
                try {
                    wikitty.setFqField(fqFormField, WikittyUtil.formatDate(date));
                } catch (ParseException eee) {
                    throw new VradiException("Cant parse date " + date, eee);
                }
            } else {
                bindingContext.dateParsingError++;
            }
            break;

        case NUMERIC:
            if (NumberUtils.isNumber(feedValue)) {
                Double value = Double.valueOf(feedValue);
                 wikitty.setFqField(fqFormField, value);
            } else {
                bindingContext.numberParsingError++;
            }
            break;

        default:
            Object fieldValue = wikitty.getFqField(fqFormField);
            String newValue = null;

            if (fieldValue != null) {
                newValue = fieldValue + "\n" + feedValue;
            } else {
                newValue = feedValue;
            }

             wikitty.setFqField(fqFormField, newValue);
        }
     }
    
    /**
     * Creates forms with the data of an xml stream
     *
     * @param webHarvestStream
     * @param vradiUser
     * @return An array containing :
     * - the number of created forms
     * - the number of already existing forms
     * - the number of forms created with date parsing error
     * - the number of forms created with number parsing error
     * 
     * @throws VradiException for various possible errors
     */
    public int[] getFormsFromWebHarvestStream(WebHarvestStream webHarvestStream) throws VradiException {

        log.info("getFormsFromWebHarvestStream for " + webHarvestStream.getName());

        // Create file
        File previewFile = createPreviewFile(webHarvestStream, true);

        Document document = null;
        try {
            SAXBuilder sxb = new SAXBuilder();
            document = sxb.build(previewFile);
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't read xml stream", e);
            }
            throw new VradiException("Can't read xml stream : ", e);
        }
        return getFormsFromStream(webHarvestStream, document);
    }

    /**
     * Creates file result of webharvest scrip
     *
     * @param webHarvestStream
     * @return Preview file
     * @throws VradiException for various possible errors
     */
    public File createPreviewFile(WebHarvestStream webHarvestStream) throws VradiException{
        return createPreviewFile(webHarvestStream, false);
    }

    // Create file if necessary
    protected File createPreviewFile(WebHarvestStream webHarvestStream, boolean override) throws VradiException{
        String path = webHarvestStream.getScriptUrl();
        if (path == null) {
            return null;
        }

        // Get script path
        path = Configuration.getInstance().getWebHarvestPreviewDir() +
                File.separator +
                path;

        File scriptFile = new File(path);

        if (!override && scriptFile.exists()) {
            return scriptFile;
        }

        // Init webharvest
        ScraperConfiguration config = null;
        try {
            config = new ScraperConfiguration(scriptFile);
        } catch (FileNotFoundException eee) {
            throw new VradiException("Cant open script " + webHarvestStream.getScriptUrl(), eee);
        }
        Configuration appConfig = Configuration.getInstance();
        Scraper scraper = new Scraper(config, appConfig.getDataDir());

        scraper.setDebug(log.isDebugEnabled());

        scraper.addVariableToContext("url", webHarvestStream.getUrl());

        long startTime = System.currentTimeMillis();
        scraper.execute();
        log.info("Script execution time elapsed: " + (System.currentTimeMillis() - startTime));

        // takes variable created during execution
        Variable resultVar = (Variable) scraper.getContext().get("result");
        String result = resultVar.toString();

        result = result.replaceAll("&", "&amp;");

        File previewFile = new File(path);

        try {
            FileUtils.writeStringToFile(previewFile, result);
            log.info("Store preview for script " + scriptFile.getPath());
        } catch (IOException eee) {
            log.error("Failed to write preview for script " + scriptFile.getName(), eee);
        }

        return previewFile;
    }

    /**
     * Creates forms with the data of an xml stream
     *
     * @param xmlStream
     * @return An array containing :
     * - the number of created forms
     * - the number of already existing forms
     * - the number of forms created with date parsing error
     * - the number of forms created with number parsing error
     *
     * @throws VradiException for various possible errors
     */
    public int[] getFormsFromXmlStream(XmlStream xmlStream) throws VradiException {

        Document document = null;
        try {
            
            // rome : but not done with rome :(
            //SyndFeedInput input = new SyndFeedInput();
            //URL rssUrl = new URL(xmlStream.getUrl());
            //SyndFeed feed = input.build(new XmlReader(rssUrl));

            SAXBuilder sxb = new SAXBuilder(false);
            
            // this prevent xerces to look for external dtd
            sxb.setFeature("http://apache.org/xml/features/nonvalidating/load-external-dtd", false);

            URL rssUrl = new URL(xmlStream.getUrl());
            log.info("Reading xmlStream url from: " + rssUrl);
            document = sxb.build(rssUrl);
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't read xml stream", e);
            }
            throw new VradiException("Can't read xml stream : ", e);
        }
        return getFormsFromStream(xmlStream, document);
    }

    /**
     * Creates forms with the data of an stream
     *
     * @param stream
     * @param document of rss
     * @return An array containing :
     * - the number of created forms
     * - the number of already existing forms
     * - the number of forms created with date parsing error
     * - the number of forms created with number parsing error
     *
     * @throws VradiException for various possible errors
     */
    public <E extends XmlStream> int[] getFormsFromStream(E stream, Document document) throws VradiException {
        
        // TODO EC-20100428 return a serializable structure (easier to use)
        int[] results = new int[4];
        
        String lastItemRecorded = PropertiesManager.getLastItemOfXmlStream(
                stream.getName());
        if (log.isDebugEnabled()) {
            log.debug("getFormsFromStream(" + stream.getName() + ", "
                    + lastItemRecorded);
        }

        if (stream.getFormTypeName() == null) {
            throw new VradiException("xmlStream.formTypeName is null");
        }
        
        WikittyExtension formType = formTypeManager.getFormType(
                stream.getFormTypeName());
        if (formType == null) {
            throw new VradiException("Extension of name xmlStream.formTypeName does not exists");
        }

        org.jdom.Element racine = document.getRootElement();
        // namespace should be used, otherwize, getChild(String) return
        // nothing in some feeds:
        // ex : http://www.gentoo.org/rdf/en/gentoo-news.rdf
        // https://labs.libre-entreprise.org/export/rss_sfnews.php
        Namespace namespace = racine.getNamespace("");
        List<org.jdom.Element> elements = null;

        // must start with ITEM :
        // some feed have both channel and item
        // http://www.gentoo.org/rdf/en/gentoo-news.rdf
        if (racine.getChild(VradiConstants.ITEM, namespace) != null) {
            elements = racine.getChildren(VradiConstants.ITEM, namespace);
        }
        else if (racine.getChild(VradiConstants.CHANNEL, namespace) != null) {
            org.jdom.Element channel = racine.getChild(VradiConstants.CHANNEL, namespace);
            elements = channel.getChildren(VradiConstants.ITEM, namespace);
        } else  if (racine.getChild(VradiConstants.ENTRY, namespace) != null) {
            elements = racine.getChildren(VradiConstants.ENTRY, namespace);
        }

        if (elements == null) {
            if (log.isWarnEnabled()) {
                log.warn("Enable to find items or entries in stream");
            }
            return results;
        }

        int index = 0;
        boolean itemAlreadyRecorded = false;
        List<String> encryptedElements = new ArrayList<String>();
        
        while(index < elements.size() && !itemAlreadyRecorded) {
            org.jdom.Element element = elements.get(index);
            StringBuffer sb = new StringBuffer();

            List<org.jdom.Element> fields = element.getChildren();
            for (org.jdom.Element field : fields) {
                sb.append(field.getText());
            }

            String encryptedItemValue = null;
            try {
                encryptedItemValue = StringUtil.encodeMD5(sb.toString());
            } catch (NoSuchAlgorithmException eee) {
                if (log.isFatalEnabled()) {
                    log.fatal("No MD5 algorithm found");
                }
            }

            //check if the element has not yet been read
            if (lastItemRecorded != null
                    && lastItemRecorded.equals(encryptedItemValue)) {
                itemAlreadyRecorded = true;
            } else {
                encryptedElements.add(encryptedItemValue);
                index++;
            }
        }
        index--;
        
        if(index >= 0) {
            String lastItemOfXmlStream = null;
            List<Form> forms = new ArrayList<Form>();

            List<XmlFieldBinding> bindings = proxy.restore(XmlFieldBinding.class,
                new ArrayList<String>(stream.getXmlFieldBinding()));

            String formId = VradiConstants.FORM_ID_DATE_FORMAT.format(new Date());
            String toTreatId = formManager.getNonTraiteStatus().getWikittyId();
            BindingContext bindingContext = new BindingContext();
            
            // for each element of the xml stream
            for (int i = index ; i >= 0 ; i--) {
                org.jdom.Element feed = elements.get(i);

                //create the form with the info from the xml stream
                FormImpl form = createForm(formType, bindings, feed, namespace, bindingContext);
                bindingContext.nbCreated++;
                
                form.setId(formId + form.getWikittyId());
                form.setXmlStream(stream.getWikittyId());
                form.setStatus(toTreatId);
                forms.add(form);
                lastItemOfXmlStream = encryptedElements.get(i);
                if(forms.size() > 1000) {
                    formManager.updateForms(forms);
                    PropertiesManager.setLastItemOfXmlStream(stream.getName(), lastItemOfXmlStream);
                    forms.clear();
                }
            }
            
            if (!forms.isEmpty()) {
                formManager.updateForms(forms);
                PropertiesManager.setLastItemOfXmlStream(stream.getName(), lastItemOfXmlStream);
            }

            results[0] = bindingContext.nbCreated;
            results[2] = bindingContext.dateParsingError;
            results[3] = bindingContext.numberParsingError;

        }

        // equals to : elements.size() - bindingContext.nbCreated
        results[1] = elements.size() - results[0];

        // TODO EC-20100428 : redirect log output into file
        if (log.isInfoEnabled()) {
            log.info("Form import from stream, created = " + results[0]);
            log.info("Form import from stream, already existing = " + results[1]);
            log.info("Form import from stream, dateParsingError = " + results[2]);
            log.info("Form import from stream, numberParsingError = " + results[3]);
        }

        return results;
    }

    /**
     * Regularly retrieves the information from all the xml streams
     * and create new forms
     *
     * @param intervalUnit unit of te interval between 2 retrievings (minute, hour or day)
     * @param intervalValue interval value between two retrievings
     * @param hour hour of the retrieving if the unit is day
     * @param minute of the retrieving if the unit is day or hour
     * @throws VradiException
     */
    public void autoLoadFormsFromXmlStreams(String intervalUnit,
                            int intervalValue, Integer hour, Integer minute)
            throws VradiException {
        int delayInMinute = 0;
        int intervalUnitInMinutes = 0;
        Calendar cal = new GregorianCalendar();
        int calHour = cal.get(Calendar.HOUR_OF_DAY);
        int calMinute = cal.get(Calendar.MINUTE);
        int diffMinute = 0;
        int diffHour = 0;
        if(minute != null) {
            diffMinute = (60 //for the diff to be positive
                    + minute - calMinute)
                    % 60 //to have only the minutes
                    ;
            if(hour != null) {
                diffHour = (24 //for the diff to be positive
                        + hour - calHour
                        - 1 // remove one hour in case the actual minute
                            // is greater than the user-defined minute
                        + ((60 + minute - calMinute) / 60)) // readd one hour
                            //  if finally the actual minute
                            // is lower than the user-defined minute
                        % 24; //to have only the hours
            }
        }
        if(log.isDebugEnabled()) {
            log.debug("delay : " + diffHour +  " hours "
                    + diffMinute + " minutes");
        }
        if(XmlStreamConfig.HOURS.toString().equals(intervalUnit)) {
            intervalUnitInMinutes = 60;
            delayInMinute = diffMinute;
        } else if(XmlStreamConfig.DAYS.toString().equals(intervalUnit)) {
            intervalUnitInMinutes = 60 * 24;
            delayInMinute = diffMinute + diffHour * 60;
        } else if(XmlStreamConfig.MINUTES.toString().equals(intervalUnit)) {
            intervalUnitInMinutes = 1;
            delayInMinute = minute;
        }
        int interval = intervalValue * intervalUnitInMinutes * 60 * 1000;
        int delay = delayInMinute * 60 * 1000;
        autoLoadFormsFromXmlStreams(delay, interval);
        PropertiesManager.setXmlStreamConfig(intervalUnit,
                String.valueOf(intervalValue),
                hour == null ? null : String.valueOf(hour),
                minute == null ? null : String.valueOf(minute));
    }

    /**
     * Regularly retrieves the information from all the xml streams
     * and create new forms
     *
     * @param delay the delay before the first retrieving
     * @param period interval between two retrievings
     * @throws VradiException
     */
    protected void autoLoadFormsFromXmlStreams(long delay, long period)
            throws VradiException {
        if(timer == null) {
            timer = new Timer();
        }
        if(xmlStreamTask != null) {
            xmlStreamTask.cancel();
            timer.purge();
        }

        xmlStreamTask = new TimerTask() {
            @Override
            public void run() {
                if (log.isInfoEnabled()) {
                    log.info("Update xml streams task");
                }

                List<XmlStream> xmlStreams = getAllXmlStreams();
                for (XmlStream xmlStream : xmlStreams) {
                    if(log.isDebugEnabled()) {
                        log.debug("Loading forms from " + xmlStream.getName());
                    }
                    try {
                        getFormsFromXmlStream(xmlStream);
                    } catch(VradiException eee) {
                        log.error("can't create forms from stream : " + xmlStream.getName());
                    }
                }
                List<WebHarvestStream> webHarvestStreams = getAllWebHarvestStreams();
                for (WebHarvestStream webHarvestStream : webHarvestStreams) {
                    if(log.isDebugEnabled()) {
                        log.debug("Loading forms from web harvest stream " + webHarvestStream.getName());
                    }
                    try {
                        getFormsFromWebHarvestStream(webHarvestStream);
                    } catch(VradiException eee) {
                        log.error("can't create forms from web harvest stream : " + webHarvestStream.getName());
                    }
                }
            }
        };
        
        timer.scheduleAtFixedRate(xmlStreamTask, delay, period);
    }
}
