/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package com.jurismarches.vradi.services;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import com.jurismarches.vradi.VradiServiceAction;
import org.apache.commons.lang.UnhandledException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.ArgumentsParserException;
import org.nuiton.wikitty.WikittyServiceCached;
import org.nuiton.wikitty.WikittyServiceNotifier;

/**
 * Configuration.
 *
 * @author schorlet
 * @version $Revision: 1283 $ $Date: 2010-09-07 14:55:45 +0200 (mar., 07 sept. 2010) $
 * @since 26 mars 2010 21:18:08
 */
public class Configuration extends ApplicationConfig {

    /** log. */
    private static final Log log = LogFactory.getLog(Configuration.class);

    /** Singleton instance. */
    protected static Configuration instance = null;

    protected Configuration() {
        super();
        
        // set default option (included configuration file name : important)
        for (Option o : Option.values()) {
            if (o.defaultValue != null) {
                setDefaultOption(o.key, o.defaultValue);
            }
        }
    }

    public static synchronized Configuration getInstance() {
        if (instance == null) {
            try {
                instance = new Configuration();
                instance = loadOptionConfiguration(instance);
                // was done in parse before (but not good for tests)
                instance.setSystemDefaultProps();
            } catch (Exception e) {
                if (log.isErrorEnabled()) {
                    log.error("", e);
                }
                throw new UnhandledException(e);
            }
        }
        return instance;
    }

    /**
     * Charge les options en testant si l'environnement n'est pas "test".
     * 
     * On doit obligatoirement faire ce code ici (au lieu des test)
     * car les tests hessian (fait pendant les test) n'utilise que
     * le code principal.
     * 
     * @return filled configuration
     * @throws ArgumentsParserException if option can't be parsed
     * @throws IOException if test file can't be loaded
     */
    protected static Configuration loadOptionConfiguration(Configuration configuration) throws ArgumentsParserException, IOException {
        String environment = System.getProperty("vradi.environment");
        if (environment == null || "prod".equals(environment)) {
            configuration.parse(new String[]{});
        }
        else {
            if (log.isInfoEnabled()) {
                log.info("Using environment : " + environment);
            }
            configuration = loadOptionConfigurationTest(configuration);
        }
        return configuration;
    }
    
    /**
     * Charge les données de test.
     * 
     * Test:
     * <ul>
     *  <li>dans target
     *  <li>n'ecoute ni n'envoie d'evenement de synchronisation
     * </ul>
     * 
     * @return filled configuration
     * @throws ArgumentsParserException if option can't be parsed
     * @throws IOException if test file can't be loaded
     */
    protected static Configuration loadOptionConfigurationTest(Configuration configuration) throws IOException {
        // test, ne va pas lire d'autres fichiers...
        InputStream input = Configuration.class.getResourceAsStream("/vradi-services.properties");
        Properties options = new Properties();
        options.load(input);
        configuration.setOptions(options);

        // working directory to target
        configuration.setOption("vradi.data.dir", "target" + File.separator + "vradi");

        // don't use wikitty synchronisation
        configuration.setWikittyListenEvents(false);
        configuration.setWikittyPropagateEvents(false);
        configuration.setWikittyJgroupsChannelName("");

        if (log.isInfoEnabled()) {
            log.info("Using options : " + configuration.getFlatOptions());
        }

        return configuration;
    }

    /**
     *
     * Init all actions
     *
     */
    public void initActions() {

        // creation des actions disponibles
        for (VradiServiceAction.ActionDefinition a : VradiServiceAction.ActionDefinition.values()) {
            for (String alias : a.aliases) {
                addActionAlias(alias, a.action);
            }
        }
    }

    protected void addActionAlias(String action, String... aliases) {
        for(String alias : aliases) {
            super.addActionAlias(alias, action);
        }
    }
                
    /**
     * @deprecated redefined to not report a bug in nuiton-utils #628
     */
    @Deprecated
    public File getOptionAsFile2(String key) {
        String optionAsString = getOption(key);
        if (optionAsString != null) {
            File optionAsFile = new File(optionAsString);
            return optionAsFile.getAbsoluteFile();
        } else {
            return null;
        }
    }

    /**
     * Get current application version as string.
     * 
     * @return version
     */
    public String getApplicationVersion() {
        return getOption(Option.APPLICATION_VERSION.key);
    }

    /**
     * Get last saved version as string (last launch).
     * 
     * @return version
     */
    public String getServiceVersion() {
        return getOption(Option.SERVICE_VERSION.key);
    }

    /**
     * Set service version.
     * 
     * @param version version to set
     */
    public void setServiceVersion(String version) {
        setOption(Option.SERVICE_VERSION.key, version);
    }

    public String getDataDir() {
        String option = getOption(Option.DATA_DIR.key);
        return option;
    }
    public File getDataDirAsFile() {
        File option = getOptionAsFile(Option.DATA_DIR.key);
        return option;
    }

    @Deprecated
    public File getCommonsPropertiesFile() {
        return getOptionAsFile("vradi.data.dir", ".vradi-properties");
    }

    public File getAttachmentsDir() {
        return getOptionAsFile("vradi.data.dir", "attachments");
    }

    public File getEmbeddedFilesDir() {
        return getOptionAsFile("vradi.data.dir", "embeddedFiles");
    }

    public File getQueryHistoryDir() {
        return getOptionAsFile("vradi.data.dir", "queryHistory");
    }

    public File getTemplatesDir() {
        return getOptionAsFile("vradi.data.dir", "templates");
    }

    public File getPdfDir() {
        return getOptionAsFile("vradi.data.dir", "pdf");
    }

    public File getWebHarvestPreviewDir() {
        return getOptionAsFile("vradi.data.dir", "WebHarvestPreview");
    }

    public File getWebHarvestScriptDir() {
        return getOptionAsFile("vradi.data.dir", "WebHarvestScript");
    }

    public String getOpenOfficeExecDir() {
        return getOption("oOo.exec.folder");
    }

    // Email config
    /*public String getMailerDaemonEmail() {
        return getOption("vradi.mail.mailerdaemonemail");
    }*/
    
    public String getSmtpHost() {
        return getOption("vradi.smtp.host");
    }

    public int getSmtpPort() {
        return getOptionAsInt("vradi.smtp.port");
    }

    public String getImapHost() {
        return getOption("vradi.imap.host");
    }

    public int getImapPort() {
        return getOptionAsInt("vradi.imap.port");
    }

    public String getMailUser() {
        return getOption("vradi.mail.user");
    }
    
    public String getMailFrom() {
        return getOption("vradi.mail.from");
    }
    
    public String getMailFromName() {
        return getOption("vradi.mail.fromname");
    }

    public String getMailPassword() {
        return getOption("vradi.mail.password");
    }

    public String getOfferMailSubject() {
        String result = getOption("vradi.mail.offer.subject");
        return result;
    }

    public boolean isMailDebug() {
        return getOptionAsBoolean("vradi.mail.debug");
    }

    public String getSessionDefaultParagraph() {
        return getOption("vradi.session.defaultParagraph");
    }

    public boolean skipMigration() {
        return getOptionAsBoolean("vradi.migration.skip");
    }

    /**
     * Change jgroups channel name.
     * 
     * @param channelName new channel name
     * 
     * @see WikittyServiceNotifier#WIKITTY_EVENT_JGROUPCHANNELNAME_OPTION
     */
    public void setWikittyJgroupsChannelName(String channelName) {
        setOption(Option.WIKITTY_EVENT_JGROUPCHANNELNAME_OPTION.key, channelName);
    }

    /**
     * Change propagate events use.
     * 
     * @param b new value
     * 
     * @see WikittyServiceNotifier#WIKITTY_EVENT_PROPAGATE_OPTION
     */
    public void setWikittyPropagateEvents(boolean b) {
        setOption(Option.WIKITTY_EVENT_PROPAGATE_OPTION.key, String.valueOf(b));
    }

    /**
     * Change listen events on cache use.
     * 
     * @param b new value
     * 
     * @see WikittyServiceCached#WIKITTY_CACHE_LISTENEVENTS_OPTION
     */
    public void setWikittyListenEvents(boolean b) {
        setOption(Option.WIKITTY_CACHE_LISTENEVENTS_OPTION.key, String.valueOf(b));
    }

    /**
     * Change wikitty cache policy.
     * 
     * @param b
     * 
     * @see WikittyServiceCached#WIKITTY_CACHE_ALLWAYS_RESTORE_COPIES_POLICY_OPTION
     */
    public void setWikittyCachePolicy(boolean b) {
        setOption(Option.WIKITTY_CACHE_ALLWAYS_RESTORE_COPIES.key, String.valueOf(b));
    }

    /*public Properties getJdbcConfigProperties() {
        File optionAsFile = getOptionAsFile2("vradi.jdbc.config.file");
        return getFileAsProperties(optionAsFile);
    }*/

    /*public Properties getJmsPublisherProperties() {
        File optionAsFile = getOptionAsFile2("vradi.jms.publisher.file");
        return getFileAsProperties(optionAsFile);
    }*/

    /*public URL getJmsPublisherURL() {
        File optionAsFile = getOptionAsFile2("vradi.jms.publisher.file");
        try {
            if (optionAsFile != null && optionAsFile.exists() && optionAsFile.isFile()) {
                return optionAsFile.toURI().toURL();
                
            } else if (optionAsFile != null) {
                String configFileName = optionAsFile.getName();
                URL url = ClassLoader.getSystemResource(configFileName);
                
                if (url == null) {
                    ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
                    url = contextClassLoader.getResource(configFileName);
                }
                
                return url;
            }
        } catch (Exception e) {
            log.warn(e.getMessage(), e);
        }

        return null;
    }*/

    /*public Properties getJmsSubscriberProperties() {
        File optionAsFile = getOptionAsFile2("vradi.jms.subscriber.file");
        return getFileAsProperties(optionAsFile);
    }*/

    protected File getOptionAsFile(String parent, String child) {
        String parentDir = getOption(parent);
        File option = new File(parentDir, child);
        return option;
    }

    /**
     * Set {@code solr} and {@code jms} system configuration.
     *
     * This is the "only" way to configure embedded solr.
     */
    protected void setSystemDefaultProps() {
        String[] datadirs = new String[] { "solr.data.dir", "vradi.objectStore.dir"};

        for (String datadir : datadirs) {
            String value = System.getProperty(datadir, null);
            if (value == null) {
                value = getOption(datadir);
                if (log.isInfoEnabled()) {
                    log.info("Setting system property " + datadir + " : " + value);
                }
                System.setProperty(datadir, value);
                env.put(datadir, value);
            }
        }
    }

    public File getTempDir() {
        String tempDir = getOption(Option.TEMP_DIR.key);
        return new File(tempDir);
    }

    /**
     * Vradi option definition.
     */
    public static enum Option {

        CONFIG_FILE(CONFIG_FILE_NAME, _("vradi.service.config.configFileName.description"), "vradi-services.properties", String.class, true, true),
        APPLICATION_VERSION("application.version", _("vradi.service.config.application.version.description"), null, String.class, false, false),
        SERVICE_VERSION("vradi.service.version", _("vradi.service.config.version.description"), null, String.class, false, false),
        DATA_DIR("vradi.data.dir", _("vradi.service.config.data.dir.description"), null, Boolean.class, false, false),
        TEMP_DIR("vradi.temp.dir", _("vradi.service.config.temp.dir.description"), null, Boolean.class, false, false),

        WIKITTY_EVENT_JGROUPCHANNELNAME_OPTION(WikittyServiceNotifier.WIKITTY_EVENT_JGROUPCHANNELNAME_OPTION, _("vradi.service.config.wikitty.jgroupschannelname.description"), "wikitty-vradi", Boolean.class, false, false),
        WIKITTY_EVENT_PROPAGATE_OPTION(WikittyServiceNotifier.WIKITTY_EVENT_PROPAGATE_OPTION, _("vradi.service.config.wikitty.propagate.description"), "true", Boolean.class, false, false),
        WIKITTY_CACHE_LISTENEVENTS_OPTION(WikittyServiceCached.WIKITTY_CACHE_LISTENEVENTS_OPTION, _("vradi.service.config.listenevents.description"), "true", Boolean.class, false, false),
        WIKITTY_CACHE_ALLWAYS_RESTORE_COPIES(WikittyServiceCached.WIKITTY_CACHE_ALLWAYS_RESTORE_COPIES_POLICY_OPTION, _("vradi.service.config.cacheusecopies.description"), "true", Boolean.class, false, false);

        public final String key;
        public final String description;
        public final String defaultValue;
        public final Class<?> type;
        public final boolean isTransient;

        public final boolean isFinal;
        private Option(String key, String description, String defaultValue,
                Class<?> type, boolean isTransient, boolean isFinal) {
            this.key = key;
            this.description = description;
            this.defaultValue = defaultValue;
            this.type = type;
            this.isFinal = isFinal;
            this.isTransient = isTransient;
        }

    }
}
