/*
 * *##% 
 * vradi-services
 * Copyright (C) 2009 JurisMarches
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package com.jurismarches.vradi.services;

import com.jurismarches.vradi.entities.*;
import com.jurismarches.vradi.services.dto.VradiFormPageDTO;
import com.jurismarches.vradi.services.dto.VradiSendingDTO;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdom.Document;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.XMLOutputter;
import static org.nuiton.i18n.I18n._;
import org.nuiton.util.*;
import org.sharengo.exceptions.TechnicalException;
import org.sharengo.wikitty.*;
import org.sharengo.wikitty.jdbc.WikittyServiceJDBC;
import org.sharengo.wikitty.search.Element;
import org.sharengo.wikitty.search.Search;
import org.sharengo.wikitty.solr.WikittySearchEngineSolr;

import java.io.*;
import java.math.BigDecimal;
import java.net.URL;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author morin
 */
public class VradiStorageServiceImpl implements VradiStorageService {

    static private final Log log =
            LogFactory.getLog(VradiStorageServiceImpl.class);

    /**
     * Proxy to store and retrieve the data
     */
    protected WikittyProxy proxy;

    //patterns to parse the queries
    /**
     * Pattern corresponding to a field name :
     * a field name can contain any letter, number, blank,
     * but no brackets neither "AND" or "OR".
     * The field name is followed by a double dot
     */
    protected static final String fieldPattern =
            "[\\p{Graph}\\p{Blank}\\p{L}\\p{Lu}&&[^\\(\\)(AND)(OR)]]+:";
    /**
     * Pattern corresponding to a value : a value can contain any letter,
     * number, blank, but no brackets.
     */
    protected static final String valuePattern =
            "[\\p{Graph}\\p{Blank}\\p{L}\\p{Lu}&&[^\\(\\)]]+";
    /**
     * Pattern corresponding to the "NOT" condition : "NOT "
     */
    protected static final String notPattern = "(NOT\\p{Blank})?";
    /**
     * General pattern to parse the queries. A query is like :
     * "field name:(value OR value OR value)
     * AND field name:((value OR value) AND value)"
     */
    protected static final Pattern queryPattern = Pattern.compile(
            "(" + fieldPattern + ")?" +
                    "(" +
                    "\\(" + notPattern + valuePattern +
                    "(" +
                    "\\p{Blank}((AND)|(OR))\\p{Blank}" + notPattern
                    + valuePattern +
                    ")*" + "\\)" +
                    ")" +
                    "(\\p{Blank}((AND)|(OR))\\p{Blank}" +
                    "(" + fieldPattern + ")?" +
                    "(" +
                    "\\(" + notPattern + valuePattern +
                    "(" +
                    "\\p{Blank}((AND)|(OR))\\p{Blank}" + notPattern
                    + valuePattern +
                    ")*" + "\\)" +
                    ")" +
                    ")*?");

    /**
     * Thesaurus root node name
     */
    public static final String ROOT_THESAURUS_NAME = "Thesaurus";

    //name of the rss tags
    protected static final String channel = "channel";
    protected static final String entry = "entry";
    protected static final String feed = "feed";
    protected static final String item = "item";
    protected static final String rdf = "rdf";
    /**
     * RSS date format
     */
    public static final SimpleDateFormat RSS_DATE_FORMAT =
            new SimpleDateFormat("E, d MMM yyyy HH:mm:ss Z",
                    java.util.Locale.US);
    /**
     * default value of the sending status
     */
    public final static int DEFAULT_SENDING_STATUS = 0;
    /**
     * default value of the sending paragraph : cannot be null or empty
     * to be able to look for it in the solr index
     */
    public final static String DEFAULT_SENDING_PARAGRAPH = "null";
    /**
     * Vradi data directory
     */
    protected String vradiDataDirName = System.getProperty("user.home")
            + File.separator + ".vradi/vradi-0.0.3";
    /**
     * Cache of the queries in the database
     */
    protected List<String> queriesInTheDB = null;
    /**
     * Cache of the entities
     */
    protected Map<Class, Map> cacheMap = new HashMap<Class, Map>();

    /**
     * Format of the automatically created form id
     */
    public static class FormIdDateFormat extends SimpleDateFormat {
        public FormIdDateFormat() {
            super("yyyy-MM-dd");
        }
    }

    /**
     * Constructor :
     * - creates (if not exists) the vradi data directory
     * and the solr config file
     * - init the proxy
     */
    public VradiStorageServiceImpl() {
        proxy = new WikittyProxy();
        try {
            File vradiDataDir = new File(vradiDataDirName);
            vradiDataDir.mkdirs();
            File userSolrConfig = new File(vradiDataDirName,
                    WikittySearchEngineSolr.SOLRCONFIG);
            if (!userSolrConfig.exists()) {
                InputStream solrConfig = getClass().getClassLoader()
                        .getResourceAsStream(
                                WikittySearchEngineSolr.SOLRCONFIG);
                FileWriter fw = new FileWriter(userSolrConfig);
                SAXBuilder sxb = new SAXBuilder();
                Document document = sxb.build(solrConfig);
                org.jdom.Element racine = document.getRootElement();
                racine.getChild("dataDir").setText("${solr.data.dir:"
                        + vradiDataDirName + "/solr/data}");
                XMLOutputter outputter = new XMLOutputter("    ", true);
                outputter.output(document, fw);
                fw.close();
            }

        } catch (JDOMException eee) {
            if (log.isErrorEnabled()) {
                log.error(eee);
            }
        } catch (IOException eee) {
            if (log.isErrorEnabled()) {
                log.error(eee);
            }
        }
        proxy.setWikittyService(new WikittyServiceJDBC(vradiDataDirName));
//        Logger.getLogger("org.apache.solr").setLevel(Level.FINEST);
    }

    /**
     * Initialize the cache of the entities whose class is clazz.
     *
     * @param clazz the class of the entities to put in the cache
     * @return If clazz is an implementation, then it returns its interface,
     *         otherwise it returns clazz
     * @throws TechnicalException
     */
    protected Class initCacheMap(Class clazz) throws TechnicalException {
//        if(cacheMap.get(clazz) == null) {
        //test class and add all
        if (User.class.isAssignableFrom(clazz)) {
            getAllUsers();
            return User.class;
        } else if (Client.class.isAssignableFrom(clazz)) {
            getAllClients();
            return Client.class;
        } else if (Group.class.isAssignableFrom(clazz)) {
            getAllGroups();
            return Group.class;
        } else if (XmlStream.class.isAssignableFrom(clazz)) {
            getAllXmlStreams();
            return XmlStream.class;
        } else if (TreeNode.class.isAssignableFrom(clazz)) {
            getAllThesaurus();
            return TreeNode.class;
        }
        return null;
    }

    /**
     * Retrieve from the cache the entity of id 'id' and whose class is 'clazz'
     * after having initialized the cache
     *
     * @param id    the id of the entity to retrieve
     * @param clazz the class of the entity to retrieve
     * @return the entity of id 'id' and whose class is 'clazz'
     * @throws TechnicalException
     */
    @Override
    public BusinessEntity getEntity(String id, Class clazz)
            throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getEntity");
        }
        initCacheMap(clazz);
        if (id != null) {
            return (BusinessEntity) cacheMap.get(clazz).get(id);
        }
        return null;
    }

    /**
     * Retrieves from the cache all the entities whose class is clazz.
     * If the cache is empty, then it fills it with the data base data.
     *
     * @param clazz the class of the entities to retrieve
     * @param ext   the WikittyExtension corresponding to clazz
     * @return all the entities whose class is clazz
     */
    protected List<BusinessEntity> getAllEntities(Class clazz, String ext) {
        if (cacheMap.get(clazz) == null) {
            Criteria criteria =
                    Search.query().eq(Element.ELT_EXTENSION, ext).criteria();
            List<BusinessEntity> entities =
                    proxy.findAllByCriteria(clazz, criteria).getAll();
            Map<String, BusinessEntity> entitiesMap
                    = new HashMap<String, BusinessEntity>();
            for (BusinessEntity entity : entities) {
                log.debug(entity.getWikittyId() + " : "
                        + entity.getExtensionNames());
                entitiesMap.put(entity.getWikittyId(), entity);
            }
            cacheMap.put(clazz, entitiesMap);
            return new ArrayList<BusinessEntity>(entities);
        }
        log.debug(clazz + " loaded");
        return new ArrayList<BusinessEntity>(cacheMap.get(clazz).values());

    }

    /**
     * Updates the entity 'entity' in the database,
     * the solr index and the cache.
     *
     * @param entity the entity to update
     * @return the updated entity
     * @throws TechnicalException
     */
    @Override
    public BusinessEntity updateEntity(BusinessEntity entity)
            throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("updateEntity");
        }
        if (entity != null) {
            //if the netity is a query maker,
            // then put its new queries in the queries cache
            if (QueryMaker.class.isAssignableFrom(entity.getClass())
                    && queriesInTheDB != null) {
                Set<String> queries = ((QueryMaker) entity).getQueries();
                if (queries != null) {
                    for (String query : queries) {
                        if (!queriesInTheDB.contains(query)) {
                            queriesInTheDB.add(query);
                        }
                    }
                }
            }
            entity.setWikittyVersion(WikittyUtil.incrementMajorRevision(
                    entity.getWikittyVersion()));
            // update the entity in the database and the solr index
            BusinessEntity result = proxy.store(entity);

            //get the interface of the entity
            Class clazz = initCacheMap(entity.getClass());
            // update the entity in the cache
            log.debug("update entity : " + result.getWikittyId());
            if (TreeNode.class.isAssignableFrom(entity.getClass())) {
                log.debug(((TreeNode) entity).getChildren());
            }
            cacheMap.get(clazz).put(result.getWikittyId(), result);
            return result;
        }
        return null;
    }

    /**
     * Delete the entity from the database, the solr index and the cache
     *
     * @param entity the entity to delete
     * @throws TechnicalException
     */
    @Override
    public void deleteEntity(BusinessEntity entity) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("deleteEntity");
        }
        if (entity != null) {
            proxy.delete(entity.getWikittyId());
            Class clazz = initCacheMap(entity.getClass());
            cacheMap.get(clazz).remove(entity);
        }
    }

    /**
     * Retrieves the user whose id is 'id'
     *
     * @param userId the id of the user to retrieve
     * @return the user whose id is 'id'
     * @throws TechnicalException
     */
    @Override
    public User getUser(String userId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getUser");
        }
        return (User) getEntity(userId, User.class);
    }

    /**
     * Retrieves the client whose id is 'id'
     *
     * @param clientId the id of the client to retrieve
     * @return the client whose id is 'id'
     * @throws TechnicalException
     */
    @Override
    public Client getClient(String clientId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getClient");
        }
        return (Client) getEntity(clientId, Client.class);
    }

    /**
     * Retrieves the group whose id is 'id'
     *
     * @param groupId the id of the group to retrieve
     * @return the group whose id is 'id'
     * @throws TechnicalException
     */
    @Override
    public Group getGroup(String groupId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getGroup");
        }
        return (Group) getEntity(groupId, Group.class);
    }

    /**
     * Retrieves the users from the group whose id is 'id'
     *
     * @param groupId the id of the group which contains the users to retrieve
     * @return the users from the group whose id is 'id'
     * @throws TechnicalException
     */
    @Override
    public List<User> getGroupUsers(String groupId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getGroupUsers");
        }
        List<User> result = new ArrayList<User>();
        Group group = getGroup(groupId);
        if (group != null) {
            for (String userId : group.getUser()) {
                User user = getUser(userId);
                result.add(user);
            }
        }
        return result;
    }

    /**
     * Retrieves the users from the client whose id is 'id'
     *
     * @param clientId the id of the client which contains the users to retrieve
     * @return the users from the client whose id is 'id'
     * @throws TechnicalException
     */
    @Override
    public List<User> getClientUsers(String clientId)
            throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getClientUsers");
        }
        List<User> result = new ArrayList<User>();
        Client client = getClient(clientId);
        if (client != null && client.getUser() != null) {
            for (String userId : client.getUser()) {
                User user = getUser(userId);
                result.add(user);
            }
        }
        return result;
    }

    /**
     * Retrieves the clients from the group whose id is 'id'
     *
     * @param groupId the id of the group which contains the clients to retrieve
     * @return the clients from the group whose id is 'id'
     * @throws TechnicalException
     */
    @Override
    public List<Client> getGroupClients(String groupId)
            throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getGroupClients");
        }
        List<Client> result = new ArrayList<Client>();
        Group group = getGroup(groupId);
        if (group != null && group.getClient() != null) {
            for (String clientId : group.getClient()) {
                Client client = getClient(clientId);
                result.add(client);
            }
        }
        return result;
    }

    /**
     * Retrieves the client containing the user whose id is 'id'
     *
     * @param userId the id of the user contained by the client to retrieve
     * @return the client containing the user whose id is 'id'
     * @throws TechnicalException
     */
    @Override
    public Client getClientByUserId(String userId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getClientByUserId");
        }
        User user = getUser(userId);
        String clientId = user.getClient();
        return getClient(clientId);
    }

    /**
     * Retrieves the group containing the user whose id is 'id'
     *
     * @param userId the id of the user contained by the group to retrieve
     * @return the group containing the user whose id is 'id'
     * @throws TechnicalException
     */
    @Override
    public List<Group> getGroupsByUserId(String userId)
            throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getGroupsByUserId");
        }
        User user = getUser(userId);
        List<String> groupIds = new ArrayList(user.getGroup());
        List<Group> result = new ArrayList<Group>();
        for (String groupId : groupIds) {
            result.add(getGroup(groupId));
        }
        return result;
    }

    /**
     * Retrieves the group containing the client whose id is 'id'
     *
     * @param clientId the id of the client contained by the group to retrieve
     * @return the group containing the client whose id is 'id'
     * @throws TechnicalException
     */
    @Override
    public List<Group> getGroupsByClientId(String clientId)
            throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getGroupsByClientId");
        }
        Client client = getClient(clientId);
        List<String> groupIds = new ArrayList(client.getGroup());
        List<Group> result = new ArrayList<Group>();
        for (String groupId : groupIds) {
            result.add(getGroup(groupId));
        }
        return result;
    }

    /**
     * Retrieves all the clients from the cache
     *
     * @return all the clients in the cache
     * @throws TechnicalException
     */
    @Override
    public List<Client> getAllClients() throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getAllClients");
        }
        List<Client> result = new ArrayList<Client>();
        List<BusinessEntity> entities =
                getAllEntities(Client.class, Client.EXT_CLIENT);
        for (BusinessEntity entity : entities) {
            result.add((Client) entity);
        }
        return result;
    }

    /**
     * Retrieves all the users from the cache
     *
     * @return all the users in the cache
     * @throws TechnicalException
     */
    @Override
    public List<User> getAllUsers() throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getAllUsers");
        }
        List<User> result = new ArrayList<User>();
        List<BusinessEntity> entities =
                getAllEntities(User.class, User.EXT_USER);
        for (BusinessEntity entity : entities) {
            result.add((User) entity);
        }
        return result;
    }

    /**
     * Retrieves all the groups from the cache
     *
     * @return all the groups in the cache
     * @throws TechnicalException
     */
    @Override
    public List<Group> getAllGroups() throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getAllGroups");
        }
        List<Group> result = new ArrayList<Group>();
        List<BusinessEntity> entities =
                getAllEntities(Group.class, Group.EXT_GROUP);
        for (BusinessEntity entity : entities) {
            result.add((Group) entity);
        }
        return result;
    }

    /**
     * Saves a form
     *
     * @param form the form to insert or update
     * @return the form stored
     */
    @Override
    public Form updateForm(Form form) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("updateForm");
        }
        if (form != null) {
            if (form.getDatePub() == null) {
                form.setDatePub(new Date());
            }
            Set<String> thesaurus = form.getThesaurus();
            if (thesaurus != null) {
                for (String thesaurusId : thesaurus) {
                    TreeNode node = getThesaurus(thesaurusId);
                    if (node.getChildren() == null ||
                            !node.getChildren().contains(form.getWikittyId())) {
                        node.addChildren(form.getWikittyId());
                        updateEntity(node);
                    }
                }
            }
            form.setWikittyVersion(
                    WikittyUtil.incrementMajorRevision(
                            form.getWikittyVersion()));
            return proxy.store(form);
        }
        return null;
    }

    /**
     * Saves a form list
     *
     * @param forms the list of forms to insert or update
     * @return the forms stored
     */
    @Override
    public List<Form> updateForms(List<Form> forms) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("updateForm");
        }
        List<TreeNode> thesaurusList = new ArrayList<TreeNode>();
        for (Form form : forms) {
            if (form != null) {
                if (form.getDatePub() == null) {
                    form.setDatePub(new Date());
                }
                Set<String> thesaurus = form.getThesaurus();
                if (thesaurus != null) {
                    for (String thesaurusId : thesaurus) {
                        TreeNode node = getThesaurus(thesaurusId);
                        if (node.getChildren() == null ||
                                !node.getChildren().contains(
                                        form.getWikittyId())) {
                            node.addChildren(form.getWikittyId());
                            if (!thesaurusList.contains(node)) {
                                thesaurusList.add(node);
                            }
                        }
                    }
                }
                form.setWikittyVersion(
                        WikittyUtil.incrementMajorRevision(
                                form.getWikittyVersion()));
            }
        }
        proxy.store(thesaurusList);
        List<Form> result = proxy.store(forms);
        return result;
    }

    /**
     * Deletes a Form
     *
     * @param formId the id of the Form to delete
     */
    @Override
    public void deleteForm(String formId) {
        if (log.isDebugEnabled()) {
            log.debug("deleteForm");
        }
        if (formId != null) {
            proxy.delete(formId);
        }
    }

    /**
     * Find the forms filtered by the parameter query
     *
     * @param query the query to filter the forms
     * @return the forms filtered by query
     */
    @Override
    public void findForms(String query, VradiFormPageDTO formPageDTO) {
        if (log.isDebugEnabled()) {
            log.debug("findForms");
        }
        findForms(query, null, null, null, null, null, formPageDTO);
    }

    /**
     * Find the forms filtered by the parameter query
     *
     * @param query     the query to filter the forms
     * @param extension the extension filter
     * @param dateType  the date type of the filter
     * @param beginDate the minimum value of the date type field
     * @param endDate   the maximum value of the date type field
     * @param thesaurus the thesaurus associated with filter
     * @return the forms filtered by query
     */
    @Override
    public void findForms(String query, WikittyExtension extension,
                          String dateType, Date beginDate, Date endDate,
                          List[] thesaurus, VradiFormPageDTO formPageDTO) {
        if (log.isDebugEnabled()) {
            log.debug("findForms " + dateType + " " + beginDate
                    + " - " + endDate + "  : " + extension);
        }
        Search search = Search.query();
        // map of the group caught by the patterns
        Map<String, String> caught = parseQuery(query);
        if (caught != null && caught.size() > 1) {
            //creates the query
            createQuery(search, caught, "_search" + (caught.size() - 1) + "_",
                    null);
        } else if (query != null) {
            //the query are only keywords
            String[] tokens = query.split("\\s");
            if (tokens.length > 0) {
                for (String token : tokens) {
                    if (!token.isEmpty()) {
                        Search subSearch = search.or();
                        subSearch.keyword(token);
                        subSearch.keyword(token + "*");
                    } else if (tokens.length == 1) {
                        search.keyword("*");
                    }
                }
            } else {
                search.keyword("*");
            }
        }
        //add the extension in the criteria
        if (extension != null) {
            search.eq(Element.ELT_EXTENSION, extension.getName());
        }
        //add the date in the criteria
        if (dateType != null) {
            if (beginDate != null && endDate != null) {
                log.info(dateType);
                String bDateS = WikittyUtil.solrDateFormat.format(beginDate);
                String eDateS = WikittyUtil.solrDateFormat.format(endDate);
                if (!beginDate.equals(endDate)) {
                    search.bw(dateType, bDateS, eDateS);
                }
            }
        }
        //add the thesaurus in the criteria
        if (thesaurus != null) {
            for (int i = 0; i < thesaurus.length; i++) {
                if (thesaurus[i] != null) {
                    Search subSearch = search.or();
                    for (Object th : thesaurus[i]) {
                        subSearch.eq(Form.FQ_FIELD_THESAURUS, (String) th);
                    }
                }
            }
        }
        Criteria criteria = search.criteria();
        //sets the pagination parameters
        int firstIndex = (formPageDTO.getPageToShow() - 1)
                * formPageDTO.getNbFormsToShow();
        int lastIndex = formPageDTO.getPageToShow()
                * formPageDTO.getNbFormsToShow() - 1;
        if (firstIndex >= 0 && lastIndex > 0 && lastIndex > firstIndex) {
            criteria =
                    criteria.setFirstIndex(firstIndex).setEndIndex(lastIndex);
        }
        if (formPageDTO.getFieldToSort() != null) {
            if (!formPageDTO.isAscending()) {
                criteria = criteria.addSortDescending(
                        formPageDTO.getFieldToSort());
            } else {
                criteria = criteria.addSortAscending(
                        formPageDTO.getFieldToSort());
            }
        }
        criteria = criteria.addSortAscending(Form.FIELD_NAME);
        //finds the forms
        PagedResult<Form> queryResult =
                proxy.findAllByCriteria(Form.class, criteria);
        List<Form> result = new ArrayList<Form>(queryResult.getAll());
        for (Form form : result) {
            if (log.isDebugEnabled()) {
                log.debug("found: " + form.toString());
            }
        }
        formPageDTO.setFormsToShow(result);
        formPageDTO.setTotalFoundFormNb(queryResult.getNumFound());
    }

    /**
     * Gets the recorded queries that returns the form 'form'.
     *
     * @param form the form which is returned by the queries
     * @result the queries of the query makers which returns the form
     */
    @Override
    public List<String> getQueriesReturningForm(Form form)
            throws TechnicalException {
        List<String> result = new ArrayList<String>();
        //get all the queries of the query makers
        if (queriesInTheDB == null) {
            List<QueryMaker> queryMakers = new ArrayList<QueryMaker>();
            queryMakers.addAll(getAllClients());
            queryMakers.addAll(getAllGroups());
            queryMakers.addAll(getAllUsers());

            queriesInTheDB = new ArrayList<String>();
            for (QueryMaker queryMaker : queryMakers) {
                if (queryMaker.getQueries() != null) {
                    for (String query : queryMaker.getQueries()) {
                        if (!queriesInTheDB.contains(query)) {
                            queriesInTheDB.add(query);
                        }
                    }
                }
            }
        }

        for (String query : queriesInTheDB) {
            //parse the query
            Map<String, String> caught = parseQuery(query);
            if (caught != null) {
                for (Map.Entry<String, String> stringStringEntry :
                        caught.entrySet()) {
                    log.info(stringStringEntry.getKey() + " : "
                            + stringStringEntry.getValue());
                }
                //if the query is returning the form, add it to the result
                Boolean formFound = isQueryReturningForm(caught,
                        "_search" + (caught.size() - 1) + "_", null, form);
                if (formFound != null && formFound) {
                    result.add(query);
                }
            }
        }
        return result;
    }

    /**
     * Separates the different token of the query into groups
     *
     * @param query the query to parse
     * @return a map linking the group names to the token groups
     */
    protected Map<String, String> parseQuery(String query) {
        if (query != null && !query.isEmpty()) {
            String s = query;
            boolean hasMatched = true;
            int k = 0;
            //map of the query elements found
            Map<String, String> caught = new HashMap<String, String>();
            //while the matcher matches the pattern
            while (hasMatched) {
                hasMatched = false;
                Matcher m = queryPattern.matcher(s);
                while (m.find()) {
                    hasMatched = true;
                    //"_search" + k + "_" is a kind of id
                    // for the pattern detected
                    caught.put("_search" + k + "_", m.group(0).trim());
                    //replace the pattern by the id
                    s = s.replace(m.group(0).trim(), "_search" + k++ + "_");
                }
            }
            caught.put("_search" + k + "_", "(" + s + ")");
            return caught;
        }
        return null;
    }

    /**
     * Get all the form types
     *
     * @return the list of the wikittyExtension representing the form types
     */
    @Override
    public List<WikittyExtension> getAllFormTypes() {
        if (log.isDebugEnabled()) {
            log.debug("getAllFormTypes");
        }
        Map map = cacheMap.get(WikittyExtension.class);
        if (map == null) {
            List<WikittyExtension> result = new ArrayList<WikittyExtension>();
            WikittyService wikittyService = proxy.getWikittyService();
            result.addAll(wikittyService.getAllExtensions(true));

            List<WikittyExtension> formTypes
                    = new ArrayList<WikittyExtension>();
            Map<String, WikittyExtension> extensionMap =
                    new HashMap<String, WikittyExtension>();
            for (WikittyExtension extension : result) {
                if (extension != null
                        && Form.EXT_FORM.equals(extension.getRequires())) {
                    formTypes.add(extension);
                    extensionMap.put(extension.getName(), extension);
                }
            }
            cacheMap.put(WikittyExtension.class, extensionMap);
            if (log.isDebugEnabled()) {
                log.debug("Number of FormType found : " + result.size());
            }
            return formTypes;
        }
        return new ArrayList<WikittyExtension>(map.values());
    }

    /**
     * Get the last version of the extension named name
     *
     * @param name the name of the extension
     * @return the last version of the extension
     */
    @Override
    public WikittyExtension getFormType(String name) {
        if (log.isDebugEnabled()) {
            log.debug("getFormType");
        }
        if (cacheMap.get(WikittyExtension.class) == null) {
            getAllFormTypes();
        }
        return name == null ? null :
                (WikittyExtension) cacheMap.get(WikittyExtension.class)
                        .get(name);
    }

    /**
     * Get the fields of a form type
     *
     * @param name the name of the field type
     * @return a map containing the names of the fields and their type
     */
    @Override
    public Map<String, FieldType> getFormTypeFields(String name) {
        if (log.isDebugEnabled()) {
            log.debug("getFormTypeFields");
        }
        WikittyExtension ext = getFormType(name);
        Map<String, FieldType> result = new HashMap<String, FieldType>();
        for (String fieldName : ext.getFieldNames()) {
            result.put(fieldName, ext.getFieldType(fieldName));
        }

        return result;
    }

    /**
     * Inserts a form type. If the form type is modified, then the version
     * is automaticaly incremented.
     *
     * @param name      the name of the form type
     * @param fields    the fields of the form type
     * @param requires  the extension requred by the extension to update
     * @param tagValues the map containing the tags and their values
     * @return the WikittyExtension corresponding to the form type
     */
    @Override
    public WikittyExtension updateFormType(String name,
                                           Map<String, FieldType> fields,
                                           String requires,
                                           Map<String, String> tagValues) {
        if (log.isDebugEnabled()) {
            log.debug("updateFormType");
        }
        if (name != null) {
            WikittyExtension result = null;
            WikittyExtension lastVersion = getFormType(name);
            if (log.isDebugEnabled()) {
                log.debug("lastVersion : " + lastVersion);
            }
            if (lastVersion != null) {
                result = new WikittyExtension(name,
                        WikittyUtil.incrementMajorRevision(
                                lastVersion.getVersion()),
                        requires, (LinkedHashMap) fields);
            } else {
                result = new WikittyExtension(name, WikittyUtil.DEFAULT_VERSION,
                        requires, (LinkedHashMap) fields);
            }
            for (Map.Entry entry : tagValues.entrySet()) {
                result.addTagValue((String) entry.getKey(),
                        (String) entry.getValue());
            }
            List<WikittyExtension> extensions = Arrays.asList(result);
            proxy.getWikittyService().storeExtension(extensions);
            if (log.isDebugEnabled()) {
                log.debug("formType : " + result.getId());
            }
            cacheMap.get(WikittyExtension.class).put(name, result);
            return result;
        }
        return null;
    }

    /**
     * Inserts a form type. If the form type is modified, then the version
     * is automaticaly incremented.
     *
     * @param extension to store
     * @return the WikittyExtension corresponding to the form type
     */
    @Override
    public WikittyExtension updateFormType(WikittyExtension extension) {
        if (log.isDebugEnabled()) {
            log.debug("updateFormType");
        }
        Map<String, FieldType> fields = new LinkedHashMap<String, FieldType>();
        for (String fieldName : extension.getFieldNames()) {
            FieldType fieldType = extension.getFieldType(fieldName);
            log.debug(fieldName + " : " + fieldType.getTagValue("description"));
            fields.put(fieldName, fieldType);
        }

        return updateFormType(extension.getName(), fields,
                extension.getRequires(), extension.getTagValues());
    }

    /**
     * Get the form whose id is id.
     *
     * @param formId the id of the form
     * @return the form whose id is id
     */
    @Override
    public Form getForm(String formId) {
        if (log.isDebugEnabled()) {
            log.debug("getForm");
        }
        Form result = null;
        if (formId != null) {
            result = proxy.restore(Form.class, formId);
        }
        return result;
    }

    /**
     * Get the form whose id is id.
     *
     * @param formIds the ids of the forms
     * @return the forms whose id is in the list of ids
     */
    @Override
    public List<Form> getForms(List<String> formIds) {
        if (log.isDebugEnabled()) {
            log.debug("getForm");
        }
        List<Form> result = null;
        if (formIds != null) {
            result = proxy.restore(Form.class, formIds);
        }
        return result;
    }

    /**
     * Get all the forms recorded
     *
     * @return the list of all the forms
     */
    @Override
    public List<Form> getAllForms() {
        if (log.isDebugEnabled()) {
            log.debug("getAllForms");
        }
        Criteria criteria = Search.query()
                .eq(Element.ELT_EXTENSION, Form.EXT_FORM).criteria();
        return new ArrayList<Form>(
                proxy.findAllByCriteria(Form.class, criteria).getAll());
    }

    /**
     * Add the AND or OR to the search
     *
     * @param search the Search instance to add the conditions
     * @param caught the map of the patterns with their id
     * @param sss    the id of the pattern matched
     * @param field  the field which must (or not) be equals to the values
     */
    protected void createQuery(Search search, Map<String, String> caught,
                               String sss, String field) {
        if (log.isDebugEnabled()) {
            log.debug("createQuery");
        }
        String si = caught.get(sss);
        String expression;
        if (si.startsWith("(")) {
            expression = si.substring(1, si.length() - 1);
        } else {
            field = si.substring(0, si.indexOf(":"));
            expression = si.substring(si.indexOf("(") + 1, si.length() - 1);
        }
        String[] orSplitted = expression.split(" OR ");
        if (orSplitted.length > 1) {
            addTokens(search.or(), orSplitted, field, caught);
        } else {
            String[] andSplitted = expression.split(" AND ");
            if (andSplitted.length > 1) {
                addTokens(search.and(), andSplitted, field, caught);
            } else {
                addTokens(search, new String[]{expression}, field, caught);
            }
        }
    }

    /**
     * Add the equals or not equals conditions to the search
     *
     * @param search the Search instance to add the conditions
     * @param tokens the list of values that must equals or not the search
     * @param field  the field which must (or not) be equals to the values
     * @param caught the map of the patterns with their id
     */
    protected void addTokens(Search search, String[] tokens, String field,
                             Map<String, String> caught) {
        if (log.isDebugEnabled()) {
            log.debug("equalsFieldValue");
        }
        for (String token : tokens) {
            if (token.trim().startsWith("NOT ")) {
                if(field == null) {
                    search.not().keyword(token.trim());        
                } else {
                    Search subSearch = search.and();
                    subSearch.neq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                            + field + Criteria.SEPARATOR
                            + Element.ElementType.TEXT, token.trim().substring(4));
                    subSearch.neq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                            + field + Criteria.SEPARATOR
                            + Element.ElementType.TEXT, token.trim().substring(4)
                            + "*");
                    try {
                        subSearch.neq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                                + field + Criteria.SEPARATOR
                                + Element.ElementType.DATE,
                                WikittyUtil.solrDateFormat.format(
                                        DateFormat.getInstance()
                                                .parse(token.trim().substring(4))));
                    } catch (ParseException eee) {
                        if (log.isDebugEnabled()) {
                            log.debug(token + " cannot be a date.");
                        }
                    }
                    try {
                        Boolean.parseBoolean(token.trim().substring(4));
                        subSearch.neq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                                + field + Criteria.SEPARATOR
                                + Element.ElementType.BOOLEAN,
                                token.trim().substring(4).toLowerCase());
                    } catch (Exception eee) {
                        if (log.isDebugEnabled()) {
                            log.debug(token + " cannot be a boolean.");
                        }
                    }
                    try {
                        Double.valueOf(token.trim().substring(4));
                        subSearch.neq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                                + field + Criteria.SEPARATOR
                                + Element.ElementType.NUMERIC,
                                token.trim().substring(4));
                    } catch (NumberFormatException eee) {
                        if (log.isDebugEnabled()) {
                            log.debug(token + " cannot be a number.");
                        }
                    }
                    subSearch.neq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                            + field + Criteria.SEPARATOR
                            + Element.ElementType.WIKITTY,
                            token.trim().substring(4));
                }
            } else {
                if (caught.containsKey(token.trim())) {
                    createQuery(search, caught, token.trim(), field);
                } else if(field == null) {
                    search.keyword(token.trim());
                } else {
                    Search subSearch = search.or();
                    subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                            + field + Criteria.SEPARATOR
                            + Element.ElementType.TEXT, token.trim());
                    subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                            + field + Criteria.SEPARATOR
                            + Element.ElementType.TEXT, token.trim() + "*");
                    try {
                        Calendar calendar = new GregorianCalendar();
                        calendar.setTime(DateFormat
                                .getDateInstance(DateFormat.SHORT,
                                        Locale.FRANCE)
                                .parse(token.trim()));
                        Calendar beginCal = new GregorianCalendar();
                        beginCal.set(calendar.get(Calendar.YEAR),
                                calendar.get(Calendar.MONTH),
                                calendar.get(Calendar.DAY_OF_MONTH), 0, 0, 0);
                        Calendar endCal = new GregorianCalendar();
                        endCal.set(calendar.get(Calendar.YEAR),
                                calendar.get(Calendar.MONTH),
                                calendar.get(Calendar.DAY_OF_MONTH), 0, 0, 0);
                        endCal.add(Calendar.DAY_OF_MONTH, 1);
                        log.info(beginCal.getTime() + " - " + endCal.getTime());
                        subSearch.bw(Criteria.ALL_EXTENSIONS
                                + Criteria.SEPARATOR
                                + field + Criteria.SEPARATOR
                                + Element.ElementType.DATE,
                                WikittyUtil.solrDateFormat
                                        .format(beginCal.getTime()),
                                WikittyUtil.solrDateFormat
                                        .format(endCal.getTime()));
                    } catch (ParseException eee) {
                        if (log.isDebugEnabled()) {
                            log.debug(token + " cannot be a date.");
                        }
                    }
                    try {
                        Boolean.parseBoolean(token.trim());
                        subSearch.eq(Criteria.ALL_EXTENSIONS
                                + Criteria.SEPARATOR
                                + field + Criteria.SEPARATOR
                                + Element.ElementType.BOOLEAN, token.trim());
                    } catch (Exception eee) {
                        if (log.isDebugEnabled()) {
                            log.debug(token.toLowerCase()
                                    + " cannot be a boolean.");
                        }
                    }
                    try {
                        Double.valueOf(token.trim());
                        subSearch.eq(Criteria.ALL_EXTENSIONS
                                + Criteria.SEPARATOR
                                + field + Criteria.SEPARATOR
                                + Element.ElementType.NUMERIC, token.trim());
                    } catch (NumberFormatException eee) {
                        if (log.isDebugEnabled()) {
                            log.debug(token + " cannot be a number.");
                        }
                    }
                    subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                            + field + Criteria.SEPARATOR
                            + Element.ElementType.WIKITTY, token.trim());
                }
            }
        }
    }

    /**
     * Add the AND or OR to the search
     *
     * @param caught the map of the patterns with their id
     * @param sss    the id of the pattern matched
     * @param field  the field which must (or not) be equals to the values
     * @param form   the form we are looking for in the queries
     * @return null if the form is null,
     *         true if the form is returned by the query, false otherwise
     */
    protected Boolean isQueryReturningForm(Map<String, String> caught,
                                           String sss, String field,
                                           Form form) {
        Boolean result;
        if (log.isDebugEnabled()) {
            log.debug("createQuery");
        }
        String si = caught.get(sss);
        String expression;
        if (si.startsWith("(")) {
            expression = si.substring(1, si.length() - 1);
        } else {
            field = si.substring(0, si.indexOf(":"));
            expression = si.substring(si.indexOf("(") + 1, si.length() - 1);
        }
        String[] orSplitted = expression.split(" OR ");
        if (orSplitted.length > 1) {
            result = equalsFieldValue(orSplitted, field, caught, form, false);
        } else {
            String[] andSplitted = expression.split(" AND ");
            if (andSplitted.length > 1) {
                result = equalsFieldValue(andSplitted, field, caught, form,
                        true);
            } else {
                result = equalsFieldValue(new String[]{expression}, field,
                        caught, form, true);
            }
        }
        return result;
    }

    /**
     * Add the equals or not equals conditions to the search
     *
     * @param tokens the list of values that must equals or not the search
     * @param field  the field which must (or not) be equals to the values
     * @param caught the map of the patterns with their id
     * @param form   the form we are looking for in the queries
     * @param and    if true, an 'and' condition is beetwen the tokens,
     *               an 'or' otherwise
     * @return null if the form is null,
     *         true if the form is returned by the query, false otherwise
     */
    protected Boolean equalsFieldValue(String[] tokens, String field,
                                       Map<String, String> caught, Form form,
                                       Boolean and) {
        if (log.isDebugEnabled()) {
            log.debug("equalsFieldValue");
        }
        Boolean result = null;
        if (form != null) {
            //if a field is defined in the query
            if (field != null) {
                Map<String, FieldType> extensions
                        = new HashMap<String, FieldType>();
                for (WikittyExtension ext : form.getExtensions()) {
                    if (ext.getFieldType(field) != null) {
                        extensions.put(ext.getName(), ext.getFieldType(field));
                    }
                }
                for (String token : tokens) {
                    Boolean tokenResult = null;
                    if (token.trim().startsWith("NOT ")) {
                        for (String extensionName : extensions.keySet()) {
                            switch (extensions.get(extensionName).getType()) {
                                case DATE:
                                    Date formDate = (Date) form
                                            .getField(extensionName, field);
                                    try {
                                        Date queryDate = DateFormat
                                                .getInstance()
                                                .parse(token.trim()
                                                        .substring(4));
                                        tokenResult = !formDate
                                                .equals(queryDate);
                                    } catch (ParseException eee) {
                                        if (log.isDebugEnabled()) {
                                            log.debug(token
                                                    + " cannot be a date.");
                                        }
                                    }
                                    break;
                                case BOOLEAN:
                                    Boolean formBoolean =
                                            (Boolean) form.getField(
                                                    extensionName, field);
                                    try {
                                        Boolean queryBoolean = Boolean
                                                .parseBoolean(token.trim()
                                                        .substring(4));
                                        tokenResult = !formBoolean
                                                .equals(queryBoolean);
                                    } catch (Exception eee) {
                                        if (log.isDebugEnabled()) {
                                            log.debug(token
                                                    + " cannot be a boolean.");
                                        }
                                    }
                                    break;
                                case NUMERIC:
                                    BigDecimal formNumber = (BigDecimal) form
                                            .getField(extensionName, field);
                                    try {
                                        BigDecimal queryNumber = BigDecimal
                                                .valueOf(Double
                                                        .valueOf(token.trim()
                                                        .substring(4)));
                                        tokenResult = !formNumber
                                                .equals(queryNumber);

                                    } catch (NumberFormatException eee) {
                                        if (log.isDebugEnabled()) {
                                            log.debug(token
                                                    + " cannot be a number.");
                                        }
                                    }
                                    break;
                                default:
                                    String formString = (String) form
                                            .getField(extensionName, field);
                                    String queryString =
                                            token.trim().substring(4);
                                    tokenResult =
                                            !formString.equals(queryString);
                            }
                        }
                    } else {
                        if (caught.containsKey(token.trim())) {
                            tokenResult = isQueryReturningForm(caught,
                                    token.trim(), field, form);
                        } else {
                            for (String extensionName : extensions.keySet()) {
                                FieldType type = extensions.get(extensionName);
                                switch (type.getType()) {
                                    case DATE:
                                        Date formDate = (Date) form
                                                .getField(extensionName, field);
                                        try {
                                            Date queryDate = DateFormat
                                                    .getInstance()
                                                    .parse(token.trim());
                                            tokenResult =
                                                    formDate.equals(queryDate);
                                        } catch (ParseException eee) {
                                            if (log.isDebugEnabled()) {
                                                log.debug(token
                                                        + " cannot be a date.");
                                            }
                                        }
                                        break;
                                    case BOOLEAN:
                                        Boolean formBoolean = (Boolean) form
                                                .getField(extensionName, field);
                                        try {
                                            Boolean queryBoolean = Boolean
                                                    .parseBoolean(token.trim());
                                            tokenResult = formBoolean
                                                    .equals(queryBoolean);
                                        } catch (Exception eee) {
                                            if (log.isDebugEnabled()) {
                                                log.debug(token + " cannot be" +
                                                        " a boolean.");
                                            }
                                        }
                                        break;
                                    case NUMERIC:
                                        BigDecimal formNumber =
                                                (BigDecimal) form.getField(
                                                        extensionName, field);
                                        try {
                                            BigDecimal queryNumber =
                                                    BigDecimal.valueOf(
                                                            Double.valueOf(
                                                                    token.trim()));
                                            tokenResult = formNumber
                                                    .equals(queryNumber);

                                        } catch (NumberFormatException eee) {
                                            if (log.isDebugEnabled()) {
                                                log.debug(token +
                                                        " cannot be a number.");
                                            }
                                        }
                                        break;
                                    default:
                                        String formString = (String) form
                                                .getField(extensionName, field);
                                        String queryString = token.trim();
                                        tokenResult = formString
                                                .equals(queryString);
                                }
                            }
                        }
                    }
                    if (result == null) {
                        result = tokenResult;
                    } else if (and) {
                        result = result && tokenResult;
                    } else {
                        result = result || tokenResult;
                    }
                }
            }
            // if no field is defined in the query
            else {
                String tokenValue = tokens[0];
                Boolean fieldResult = null;
                //check each field of each extension
                for (WikittyExtension ext : form.getExtensions()) {
                    String extensionName = ext.getName();
                    for (String fieldName : ext.getFieldNames()) {
                        switch (ext.getFieldType(fieldName).getType()) {
                            case DATE:
                                Date formDate = (Date) form.getField(
                                        extensionName, fieldName);
                                try {
                                    Date queryDate = DateFormat.getInstance()
                                            .parse(tokenValue.trim());
                                    fieldResult = formDate != null ?
                                            formDate.equals(queryDate) : false;
                                } catch (ParseException eee) {
                                    if (log.isDebugEnabled()) {
                                        log.debug(tokenValue
                                                + " cannot be a date.");
                                    }
                                }
                                break;
                            case BOOLEAN:
                                Boolean formBoolean = (Boolean) form
                                        .getField(extensionName, fieldName);
                                try {
                                    Boolean queryBoolean = Boolean
                                            .parseBoolean(tokenValue.trim());
                                    fieldResult = formBoolean != null
                                            ? formBoolean.equals(queryBoolean)
                                            : false;
                                } catch (Exception eee) {
                                    if (log.isDebugEnabled()) {
                                        log.debug(tokenValue
                                                + " cannot be a boolean.");
                                    }
                                }
                                break;
                            case NUMERIC:
                                BigDecimal formNumber = (BigDecimal) form
                                        .getField(extensionName, fieldName);
                                try {
                                    BigDecimal queryNumber = BigDecimal.valueOf(
                                            Double.valueOf(tokenValue.trim()));
                                    fieldResult = formNumber != null
                                            ? formNumber.equals(queryNumber)
                                            : false;

                                } catch (NumberFormatException eee) {
                                    if (log.isDebugEnabled()) {
                                        log.debug(tokenValue
                                                + " cannot be a number.");
                                    }
                                }
                                break;
                            default:
                                String formString = (String) form
                                        .getField(extensionName, fieldName);
                                String queryString = tokenValue.trim();
                                fieldResult = formString != null ? formString
                                        .contains(queryString) : false;
                        }
                        if (result == null) {
                            result = fieldResult;
                        } else if (fieldResult != null) {
                            result = result || fieldResult;
                        }
                    }
                }
            }
        }
        return result;
    }

    /**
     * @return all the XmlStreams
     */
    @Override
    public List<XmlStream> getAllXmlStreams() {
        if (log.isDebugEnabled()) {
            log.debug("getAllXmlStreams");
        }
        List<XmlStream> result = new ArrayList<XmlStream>();
        for (BusinessEntity entity : getAllEntities(XmlStream.class,
                XmlStream.EXT_XMLSTREAM)) {
            result.add((XmlStream) entity);
        }
        return result;
    }

    /**
     * Update a list of XmlFieldBinding
     *
     * @param bindings the list of XmlFieldBinding to update
     * @return the list of updated XmlFieldBinding
     */
    @Override
    public List<XmlFieldBinding> updateXmlFieldBindings(
            List<XmlFieldBinding> bindings) {
        if (log.isDebugEnabled()) {
            log.debug("updateXmlFieldBindings");
        }
        if (bindings != null) {
            List<XmlFieldBinding> result = new ArrayList<XmlFieldBinding>();
            for (XmlFieldBinding binding : bindings) {
                if (binding != null) {
                    result.add(binding);
                }
            }
            result = proxy.store(result);
            return new ArrayList<XmlFieldBinding>(result);
        }
        return null;
    }

    /**
     * @return the root thesaurus node
     * @throws TechnicalException
     */
    @Override
    public TreeNode getRootThesaurus() throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getRootThesaurus");
        }
        Criteria criteria = Search.query()
                .eq(Element.ELT_EXTENSION, TreeNode.EXT_TREENODE)
                .eq(TreeNode.FQ_FIELD_NAME, ROOT_THESAURUS_NAME).criteria();
        List<TreeNode> rootThesaurus = proxy
                .findAllByCriteria(TreeNode.class, criteria).getAll();

        log.debug("Thesaurus find " + rootThesaurus);
        if (rootThesaurus.isEmpty()) {
            TreeNode thesaurusImpl = new TreeNodeImpl();
            thesaurusImpl.setName(ROOT_THESAURUS_NAME);
            updateEntity(thesaurusImpl);

            return thesaurusImpl;
        }
        return rootThesaurus.get(0);
    }

    /**
     * @return all the thesaurus nodes
     * @throws TechnicalException
     */
    @Override
    public List<TreeNode> getAllThesaurus() throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getAllThesaurus");
        }
        List<TreeNode> result = new ArrayList<TreeNode>();
        for (BusinessEntity entity : getAllEntities(TreeNode.class,
                TreeNode.EXT_TREENODE)) {
            result.add((TreeNode) entity);
            log.debug(((TreeNode) entity).getName() + " " + ((TreeNode) entity)
                    .getChildren());
        }
        return result;
    }

    /**
     * Get the thesaurus node whose id is 'thesaurusId'
     *
     * @param thesaurusId the id of the thesaurus node we are retrieving
     * @return the thesaurus node whose id is 'thesaurusId'
     * @throws TechnicalException
     */
    @Override
    public TreeNode getThesaurus(String thesaurusId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getThesaurus");
        }
        return (TreeNode) getEntity(thesaurusId, TreeNode.class);
    }

    /**
     * @return all the thesaurus nodes
     * @throws TechnicalException TODO : To optimize
     */
    @Override
    public List<TreeNode> getChildrenThesaurus(String thesaurusId)
            throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getChildrenThesaurus");
        }
        List<TreeNode> result = new ArrayList<TreeNode>();
        for (BusinessEntity entity : getAllEntities(TreeNode.class,
                TreeNode.EXT_TREENODE)) {
            TreeNode node = (TreeNode) entity;
            if (node.getParent() != null && node.getParent()
                    .equals(thesaurusId)) {
                result.add(node);
            }
        }
        return result;
    }

    /**
     * Get the XmlStream whose id is 'xmlStreamId'
     *
     * @param xmlStreamId the id of the XmlStream we are retrieving
     * @return the XmlStream whose id is 'xmlStreamId'
     * @throws TechnicalException
     */
    @Override
    public XmlStream getXmlStream(String xmlStreamId)
            throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getXmlStream");
        }
        return (XmlStream) getEntity(xmlStreamId, XmlStream.class);
    }

    /**
     * Get the XmlFieldBinding whose id is 'xmlFieldBindingId'
     *
     * @param xmlFieldBindingId the id of the XmlFieldBinding we are retrieving
     * @return the XmlFieldBinding whose id is 'xmlFieldBindingId'
     */
    @Override
    public XmlFieldBinding getXmlFieldBinding(String xmlFieldBindingId) {
        if (log.isDebugEnabled()) {
            log.debug("getXmlFieldBinding");
        }
        XmlFieldBinding result = null;
        if (xmlFieldBindingId != null) {
            result = proxy.restore(XmlFieldBinding.class, xmlFieldBindingId);
        }
        return result;
    }

    /**
     * Get the XmlFieldBindings linked to xmlStream
     *
     * @param xmlStream the XmlStream linked
     *                  to the XmlFieldBinding we are looking for
     * @return the list of XmlFieldBindings linked to xmlStream
     */
    @Override
    public List<XmlFieldBinding> getXmlFieldBindings(XmlStream xmlStream) {
        if (log.isDebugEnabled()) {
            log.debug("getXmlFieldBindings");
            log.debug(xmlStream.getXmlFieldBinding());
        }
        if (xmlStream != null && xmlStream.getXmlFieldBinding() != null) {
            List<String> bindings = new ArrayList<String>();
            bindings.addAll(xmlStream.getXmlFieldBinding());
            return proxy.restore(XmlFieldBinding.class, bindings);
        }
        return null;
    }

    /**
     * Binds the clients with the forms found by their queries
     * and not already sent
     */
    @Override
    public void bindFormsToClients() {
        List<Sending> toSend = new ArrayList<Sending>();
        try {
            List<QueryMaker> queryMakers = new ArrayList<QueryMaker>();
            //get all the clients
            List<Client> clients = getAllClients();
            for (Client client : clients) {
                // look for all the already bound forms
                Criteria criteria = Search.query()
                        .eq(Element.ELT_EXTENSION, Sending.EXT_SENDING)
                        .eq(Sending.FQ_FIELD_CLIENT, client.getWikittyId())
                        .criteria();
                List<Sending> sendings = proxy
                        .findAllByCriteria(Sending.class, criteria).getAll();
                // List of all the already bound forms
                List<String> alreadyBoundFormIds = new ArrayList<String>();
                for (Sending sending : sendings) {
                    if (sending.getForm() != null) {
                        alreadyBoundFormIds.addAll(sending.getForm());
                    }
                }
                queryMakers.add(client);
                // get all the client users
                List<User> users = getClientUsers(client.getWikittyId());
                queryMakers.addAll(users);
                // List of the forms to bind ( = not yet bound)
                List<String> formsToBindIds = new ArrayList<String>();
                // iterate on the client and its users
                for (QueryMaker queryMaker : queryMakers) {
                    //if the query maker has queries
                    if (queryMaker.getQueries() != null) {
                        for (String query : queryMaker.getQueries()) {
                            //create a new VradiFormPageDTO to find
                            //  all the forms corresponding to the query
                            VradiFormPageDTO vradiFormPageDTO
                                    = new VradiFormPageDTO();
                            findForms(query, vradiFormPageDTO);
                            List<Form> forms = vradiFormPageDTO
                                    .getFormsToShow();
                            for (Form form : forms) {
                                //if the form has not yet been bound,
                                //  add it to the forms to bind
                                if (!alreadyBoundFormIds
                                        .contains(form.getWikittyId())) {
                                    formsToBindIds.add(form.getWikittyId());
                                }
                            }
                        }
                    }
                }
                // create a new Sending
                Sending newSending = new SendingImpl();
                // look for a sending with not yet sent forms
                for (Sending sending : sendings) {
                    if (sending.getStatus() == DEFAULT_SENDING_STATUS) {
                        newSending = sending;
                        break;
                    }
                }
                // if some forms are to be bound, bind them
                if (!formsToBindIds.isEmpty()) {
                    newSending.setClient(client.getWikittyId());
                    for (String formsToSendId : formsToBindIds) {
                        newSending.addForm(formsToSendId);
                        alreadyBoundFormIds.add(formsToSendId);
                    }
                    newSending.setSentDate(null);
                    newSending.setReceptionDate(null);
                    newSending.setParagraph(DEFAULT_SENDING_PARAGRAPH);
                    newSending.setReceptionProof(false);
                    newSending.setStatus(DEFAULT_SENDING_STATUS);
                    toSend.add(newSending);
                }
                queryMakers.clear();
            }
            // store the data
            proxy.store(toSend);

        } catch (TechnicalException eee) {
            if (log.isErrorEnabled()) {
                log.error(eee);
            }
        }
    }

    /**
     * Gets the forms by client, status and sent date with a filter.
     *
     * @param dateType       the date type of the filter
     * @param beginDate      the minimum value of the date of the filter
     * @param endDate        the maximum value of the date of the filter
     * @param extension      the WikittyExtension of the filter
     * @param receptionProof true if the sending must have a reception proof
     * @param paragraph      the paragraph sent with the forms
     * @param status         the status of the sending
     * @return a list of VradiSendingDTO representing
     *         the forms sent to the clients
     * @throws TechnicalException
     */
    @Override
    public List<VradiSendingDTO> getFormsByClients(String dateType,
                                                   Date beginDate, Date endDate,
                                                   WikittyExtension extension,
                                                   Boolean receptionProof,
                                                   Boolean paragraph,
                                                   int status)
            throws TechnicalException {

        List<VradiSendingDTO> result = new ArrayList<VradiSendingDTO>();
        List<Client> clients = getAllClients();
        for (Client client : clients) {
            Search search = Search.query()
                    .eq(Element.ELT_EXTENSION, Sending.EXT_SENDING)
                    .eq(Sending.FQ_FIELD_CLIENT, client.getWikittyId());
            if (receptionProof != null) {
                search.eq(Sending.FQ_FIELD_RECEPTIONPROOF,
                        receptionProof.toString());
            }
            if (paragraph != null) {
                if (paragraph) {
                    search.neq(Sending.FQ_FIELD_PARAGRAPH,
                            DEFAULT_SENDING_PARAGRAPH);
                } else {
                    search.eq(Sending.FQ_FIELD_PARAGRAPH,
                            DEFAULT_SENDING_PARAGRAPH);
                }
            }
            if (status >= 0) {
                search.eq(Sending.FQ_FIELD_STATUS, String.valueOf(status));
            }
            Criteria criteria = search.criteria();
            List<Sending> sendings = proxy
                    .findAllByCriteria(Sending.class, criteria).getAll();
            for (Sending sending : sendings) {
                List<Form> forms = getForms(sending.getForm() != null ?
                        new ArrayList(sending.getForm()) : new ArrayList());
                String dateExtension = null;
                String dateTypeName = null;
                if (dateType != null) {
                    int dot = dateType.indexOf('.');
                    if (dot > 0) {
                        dateExtension = dateType.substring(0, dot);
                        dateTypeName = dateType.substring(dot + 1);
                    }
                }
                List<Form> formsToSend = new ArrayList<Form>();
                for (Form form : forms) {
                    if ((extension == null || form.getExtensions()
                            .contains(extension))
                            && (dateType == null || beginDate == null
                            || !beginDate.after((Date) form
                            .getField(dateExtension, dateTypeName)))
                            && (dateType == null || endDate == null || !endDate
                            .before((Date) form
                                    .getField(dateExtension, dateTypeName)))) {
                        formsToSend.add(form);
                    }
                }
                if (formsToSend != null && !formsToSend.isEmpty()) {
                    VradiSendingDTO formsByClientDTO = new VradiSendingDTO(
                            sending);
                    formsByClientDTO.setFormDTOs(formsToSend);
                    result.add(formsByClientDTO);
                }
            }
        }
        return result;
    }

    /**
     * Gets the client associated with each form
     *
     * @param dateType  the date type of the filter
     * @param beginDate the minimum value of the date of the filter
     * @param endDate   the maximum value of the date of the filter
     * @param extension the WikittyExtension of the filter
     * @return a map containg the forms and their associated list of clients
     */
    @Override
    public Map<Form, List<Client>> getClientsByForms(String dateType,
                                                     Date beginDate,
                                                     Date endDate,
                                                     WikittyExtension extension) {
        Map<Form, List<Client>> result = new HashMap<Form, List<Client>>();
        List<Form> forms = getAllForms();
        for (Form form : forms) {
            String dateExtension = null;
            String dateTypeName = null;
            if (dateType != null) {
                int dot = dateType.indexOf('.');
                if (dot > 0) {
                    dateExtension = dateType.substring(0, dot);
                    dateTypeName = dateType.substring(dot + 1);
                }
            }
            if ((extension == null || form.getExtensions().contains(extension))
                    && (dateType == null || beginDate == null || !beginDate
                    .after((Date) form.getField(dateExtension, dateTypeName)))
                    && (dateType == null || endDate == null || !endDate
                    .before((Date) form
                            .getField(dateExtension, dateTypeName)))) {
                List<String> clientIds = new ArrayList<String>();
                Criteria criteria = Search.query()
                        .eq(Element.ELT_EXTENSION, Sending.EXT_SENDING)
                        .eq(Sending.FQ_FIELD_FORM, form.getWikittyId())
                        .criteria();
                List<Sending> sendings = proxy
                        .findAllByCriteria(Sending.class, criteria).getAll();
                for (Sending sending : sendings) {
                    clientIds.add(sending.getClient());
                }
                if (clientIds != null && !clientIds.isEmpty()) {
                    result.put(form, proxy.restore(Client.class, clientIds));
                }
            }
        }
        return result;
    }

    /**
     * Creates and store forms from an Xml Stream by using the XmlStreamBinding
     * to link xml stream field values
     * with form fields.
     *
     * @param xmlStream        the xml stream to read
     * @param lastItemRecorded the encrypted value of the last read item
     *                         of the xml stream
     * @param vradiUser        the VradiUser that launched the import
     * @return the encrypted value of the last read item of the xml stream
     */
    @Override
    public String getFormsFromXmlStream(XmlStream xmlStream,
                                        String lastItemRecorded,
                                        VradiUser vradiUser) {
        // map linking the form fields with the xmlstream fields
        Map<String, Set<String>> streamBinding
                = new HashMap<String, Set<String>>();
        WikittyExtension formType = null;
        List<Form> forms = new ArrayList<Form>();
        List<XmlFieldBinding> bindings = proxy.restore(XmlFieldBinding.class,
                new ArrayList<String>(xmlStream.getXmlFieldBinding()));
        //look for the WikittyExtension associated with the xmlstream
        for (XmlFieldBinding binding : bindings) {
            //get the first part of the fq field
            String formTypeName = binding.getFormField()
                    .substring(0, binding.getFormField().indexOf('.'));
            if (formType == null && !formTypeName.equals(Form.EXT_FORM)) {
                formType = getFormType(formTypeName);
            }
            streamBinding.put(binding.getFormField(), binding.getXmlField());
        }
        if (formType != null) {
            try {
                String result = null;
                SAXBuilder sxb = new SAXBuilder();
                URL rssUrl = new URL(xmlStream.getUrl());
                Document document = sxb.build(rssUrl);
                org.jdom.Element racine = document.getRootElement();
                List<org.jdom.Element> itemElt = null;
                if (racine.getChild(channel) != null) {
                    itemElt = racine.getChild(channel).getChildren(item);
                } else if (racine.getChild(item) != null) {
                    itemElt = racine.getChildren(item);
                } else if (racine.getChild(entry) != null) {
                    itemElt = racine.getChildren(entry);
                }
                if (itemElt != null) {
                    // for each element of the xml stream
                    for (org.jdom.Element element : itemElt) {
                        StringBuffer sb = new StringBuffer();
                        List<org.jdom.Element> fields = element.getChildren();
                        for (org.jdom.Element field : fields) {
                            sb.append(field.getText());
                        }
                        MD5 lastItem = new MD5(sb.toString());
                        //get the encrypted value of the newest element
                        if (result == null) {
                            result = MD5.asHex(lastItem.Final());
                        }
                        //check if the element has not yet been read
                        if (lastItemRecorded != null
                                && lastItemRecorded
                                .equals(MD5.asHex(lastItem.Final()))) {
                            break;
                        }
                        //create the form with the info from the xml stream
                        Form form = new FormImpl();
                        form.addExtension(formType);
                        form.addExtension(ModificationTag.MODIFICATION_TAG);
                        for (String field : streamBinding.keySet()) {
                            int dot = field.indexOf('.');
                            String extName = field.substring(0, dot);
                            String fieldName = field.substring(dot + 1);
                            FieldType.TYPE fieldType = null;
                            //get the field type
                            if (extName.equals(Form.EXT_FORM)) {
                                fieldType = FormImpl.extensionForm
                                        .getFieldType(fieldName).getType();
                            } else if (formType != null) {
                                fieldType = formType.getFieldType(fieldName)
                                        .getType();
                            }
                            // format the data
                            if (fieldType != null
                                    && streamBinding.get(field) != null) {
                                switch (fieldType) {
                                    case DATE:
                                        for (String xmlField : streamBinding
                                                .get(field)) {
                                            if (element.getChild(xmlField)
                                                    != null) {
                                                form.setField(extName,
                                                        fieldName,
                                                        WikittyUtil.solrDateFormat.format(
                                                                RSS_DATE_FORMAT.parse(
                                                                        element.getChild(
                                                                                xmlField).getText())));
                                            }
                                        }
                                        break;
                                    default:
                                        for (String xmlField : streamBinding
                                                .get(field)) {
                                            if (element.getChild(xmlField)
                                                    != null) {
                                                form.setField(extName,
                                                        fieldName,
                                                        (form.getField(extName,
                                                                fieldName)
                                                                != null ?
                                                                form.getField(
                                                                        extName,
                                                                        fieldName)
                                                                        + "\n"
                                                                : "")
                                                                + element
                                                                .getChild(
                                                                        xmlField)
                                                                .getText());
                                            }
                                        }
                                        break;
                                }
                            }
                        }
                        if (form.getId() == null) {
                            form.setId(new FormIdDateFormat().format(
                                    new GregorianCalendar().getTime()));
                        }
                        if (vradiUser != null) {
                            form.setField(ModificationTag.EXT_MODIFICATION_TAG,
                                    ModificationTag.FIELD_LAST_MODIFIER,
                                    vradiUser.getName());
                        }
                        forms.add(form);
                    }
                }
                updateForms(forms);
                return result;
            } catch (Exception eee) {
                if (log.isErrorEnabled()) {
                    log.error(eee);
                }
            }
        }
        return null;
    }

    /**
     * Updates a VradiUser
     *
     * @param vradiUser the VradiUser to update
     * @return the vradiUser updated
     * @throws TechnicalException
     */
    @Override
    public VradiUser updateVradiUser(VradiUser vradiUser)
            throws TechnicalException {
        Criteria criteria = Search.query()
                .eq(VradiUser.FQ_FIELD_NAME, vradiUser.getName()).criteria();
        List<VradiUser> result = proxy
                .findAllByCriteria(VradiUser.class, criteria).getAll();
        if (result.size() > 0) {
            log.debug("user " + vradiUser + " already exists");
            return null;
        } else {
            return proxy.store(vradiUser);
        }
    }

    /**
     * Logs a VradiUser with its username and password.
     *
     * @param vradiUserName     the username of the VradiUser
     * @param vradiUserPassword the password of the VradiUser
     * @return the VradiUser having vradiUserName as username
     *         and vradiUserPassword as password
     */
    @Override
    public VradiUser logVradiUser(String vradiUserName,
                                  String vradiUserPassword) {
        Criteria criteria = Search.query()
                .eq(VradiUser.FQ_FIELD_NAME, vradiUserName)
                .eq(VradiUser.FQ_FIELD_PASSWORD, vradiUserPassword).criteria();
        List<VradiUser> result = proxy
                .findAllByCriteria(VradiUser.class, criteria).getAll();
        if (result.size() > 0) {
            return result.get(0);
        }
        return null;
    }

    /**
     * Update a list of Sending
     *
     * @param sendings the list of Sending to store
     */
    @Override
    public void updateSendings(
            List<com.jurismarches.vradi.entities.Sending> sendings) {
        proxy.store(sendings);
    }

    /**
     * Import data from an XML file
     *
     * @param file the XML file containing the data
     * @throws TechnicalException
     */
    @Override
    public void importData(File file) throws TechnicalException {
        proxy.syncImportFromUri(file.toURI().toString());
    }

    /**
     * Export the data into an XML format
     *
     * @return the data formatted for an XML file
     * @throws TechnicalException
     */
    @Override
    public String exportData() throws TechnicalException {
        Search search = Search.query();
        search.keyword("*");
        Criteria criteria = search.criteria();
        String result = proxy.syncExportAllByCriteria(criteria);
        return result;
    }

    /**
     * reindex the data (solr)
     */
    @Override
    public void reindexData() {
        UpdateResponse response = proxy.getWikittyService().syncEngin();
        if (log.isDebugEnabled()) {
            log.debug(response.toString());
        }
    }

    /**
     * Changes the data directory
     *
     * @param newDataDir the new data directory path
     * @param oldDataDir the old data directory path.
     *                   If null, the data in the old directory will not be copied.
     */
    @Override
    public void changeDataDir(String newDataDir, String oldDataDir) {
        proxy.changeDataDir(newDataDir, oldDataDir);
    }

    /**
     * Gets the number of forms associated with every thesaurus node
     *
     * @return a map containig the thesaurus nodes and the number of forms
     *         associated with it
     * @throws TechnicalException
     */
    @Override
    public Map<TreeNode, Integer> getNbFormsByThesaurus()
            throws TechnicalException {
        Map<TreeNode, Integer> result = new HashMap<TreeNode, Integer>();
        for (TreeNode node : getAllThesaurus()) {
            Map.Entry<TreeNode, Integer> entry = proxy
                    .restoreNode(TreeNode.class, node.getWikittyId());
            result.put(node, entry == null ? 0 : entry.getValue());
        }
        return result;
    }

    /**
     * Gets the number of forms associated with the thesaurus node
     * whose id is 'thesaurusId'
     *
     * @return the number of forms associated with the thesaurus node
     *         whose id is 'thesaurusId'
     * @throws TechnicalException
     */
    @Override
    public int getNbFormsForThesaurus(String thesaurusId)
            throws TechnicalException {
        Map.Entry<TreeNode, Integer> entry = proxy
                .restoreNode(TreeNode.class, thesaurusId);
        return entry == null ? 0 : entry.getValue();
    }

}
